/*
                          Search and Rescue Core
 */

#ifndef SAR_H
#define SAR_H

#include <stdio.h>

#include <sys/types.h>
#ifndef __MSW__
# include <sys/time.h>
#endif
#include <limits.h>

#include "../include/os.h"
#include "gw.h"
#include "stategl.h"
#include "obj.h"
#include "human.h"
#include "weather.h"
#include "gctl.h"
#include "menu.h"
#include "sound.h"
#include "musiclistio.h"
#include "mission.h"
#include "texturelistio.h"
#include "sarfps.h"
#include "sarfio.h"


/*
 *	Menu system op codes (for button id codes):
 */
#define SAR_OP_MENU_TO_MAIN		0
#define SAR_OP_MENU_EXIT		1

#define SAR_OP_MENU_TO_MISSION		10
#define SAR_OP_MENU_TO_CAMPAIGN		11
#define SAR_OP_MENU_TO_FREE_FLIGHT      12
#define SAR_OP_MENU_TO_OPTIONS		13

#define SAR_OP_MENU_TO_MISSION_BRIEF		20
#define SAR_OP_MENU_TO_MISSION_BEGIN		21
#define SAR_OP_MENU_TO_MISSION_MAP		22
#define SAR_OP_MISSION_MAP_LEFT			23
#define SAR_OP_MISSION_MAP_RIGHT		24
#define SAR_OP_MISSION_MAP_UP			25
#define SAR_OP_MISSION_MAP_DOWN			26
#define SAR_OP_MISSION_MAP_ZOOM_IN		27
#define SAR_OP_MISSION_MAP_ZOOM_OUT		28
#define SAR_OP_MISSION_LOG_MAP_LEFT		29
#define SAR_OP_MISSION_LOG_MAP_RIGHT		30
#define SAR_OP_MISSION_LOG_MAP_UP		31
#define SAR_OP_MISSION_LOG_MAP_DOWN		32
#define SAR_OP_MISSION_LOG_MAP_ZOOM_IN		33
#define SAR_OP_MISSION_LOG_MAP_ZOOM_OUT		34
#define SAR_OP_MISSION_LOG_MAP_NEXT		35
#define SAR_OP_MISSION_LOG_MAP_PREVIOUS		36

#define SAR_OP_MENU_TO_FREE_FLIGHT_AIRCRAFT	40
#define SAR_OP_MENU_TO_FREE_FLIGHT_BEGIN	41
#define SAR_OP_MENU_TO_FREE_FLIGHT_WEATHER	42

#define SAR_OP_MENU_TO_OPTIONS_SIMULATION	50
#define SAR_OP_MENU_TO_OPTIONS_CONTROLLER	51
#define SAR_OP_MENU_TO_OPTIONS_CONTROLLER_JS_BTN	52
#define SAR_OP_MENU_TO_OPTIONS_GRAPHICS		53
#define SAR_OP_MENU_TO_OPTIONS_SOUNDS		54


/*
 *	Menu names:
 */
#define SAR_MENU_NAME_MAIN			"Main"

#define SAR_MENU_NAME_FREE_FLIGHT		"Free Flight"
#define SAR_MENU_NAME_FREE_FLIGHT_AIRCRAFT	"Free Flight Aircraft"
#define SAR_MENU_NAME_FREE_FLIGHT_WEATHER	"Free Flight Weather"

#define SAR_MENU_NAME_MISSION			"Mission"
#define SAR_MENU_NAME_MISSION_BRIEF		"Mission Brief"
#define SAR_MENU_NAME_MISSION_MAP		"Mission Map"
#define SAR_MENU_NAME_MISSION_LOG_MAP		"Mission Log Map"

#define SAR_MENU_NAME_CAMPAIGN			"Campaign"

#define SAR_MENU_NAME_OPTIONS			"Options"
#define SAR_MENU_NAME_OPTIONS_SIMULATION	"Options Simulation"
#define SAR_MENU_NAME_OPTIONS_CONTROLLER	"Options Controler"
#define SAR_MENU_NAME_OPTIONS_CONTROLLER_JS_BTN	"Options Controler JS Buttons"
#define SAR_MENU_NAME_OPTIONS_GRAPHICS		"Options Graphics"
#define SAR_MENU_NAME_OPTIONS_SOUND		"Options Sound"

#define SAR_MENU_NAME_LOADING_SIMULATION	"Loading Simulation"


/*
 *	Options menu object ID codes (should not conflict
 *	with any existing SAR_OP_MENU_* codes). These ID codes
 *	are for various switch and spin objects on the options
 *	menus.
 */
#define SAR_OPT_SELECT_HOIST_CONTACT		510
#define SAR_OPT_SELECT_DAMAGE_RESISTANCE	511
#define SAR_OPT_SELECT_FLIGHT_PHYSICS		512

#define SAR_OPT_SELECT_JS0_AXISES		520
#define SAR_OPT_SELECT_JS1_AXISES		521
#define SAR_OPT_SELECT_JS0_BUTTON_ACTION	522
#define SAR_OPT_SELECT_JS0_BUTTON_NUMBER	523
#define SAR_OPT_SELECT_JS1_BUTTON_ACTION	524
#define SAR_OPT_SELECT_JS1_BUTTON_NUMBER	525

#define SAR_OPT_SELECT_GROUND_TEXTURE		530
#define SAR_OPT_SELECT_OBJECT_TEXTURE		531
#define SAR_OPT_SELECT_CLOUDS			532
#define SAR_OPT_SELECT_ATMOSPHERE               533
#define SAR_OPT_SELECT_DUAL_PASS_DEPTH		534
#define SAR_OPT_SELECT_MAX_VISIBILITY		535

#define SAR_OPT_SELECT_SOUND_LEVEL		540
#define SAR_OPT_SELECT_MUSIC			541


/*
 *	Background music codes, these correspond to an index on the
 *	core structure's list of music file refs.
 */
#define SAR_MUSIC_CODE_NONE		0
#define SAR_MUSIC_CODE_SPLASH		1	/* Default. */
#define SAR_MUSIC_CODE_MENUS		2
#define SAR_MUSIC_CODE_SIMULATION_START	3
#define SAR_MUSIC_CODE_MISSION_FAILED	4
#define SAR_MUSIC_CODE_MISSION_SUCCESS	5

#define SAR_MUSIC_CODE_SIMULATION_ONGROUND		6
#define SAR_MUSIC_CODE_SIMULATION_ONGROUND_ENTER	7
#define SAR_MUSIC_CODE_SIMULATION_INFLIGHT		8
#define SAR_MUSIC_CODE_SIMULATION_INFLIGHT_ENTER	9
#define SAR_MUSIC_CODE_SIMULATION_RESCUE		10
#define SAR_MUSIC_CODE_SIMULATION_RESCUE_ENTER		11

/* Total number of music file refs. */
#define SAR_MUSIC_MAX			12



/*
 *	Options:
 */
typedef struct {

	Boolean menu_backgrounds;

	Boolean	console_quiet,
		internal_debug,
		runtime_debug,
		prioritize_memory;

	/* Graphics options. */
	Boolean	textured_ground,
		textured_objects,
		textured_clouds,
		atmosphere,
		dual_pass_depth,
		prop_wash,
		smoke_trails;

	double gl_polygon_offset_factor;	/* For glPolygonOffset(). */

	/* GL shade model, can be GL_FLAT or GL_SMOOTH. */
	int gl_shade_model;

	/* Maximum drawing visibility, value from 0 to 6.
	 * Computed by; miles = 3 + (option.visibility_max * 3).
	 */
	int visibility_max;

	/* Sound options. */
	Boolean engine_sounds,
		event_sounds,
		voice_sounds,
		music;

	/* How far can an aircraft engine sound be heard, in meters. */
	double aircraft_engine_sound_distance;

	/* Cocky crash/failure messages. */
	Boolean cocky_crash_messages;

	/* HUD color and font. */
	sar_color_struct hud_color;
	GWFont *hud_font;

	/* Standard messages color and font. */
	sar_color_struct message_color;
	GWFont *message_font;
	GWFont *menu_font;	/* Menu standard font (except menu values). */

	/* Sticky banner text (first line) big font. */
	GWFont *banner_font;


	/* Intervals in milliseconds. */
	time_t	explosion_frame_int,	/* Explosion frame inc interval. */
		splash_frame_int;	/* Splash frame inc interval. */

	time_t	crash_explosion_life_span,	/* Explosions by crashes. */
		fuel_tank_life_span;	/* Dropped tanks landed on ground. */

	double rotor_wash_vis_coeff;	/* 0.0 to 1.0. */

	/* Game controller type, one of GCTL_CONTROLLER_*. */
	int gctl_controller_type;

	/* Joystick button mappings. */
	int	js0_btn_rotate,		/* Treat bank to heading axis. */
		js0_btn_air_brakes,
		js0_btn_wheel_brakes,
		js0_btn_zoom_in,
		js0_btn_zoom_out,
		js0_btn_hoist_up,
		js0_btn_hoist_down;

	int	js1_btn_rotate,         /* Treat bank as heading axis. */
                js1_btn_air_brakes,
                js1_btn_wheel_brakes,
                js1_btn_zoom_in,
                js1_btn_zoom_out,
                js1_btn_hoist_up,
                js1_btn_hoist_down;

	/* Game controller options. */
	gctl_flags_t gctl_options;

	/* Simulation difficulty. */
	double hoist_contact_expansion_coeff;	/* Hoist contact radius. */
	double damage_resistance_coeff;
#define FLIGHT_PHYSICS_EASY	0
#define FLIGHT_PHYSICS_MODERATE	1
#define FLIGHT_PHYSICS_REALISTIC	2
	int flight_physics;	/* One of FLIGHT_PHYSICS_*. */


	/* Last run selections. */
	int	last_selected_mission,	/* Matches menu list obj sel pos. */
		last_selected_ffscene,
		last_selected_ffaircraft,
		last_selected_ffweather,
		last_x, last_y, last_width, last_height;

} sar_option_struct;
extern sar_option_struct option;

/*
 *	Directory names:
 */
typedef struct {

	char local_data[PATH_MAX];
	char global_data[PATH_MAX];

} sar_dname_struct;
extern sar_dname_struct dname;

/*
 *	File names:
 */
typedef struct {

	/* Run time preferances. */
	char preferances[PATH_MAX + NAME_MAX];

	/* Human preset definations. */
	char human[PATH_MAX + NAME_MAX];

        /* Music file referances list. */
        char music[PATH_MAX + NAME_MAX];

        /* Sound files referances list. */
        char sounds[PATH_MAX + NAME_MAX];

	/* Basic textures list. */
	char textures[PATH_MAX + NAME_MAX];

	/* Weather preset definations. */
	char weather[PATH_MAX + NAME_MAX];

} sar_fname_struct;
extern sar_fname_struct fname;


/*
 *	Timmers:
 */
typedef struct {

	int foo;		/* Dummy place holder. */

} sar_next_struct;
extern sar_next_struct next;


/*
 *	SARDrawMap() name matching structure:
 *
 *	Map draw OpenGL names to string names record structure, used in
 *	sardraw.c's map drawing function. This is used for selection
 *	matching.
 *
 *	Note that the structure's index is not the matched object's index.
 *	The matched object's index will be dmon_ptr->gl_name if and only
 *	if it is 0 <= dmon_ptr->gl_name < total_objects.
 */
typedef struct {

	/* The object's index number or some special code. A special
	 * code is when: gl_name >= total_objects AND when total_objects
	 * is subtracted from gl_name: special_code = gl_name - 
	 * total_objects.
	 */
	GLuint gl_name;

	/* Coppied pointer to the object's name. */
	char *obj_name;

} sar_drawmap_objname_struct;


/*
 *	Core structure:
 */
typedef struct {

	/* Program file (name only) and full path. */
	char *prog_file;
	char *prog_file_full_path;

	/* Video, sound, and game controller. */
	gw_display_struct *display;		/* Graphics wrapper. */
	char *recorder_address;			/* Sound server address. */
	snd_recorder_struct *recorder;		/* Sound server. */
        gctl_struct *gctl;			/* Game controller. */

	/* Current audio mode name the sound server is in. */
	char *audio_mode_name;

        /* Current music sound object being played (NULL for none). */
        void *cur_music_sndobj;

        /* Our code to indicate what background music is being played
         * (one of SAR_MUSIC_CODE_*).
         */
        int cur_music_code;

	/* Music file referances list. */
	sar_music_ref_struct **music_ref;
	int total_music_refs;


	/* Human presets data. */
	sar_human_data_struct *human_data;

	/* Weather presets data. */
	sar_weather_data_struct *weather_data;

	/* SAR scenery, objects, and mission. */
	sar_scene_struct *scene;		/* SAR scene. */
	sar_mission_struct *mission;		/* SAR mission. */
	sar_object_struct **object;		/* SAR objects. */
	int total_objects;


	/* Menu and shared data for menus. */
	sar_menu_struct **menu;
	int total_menus, cur_menu;
	/* Menu images (shared amoung menu object/widgets. */
	sar_image_struct	**menu_list_bg_img,	/* 9 images. */
				*menu_button_armed_img,
				*menu_button_unarmed_img,
				*menu_button_highlighted_img,
				*menu_button_label_img,
				*menu_label_bg_img,
				*menu_switch_bg_img,
				*menu_switch_off_img,
				*menu_switch_on_img,
				*menu_spin_label_img,
				*menu_spin_value_img,
				*menu_progress_bg_img,
				*menu_progress_fg_img,

				*menu_button_pan_up_armed_img,
				*menu_button_pan_up_unarmed_img,
				*menu_button_pan_down_armed_img,
				*menu_button_pan_down_unarmed_img,
                                *menu_button_pan_left_armed_img,
                                *menu_button_pan_left_unarmed_img,
                                *menu_button_pan_right_armed_img,
				*menu_button_pan_right_unarmed_img,

				*menu_button_zoom_in_armed_img,
				*menu_button_zoom_in_unarmed_img,
				*menu_button_zoom_out_armed_img,
				*menu_button_zoom_out_unarmed_img,

				*menumap_helipad_img,
				*menumap_intercept_img,	/* Interecpt point. */
				*menumap_helicopter_img,
				*menumap_victim_img,
				*menumap_boat_img;

	/* Global list of standard texture referance names and file names
	 * that are to be loaded *when* a scene is loaded so that it
	 * has a standard set of textures to use.
	 */
	sar_texture_name_struct **texture_list;
	int total_texture_list;

	/* Selected files. */
	char *cur_mission_file;
	char *cur_player_model_file;
	char *cur_scene_file;

	/* Display help during simulation, value of 0 means display
	 * nothing while any positive value corresponds to the `page'
	 * being displayed starting from page 1 being a value of 1.
	 */
	int display_help;

	/* Draw map object name referance structures, used in sardraw.c's
	 * map drawing function.
	 */
	sar_drawmap_objname_struct **drawmap_objname;
	int total_drawmap_objnames;
	/* Draw map ground hit contact result, a value from 0.0 to 1.0
	 * which corresponds to the alpha value from reading back
	 * the pixels.
	 */
	double drawmap_ghc_result;

	/* Text prompt input callback function, when this pointer
	 * is not NULL, text is gathered into the text input buffer.
	 * When enter is pressed, the callback function is called
	 * and then set back to NULL.
	 */
	char *text_input_prompt, *text_input_buffer;
	void (*text_input_cb)(void *, const char *);

	/* Frames per second counter and tallying structure. Note
	 * that we have one fps structure per graphics wrapper structure
	 * (which we only have one) so we only have one fps structure.
	 */
	sar_fps_struct fps;

} sar_core_struct;


/*
 *	Client data for menu list widget/object items.
 */
typedef struct {

	sar_core_struct *core_ptr;

	/* For aircraft, mission, and scene listings. */
	char *filename;

	/* For scene start locations. */
	char *name;
	sar_position_struct pos;
	sar_direction_struct dir;

} sar_menu_list_item_data_struct;


/*
 *	Callback data for load progress menu updates:
 */
typedef struct {

	sar_core_struct *core_ptr;

	double coeff_offset;	/* From 0.0 to 1.0. */
	double coeff_range;	/* From 0.0 to 1.0. */
	Boolean can_abort;	/* True if operation can be aborted. */

} sar_progress_cb_struct;





extern double	debug_value;		/* Arbitary global debug value. */

extern int	runlevel;		/* Program run level from 0 to 4. */

extern time_t	cur_millitime,		/* Current time in ms. */
		cur_systime,		/* Current time in systime seconds. */
		lapsed_millitime;	/* Delta ms between main loop. */

extern double	time_compensation,	/* Coeff of delta ms of loop vs 1 second. */
		time_compression;	/* Slow motion or acceleration, 0.25 to 8.0. */



/* In main.c */
extern void SARHandleSignal(int s);
extern void SARUpdateBackgroundMusic(sar_core_struct *core_ptr);
extern int SARInitGCTL(sar_core_struct *core_ptr);
/* DELETE THIS LATER. */
extern void SARTextInputCBSendMessage(void *client_data, const char *buffer);
extern void SARTextInputCBQuitSimulation(void *client_data, const char *buffer);
extern void SARDrawCB(void *ptr);
extern void SARReshapeCB(void *ptr, int x, int y, int width, int height);
extern void SARKeyBoardCB(void *ptr, int c, Boolean state, long t);
extern void SARPointerCB(void *ptr, int x, int y, int state, int btn_num, long t);
extern void SARVisibilityCB(void *ptr, int v);
extern void SARSaveYourselfCB(void *ptr);
extern void SARCloseCB(void *ptr, void *data);
extern void SARResetTimmersCB(sar_core_struct *core_ptr, time_t t_new);

extern sar_core_struct *SARInit(int argc, char **argv);
extern void SARManage(void *ptr);
extern void SARShutdown(sar_core_struct *core_ptr);


/* In mission.c (mission callbacks, utils, and procedures) */
extern void SARMissionPrintStats(
	sar_core_struct *core_ptr,
	sar_scene_struct *scene,
	sar_mission_struct *mission,
	sar_object_struct *obj_ptr
);
extern void SARMissionDestroyNotify(
	sar_core_struct *core_ptr,
	sar_object_struct *obj_ptr
);
void SARMissionPassengersLeaveNotify(
	sar_core_struct *core_ptr, sar_object_struct *obj_ptr,
	int passengers_left
);
extern void SARMissionLandNotify(
	sar_core_struct *core_ptr, sar_object_struct *obj_ptr,
	int *list, int list_total
);
extern int SARMissionManage(sar_core_struct *core_ptr);

/* In missionio.c (mission file io) */
extern void SARMissionDestroy(sar_mission_struct *mission);
extern char *SARMissionLoadDescription(const char *filename);
extern void SARMissionLoadMenuMap(
        sar_core_struct *core_ptr,
        sar_menu_struct *m,
        const char *filename          /* Mission file name. */
);
extern sar_mission_struct *SARMissionLoadFromFile(
	sar_core_struct *core_ptr,
        const char *filename,
        void *client_data,
        int (*progress_func)(void *, long, long)
);

/* In objio.c (object model file io) */
extern char *SARObjFSeekGetOpString(FILE *fp);
extern int SARObjLoadTranslate(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
	sar_parm_translate_struct *p_translate
);
extern int SARObjLoadTranslateRandom(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
	sar_parm_translate_random_struct *p_translate_random
);
extern int SARObjLoadSound(
        snd_recorder_struct *recorder,
        snd_play_struct **snd_play_rtn,
        const char *path, int options
);
extern int SARObjLoadTexture(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
	sar_parm_texture_load_struct *p_texture_load
);
extern int SARObjLoadHelipad(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_helipad_struct *p_new_helipad
);
extern int SARObjLoadHuman(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_human_struct *p_new_human
);
extern int SARObjLoadFire(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_fire_struct *p_new_fire
);
extern int SARObjLoadSmoke(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_smoke_struct *p_new_smoke
);
extern int SARObjLoadLineHeightField(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        const char *line,
	sar_object_struct *obj_ptr
);
extern int SARObjLoadFromFile(
        sar_core_struct *core_struct,
        int n,                  /* Object number. */
        const char *filename
);

/* In objiocb.c (object model file io callbacks) */
extern int SARSceneLoadProgressCB(
	void *ptr,
	long file_pos, long file_size
);

/* In objiopremodeled.c (sar premodeled object io/creation) */
extern int SARObjCreatePremodeled(
	sar_core_struct *core_ptr, sar_scene_struct *scene,
	const char *type_name, int argc, char **argv
);

/* In optionio.c (options/preferances file io) */
extern int SAROptionsLoadFromFile(
	sar_option_struct *opt, const char *filename
);
extern int SAROptionsSaveToFile(
	sar_option_struct *opt, const char *filename
);

/* In sardraw.c (primarly simulation drawing) */
extern void SARDraw(sar_core_struct *core_ptr);
extern void SARDrawMap(
	sar_core_struct *core_ptr,
	Boolean draw_for_gcc, Boolean draw_for_ghc, int gcc_obj_num
);

/* In sardrawpremodeled.c (drawing of sar premodeled objects) */
extern void SARDrawPremodeled(
	gw_display_struct *display,
	sar_scene_struct *scene,
	sar_object_struct *obj_ptr,
	sar_object_premodeled_struct *obj_premodeled_ptr,
	sar_position_struct *camera_pos,
	sar_direction_struct *camera_dir,
	Boolean draw_for_gcc
);

/* In sarinstall.c (checking and installation of data files) */
extern Boolean SARIsInstalledLocal(
        sar_dname_struct *dn,
        sar_fname_struct *fn
);
extern Boolean SARIsInstalledGlobal(
        sar_dname_struct *dn,
        sar_fname_struct *fn
);
extern int SARDoInstallLocal(
        sar_dname_struct *dn,
        sar_fname_struct *fn,
        sar_option_struct *opt
);

/* In sarkey.c (sar simulation keyboard key event handling) */
extern void SARKey(
	sar_core_struct *core_ptr,
	int c,		/* GW Key value or character. */
	Boolean state,	/* Pressed (True) or released (False). */
	long t		/* Time stamp. */
);

/* In sarmenubuild.c (creates all menus) */
extern int SARBuildMenus(sar_core_struct *core_ptr);

/* In sarmenucb.c (menu object/widget callbacks) */
extern void SARMenuButtonCBSelectMenu(
        sar_core_struct *core_ptr,
        const char *name
);
extern sar_menu_list_struct *SARMenuGetList(
	sar_menu_struct *menu_ptr, int skip, int *list_num
);
extern void SARMenuButtonCB(
	void *object, void *client_data,
	int id_code
);
extern void SARMenuListSelectCB(
        void *object, void *client_data,
        int item_num, void *item_ptr
);
extern void SARMenuSpinSelectCB(
        void *object, void *client_data,
        int id_code, char *value
);

/* In saroptions.c (sar options/preferances procedures and callbacks) */
extern void SARApplyOptionsFromMenus(sar_core_struct *core_ptr);
extern void SARFetchOptionsToMenus(sar_core_struct *core_ptr);

extern void SARMenuOptionsSwitchCB(
        void *object, void *client_data,
        int id_code, Boolean state
);
extern void SARMenuOptionsSpinCB(
        void *object, void *client_data,
        int id_code, char *value
);

/* In sarsimend.c (sar simulation end tabulation and procedures) */
extern void SARSimEnd(sar_core_struct *core_ptr);

/* In sarutils.c (sar specific utility functions) */
extern time_t SARGetCurMilliTime(void);
extern int SARRandom(void);

extern Boolean SARGetGLVersion(int *major, int *minor, int *release);
extern char *SARGetGLVendorName(void);
extern char *SARGetGLRendererName(void);
extern char **SARGelGLExtensionNames(int *strc);

extern int SARIsMenuAllocated(sar_core_struct *core_ptr, int n);
extern int SARMatchMenuByName(
        sar_core_struct *core_ptr,
        const char *name
);
extern sar_menu_struct *SARGetCurrentMenuPtr(sar_core_struct *core_ptr);
extern void SARDeleteListItemData(
        sar_menu_list_item_struct *item_ptr
);

extern char *SARTimeOfDayString(sar_core_struct *core_ptr, double t);
extern char *SARDeltaTimeString(sar_core_struct *core_ptr, time_t t);

extern int SARParseTimeOfDay( 
        const char *string,
        int *h, int *m, int *s
);
extern int SARParseLatitudeDMS(
        const char *string,
        double *dms
);
extern int SARParseLongitudeDMS(
        const char *string,
        double *dms
);

extern void SARDrawMapProcessHits(
	sar_core_struct *core_ptr, GLint hits, GLuint buffer[]
);
extern int SARDrawMapObjNameListAppend(
        sar_drawmap_objname_struct ***ptr, int *total,
        GLuint gl_name, const char *obj_name
);
extern void SARDrawMapObjNameListDelete(
	sar_drawmap_objname_struct ***ptr, int *total
);
extern int *SARGetGCCHitList(   
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
        int obj_num,
        int *hits
);
extern int SARGetGHCOverWater(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
        int obj_num, Boolean *got_hit, Boolean *over_water
);
extern void SARReportGLError(
        sar_core_struct *core_ptr, GLenum error_code
);

/* In sceneio.c (scenery file io) */
extern void SARSceneDestroy(    
        sar_core_struct *core_ptr,
        sar_scene_struct *scene,  
        sar_object_struct ***ptr,
        int *total
);
extern void SARSceneLoadLocationsToList(
        sar_core_struct *core_ptr,
        sar_menu_struct *menu_ptr,
        sar_menu_list_struct *list_ptr,
        int list_num,
        const char *filename
);
extern int SARSceneAddPlayerObject(
        sar_core_struct *core_ptr,
        sar_scene_struct *scene,
        const char *model_file,
        sar_position_struct *pos,
        sar_direction_struct *dir
);
extern int SARSceneLoadFromFile(
        sar_core_struct *core_struct,   
        sar_scene_struct *scene,
        const char *filename,
        const char *weather_preset_name,
        void *client_data,
        int (*progress_func)(void *, long, long)
);

/* In scenesound.c (scene sound reloading/updating) */
extern void SARSceneSoundUpdate(
        sar_core_struct *core_ptr,
        Boolean engine_sounds,
        Boolean event_sounds,
        Boolean voice_sounds,
        Boolean music
);



#endif	/* SAR_H */
