/*
 * msgget.c
 *
 * Author: Howard Holm (NSA), <hdholm@epoch.ncsc.mil>
 * 
 * Create a message queue (possibly at a different context.)  If a queue already
 * exists with the (optional) key then return it's identifier.
 *
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <linux/flask/flask.h>
#include <sys/msg.h>
#include <ipc_secure.h>

/* print the usage information for this program */
void usage(char *progname) 
{
  fprintf(stderr, "usage: %s [-k key] [-c context] [-s sid]\n", progname);
  exit(1);
}


int main(int argc, char **argv)
{
  int have_sid = 0;
  int valid_ipc_key = 0;
  key_t ipc_key;
  int ret;
  security_id_t sid = SECSID_NULL;
  char c;
  int id;

  while ( (c = getopt(argc, argv, "k:c:s:")) != EOF) {
    switch (c) {
    case 'k':
      ipc_key = atoi(optarg);
      valid_ipc_key = 1;
      break;
    case 'c':
      if (have_sid) {
	fprintf (stderr, "May specify only a sid (-s) or context (-c) not both\n");
	usage (argv[0]);
      } else {
	have_sid = 1;
	ret = security_context_to_sid(optarg,strlen(optarg)+1,&sid);
	if (ret) {
	  perror("Invalid context (-c) given");
	  exit(ret);
	}
      }
      break;
    case 's':
      if (have_sid) {
	fprintf (stderr, "May specify only a sid (-s) or context (-c) not both\n");
	usage (argv[0]);
      } else {
	have_sid = 1;
	sid = atoi(optarg);
      }
      break;
    default:
      usage(argv[0]);
    }
  }

  if (sid != SECSID_NULL) {
    if (!valid_ipc_key)
      ipc_key = IPC_PRIVATE;
    if ((id = msgget_secure (ipc_key, IPC_CREAT|0777, sid)) == -1){
      perror ("msgget_secure failed");
      exit(1);
    } else
      fprintf (stderr, "%s:msgget_secure succeeded key = %d, id = %d\n", argv[0], ipc_key, id);
  } else {
    if (!valid_ipc_key)
      ipc_key = IPC_PRIVATE;
    if ((id = msgget (ipc_key, IPC_CREAT|0777)) == -1){
      perror ("msgget failed");
      exit (1);
    } else
      fprintf (stderr, "%s:msgget succeeded key = %d, id = %d\n", argv[0], ipc_key, id);
  }
}
