/*
 * Security Context Mapping Protocol Daemon
 *
 * Copyright (c) 2001-2002 James Morris <jmorris@intercode.com.au>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <getopt.h>
#include <libgen.h>
#include <errno.h>
#include <pwd.h>
#include <syslog.h>
#include <sys/poll.h>
#include <netinet/in.h>
#include <linux/flask/flask_types.h>
#include <linux/flask/flnetlink.h>

#include "list.h"
#include "server.h"
#include "flnetlink.h"
#include "transport.h"
#include "perimtab.h"

#define SHOW_USAGE	1
#define POLL_TIMEOUT	(1000 * 4)

unsigned char verbose	 = 0;
unsigned char debugging	 = 0;
unsigned char *appname	 = NULL;
u_int32_t send_sequence  = 0;

/* Poll users */
enum {
	P_FLNETLINK = 0,
	P_TRANSPORT,
	P_MAX
};

static void (*polltab[P_MAX])(void) = {
	[P_FLNETLINK]	flnetlink_pollin,
	[P_TRANSPORT]	transport_pollin,
};

static struct option longopts[] = {
	{ "debug",     0, 0, 'd' },
	{ "help",      0, 0, 'h' },
	{ "verbose",   0, 0, 'v' },
	{ "version",   0, 0, 'V' },
	{ 0 }
};

static void usage(void)
{
	fprintf(stderr,
"\nUsage: %s [options]\n\n"
"\n"
"Options:\n"
"  --debug,    -d              Log debugging information.\n"
"  --help,     -h              Display this help.\n"
"  --verbose,  -v              Act verbosely.\n"
"  --version,  -V              Display version and exit.\n"
"\n"
		, appname);
}

static void cleanup(void)
{
	transport_exit();
	perimtab_exit();
	flnetlink_exit();

	if (appname) {
		free(appname);
		appname = NULL;
	}
	
	closelog();
}

static void parse_commandline(int argc, char **argv)
{
	int i;

	while ((i = getopt_long(argc, argv, "dhvV", longopts, NULL)) != -1) {
	
		switch (i) {
		
			case 'h':
				usage();
				exit(0);
			
			case 'd':
				debugging = 1;
				break;
				
			case 'v':
				verbose = 1;
				break;
			
			case 'V':
				printf("%s version %s\n", appname, SCMPD_VERSION);
				exit(0);
				
			case ':':
			case '?':
			default:
				die(SHOW_USAGE, "Invalid option: '%c'", i);
		}
	}
}

static void event_loop(void)
{
	int done = 0;
	struct pollfd pfds[2];
	
	pfds[P_FLNETLINK].fd = flnetlink_fd;
	pfds[P_FLNETLINK].events = POLLIN;
	
	pfds[P_TRANSPORT].fd = transport_fd;
	pfds[P_TRANSPORT].events = POLLIN;
	
	while (!done) {
		int rc, i;
		
		rc = poll(pfds, P_MAX, POLL_TIMEOUT);
		
		if (rc == 0)
			continue;
			
		if (rc < 0) {
			if (errno == EINTR)
				continue;
				
			die(0, "poll: %m");
		}
		
		for (i = 0; i < P_MAX; i++) {
			short revents = pfds[i].revents;
			
			if (revents & (POLLERR|POLLHUP|POLLNVAL))
				die(0, "poll: %d revents=%#x\n", i, revents);
				
			if (revents & POLLIN)
				polltab[i]();
		}
	}
	
	return;
}

static char *euname(void)
{
	struct passwd *p;
	
	p = getpwuid(geteuid());
	if (p == NULL)
		die(0, "getpwuid: %m");
		
	return p->pw_name;
}

void die(int show_usage, char *fmt, ...)
{
	va_list args;
	
	va_start(args, fmt);
	vsyslog(LOG_ERR, fmt, args);
	va_end(args);
	
	if (show_usage)
		usage();
		
	exit(1);
}

/*
 * Current serial policy number.
 */
u_int32_t current_serial(void)
{
	return 0x12345678;
}

int main(int argc, char **argv)
{
	int rc;
	
	appname = strdup(basename(argv[0]));
	if (appname == NULL) {
		perror("strdup");
		exit(1);
	}
	
	openlog(appname, LOG_CONS|LOG_NDELAY|LOG_PID, LOG_AUTHPRIV);
	
	if (atexit(cleanup))
		die(0, "atexit failed");

	parse_commandline(argc, argv);
	
	rc = flnetlink_init();
	if (rc)
		die(0, "flnetlink_init failed");

	rc = perimtab_init();
	if (rc)
		die(0, "perimtab_init failed");
	
	rc = transport_init();
	if (rc)
		die(0, "transport_init failed");
	
	syslog(LOG_NOTICE, "started by %s", euname());
	
	event_loop();
	
	return 0;
}
