/*
 * ct - NSID mapping cache tool.
 *
 * Copyright (c) 2001-2002 James Morris <jmorris@intercode.com.au>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <getopt.h>
#include <libgen.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
          
#include "libflnetlink.h"
#include "libflutil.h"

#define BUFSIZE		8192		/* Needs to be page-sized for netlink dumps. */
#define SHOW_USAGE	1

enum command_t {
	CMD_NONE = 0,
	CMD_DUMP,
	CMD_FLUSH,
	CMD_MAX
};

static enum command_t command = CMD_NONE;
static unsigned char verbose = 0;
static unsigned char *appname = NULL;
static struct fln_handle *h = NULL;

static unsigned char *cmdname[] = {
	[CMD_DUMP]	"dump",
	[CMD_FLUSH]	"flush",
};

static struct option longopts[] = {
	{ "help",     0, 0, 'h' },
	{ "verbose",  0, 0, 'v'},
	{ 0 }
};

static void cleanup(void)
{
	if (h) {
		fln_destroy_handle(h);
		h = NULL;
	}

	if (appname) {
		free(appname);
		appname = NULL;
	}
}

static void warn(char *msg, ...)
{
	va_list args;
	
	va_start(args, msg);
	vfprintf(stderr, msg, args);
	va_end(args);
	
	fputc('\n', stderr);
}

static void usage(void)
{
	warn(
"\nUsage: %s [options] command [params]\n\n"
"Commands:\n"
"  dump                                Dump NSID mapping cache.\n"
"  flush                               Flush all NSID mapping cache entries.\n"
"\n"
"Options:\n"
"  --help, -h                          Display help (this).\n"
"  --verbose, -v                       Act verbosely.\n"
"\n"
		, appname);

}

static void die(int show_usage, char *msg, ...)
{
	va_list args;
	
	fputs("Fatal: ", stderr);
	
	va_start(args, msg);
	vfprintf(stderr, msg, args);
	va_end(args);
	
	if (h)
		fprintf(stderr, ": %s", fln_errstr());

	if (errno)
		fprintf(stderr, ": %s", strerror(errno));
		
	fputc('\n', stderr);
	
	if (show_usage)
		usage();
		
	exit(1);
}

static enum command_t parse_command(unsigned char *cmdstr)
{
	enum command_t i;
	
	for (i = CMD_NONE + 1; i < CMD_MAX; i++) {
		if (strcmp(cmdname[i], cmdstr) == 0)
			return i;
	}
	return CMD_NONE;
}

static void parse_commandline(int argc, char **argv)
{
	int i, diff;
	
	appname = strdup(basename(argv[0]));
	if (appname == NULL)
		die(0, "strdup");

	while ((i = getopt_long(argc, argv, "hv", longopts, NULL)) != -1) {
	
		switch (i) {
		
			case 'h':
				usage();
				exit(0);
			
			case 'v':
				verbose = 1;
				break;
			
			case ':':
			case '?':
			default:
				die(SHOW_USAGE, "Invalid option: '%c'", i);
		}
	}
	
	diff = argc - optind;
	if (diff < 1)
		die(SHOW_USAGE, "Command required");

	command = parse_command(argv[optind]);
	if (command == CMD_NONE)
		die(SHOW_USAGE, "Invalid command");
	
}

int main(int argc, char **argv)
{
	int status, done;
	unsigned char buf[BUFSIZE];

	if (atexit(cleanup)) {
		warn("Error: Unable to register exit handler.");
		exit(1);
	}

	parse_commandline(argc, argv);
	
	h = fln_create_handle(FLN_F_ACKS, FLN_G_NONE);
	if (!h)
		die(0, "Unable to create handle");

	switch (command) {
		
	case CMD_DUMP:
		status = fln_cache_dump(h);
		if (status < 0)
			die(0, cmdname[command]);
		break;
		
	case CMD_FLUSH:
		status = fln_cache_flush(h);
		if (status < 0)
			die(0, cmdname[command]);
		break;
	
	default:
		die(0, "Internal error: Invalid command %d\n", command);
	}
	
	for (done = 0; !done; ) {
		int type;
		
		status = fln_read(h, buf, BUFSIZE, 0);
		if (status < 0)
			die(0, "fln_read");
		
		type = fln_message_type(buf);
		
		switch (type) {
		
		case NLMSG_ERROR:
		{
			int errval = fln_get_msgerr(buf);
			
			/* netlink ack */
			if (errval == 0)
				break;
			
			warn("Netlink error report: %s", strerror(fln_get_msgerr(buf)));
			break;
		}
		
		case FLMSG_CACHE_GET:
			fln_parse_cache_map(h, buf, status, print_mapres);
			break;
			
		case NLMSG_DONE:
			done = 1;
			break;
		
		default:
			warn("Received unknown Netlink message type: %d\n", type);
			break;
		}
		
		if (command != CMD_DUMP)
			done = 1;
	}

	return 0;
}
