#include "hashtab.h"

#include <unistd.h>
#include <sys/types.h>
#include <stdlib.h>

#define HASHTAB_DEBUG 0

hashtab_t 
hashtab_create(void)
{
	hashtab_t       p;
	int             i;


	p = (hashtab_t) malloc(sizeof(hashtab_val_t));
	if (p == NULL)
		return p;

	p->htable = (hashtab_ptr_t *) 
	  malloc(sizeof(hashtab_ptr_t)*INOTAB_SIZE);
	if (p->htable == NULL) {
		free(p);
		return NULL;
	}
	for (i = 0; i < INOTAB_SIZE; i++)
		p->htable[i] = (hashtab_ptr_t) NULL;

	return p;
}


int 
hashtab_insert(hashtab_t h, hashtab_key_t key, hashtab_datum_t datum)
{
	int             hvalue;
	hashtab_ptr_t   cur, newnode;


	if (!h)
		return HASHTAB_OVERFLOW;

	hvalue = HASH_VALUE(key);
	cur = h->htable[hvalue];
	while (cur != NULL && KEYCMP(cur->key, key))
		cur = cur->next;

	if (cur != NULL)
		return HASHTAB_PRESENT;

	newnode = (hashtab_ptr_t) malloc(sizeof(struct hashtab_node_t));
	if (newnode == NULL)
		return HASHTAB_OVERFLOW;
	newnode->key = key;
	newnode->datum = datum;
	newnode->next = h->htable[hvalue];
	h->htable[hvalue] = newnode;
	return HASHTAB_SUCCESS;
}


int 
hashtab_remove(hashtab_t h, hashtab_key_t key)
{
	int             hvalue;
	hashtab_ptr_t   cur, last;


	if (!h)
		return HASHTAB_MISSING;

	hvalue = HASH_VALUE(key);
	last = NULL;
	cur = h->htable[hvalue];
	while (cur != NULL && KEYCMP(cur->key, key)) {
		last = cur;
		cur = cur->next;
	}

	if (cur == NULL)
		return HASHTAB_MISSING;

	if (last == NULL)
		h->htable[hvalue] = cur->next;
	else
		last->next = cur->next;

	free(cur);
	return HASHTAB_SUCCESS;
}


int 
hashtab_replace(hashtab_t h, hashtab_key_t key, hashtab_datum_t datum)
{
	int             hvalue;
	hashtab_ptr_t   cur, newnode;


	if (!h)
		return HASHTAB_OVERFLOW;

	hvalue = HASH_VALUE(key);
	cur = h->htable[hvalue];
	while (cur != NULL && KEYCMP(cur->key, key) != 0)
		cur = cur->next;

	if (cur) {
		cur->key = key;
		cur->datum = datum;
	} else {
		newnode = (hashtab_ptr_t) malloc(sizeof(struct hashtab_node_t));

		if (newnode == NULL)
			return HASHTAB_OVERFLOW;
		newnode->key = key;
		newnode->datum = datum;
		newnode->next = h->htable[hvalue];
		h->htable[hvalue] = newnode;
	}

	return HASHTAB_SUCCESS;
}


hashtab_datum_t 
hashtab_search(hashtab_t h, hashtab_key_t key)
{
	int             hvalue;
	hashtab_ptr_t   cur;


	if (!h)
		return 0;

	hvalue = HASH_VALUE(key);
	cur = h->htable[hvalue];
	while (cur != NULL && KEYCMP(cur->key, key))
		cur = cur->next;

	if (cur == NULL)
		return 0;

	return cur->datum;
}


hashtab_key_t 
hashtab_reverse_search(hashtab_t h, hashtab_datum_t datum)
{
	int             i;
	hashtab_ptr_t   cur;


	if (!h)
		return 0;

	for (i = 0; i < INOTAB_SIZE; i++) {
		cur = h->htable[i];
		while (cur != NULL) {
			if (cur->key && (cur->datum == datum))
				return cur->key;
			cur = cur->next;
		}
	}
	return 0;
}


void 
hashtab_destroy(hashtab_t h)
{
	int             i;
	hashtab_ptr_t   cur, temp;


	if (!h)
		return;

	for (i = 0; i < INOTAB_SIZE; i++) {
		cur = h->htable[i];
		while (cur != NULL) {
			temp = cur;
			cur = cur->next;
			free(temp);
		}
	}

	free(h->htable);
	free(h);
}


int 
hashtab_map(hashtab_t h,
	    int (*apply) (hashtab_key_t k,
			  hashtab_datum_t d,
			  void *args),
	    void *args)
{
	int             i, ret;
	hashtab_ptr_t   cur;


	if (!h)
		return HASHTAB_SUCCESS;

	for (i = 0; i < INOTAB_SIZE; i++) {
		cur = h->htable[i];
		while (cur != NULL) {
			ret = apply(cur->key, cur->datum, args);
			if (ret)
				return ret;
			cur = cur->next;
		}
	}
	return HASHTAB_SUCCESS;
}


void 
hashtab_map_remove_on_error(hashtab_t h,
			    int (*apply) (hashtab_key_t k,
					  hashtab_datum_t d,
					  void *args),
			    void *args)
{
	int             i, ret;
	hashtab_ptr_t   last, cur, temp;


	if (!h)
		return;

	for (i = 0; i < INOTAB_SIZE; i++) {
		last = NULL;
		cur = h->htable[i];
		while (cur != NULL) {
			ret = apply(cur->key, cur->datum, args);
			if (ret) {
				if (last) {
					last->next = cur->next;
				} else {
					h->htable[i] = cur->next;
				}

				temp = cur;
				cur = cur->next;
				free(temp);
			} else {
				last = cur;
				cur = cur->next;
			}
		}
	}

	return;
}
