/* 
 * setfiles
 *
 * AUTHOR:  Stephen Smalley, NAI Labs <ssmalley@nai.com>
 * This program was derived in part from the setfiles.pl script
 * developed by Secure Computing Corporation.
 *
 * PURPOSE:
 * This program reads a set of file security context specifications
 * based on pathname regular expressions and labels files
 * accordingly, traversing a set of file systems specified by
 * the user.  The program does not cross file system boundaries.
 *
 * USAGE:
 * setfiles [-dnvR] spec_file pathname...
 * 
 * -d   Show what specification matched each file.
 * -n	Do not change any file labels.
 * -v	Show changes in file labels.  
 * -R   Recreate the persistent label mappings from scratch.
 *      Must reboot to take affect.
 * spec_file	The specification file.
 * pathname...	The file systems to label.	
 *
 * EXAMPLE USAGE:
 * ./setfiles -v file_contexts `mount | awk '/ext2/{print $3}'`
 *
 * SPECIFICATION FILE:
 * Each specification has the form:
 *       regexp [ -type ] ( context | <<none>> )
 *
 * By default, the regexp is an anchored match on both ends (i.e. a 
 * caret (^) is prepended and a dollar sign ($) is appended automatically).
 * This default may be overridden by using .* at the beginning and/or
 * end of the regular expression.  
 *
 * The optional type field specifies the file type as shown in the mode
 * field by ls, e.g. use -d to match only directories or -- to match only
 * regular files.
 * 
 * The value of <<none> may be used to indicate that matching files
 * should not be relabeled.
 *
 * The last matching specification is used.
 *
 * If there are multiple hard links to a file that match 
 * different specifications and those specifications indicate
 * different security contexts, then a warning is displayed
 * but the file is still labeled based on the last matching
 * specification other than <<none>>.
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <regex.h>
#include <sys/vfs.h>
#define __USE_XOPEN_EXTENDED 1	/* nftw */
#include <ftw.h>
#include <flask_util.h>
#include <fs_secure.h>
#include <ss.h>
#include "psid.h"

struct psidtab *psids;

#define FILE_CONTEXT "system_u:object_r:file_t"
#define FS_CONTEXT "system_u:object_r:fs_t"

static int flask_enabled = 0;

/*
 * Command-line options.
 */
static int change = 1;
static int debug = 0;
static int verbose = 0;
static int reset = 0;

/*
 * Program name and error message buffer.
 */
static char *progname;
static char errbuf[255 + 1];

/*
 * A file security context specification.
 */
typedef struct spec {
	char *regex_str;	/* regular expession string for diagnostic messages */
	char *type_str;		/* type string for diagnostic messages */
	char *context;		/* context string */
	regex_t regex;		/* compiled regular expression */
	mode_t mode;		/* mode format value */
	security_id_t sid;      /* SID */
	int matches;		/* number of matching pathnames */
} spec_t;

/*
 * The array of specifications, in the
 * same order as in the specification file.
 */
static spec_t *spec;
static int nspec;

/*
 * An association between an inode and a 
 * specification.  
 */
typedef struct file_spec {
	ino_t ino;		/* inode number */
	int specind;		/* index of specification in spec */
	char *file;		/* full pathname for diagnostic messages about conflicts */
	struct file_spec *next;	/* next association in hash bucket chain */
} file_spec_t;

/*
 * The hash table of associations, hashed by inode number.
 * Chaining is used for collisions, with elements ordered
 * by inode number in each bucket.  Each hash bucket has a dummy 
 * header.
 */
#define HASH_BITS 16
#define HASH_BUCKETS (1 << HASH_BITS)
#define HASH_MASK (HASH_BUCKETS-1)
static file_spec_t fl_head[HASH_BUCKETS];

/*
 * Try to add an association between an inode and
 * a specification.  If there is already an association
 * for the inode and it conflicts with this specification,
 * then use the specification that occurs later in the
 * specification array.
 */
static file_spec_t *file_spec_add(ino_t ino, int specind, const char *file)
{
	file_spec_t *prevfl, *fl;
	int h, no_conflict;

	h = (ino + (ino >> HASH_BITS)) & HASH_MASK;
	for (prevfl = &fl_head[h], fl = fl_head[h].next; fl;
	     prevfl = fl, fl = fl->next) {
		if (ino == fl->ino) {
			if (flask_enabled) 
				no_conflict = (spec[fl->specind].sid == spec[specind].sid);
			else
				no_conflict = (strcmp(spec[fl->specind].context,spec[specind].context) == 0);
			if (no_conflict)
				return fl;

			fprintf(stderr,
				"%s:  conflicting specifications for %s and %s, using %s.\n",
				progname, file, fl->file,
				((specind > fl->specind) ? spec[specind].
				 context : spec[fl->specind].context));
			fl->specind =
			    (specind >
			     fl->specind) ? specind : fl->specind;
			free(fl->file);
			fl->file = malloc(strlen(file) + 1);
			if (!fl->file) {
				fprintf(stderr,
					"%s:  insufficient memory for file label entry for %s\n",
					progname, file);
				return NULL;
			}
			strcpy(fl->file, file);
			return fl;
		}

		if (ino > fl->ino)
			break;
	}

	fl = malloc(sizeof(file_spec_t));
	if (!fl) {
		fprintf(stderr,
			"%s:  insufficient memory for file label entry for %s\n",
			progname, file);
		return NULL;
	}
	fl->ino = ino;
	fl->specind = specind;
	fl->file = malloc(strlen(file) + 1);
	if (!fl->file) {
		fprintf(stderr,
			"%s:  insufficient memory for file label entry for %s\n",
			progname, file);
		return NULL;
	}
	strcpy(fl->file, file);
	fl->next = prevfl->next;
	prevfl->next = fl;
	return fl;
}

/*
 * Evaluate the association hash table distribution.
 */
static void file_spec_eval(void)
{
	file_spec_t *fl;
	int h, used, nel, len, longest;

	used = 0;
	longest = 0;
	nel = 0;
	for (h = 0; h < HASH_BUCKETS; h++) {
		len = 0;
		for (fl = fl_head[h].next; fl; fl = fl->next) {
			len++;
		}
		if (len)
			used++;
		if (len > longest)
			longest = len;
		nel += len;
	}

	printf
	    ("%s:  hash table stats: %d elements, %d/%d buckets used, longest chain length %d\n",
	     progname, nel, used, HASH_BUCKETS, longest);
}


/*
 * Destroy the association hash table.
 */
static void file_spec_destroy(void)
{
	file_spec_t *fl, *tmp;
	int h;

	for (h = 0; h < HASH_BUCKETS; h++) {
		fl = fl_head[h].next;
		while (fl) {
			tmp = fl;
			fl = fl->next;
			free(tmp->file);
			free(tmp);
		}
		fl_head[h].next = NULL;
	}
}


int match(const char *name, struct stat *sb, security_id_t *out_sid) 
{
	int i, ret;

	if (flask_enabled) {
		ret = lstat_secure(name, sb, out_sid);
	} else {
		ret = lstat(name, sb);
	} 
	if (ret) {
		fprintf(stderr, "%s:  unable to stat file %s\n", progname,
			name);
		return -1;
	}

	/* 
	 * Check for matching specifications in reverse order, so that
	 * the last matching specification is used.
	 */
	for (i = nspec - 1; i >= 0; i--) {
		ret = regexec(&spec[i].regex, name, 0, NULL, 0);
		if (ret == 0 &&
		    (!spec[i].mode
		     || (sb->st_mode & S_IFMT) == spec[i].mode)) break;
		if (ret) {
			if (ret == REG_NOMATCH)
				continue;
			regerror(ret, &spec[i].regex, errbuf,
				 sizeof errbuf);
			fprintf(stderr,
				"%s:  unable to match %s against %s:  %s\n",
				progname, name, spec[i].regex_str,
				errbuf);
			return -1;
		}
	}

	if (i < 0)
		/* No matching specification. */
		return -1;

	spec[i].matches++;

	return i;
}

#define SZ 255

/*
 * Apply the last matching specification to a file.
 * This function is called by nftw on each file during
 * the directory traversal.
 */
static int apply_spec(const char *file,
		      const struct stat *sb, int flag, struct FTW *s)
{
	const char *my_file;
	file_spec_t *fl;
	struct stat my_sb;
	security_id_t sid;
	int i, len, ret;
	char *context;

	/* Skip the extra slash at the beginning, if present. */
	if (file[0] == '/' && file[1] == '/')
		my_file = &file[1];
	else
		my_file = file;

	if (flag == FTW_DNR) {
		fprintf(stderr, "%s:  unable to read directory %s\n",
			progname, my_file);
		return 0;
	}

	i = match(my_file, &my_sb, &sid);
	if (i < 0)
		/* No matching specification. */
		return 0;

	/*
	 * Try to add an association between this inode and
	 * this specification.  If there is already an association
	 * for this inode and it conflicts with this specification,
	 * then use the last matching specification.
	 */
	fl = file_spec_add(my_sb.st_ino, i, my_file);
	if (!fl)
		/* Insufficient memory to proceed. */
		return 1;

	if (fl->specind != i)
		/* There was already an association and it took precedence. */
		return 0;

	if (debug) {
		if (spec[i].type_str) {
			printf("%s:  %s matched by (%s,%s,%s)\n", progname,
			       my_file, spec[i].regex_str,
			       spec[i].type_str, spec[i].context);
		} else {
			printf("%s:  %s matched by (%s,%s)\n", progname,
			       my_file, spec[i].regex_str,
			       spec[i].context);
		}
	}

	/* Get the current SID or context of the file. */
	if (flask_enabled) {
		if (lstat_secure(my_file, &my_sb, &sid)) {
			fprintf(stderr, "%s:  unable to stat file %s\n", progname,
				my_file);
			return 0;
		}
	} else {
		context = psid_to_context(psids, my_sb.st_ino);
		if (!context) {
			fprintf(stderr, "%s:  unable to obtain context of %s\n", progname,
				my_file);
			return 0;
		}
	}

	/*
	 * Do not relabel the file if the matching specification is 
	 * <<none>> or the file is already labeled according to the 
	 * specification.
	 */
	if (flask_enabled) {
		if (!spec[i].sid || sid == spec[i].sid)
			return 0;
	} else {
		if ((strcmp(spec[i].context, "<<none>>") == 0) || 
		    (strcmp(context,spec[i].context) == 0))
			return 0;
	}

	if (verbose) {
		if (flask_enabled) {
			context = malloc(SZ);
			if (!context) {
				fprintf(stderr,
					"%s:  insufficient memory for context for %s\n",
					progname, my_file);
				return 1;
			}
			len = SZ;
			ret = security_sid_to_context(sid, context, &len);
			if (ret && (len > SZ)) {
				context = realloc(context, len);
				if (!context) {
					fprintf(stderr,
						"%s:  insufficient memory for context for %s\n",
						progname, my_file);
					return 1;
				}
				ret =
				    security_sid_to_context(sid, context,
							    &len);
			}
			if (ret) {
				fprintf(stderr,
					"%s:  unable to obtain context for SID %d on file %s\n",
					progname, sid, my_file);
				return 1;
			}
		}
		printf("%s:  relabeling %s from %s to %s\n", progname,
		       my_file, context, spec[i].context);
	}

	/*
	 * Do not relabel the file if -n was used.
	 */
	if (!change)
		return 0;

	/*
	 * Relabel the file to the specified context.
	 */
	if (flask_enabled) 
		ret = lchsid(my_file, spec[i].sid);
	else 
		ret = context_to_psid(psids, my_sb.st_ino, spec[i].context);
	if (ret) {
		if (flask_enabled)
			perror(my_file);
		fprintf(stderr, "%s:  unable to relabel %s to %s\n",
			progname, my_file, spec[i].context);
		return 1;
	}

	return 0;
}


int main(int argc, char **argv)
{
	FILE *fp;
	char buf[255 + 1], *buf_p;
	char *regex, *type, *context;
	char *anchored_regex;
	char pathname[1024];
	int opt, items, len, lineno, pass, nerr, regerr, i;
	security_id_t sid;
	struct stat sb;

	/* Process any options. */
	while ((opt = getopt(argc, argv, "dnvR")) > 0) {
		switch (opt) {
		case 'd':
			debug = 1;
			break;
		case 'n':
			change = 0;
			break;
		case 'v':
			verbose = 1;
			break;
		case 'R':
			reset = 1;
			break;
		}
	}

	if (optind > (argc - 2)) {
		fprintf(stderr,
			"usage:  %s [-dnvR] spec_file pathname...\n",
			argv[0]);
		exit(1);
	}

	if (reset) {
		printf("%s:  Recreating persistent label mappings from scratch.\n", argv[0]);
                /* Do not use the new syscalls, even if running on SELinux,
		   since they will access the old persistent label mappings. */
		flask_enabled = 0; 
	} else {
		flask_enabled = is_flask_enabled();
		if (flask_enabled) 
			printf("%s:  Running on a SELinux kernel, using new system calls\n", argv[0]);
		else 
			printf("%s:  Running on a non-SELinux kernel, directly accessing persistent label mapping\n", argv[0]);
	}

	/* Open the specification file. */
	if ((fp = fopen(argv[optind], "r")) == NULL) {
		perror(argv[optind]);
		exit(1);
	}
	optind++;

	/* 
	 * Perform two passes over the specification file.
	 * The first pass counts the number of specifications and
	 * performs simple validation of the input.  At the end
	 * of the first pass, the spec array is allocated.
	 * The second pass performs detailed validation of the input
	 * and fills in the spec array.
	 */
	for (pass = 0; pass < 2; pass++) {
		lineno = 0;
		nspec = 0;
		nerr = 0;
		while (fgets(buf, sizeof buf, fp)) {
			lineno++;
			len = strlen(buf);
			if (buf[len - 1] != '\n') {
				fprintf(stderr,
					"%s:  no newline on line number %d (only read %s)\n",
					argv[0], lineno, buf);
				nerr++;
				continue;
			}
			buf[len - 1] = 0;
			buf_p = buf;
			while (isspace(*buf_p))
				buf_p++;
			/* Skip comment lines and empty lines. */
			if (*buf_p == '#' || *buf_p == 0)
				continue;
			items =
			    sscanf(buf, "%as %as %as", &regex, &type,
				   &context);
			if (items < 2) {
				fprintf(stderr,
					"%s:  line number %d is missing fields (only read %s)\n",
					argv[0], lineno, buf);
				nerr++;
				if (items == 1)
					free(regex);
				continue;
			} else if (items == 2) {
				/* The type field is optional. */
				free(context);
				context = type;
				type = 0;
			}

			if (pass == 1) {
				/* On the second pass, compile and store the specification in spec. */
				spec[nspec].regex_str = regex;

				/* Anchor the regular expression. */
				len = strlen(regex);
				anchored_regex = malloc(len + 3);
				if (!anchored_regex) {
					fprintf(stderr,
						"%s:  insufficient memory for anchored regexp on line %d\n",
						argv[0], lineno);
					exit(1);
				}
				sprintf(anchored_regex, "^%s$", regex);

				/* Compile the regular expression. */
				regerr =
				    regcomp(&spec[nspec].regex,
					    anchored_regex,
					    REG_EXTENDED | REG_NOSUB);
				if (regerr < 0) {
					regerror(regerr,
						 &spec[nspec].regex,
						 errbuf, sizeof errbuf);
					fprintf(stderr,
						"%s:  unable to compile regular expression %s on line number %d:  %s\n",
						argv[0], regex, lineno,
						errbuf);
					nerr++;
				}
				free(anchored_regex);

				/* Convert the type string to a mode format */
				spec[nspec].type_str = type;
				spec[nspec].mode = 0;
				if (!type)
					goto skip_type;
				len = strlen(type);
				if (type[0] != '-' || len != 2) {
					fprintf(stderr,
						"%s:  invalid type specifier %s on line number %d\n",
						argv[0], type, lineno);
					nerr++;
					goto skip_type;
				}
				switch (type[1]) {
				case 'b':
					spec[nspec].mode = S_IFBLK;
					break;
				case 'c':
					spec[nspec].mode = S_IFCHR;
					break;
				case 'd':
					spec[nspec].mode = S_IFDIR;
					break;
				case 'p':
					spec[nspec].mode = S_IFIFO;
					break;
				case 'l':
					spec[nspec].mode = S_IFLNK;
					break;
				case 's':
					spec[nspec].mode = S_IFSOCK;
					break;
				case '-':
					spec[nspec].mode = S_IFREG;
					break;
				default:
					fprintf(stderr,
						"%s:  invalid type specifier %s on line number %d\n",
						argv[0], type, lineno);
					nerr++;
				}

			      skip_type:

				spec[nspec].context = context;
				if (flask_enabled) {
					if (strcmp(context, "<<none>>") == 0) {
						spec[nspec].sid = 0;
					} else {
						len = strlen(context);
						if (security_context_to_sid
						    (context, len + 1,
						     &spec[nspec].sid)) {
							fprintf(stderr,
								"%s:  invalid context %s on line number %d\n",
								argv[0], context,
								lineno);
							nerr++;
						}
					}
				}
			}

			nspec++;
			if (pass == 0) {
				free(regex);
				if (type)
					free(type);
				free(context);
			}
		}

		if (nerr)
			exit(1);

		if (pass == 0) {
			printf("%s:  read %d specifications\n", argv[0],
			       nspec);
			if (nspec == 0)
				exit(0);
			if ((spec = malloc(sizeof(spec_t) * nspec)) ==
			    NULL) {
				fprintf(stderr,
					"%s:  insufficient memory for specifications\n",
					argv[0]);
				exit(1);
			}
			bzero(spec, sizeof(spec_t) * nspec);
			rewind(fp);
		}
	}
	fclose(fp);

	/*
	 * Apply the specifications to the file systems.
	 */
	progname = argv[0];
	for (; optind < argc; optind++) {

		if (flask_enabled) {
			printf("%s:  labeling files under %s\n", argv[0],
			       argv[optind]);
		} else {
			sprintf(pathname, "%s/%s", argv[optind], ".");
			i = match(pathname, &sb, &sid);
			if (i < 0) 
				context = FILE_CONTEXT;
			else
				context = spec[i].context;

			printf("%s:  labeling files under %s (default file context %s)\n", argv[0],
			       argv[optind], context);

			psids = psidtab_init(argv[optind], context, FS_CONTEXT, reset);
			if (psids == NULL) {
				fprintf(stderr,
					"%s:  unable to initialize PSID mapping for %s\n",
					argv[0], argv[optind]);
				exit(1);
			}
		}
		
		/* Walk the file tree, calling apply_spec on each file. */
		if (nftw
		    (argv[optind], apply_spec, 1024,
		     FTW_PHYS | FTW_MOUNT)) {
			fprintf(stderr,
				"%s:  error while labeling files under %s\n",
				argv[0], argv[optind]);
			exit(1);
		}

		if (!flask_enabled)
			psidtab_destroy(psids);

		/*
		 * Evaluate the association hash table distribution for the
		 * directory tree just traversed.
		 */
		file_spec_eval();

		/* Reset the association hash table for the next directory tree. */
		file_spec_destroy();
	}

	for (i = 0; i < nspec; i++) {
		if (spec[i].matches == 0) {
			if (spec[i].type_str) {
				printf
				    ("%s:  Warning!  No matches for (%s, %s, %s)\n",
				     argv[0], spec[i].regex_str,
				     spec[i].type_str, spec[i].context);
			} else {
				printf
				    ("%s:  Warning!  No matches for (%s, %s)\n",
				     argv[0], spec[i].regex_str,
				     spec[i].context);
			}
		}
	}

	if (!flask_enabled) {
		printf("%s:  Done.  Please %sboot your SELinux kernel and then run 'make verbose' to relabel any files that were created during the shutdown.\n", argv[0], reset ? "re" : "");
	} else {
		printf("%s:  Done.\n", argv[0]);
	}

	exit(0);
}
