/* properties.c
All functions related to the "Properties" section */

/* Includes */
#include <applet-widget.h>

/* Sensor Includes */
#include <sensors/sensors.h>
#include <sensors/error.h>
#include <sensors/chips.h>
/* #include <linux/sensors.h> */

/* Local Includes */
#include "updateFunctions.h"
#include "structures.h"

/* Prototypes */
void properties(AppletWidget *caller, gpointer data);
void updateSensorsCList(GList *list, GtkWidget *propertiesCList);
void modifySensorDialog(GtkWidget *widget, gpointer data);
void addSensorDialog(GtkWidget *widget, gpointer data);
void modifySensor(GtkWidget *widget, gpointer data);
void addSensor(GtkWidget *widget, gpointer data);
void writeConfig(GtkWidget *widget, gint page_num, gpointer data);
gint deleteSensor(GtkWidget *widget, gpointer data);
void cancelConfig(GtkWidget *widget, gpointer data);
GtkWidget *makeSensorsTab(struct appletInfo *ai, struct tempAppletInfo *tai, GtkWidget *propertiesBox, struct deleteSensorInfo *deleteInfo, struct drawAddSensorInfo *drawInfo);
GtkWidget *makeDisplayTab(struct propDisplayInfo *disInfo);
void propHeightChange(GtkWidget *widget, gpointer data);
void propWidthChange(GtkWidget *widget, gpointer data);
void propSensorFreqChange(GtkWidget *widget, gpointer data);
void propDisplayFreqChange(GtkWidget *widget, gpointer data);
void makeChanges(gpointer *data);
void windowDestroy(GtkWidget *widget, gpointer *data);
gint initTAI(struct appletInfo *ai, struct tempAppletInfo *tai);
#if DEBUG == 1
void showSensorList(gchar *label, GList *list);
#endif

void properties(AppletWidget *caller, gpointer data) {
	GtkWidget *propertiesWindow, *sensorsTab, *displayTab;
	struct appletInfo *ai; 
	struct tempAppletInfo *tai;
	struct propDisplayInfo *disInfo;
	struct makeChangesInfo *info;
	struct cancelConfigInfo *cancelInfo;
	struct deleteSensorInfo *deleteInfo;
	struct drawAddSensorInfo *drawInfo;

	/* Casting */
	ai = (struct appletInfo *) data;

	#if DEBUG == 1
		g_print("Entering properties()\n");
	#endif

	/* Allocate some memory for structures*/
	/* disInfo is used in makeDisplay and to pass on */
	disInfo = g_malloc(sizeof(struct propDisplayInfo));
	/* cancelInfo holds structures to be freed when properties is closed */
	cancelInfo = g_malloc(sizeof(struct cancelConfigInfo));
	/* deleteSensorInfo is used in deleteSensor pass through makeSensorsTab */
	deleteInfo = g_malloc(sizeof(struct deleteSensorInfo));
	/* drawInfo is used in addSensorDialog & modifySensorDialog passed through makeSensorsTab */
        drawInfo = g_malloc(sizeof(struct drawAddSensorInfo));
	
	/* tai is out temporary settings structure */
	/* It holds the settings before we write them */
	tai = g_malloc(sizeof(struct tempAppletInfo));
	initTAI(ai, tai);

	/* Info struct for makeChanges() */
	info = g_malloc(sizeof(struct makeChangesInfo));
	info->tai = tai;
	info->ai = ai;

	/* Setup the cancelConfigInfo struct */
	cancelInfo->tai = tai;
	cancelInfo->minfo = info;
	cancelInfo->deleteInfo = deleteInfo;
	cancelInfo->drawInfo = drawInfo;

	/* Create the Gnome Properties dialog */
	propertiesWindow = gnome_property_box_new();

	/* Create the property tabs */
	sensorsTab = makeSensorsTab(ai, tai, propertiesWindow, deleteInfo, drawInfo);
	disInfo->propertiesBox = propertiesWindow;
	disInfo->ai = ai;
	disInfo->tai = tai;
	displayTab = makeDisplayTab(disInfo);

	/* Insert the current config to the Clist */
        updateSensorsCList(ai->sensorList, ai->sensorsCList);

	/* Add it to the notebook */
	gnome_property_box_append_page(GNOME_PROPERTY_BOX(propertiesWindow), sensorsTab, gtk_label_new("Sensors"));
	gnome_property_box_append_page(GNOME_PROPERTY_BOX(propertiesWindow), displayTab, gtk_label_new("Display"));

	/* Signal time */
	gtk_signal_connect(GTK_OBJECT(propertiesWindow), "apply", GTK_SIGNAL_FUNC(writeConfig), info);
	gtk_signal_connect(GTK_OBJECT(propertiesWindow), "destroy", GTK_SIGNAL_FUNC(cancelConfig), cancelInfo);

	/* Show the widgets */
	gtk_widget_show(propertiesWindow);
}

GtkWidget *makeSensorsTab(struct appletInfo *ai, struct tempAppletInfo *tai, GtkWidget *propertiesBox, struct deleteSensorInfo *deleteInfo, struct drawAddSensorInfo *drawInfo) {
	/* Create and setup the "sensors" tab of the properties */
	/* The deleteInfo is passed to the deleteSensor func. */
	/* It is setup in the properties() function so it can be free'd by */
	/* the cancelConfig() func */
	GtkWidget *sensorsFrame, *sensorsVBox, *sensorsHBox1, *sensorsHBox2;
	GtkWidget *sensorsScrolledWindow, *addSensorButton;
	GtkWidget *deleteSensorButton, *modifySensorButton;
	gchar *colTitles[3] = { "Display Name", "Chip Name", "Sensor Name" };

	#if DEBUG == 1
		g_print("Entering makeSensorsTab()\n");
	#endif

	/* Create the Frame */
        sensorsFrame = gtk_frame_new("Sensor Configuration");
	gtk_widget_set_usize(GTK_WIDGET(sensorsFrame), 500, 300);
        gtk_container_set_border_width(GTK_CONTAINER(sensorsFrame), 3);

        /* Create the vbox and add it to the sensorsFrame */
        sensorsVBox = gtk_vbox_new(FALSE, 3);
        gtk_container_add(GTK_CONTAINER(sensorsFrame), sensorsVBox);

        /* Create the hBox's and add them to the vBox */
        sensorsHBox1 = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(sensorsVBox), sensorsHBox1, TRUE, TRUE, 0);
        sensorsHBox2 = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(sensorsVBox), sensorsHBox2, FALSE, FALSE, 0);

        /* Create the scrolled window and add it to the first hBox */
        sensorsScrolledWindow = gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sensorsScrolledWindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
        gtk_box_pack_start(GTK_BOX(sensorsHBox1), sensorsScrolledWindow, TRUE, TRUE, 0);

        /* Create the CList and add it to the scrolled window */
        ai->sensorsCList = gtk_clist_new_with_titles(3, colTitles);
        gtk_clist_set_selection_mode(GTK_CLIST(ai->sensorsCList), GTK_SELECTION_SINGLE);
	gtk_clist_set_column_width(GTK_CLIST(ai->sensorsCList), 0, 150);
	gtk_clist_set_column_width(GTK_CLIST(ai->sensorsCList), 1, 150);
        gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(sensorsScrolledWindow), ai->sensorsCList);

	/* Fill out the struct for passing with the addSensorButton & modifySensorButton */
	drawInfo->propertiesBox = propertiesBox;
	drawInfo->clist = ai->sensorsCList;
	drawInfo->startList = tai->sensorList;

	/* Fill out the struct for passing with the deleteSensorButton */
	deleteInfo->clist = ai->sensorsCList;
	deleteInfo->propertiesBox = propertiesBox;
	deleteInfo->tai = tai;
	
        /* Create and setup the add and delete sensors buttons */
        addSensorButton = gtk_button_new_with_label(" Add Sensor ");
	modifySensorButton = gtk_button_new_with_label(" Modify Sensor ");
        deleteSensorButton = gtk_button_new_with_label(" Delete Sensor ");
        gtk_signal_connect(GTK_OBJECT(addSensorButton), "clicked", GTK_SIGNAL_FUNC(addSensorDialog), drawInfo);
	gtk_signal_connect(GTK_OBJECT(modifySensorButton), "clicked", GTK_SIGNAL_FUNC(modifySensorDialog), drawInfo);
        gtk_signal_connect(GTK_OBJECT(deleteSensorButton), "clicked", GTK_SIGNAL_FUNC(deleteSensor), deleteInfo);
        gtk_box_pack_start(GTK_BOX(sensorsHBox2), addSensorButton, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(sensorsHBox2), modifySensorButton, FALSE, FALSE, 0);
        gtk_box_pack_start(GTK_BOX(sensorsHBox2), deleteSensorButton, FALSE, FALSE, 0);

	/* Show the widgets */
        gtk_widget_show(addSensorButton);
	gtk_widget_show(modifySensorButton);
        gtk_widget_show(deleteSensorButton);
        gtk_widget_show(ai->sensorsCList);
        gtk_widget_show(sensorsScrolledWindow);
        gtk_widget_show(sensorsHBox1);
        gtk_widget_show(sensorsHBox2);
        gtk_widget_show(sensorsVBox);
        gtk_widget_show(sensorsFrame);

	return(sensorsFrame);
}

GtkWidget *makeDisplayTab(struct propDisplayInfo *disInfo) {
	GtkWidget *displayFrame, *heightLabel, *widthLabel, *sensorFreqLabel, *displayFreqLabel;
	GtkWidget *heightEntry, *widthEntry, *sensorFreqEntry, *displayFreqEntry;
	GtkWidget *vbox, *hbox1, *hbox2, *hbox3, *hbox4;
	gchar *heightText, *widthText, *sensorFreqText, *displayFreqText;
	struct appletInfo *ai;

	#if DEBUG == 1
		g_print("Entering makeDisplayTab()\n");
	#endif

	/* Lil casting */
	ai = disInfo->ai;
	
	/* Create the Frame */
	displayFrame = gtk_frame_new("Display Configuration");
	gtk_container_set_border_width(GTK_CONTAINER(displayFrame), 3);

	/* Create the boxes */
	vbox = gtk_vbox_new(TRUE, 0);
	gtk_container_add(GTK_CONTAINER(displayFrame), vbox);
	
	hbox1 = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox1, FALSE, FALSE, 0);
	hbox2 = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox2, FALSE, FALSE, 0);
	hbox3 = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox3, FALSE, FALSE, 0);
	hbox4 = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox4, FALSE, FALSE, 0);

	/* Add the labels and the text entries */
	heightLabel = gtk_label_new("Applet Height:\n(Used when panel set to vertical orientation)");
	widthLabel = gtk_label_new("Applet Width:\n(Used when panel set to horizontal orientation)");
	sensorFreqLabel = gtk_label_new("Sensors Update Frequency (ms): ");
	displayFreqLabel = gtk_label_new("Display Update Frequency (ms): ");
	heightEntry = gtk_entry_new();
	heightText = g_strdup_printf("%i", ai->height);
	gtk_entry_set_text(GTK_ENTRY(heightEntry), heightText);
	widthEntry = gtk_entry_new();
	widthText = g_strdup_printf("%i", ai->width);
	gtk_entry_set_text(GTK_ENTRY(widthEntry), widthText);
	sensorFreqEntry = gtk_entry_new();
	sensorFreqText = g_strdup_printf("%i", ai->sensorFreq);
	gtk_entry_set_text(GTK_ENTRY(sensorFreqEntry), sensorFreqText);
	displayFreqEntry = gtk_entry_new();
	displayFreqText = g_strdup_printf("%i", ai->displayFreq);
	gtk_entry_set_text(GTK_ENTRY(displayFreqEntry), displayFreqText);

	/* Free the temp strings */
	g_free(heightText);
	g_free(widthText);
	g_free(sensorFreqText);
	g_free(displayFreqText);

	/* Pack the labels and the text entries */
	gtk_box_pack_start(GTK_BOX(hbox1), heightLabel, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox1), heightEntry, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox2), widthLabel, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox2), widthEntry, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox3), sensorFreqLabel, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox3), sensorFreqEntry, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox4), displayFreqLabel, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox4), displayFreqEntry, FALSE, FALSE, 0);

	/* Connect the callbacks to the text boxes */
	gtk_signal_connect(GTK_OBJECT(heightEntry), "changed", GTK_SIGNAL_FUNC(propHeightChange), disInfo);
	gtk_signal_connect(GTK_OBJECT(widthEntry), "changed", GTK_SIGNAL_FUNC(propWidthChange), disInfo);
	gtk_signal_connect(GTK_OBJECT(sensorFreqEntry), "changed", GTK_SIGNAL_FUNC(propSensorFreqChange), disInfo);
	gtk_signal_connect(GTK_OBJECT(displayFreqEntry), "changed", GTK_SIGNAL_FUNC(propDisplayFreqChange), disInfo);

	/* Show the widgets */
	gtk_widget_show(heightLabel);
	gtk_widget_show(heightEntry);
	gtk_widget_show(widthLabel);
	gtk_widget_show(widthEntry);
	gtk_widget_show(sensorFreqLabel);
	gtk_widget_show(sensorFreqEntry);
	gtk_widget_show(displayFreqLabel);
	gtk_widget_show(displayFreqEntry);
	gtk_widget_show(hbox4);
	gtk_widget_show(hbox3);
	gtk_widget_show(hbox2);
	gtk_widget_show(hbox1);
	gtk_widget_show(vbox);
	gtk_widget_show(displayFrame);

	return(displayFrame);
}

void updateSensorsCList(GList *list, GtkWidget *propertiesCList) {
	/* Fill in the CList that has all of our sensors */
	GList *temp;
	gchar *info[3];
	gint i=0;
	struct sensor *tempSensor;

	#if DEBUG == 1
		g_print("Entering updateSensorsCList()\n");
	#endif

	/* So we don't overrite the wrong pointer :) */
	temp = list;

	/* Check if this list is empty */
	if(temp == NULL) {
		return;
	}
	
	/* Freeze the CList */
	gtk_clist_freeze(GTK_CLIST(propertiesCList));

	/* Empty out the list completely */
	gtk_clist_clear(GTK_CLIST(propertiesCList));

	/* Read in the list of sensors to the CList */
	tempSensor = (struct sensor *) temp->data;

	info[0] = tempSensor->displayName;
	info[1] = tempSensor->chipName;
	info[2] = tempSensor->sensorName;
	
	gtk_clist_insert(GTK_CLIST(propertiesCList), i, info);

	while (temp->next != NULL) {
		temp = g_list_next(temp);
		i++;

		tempSensor = (struct sensor *) temp->data;

		info[0] = tempSensor->displayName;
		info[1] = tempSensor->chipName;
		info[2] = tempSensor->sensorName;

		gtk_clist_insert(GTK_CLIST(propertiesCList), i, info);
	}

	/* Thaw the CList */
	gtk_clist_thaw(GTK_CLIST(propertiesCList));
}

void modifySensorDialog(GtkWidget *widget, gpointer data) {
	GtkWidget *window, *chipName, *sensorName, *displayName;
	GtkWidget *vbox, *hbox1, *hbox2, *hbox3, *hbox4;
	GtkWidget *OKbutton, *frame, *cancelButton;
	GtkWidget *displayLabel, *chipLabel, *sensorLabel;
	struct drawAddSensorInfo *drawInfo;
	struct addModifySensorInfo *info;
	struct sensor *tempSensor;
	GList *element;

	#if DEBUG == 1
		g_print("Entering modifySensorDialog()\n");
	#endif

	/* Cast */
	drawInfo = (struct drawAddSensorInfo *) data;

	/* Allocate the space for the struct to pass */
	info = g_malloc(sizeof(struct addModifySensorInfo));

	/* Get the element we want to modify */
	element = g_list_nth(*drawInfo->startList, GTK_CLIST(drawInfo->clist)->focus_row);
	tempSensor = element->data;

	/* Create the window */
	window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_default_size(GTK_WINDOW(window), 200, 200);

	/* Create the frame */
	frame = gtk_frame_new("Modify Sensor");
	gtk_container_add(GTK_CONTAINER(window), frame);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 3);

	/* Create the main VBox */
	vbox = gtk_vbox_new(TRUE, 2);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	/* Create the entries */
	hbox1 = gtk_hbox_new(TRUE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), hbox1, FALSE, FALSE, 0);
	displayLabel = gtk_label_new("Display Name:");
	displayName = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(displayName), tempSensor->displayName);
	gtk_box_pack_start(GTK_BOX(hbox1), displayLabel, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox1), displayName, FALSE, FALSE, 0);

	hbox2 = gtk_hbox_new(TRUE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), hbox2, FALSE, FALSE, 0);
	chipLabel = gtk_label_new("Chip Name:");
	chipName = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(chipName), tempSensor->chipName);
	gtk_box_pack_start(GTK_BOX(hbox2), chipLabel, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox2), chipName, FALSE, FALSE, 0);

	hbox3 = gtk_hbox_new(TRUE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), hbox3, FALSE, FALSE, 0);
	sensorLabel = gtk_label_new("Sensor Name:");
	sensorName = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(sensorName), tempSensor->sensorName);
	gtk_box_pack_start(GTK_BOX(hbox3), sensorLabel, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox3), sensorName, FALSE, FALSE, 0);

	/* Setup the struct for passing to the callback */
	info->displayName = displayName;
	info->chipName = chipName;
	info->sensorName = sensorName;
	info->clist = drawInfo->clist;
	info->window = window;
	info->propertiesBox = drawInfo->propertiesBox;
	info->position = GTK_CLIST(drawInfo->clist)->focus_row;
	info->startList = drawInfo->startList;

	/* Create the buttons */
	OKbutton = gtk_button_new_with_label("Modify It!");
	cancelButton = gtk_button_new_with_label("Cancel");
	gtk_signal_connect(GTK_OBJECT(OKbutton), "clicked", GTK_SIGNAL_FUNC(modifySensor), info);
	gtk_signal_connect(GTK_OBJECT(cancelButton), "clicked", GTK_SIGNAL_FUNC(windowDestroy), window);
	hbox4 = gtk_hbox_new(TRUE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), hbox4, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox4), OKbutton, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox4), cancelButton, FALSE, FALSE, 0);

	/* Show the widgets */
	gtk_widget_show(OKbutton);
	gtk_widget_show(cancelButton);
	gtk_widget_show(displayName);
	gtk_widget_show(sensorName);
	gtk_widget_show(chipName);
	gtk_widget_show(displayLabel);
	gtk_widget_show(sensorLabel);
	gtk_widget_show(chipLabel);
	gtk_widget_show(hbox1);
	gtk_widget_show(hbox2);
	gtk_widget_show(hbox3);
	gtk_widget_show(hbox4);
	gtk_widget_show(vbox);
	gtk_widget_show(frame);
	gtk_widget_show(window);
}

void modifySensor(GtkWidget *widget, gpointer data) {
	struct addModifySensorInfo *info;
	struct sensor *tempSensor;
	gchar *tempString;
	GList *element;

	#if DEBUG == 1
		g_print("modifySensor() - called\n");
	#endif

	/* Casting */
	info = (struct addModifySensorInfo *) data;
	/*list = *info->startList;*/

	#if DEBUG == 1
		showSensorList("---Start modifySensor ---", *info->startList);
	#endif

	element = g_list_nth(*info->startList, info->position);
	tempSensor = element->data;

	/* Free the old strings */
	g_free(tempSensor->displayName);
	g_free(tempSensor->chipName);
	g_free(tempSensor->sensorName);

	/* Get the values */
	tempString = gtk_entry_get_text(GTK_ENTRY(info->displayName));
	tempSensor->displayName = g_malloc(strlen(tempString) + 1);
	memcpy(tempSensor->displayName, tempString, strlen(tempString) + 1);

	tempString = gtk_entry_get_text(GTK_ENTRY(info->chipName));
	tempSensor->chipName = g_malloc(strlen(tempString) + 1);
	memcpy(tempSensor->chipName, tempString, strlen(tempString) + 1);

	tempString = gtk_entry_get_text(GTK_ENTRY(info->sensorName));
	tempSensor->sensorName = g_malloc(strlen(tempString) + 1);
	memcpy(tempSensor->sensorName, tempString, strlen(tempString) + 1);

	#if DEBUG == 1
		showSensorList("---- End startSensor----", *info->startList);
	#endif

	/* Set the Properties box status to changed. This activates the "Apply" button */
	gnome_property_box_changed(GNOME_PROPERTY_BOX(info->propertiesBox));

	/* Update the CList with the new list */
	updateSensorsCList(*info->startList, info->clist);

	/* OK now kill the window */
	gtk_widget_destroy(info->window);

	/* Free the structure that was passed to us */
	g_free(info);
}

void addSensorDialog(GtkWidget *widget, gpointer data) {
	GtkWidget *window, *chipName, *sensorName, *displayName;
	GtkWidget *vbox, *hbox1, *hbox2, *hbox3, *hbox4;
	GtkWidget *OKbutton, *frame, *cancelButton;
	GtkWidget *displayLabel, *chipLabel, *sensorLabel;
	struct drawAddSensorInfo *drawInfo;
	struct addModifySensorInfo *info;

	#if DEBUG == 1
		g_print("Entering addSensorDialog()\n");
	#endif

	/* Cast */
	drawInfo = (struct drawAddSensorInfo *) data;

	/* Allocate the space for the struct */
	info = g_malloc(sizeof(struct addModifySensorInfo));

	/* Create the window */
	window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_default_size(GTK_WINDOW(window), 200, 200);

	/* Create the frame */
	frame = gtk_frame_new("Add Sensor");
	gtk_container_add(GTK_CONTAINER(window), frame);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 3);

	/* Create the main VBox */
	vbox = gtk_vbox_new(TRUE, 2);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	/* Create the entries */
	hbox1 = gtk_hbox_new(TRUE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), hbox1, FALSE, FALSE, 0);
	displayLabel = gtk_label_new("Display Name:");
	displayName = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox1), displayLabel, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox1), displayName, FALSE, FALSE, 0);

	hbox2 = gtk_hbox_new(TRUE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), hbox2, FALSE, FALSE, 0);
	chipLabel = gtk_label_new("Chip Name:");
	chipName = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox2), chipLabel, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox2), chipName, FALSE, FALSE, 0);

	hbox3 = gtk_hbox_new(TRUE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), hbox3, FALSE, FALSE, 0);
	sensorLabel = gtk_label_new("Sensor Name:");
	sensorName = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox3), sensorLabel, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox3), sensorName, FALSE, FALSE, 0);

	/* Setup the struct for passing to the callback */
	info->displayName = displayName;
	info->chipName = chipName;
	info->sensorName = sensorName;
	info->clist = drawInfo->clist;
	info->window = window;
	info->propertiesBox = drawInfo->propertiesBox;
	info->startList = drawInfo->startList;

	/* Create the buttons */
	OKbutton = gtk_button_new_with_label("Add It!");
	cancelButton = gtk_button_new_with_label("Cancel");
	gtk_signal_connect(GTK_OBJECT(OKbutton), "clicked", GTK_SIGNAL_FUNC(addSensor), info);
	gtk_signal_connect(GTK_OBJECT(cancelButton), "clicked", GTK_SIGNAL_FUNC(windowDestroy), window);
	hbox4 = gtk_hbox_new(TRUE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), hbox4, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox4), OKbutton, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox4), cancelButton, FALSE, FALSE, 0);

	/* Show the widgets */
	gtk_widget_show(OKbutton);
	gtk_widget_show(cancelButton);
	gtk_widget_show(displayName);
	gtk_widget_show(sensorName);
	gtk_widget_show(chipName);
	gtk_widget_show(displayLabel);
	gtk_widget_show(sensorLabel);
	gtk_widget_show(chipLabel);
	gtk_widget_show(hbox1);
	gtk_widget_show(hbox2);
	gtk_widget_show(hbox3);
	gtk_widget_show(hbox4);
	gtk_widget_show(vbox);
	gtk_widget_show(frame);
	gtk_widget_show(window);
}

void addSensor(GtkWidget *widget, gpointer data) {
	struct addModifySensorInfo *info;
	struct sensor *temp;
	gchar *tempString;

	/* Casting */
	info = (struct addModifySensorInfo *) data;
	/*list = *info->startList;*/

	#if DEBUG == 1
		showSensorList("---Start addSensor ---", *info->startList);
	#endif

	/* Allocate */
	temp = g_malloc(sizeof(struct sensor));

	/* Get the values */
	tempString = gtk_entry_get_text(GTK_ENTRY(info->displayName));
	temp->displayName = g_malloc(strlen(tempString) + 1);
	memcpy(temp->displayName, tempString, strlen(tempString) + 1);

	tempString = gtk_entry_get_text(GTK_ENTRY(info->chipName));
	temp->chipName = g_malloc(strlen(tempString) + 1);
	memcpy(temp->chipName, tempString, strlen(tempString) + 1);

	tempString = gtk_entry_get_text(GTK_ENTRY(info->sensorName));
	temp->sensorName = g_malloc(strlen(tempString) + 1);
	memcpy(temp->sensorName, tempString, strlen(tempString) + 1);

	/* Append this structure to the list */
	*info->startList = g_list_append(*info->startList, temp);

	#if DEBUG == 1
		showSensorList("---- End addSensor----", *info->startList);
	#endif

	/* Set the Properties box status to changed. This activates the "Apply" button */
	gnome_property_box_changed(GNOME_PROPERTY_BOX(info->propertiesBox));

	/* Update the CList with the new list */
	updateSensorsCList(*info->startList, info->clist);

	/* OK now kill the window */
	gtk_widget_destroy(info->window);

	/* Free the structure that was passed to us */
	g_free(info);
}

void cancelConfig(GtkWidget *widget, gpointer data) {
	/* Takes the main properties window widget and destroys it */
	struct cancelConfigInfo *cancelInfo;

	#if DEBUG == 1
		g_print("Entering cancelConfig()\n");
	#endif

	cancelInfo = (struct cancelConfigInfo *)data;

	/* Free the tai structure that we made when we entered */
	/* the properties() function */
	g_free(cancelInfo->tai);
	g_free(cancelInfo->minfo);
	g_free(cancelInfo->deleteInfo);
	g_free(cancelInfo->drawInfo);

	/* Now free the structure that was passed to us */
	g_free(cancelInfo);

	/* Write the Config */
	gnome_config_drop_all();
}

void writeConfig(GtkWidget *widget, gint page_num, gpointer data) {
	/* Write the config and close the main window */
	struct makeChangesInfo *info;
	gint i;
	GList *temp;
	gchar *string;
	struct sensor *tempSensor;

	#if DEBUG == 1
		g_print("Entering writeConfig()\n");
	#endif

	/* Casting time */
	info = (struct makeChangesInfo *) data;

	/* Check that this is the Global "Apply" signal */
	if(page_num != -1) {
		return;
	}

	/* Remove the gtk_timeout functions */
        gtk_timeout_remove(info->ai->sensorsTimeoutTag);
        gtk_timeout_remove(info->ai->GUITimeoutTag);

	/* Buffer Time */
	string = g_malloc(50);

	/* Go through and clean out all of the old sensor entries */
	temp = info->ai->sensorList;
	for(i=0;temp != NULL; i++) {
		tempSensor = temp->data;
		sprintf(string, "Sensor%i", i);

		gnome_config_clean_section(string);

		temp = g_list_next(temp);
	}
	/* Commit the changes */
	gnome_config_sync();

	/* Write the global updates */
	gnome_config_set_int("Global/height", info->tai->height);
	gnome_config_set_int("Global/width", info->tai->width);
	gnome_config_set_int("Global/sensorFreq", info->tai->sensorFreq);
	gnome_config_set_int("Global/displayFreq", info->tai->displayFreq);

	/* Update the sensors data */
	temp = *info->tai->sensorList;
	for(i=0;temp != NULL;i++) {
		tempSensor = temp->data;

		/* Display Name */
		sprintf(string, "Sensor%i/displayName", i);
		gnome_config_set_string(string, tempSensor->displayName);

		/* Chip Name */
		sprintf(string, "Sensor%i/chipName", i);
		gnome_config_set_string(string, tempSensor->chipName);

		/* Sensor Name */
		sprintf(string, "Sensor%i/sensorName", i);
		gnome_config_set_string(string, tempSensor->sensorName);

		temp = g_list_next(temp);
	}

	/* Update Number_Of_Sensors */
	gnome_config_set_int("Global/Number_Of_Sensors", i);

	/* Write the data */
	gnome_config_sync();

	/* Delete the old sensors Lists */
	temp = info->ai->sensorList;
	while(temp != NULL) {
		tempSensor = (struct sensor *)temp->data;
		g_free(tempSensor->displayName);
		g_free(tempSensor->chipName);
		g_free(tempSensor->sensorName);
		g_free(tempSensor->nameStruct.prefix);
		g_free(tempSensor);

		temp = g_list_next(temp);
	}
	g_list_free(info->ai->sensorList);
	/* We do not have to free the data in this GList because */
	/* it shared the data pointers with the above GList */
	g_list_free(*info->tai->sensorList);

	/* Update the applet now */
	makeChanges(data);

	/* Update the tai structure. */
	initTAI(info->ai, info->tai);

	/* Force an update on the sensors and the GUI */
	updateSensors(info->ai);
	updateGUI(info->ai);

	/* Re-instate the timeout functions with the new freq */
	info->ai->sensorsTimeoutTag = gtk_timeout_add(info->ai->sensorFreq, updateSensors, info->ai);
	info->ai->GUITimeoutTag = gtk_timeout_add(info->ai->displayFreq, updateGUI, info->ai);

	/* Free our string */
	g_free(string);
}

gint deleteSensor(GtkWidget *widget, gpointer data) {
	struct deleteSensorInfo *info;
	GList *previous, *element, *next;
	struct tempAppletInfo *tai;
	gpointer *delete;

	#if DEBUG == 1
		g_print("Entering deleteSensor()\n");
	#endif

	/* A little casting */
	/* clist = (GtkWidget *) data;*/
	info = (struct deleteSensorInfo *) data;
	tai = (struct tempAppletInfo *) info->tai;

	if(*info->tai->sensorList == NULL) {
		#if DEBUG == 1
			g_print("deleteSensor() Empty Sensor List\n");
		#endif

		return(0);
	}

	#if DEBUG == 1
		showSensorList("Delete Sensor Start", *tai->sensorList);
	#endif

	/* Get the element we want to delete */
	element = g_list_nth(*tai->sensorList, GTK_CLIST(info->clist)->focus_row);
	previous = g_list_previous(element);
	next = g_list_next(element);

	/* Get the address of the sensor structure */
	delete = element->data;

	/* Remove the element */
	*tai->sensorList = g_list_remove(*tai->sensorList, delete);

	/* Update the GList */
	if(previous != NULL) {
		/* If this is the first GList entry this is not necessary */
		previous->next = next;
	}

	/* Remove the entry from the sensors CList */
	gtk_clist_clear(GTK_CLIST(info->clist));
	updateSensorsCList(*tai->sensorList, info->clist);

	/* Send the changed signal */
	gnome_property_box_changed(GNOME_PROPERTY_BOX(info->propertiesBox));

	#if DEBUG == 1
		showSensorList("Delete Sensor End", *tai->sensorList);
	#endif

	return(0);
}

void propHeightChange(GtkWidget *widget, gpointer data) {
	gchar *entryText;
	struct propDisplayInfo *disInfo;
	#if DEBUG == 1
		g_print("Entering propHeightChange()\n");
	#endif

	/* Casting time... */
	disInfo = (struct propDisplayInfo *) data;

	entryText = gtk_entry_get_text(GTK_ENTRY(widget));

	disInfo->tai->height = atoi(entryText);

	gnome_property_box_changed(GNOME_PROPERTY_BOX(disInfo->propertiesBox));
}

void propWidthChange(GtkWidget *widget, gpointer data) {
        gchar *entryText;
	struct propDisplayInfo *disInfo;

	#if DEBUG == 1
		g_print("Entering propWidthChange()\n");
	#endif

	/* Casting time... */
	disInfo = (struct propDisplayInfo *) data;

        entryText = gtk_entry_get_text(GTK_ENTRY(widget));

	disInfo->tai->width = atoi(entryText);

        gnome_property_box_changed(GNOME_PROPERTY_BOX(disInfo->propertiesBox));
}

void propSensorFreqChange(GtkWidget *widget, gpointer data) {
        gchar *entryText;
	struct propDisplayInfo *disInfo;

	#if DEBUG == 1
		g_print("Entering propSensorFreqChange()\n");
	#endif

	/* Casting time ... */
	disInfo = (struct propDisplayInfo *) data;

        entryText = gtk_entry_get_text(GTK_ENTRY(widget));

	disInfo->tai->sensorFreq = atoi(entryText);

        gnome_property_box_changed(GNOME_PROPERTY_BOX(disInfo->propertiesBox));
}

void propDisplayFreqChange(GtkWidget *widget, gpointer data) {
        gchar *entryText;
        struct propDisplayInfo *disInfo;

        #if DEBUG == 1
                g_print("Entering propDisplayFreqChange()\n");
        #endif

        /* Casting time ... */
        disInfo = (struct propDisplayInfo *) data;

        entryText = gtk_entry_get_text(GTK_ENTRY(widget));

        disInfo->tai->displayFreq = atoi(entryText);

        gnome_property_box_changed(GNOME_PROPERTY_BOX(disInfo->propertiesBox));
}

void makeChanges(gpointer *data) {
	/* Make the changes to the applet */
	struct makeChangesInfo *info;
	
	#if DEBUG == 1
		g_print("Entering makeChanges()\n");
	#endif

	info = (struct makeChangesInfo *) data;

	/* Get the new values */
	info->ai->height = info->tai->height;
	info->ai->width = info->tai->width;
	info->ai->sensorFreq = info->tai->sensorFreq;
	info->ai->displayFreq = info->tai->displayFreq;

	/* Update the Applet size */
	if(info->ai->orient == ORIENT_UP || info->ai->orient == ORIENT_DOWN) {
		gtk_widget_set_usize(GTK_WIDGET(info->ai->applet), info->ai->width, -1);
	} else {
		gtk_widget_set_usize(GTK_WIDGET(info->ai->frame), info->ai->panelSize, info->ai->height);
	}

	/* Update the tooltip width */
	gtk_widget_set_usize(GTK_WIDGET(info->ai->tooltip), info->ai->width,
		0);

	/* Update the sensors List */
	info->ai->sensorList = setupSensors();

	/* Reset the sensor position number */
	info->ai->displayPos = 0;
}

void windowDestroy(GtkWidget *widget, gpointer *data) {
	gtk_widget_destroy(GTK_WIDGET(data));
}

gint initTAI(struct appletInfo *ai, struct tempAppletInfo *tai) {
	/* Function to setup the tai structure */

	tai->height = ai->height;
        tai->width = ai->width;
        tai->sensorFreq = ai->sensorFreq;
	tai->displayFreq = ai->displayFreq;
        tai->temp = g_list_copy(ai->sensorList);

        /*tai->sensorList = g_list_copy(ai->sensorList);*/
        tai->sensorList = &tai->temp;

	return(0);
}

#if DEBUG == 1
/********************************************************/
/* Debuging fuctions */

void showSensorList(gchar *label, GList *list) {
	/* Little utility function to help debug */
	/* Should not been seen in production code */
	GList *tmp;
	struct sensor *sensor;
	gchar *string;

	tmp = list;
	string = g_strdup_printf("------------%s-------------List\n", label);
	g_print("%s",string);
	while(tmp != NULL)
	{
		g_print("List Address: %p\n", tmp);
		g_print("List Element: %p\n", tmp->data);
		sensor = (struct sensor *)tmp->data;
		g_print("Element->DisplayName: %s\n", sensor->displayName);
		g_print("Element->chipName: %s\n", sensor->chipName);
		g_print("Element->sensorName: %s\n", sensor->sensorName);
		g_print("List Previous: %p\n", tmp->prev);
		g_print("List Next: %p\n", tmp->next);
		tmp = g_list_next(tmp);
		g_print("-----------Next-------------\n");
	}
	g_print("-----------End----------------List\n");

	g_free(string);
}
#endif
