/* child.c  --  The child of sigit, eventualy called with a fifo pointer, 
                  which it is monitoring.  

   Copyright (C) Kenneth 'Redhead' Nielsen <kn@redhead.dk>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */



/****
 * $Id: child.c,v 1.3 2003/07/22 22:03:42 redhead Exp $ 
 * $Log: child.c,v $
 * Revision 1.3  2003/07/22 22:03:42  redhead
 * Cleaned up the code, which included rewriting every structure in the program.
 * Fixed some error checking, to make sure theres no undocumentet parts that might
 * produce an error.
 *
 * Revision 1.2  2002/10/18 18:25:23  redhead
 * Added the use of boxes to create a surrounding box around the signature
 * This introduced the --with-boxes flag to configure.
 * If that flag isn't used it will default be assumed boxes isn't installed
 * Else the configure script will search for boxes, if none should be found
 * in the search path, you can use --with-boxes=/where to specify it.
 *
 ****/



#include "child.h"

#ifdef sigit

/* 
 * This function will monitor any fifo, given the fact that the child will only
 * be run from the thread which initiated it, in this case it will open the fifo 
 * for writing, and monitor it for any opening from the other end. 
 *
 * It might give a few problems, when fetching the signatures from the data file,
 * But lets hope no one is requesting a signature from two different fifo's at the same time.
 *
 * Any temporary problems with two threads fetching from the data file at the same time
 * has been solved by locking the mutex while another thread is fetching signatures.
 */

void* monitor_fifo(void* ptr) 
{

  FIFO* fifo = (FIFO*)ptr;
  init_fifo(fifo); /* this should init the debug_fifo and the sig_fifo, if we needs them */

  while(1)
  {
    if(SUCCESS == get_sig(fifo))
      pretty_print(fifo);
    else
    {
#ifdef I__HAVE__STRNCPY
      strncpy(fifo->sig.sig, DEFAULT_SIG, NUM_LINES*LINE_SIZE);
      strncpy(fifo->sig.ident, "DEFAULT", HEXCOUNT);
#else
      strcpy(fifo->sig.sig, DEFAULT_SIG);
      strcpy(fifo->sig.ident, "DEFAULT");
#endif
      pretty_print(fifo);
    }
    sleep(1);
  }
  unlink_fifo(fifo);
  return SUCCESS;
}


/*
 *  Using stat() on the fifo files, and see if the st_atime has changed.
 *  Dont know if the access time changes on a NFS mounted system, but if it does 
 *  it's no good. Else that might be the solution, in order to see when
 *  the fifo file was last emptied..
 *
 * Ie:
 *
 * unsigned long last_st_atime = 0;
 * struct stat buf;
 * stat(fifo->fifo_name, &buf);
 * if(last_st_atime < buf.st_atime)
 * {
 *   // overwrite the current fifo file in question  
 *   // with a new signature.
 *   stat(fifo->fifo_name, &buf);
 *   last_st_atime = buf.st_atime;
 * }
 */

void* monitor_nfs_fifo(void* ptr) 
{

  unsigned long last_st_atime = 0;
  struct stat buf;
  FIFO* fifo = (FIFO*)ptr;
  
  goto first_time;
  
  while(1)
    {
      stat(fifo->fifo_name, &buf);
      if(last_st_atime < buf.st_atime)
	{
	first_time:
	  unlink(fifo->fifo_name);
	  if(SUCCESS == get_sig(fifo))
	    pretty_print(fifo);
	  else
	      {
#ifdef I__HAVE__STRNCPY
		strncpy(fifo->sig.sig, DEFAULT_SIG, NUM_LINES*LINE_SIZE);
		strncpy(fifo->sig.ident, "DEFAULT", HEXCOUNT);
#else
		strcpy(fifo->sig.sig, DEFAULT_SIG);
		strcpy(fifo->sig.ident, "DEFAULT");
#endif
		pretty_print(fifo);
	      }
	  stat(fifo->fifo_name, &buf);
	  last_st_atime = buf.st_atime;
	}
      sleep(fifo->nfs_delay);
    }
  return SUCCESS;
}

#endif
