/* -*- c++ -*-
 * __BEGIN_COPYRIGHT
 * SimpleDB API
 * 
 * Copyright (C) 2005 Eminence Technology Pty Ltd
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You can view the GNU Lesser General Public Licence at
 * http://www.gnu.org/licenses/lgpl.html or you can write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Eminence Technology Pty Ltd can be contacted by writing to
 * Eminence Technology, PO Box 118, Moorooka QLD 4105, Australia.
 * Alternatively, you may email opensource [at] eminence [dot] com [dot] au
 * __END_COPYRIGHT
 */

#ifndef __SIMPLEDB_QUERY_H_
#define __SIMPLEDB_QUERY_H_ 1

#include <sql.h>
#include <sqltypes.h>
#include <string>
#include "Column.h"
#include "Exception.h"

namespace SimpleDB {

  /** Class to represent a database query.
   * This class is created by the Database object.
   * It encapsulates details pertaining to a database query.
   */
  class Query {
  
  public:
  
    /** Query exception class.
     * Thrown whenever there is an error condition.
     */
    class Exception : public SimpleDB::Exception {
    public:
      Exception(const std::string& message) : SimpleDB::Exception(message) {};
    };
  
    /** Constructor called by the Database object.
     *
     * @param connectionHandle The connection handle from a database object.
     */ 
    Query(SQLHDBC connectionHandle);
  
    /** Destructor deallocates any memory or resources.
     */
    ~Query();
  
    /** Bind the given columns to this query context.
     *
     * @param columns An array of columns to bind to this query.
     * @param numberColumns The number of columns in the array.
     */
    void bind(Column* columns[], int numberColumns);

		     
    /** Execute the given query.
     *
     * @param sqlQuery The SQL query.
     * \exception Database::NoDataException if the return from the SQL
     * query is SQL_NO_DATA
     * \exception Exception When there is a problem with the database
     * or with the query itself.
     */
    void execute(const std::string& sqlQuery);
  
    /** Fetches a row of data returned by the SQL query.
     *
     * @return Returns false if there is no data to be returned.
     */
    bool fetchRow();
    
  protected:
  
    /** ODBC statement handle.
     */
    SQLHSTMT statementHandle;

    /** ODBC connection handle.
     */
    SQLHDBC connectionHandle;
  };

}

#endif
