/*
**  Sinek (Video Player)
**  Copyright (c) 2001 - 2002 the Sinek Team, see the AUTHORS file.
**
**  This code is free software; you can redistribute it and/or
**  modify it under the terms of the GNU General Public License.
**
**  video window
*/

#include "common.h"

#include <gdk/gdkx.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <X11/extensions/XShm.h>
#include <X11/Xatom.h>
#include <xine/video_out_x11.h>

/* missing stuff from X includes */
#ifndef XShmGetEventBase
extern int XShmGetEventBase(Display *);
#endif
#define MWM_HINTS_DECORATIONS (1L << 1)
#define PROP_MWM_HINTS_ELEMENTS 5
typedef struct _mwmhints
{
	uint32_t flags;
	uint32_t functions;
	uint32_t decorations;
	int32_t input_mode;
	uint32_t status;
} MWMHints;

static int video_visible = 42;
static char **driver_ids;
static XClassHint *xc_hint;
static XWMHints *wm_hint;
static XSizeHints xs_hint;
static Atom wm_delete_window;
static int depth;
static Visual *visual;
static Window videowin;
static Display *display;
static int screen;
static XColor black;
static Colormap colormap = 0;
static int MyShmEvent = -1;
static int xfred;
static Cursor cursor[2];

static void calc_dest_size (void *this, int vid_w, int vid_h, int *dest_w, int *dest_h);
static void adapt_size(void *this, int vid_w, int vid_h, int *dest_x, int *dest_y, int *dest_w, int *dest_h);
static void find_best_visual(void);
static int load_codecs(void);
static int translate_point(int x, int y, int *vid_x, int *vid_y);
static void create_window(int fs, int w, int h);

static gboolean handle_event(GIOChannel *gio, GIOCondition cond, gpointer data);
GIOChannel *lala;


int video_init(void)
{
	static unsigned char bm_no_data[] = { 0, 0, 0, 0, 0, 0, 0, 0 };
	Pixmap bm_no;
	int x, y, w, h;
	XColor dummy;

	XInitThreads();

	display = XOpenDisplay(gdk_display_name);
	sinek.display = display;
	xfred = ConnectionNumber(display);
	screen = gdk_screen;
	find_best_visual();
	XLockDisplay(display);

	XAllocNamedColor(display, DefaultColormap(display, screen), "black", &black, &dummy);

	sinek.fs_w = DisplayWidth(display, screen);
	sinek.fs_h = DisplayHeight(display, screen);

	xc_hint = XAllocClassHint();
	if(xc_hint)
	{
		xc_hint->res_name = "Sinek Video Window";
		xc_hint->res_class = "Sinek";
	}
	wm_hint = XAllocWMHints();
	if(!wm_hint)
	{
		printf(_("XAllocWMHints failed.\n"));
		return 0;
	}

	wm_delete_window = XInternAtom(display, "WM_DELETE_WINDOW", False);

	if(XShmQueryExtension(display) == True)
		MyShmEvent = XShmGetEventBase(display) + ShmCompletion;

	XUnlockDisplay(display);

	lala = g_io_channel_unix_new(xfred);
	g_io_add_watch(lala, G_IO_IN, handle_event, NULL);

	bm_no = XCreateBitmapFromData(display, RootWindow(display, screen), bm_no_data, 8, 8);
	cursor[0] = XCreatePixmapCursor(display, bm_no, bm_no, &black, &black, 0, 0);
	cursor[1] = XCreateFontCursor(display, XC_left_ptr);

	adapt_size(NULL, 200, 200, &x, &y, &w, &h);
	if(!load_codecs()) return(0);

	return 1;
}


void video_show(void)
{
	XMapWindow(display, videowin);
	XFlush(display);
	if(video_visible != 1) sinek.nr_mapped++;
	video_visible = 1;
}


void video_hide(void)
{
	XUnmapWindow(display, videowin);
	XFlush(display);
	if(video_visible == 1) sinek.nr_mapped--;
	video_visible = 0;
	if(0 == sinek.nr_mapped) gtk_main_quit();
}


void video_cursor(int show)
{
	static int old = 2;

	if(show == old) return;
	old = show;
	if(show == 1) sinek.cursor_timer = 0;
	XLockDisplay(display);
	XDefineCursor(display, videowin, cursor[show]);
	XFlush(display);
	XUnlockDisplay(display);
}


int video_play(char *mrl)
{
	if(!mrl) return(0);
	if(sinek.playing)
	{
		sinek.playing = 0;
		xine_stop(sinek.xine);
	}
	if(video_visible == 42) video_show();
	sinek.mrl = strdup(mrl);
	if(xine_play(sinek.xine, sinek.mrl, 0, 0))
	{
		disable_blanking();
		control_total_time(xine_get_stream_length(sinek.xine));
		control_mrl(sinek.mrl);
		sinek.playing = -1;
		return 1;
	}
	return 0;
}


void video_pause(void)
{
	xine_set_speed(sinek.xine, SPEED_PAUSE);
	sinek.pause = -1;
}


void video_seek(int secs)
{
	disable_blanking();
	if(xine_is_stream_seekable(sinek.xine))
		xine_play(sinek.xine, sinek.mrl, 0, secs);
}


void video_zoom(int fx, int fy)
{
	int tmp;

	if(fx == 0 && fy == 0)
	{
		/* reset zoom */
		tmp = sinek.vd->get_property(sinek.vd, VO_PROP_ASPECT_RATIO);
		sinek.vd->set_property(sinek.vd, VO_PROP_ASPECT_RATIO, tmp);
		return;
	}
	if(fx)
	{
		tmp = sinek.vd->get_property(sinek.vd, VO_PROP_ZOOM_X);
		sinek.vd->set_property(sinek.vd, VO_PROP_ZOOM_X, tmp + fx);
	}
	if(fy)
	{
		tmp = sinek.vd->get_property(sinek.vd, VO_PROP_ZOOM_Y);
		sinek.vd->set_property(sinek.vd, VO_PROP_ZOOM_Y, tmp + fy);
	}
}


void video_scale(float factor)
{
	int x, y;
	unsigned int w, h, b, d;
	Window root;

	if(sinek.fullscreen) return;

	XLockDisplay(display);
	XGetGeometry(display, videowin, &root, &x, &y, &w, &h, &b, &d);
	w *= factor;
	h *= factor;
	XResizeWindow(display, videowin, w, h);
	XUnlockDisplay(display);
}


void video_toggle_fullscreen(void)
{
	x11_rectangle_t t;

	sinek.fs_req = 1;
	adapt_size(NULL, sinek.vid_w, sinek.vid_h, &t.x, &t.y, &t.w, &t.h);
	sinek.vd->gui_data_exchange(sinek.vd, GUI_DATA_EX_DEST_POS_SIZE_CHANGED, &t);
	video_show();
	wm_relayer_all();
}


static gboolean handle_event(GIOChannel *gio, GIOCondition cond, gpointer data)
{
	XEvent xev;
	xine_input_event_t xinev;
	int x, y;

	XNextEvent(display, &xev);
	if(videowin == 0) return TRUE;

	switch(xev.type)
	{
		case MotionNotify:
			video_cursor(1);
			if(translate_point(xev.xmotion.x, xev.xmotion.y, &x, &y))
			{
				if(sinek.osd_place)
				{
					osd_position(10, y);
				}
				xinev.event.type = XINE_EVENT_MOUSE_MOVE;
				xinev.button = 0;
				xinev.x = x;
				xinev.y = y;
				xine_send_event(sinek.xine, (xine_event_t *)&xinev);
			}
			break;

		case ButtonPress:
			video_cursor(1);
			sinek.osd_place = 0;
			switch(xev.xbutton.button)
			{
				case 1:
					if(translate_point(xev.xbutton.x, xev.xbutton.y, &x, &y))
					{
						xinev.event.type = XINE_EVENT_MOUSE_BUTTON;
						xinev.button = 1;
						xinev.x = x;
						xinev.y = y;
						xine_send_event(sinek.xine, (xine_event_t *)&xinev);
					}
					break;
				case 3:
					popup_pop(xev.xbutton.time);
					break;
				case 4:
					audio_slide_volume(5);
					break;
				case 5:
					audio_slide_volume(-5);
					break;
			}
			break;

		case Expose:
		{
			XExposeEvent *ev = (XExposeEvent *)&xev;
			if(ev->count == 0 && ev->window == videowin)
			{
				sinek.vd->gui_data_exchange(sinek.vd, GUI_DATA_EX_EXPOSE_EVENT, &xev);
			}
			break;
		}

		case ConfigureNotify:
		{
			XConfigureEvent *ev = (XConfigureEvent *)&xev;
			x11_rectangle_t area;
			area.x = 0;
			area.y = 0;
			area.w = ev->width;
			area.h = ev->height;
			sinek.vd->gui_data_exchange(sinek.vd, GUI_DATA_EX_DEST_POS_SIZE_CHANGED, &area);
			break;
		}

		case KeyPress:
		{
			XKeyEvent *ev = (XKeyEvent *)&xev;
			KeySym key;
			char buf[256];
			int len;

//			XLockDisplay(display);
			len = XLookupString(ev, buf, sizeof(buf), &key, NULL);
//			XUnlockDisplay(display);
			key_handle(key, ev->state);

			break;
		}

		case ClientMessage:
			if(xev.xclient.data.l[0] == wm_delete_window)
				video_hide();
			break;
	}

	if(xev.type == MyShmEvent)
		sinek.vd->gui_data_exchange(sinek.vd, GUI_DATA_EX_COMPLETION_EVENT, &xev);

	return TRUE;
}


static void adapt_size(void *this, int vid_w, int vid_h, int *dest_x, int *dest_y, int *dest_w, int *dest_h)
{
	Window old = None;
	int fs = 0, w = vid_w, h = vid_h;

	sinek.vid_w = vid_w;
	sinek.vid_h = vid_h;

	XLockDisplay(display);

	if((sinek.fs_req && sinek.fullscreen == 0) || (sinek.fs_req == 0 && sinek.fullscreen))
	{
		fs = 1;
		w = sinek.fs_w;
		h = sinek.fs_h;
	}

	if(videowin == None || sinek.fs_req)
	{
		old = videowin;
		create_window(fs, w, h);
		if(old != None)
		{
			XDestroyWindow(display, old);
			XFlush(display);
		}
	}
	else
	{
		xs_hint.x = 0;
		xs_hint.y = 0;
		xs_hint.width = w;
		xs_hint.height = h;
		xs_hint.flags = PPosition | PSize;
		XResizeWindow(display, videowin, w, h);
		XSetNormalHints(display, videowin, &xs_hint);
	}

	sinek.fs_req = 0;
	sinek.fullscreen = fs;
	*dest_x = 0;
	*dest_y = 0;
	*dest_w = w;
	*dest_h = h;

	XUnlockDisplay(display);
}


static void calc_dest_size (void *this, int vid_w, int vid_h, int *dest_w, int *dest_h)
{
	if(sinek.fullscreen)
	{
		*dest_w = sinek.fs_w;
		*dest_h = sinek.fs_h;
	}
	else
	{
		*dest_w  = vid_w;
		*dest_h = vid_h;
	}
}


static int load_codecs(void)
{
	char *id;
	double res_h, res_v;
	x11_visual_t vis;

	vis.display = display;
	vis.screen = screen;
	vis.d = videowin;
	res_h = (DisplayWidth(display, screen)*1000 / DisplayWidthMM(display, screen));
	res_v = (DisplayHeight(display, screen)*1000 / DisplayHeightMM(display, screen));
	vis.display_ratio = res_h / res_v;
	vis.calc_dest_size = calc_dest_size;
	vis.request_dest_size = adapt_size;
	vis.user_data = NULL;

	if(fabs(vis.display_ratio - 1.0) < 0.01) vis.display_ratio = 1.0;

	driver_ids = xine_list_video_output_plugins(VISUAL_TYPE_X11);
	id = sinek.conf->register_string(sinek.conf, "video.driver", "auto", "video driver to use", NULL, NULL, NULL);
	if(sinek.video_id) id = sinek.video_id;

	/* try user preferred driver */
	if(strncmp(id, "auto", 4) != 0)
		sinek.vd = xine_load_video_output_plugin(sinek.conf, id, VISUAL_TYPE_X11, (void *)&vis);
	/* if that fails or user choosed auto, try any available driver */
	if(!sinek.vd)
	{
		int i = 0;
		while(driver_ids[i])
		{
			sinek.vd = xine_load_video_output_plugin(sinek.conf, driver_ids[i], VISUAL_TYPE_X11, (void *)&vis);
			if(sinek.vd) break;
			i++;
		}
	}
	/* if that fails too, exit with an error */
	if(!sinek.vd)
	{
		printf(_("Cannot load video driver.\n"));
		return 0;
	}

	sinek.conf->update_string(sinek.conf, "video.driver", id);

	return 1;
}


static void find_best_visual(void)
{
	XVisualInfo *vi, tmpl;
	int num, i, prf, best_prf = -1;

	tmpl.screen = screen;
	tmpl.class = TrueColor;

	vi = XGetVisualInfo(display, VisualScreenMask | VisualClassMask, &tmpl, &num);
	if(vi)
	{
		for(i = 0; i < num; i++)
		{
			if(vi[i].depth > 8 && vi[i].depth <= 16)
				prf = 3;
			else if(vi[i].depth > 16)
				prf = 2;
			else
				prf = 1;

			if(prf > best_prf)
			{
				depth = vi[i].depth;
				visual = vi[i].visual;
			}
		}
		XFree(vi);
	}

	if(best_prf == -1)
	{
		XWindowAttributes attr;
		XVisualInfo vinfo;

		XGetWindowAttributes(display, RootWindow(display, screen), &attr);
		depth = attr.depth;
		if(XMatchVisualInfo(display, screen, depth, TrueColor, &vinfo))
		{
			visual = vinfo.visual;
		} else {
			printf(_("couldn't find true color visual.\n"));
			depth = DefaultDepth(display, screen);
			visual = DefaultVisual(display, screen);
		}
	}
}


static int translate_point(int x, int y, int *vid_x, int *vid_y)
{
	x11_rectangle_t rect;
	int xwin, ywin;
	unsigned int wwin, hwin, bwin, dwin;
	float xf,yf;
	float scale, width_scale, height_scale, aspect;
	Window rootwin;

	rect.x = x;
	rect.y = y;
	rect.w = 0;
	rect.h = 0;

	if(sinek.vd->gui_data_exchange(sinek.vd, GUI_DATA_EX_TRANSLATE_GUI_TO_VIDEO,  (void*)&rect) != -1)
	{
		*vid_x = rect.x;
		*vid_y = rect.y;
		return 1;
	}

	if(XGetGeometry(display, videowin, &rootwin, &xwin, &ywin, &wwin, &hwin, &bwin, &dwin) == BadDrawable)
		return 0;

	/* Scale co-ordinate to image dimensions. */
	height_scale = (float)sinek.vid_h / (float)hwin;
	width_scale = (float)sinek.vid_w / (float)wwin;
	aspect = (float)sinek.vid_w / (float)sinek.vid_h;
	if(((float)wwin / (float)hwin) < aspect)
	{
		scale = width_scale;
		xf = (float)x * scale;
		yf = (float)y * scale;
		hwin = hwin * scale;
		/* FIXME: The 1.25 should really come from the NAV packets. */
		*vid_x = xf * 1.25 / aspect;
		*vid_y = yf - ((hwin - sinek.vid_h) / 2);
	}
	else
	{
		scale = height_scale;
		xf = (float)x * scale;
		yf = (float)y * scale;
		wwin=wwin * scale;
		/* FIXME: The 1.25 should really come from the NAV packets. */
		*vid_x = (xf - ((wwin - sinek.vid_w) / 2)) * 1.25 / aspect;
		*vid_y = yf;
	}

	return 1;
}


static void create_window(int fs, int w, int h)
{
	char *title;
	static Atom  XA_WIN_LAYER = None;
	XSetWindowAttributes attr;
//	XEvent xev;
	Atom prop;
	MWMHints	mwmhints;
	long data[1];

	title = _("Sinek Video Output");
	attr.background_pixel = black.pixel;
	attr.border_pixel = black.pixel;
	attr.colormap = colormap;

	videowin = XCreateWindow(display, RootWindow(display, screen),
		0, 0, w, h,
		0, depth, CopyFromParent, visual,
		CWBackPixel | CWBorderPixel | CWColormap, &attr);

	if(!videowin)
	{
		printf(_("Cannot create video window!\n"));
		exit(5);
	}
	sinek.video_win = videowin;

	if(sinek.vd) sinek.vd->gui_data_exchange(sinek.vd, GUI_DATA_EX_DRAWABLE_CHANGED, (void *)videowin);

	if(!fs && xc_hint) XSetClassHint(display, videowin, xc_hint);

	xs_hint.x = 0;
	xs_hint.y = 0;
	xs_hint.width = w;
	xs_hint.height = h;
	xs_hint.win_gravity = StaticGravity;
	xs_hint.flags = PPosition | PSize;
	if(fs) xs_hint.flags |= PWinGravity;

	XSetStandardProperties(display, videowin, title, title, None, NULL, 0, 0);
	XSetWMNormalHints(display, videowin, &xs_hint);
	XSetWMHints(display, videowin, wm_hint);

	XSelectInput(display, videowin, StructureNotifyMask | ExposureMask | KeyPressMask | ButtonPressMask | PointerMotionMask);
	if(!fs) XSetWMProtocols(display, videowin, &wm_delete_window, 1);

	if(fs)
	{
		if(XA_WIN_LAYER == None)
			XA_WIN_LAYER = XInternAtom(display, "_WIN_LAYER", False);
		data[0] = 10;
		XChangeProperty(display, videowin, XA_WIN_LAYER, XA_CARDINAL, 32, PropModeReplace, (unsigned char *)data, 1);
		prop = XInternAtom(display, "_MOTIF_WM_HINTS", False);
		mwmhints.flags = MWM_HINTS_DECORATIONS;
		mwmhints.decorations = 0;
		XChangeProperty(display, videowin, prop, prop, 32, PropModeReplace, (unsigned char *)&mwmhints, PROP_MWM_HINTS_ELEMENTS);
		XSetTransientForHint(display, videowin, None);
	}

//	XMapRaised(display, videowin);
//	do
//	{
//		XMaskEvent(display, StructureNotifyMask, &xev);
//	} while(xev.type != MapNotify || xev.xmap.event != videowin);
//	XFlush(display);
}


void video_lift(GtkWidget *win)
{
	static Atom XA_WIN_LAYER = None;
	Window w = GDK_WINDOW_XWINDOW(GTK_WIDGET(win)->window);
	XEvent ev;

	if(XA_WIN_LAYER == None)
		XA_WIN_LAYER = XInternAtom(display, "_WIN_LAYER", False);

	ev.type = ClientMessage;
	ev.xclient.type = ClientMessage;
	ev.xclient.window = w;
	ev.xclient.message_type = XA_WIN_LAYER;
	ev.xclient.format = 32;
	if(sinek.fullscreen)
		ev.xclient.data.l[0] = (long) 11;
	else
		ev.xclient.data.l[0] = (long) 4;
	ev.xclient.data.l[1] = 0;

	XSendEvent(gdk_display, RootWindow(gdk_display, screen), False, SubstructureNotifyMask, (XEvent *)&ev);
	XRaiseWindow(gdk_display, w);
}
