#include "sjog.h"
#include "sjog_list.h"
#include "sjog_list_callbacks.h"

/* -------------------------------------------------------------------------- */

/*
 * Call the configuration file parsing and creates the list window
 * (only the window, not the list).
 */
void
sjog_list_init()
{
   /* call the parsing of the config file */
   sjog_init_list_entries();

   listTimer = 0;
   /* create the list window and register callbacks */
   listWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title(GTK_WINDOW(listWindow), "S-Jog");
   gtk_signal_connect(GTK_OBJECT(listWindow), "delete-event",
                      GTK_SIGNAL_FUNC(sjog_list_signal_delete_event_callback),
                      NULL);

}

/* -------------------------------------------------------------------------- */

/*
 * Builds the widgets of the actions list.
 * @arg list the list model.
 */
void
sjog_list_set_list(GList * list)
{
   guint length, i = 0;      /* commands list length and loop index */
   sjog_cmd_t *cmd;          /* temporary pointer for commands insertion loop */
   gchar *item[1];           /* GtkCList needs an array of pointers to gchar */


   /* default is to select 1st item */
   selectedIndex = 0;

   /* creates a 1 column width list */
   cmdListUI = gtk_clist_new(1);
   gtk_clist_set_column_auto_resize(GTK_CLIST(cmdListUI), 0, TRUE);

   /* be warned when a row is selected */
   gtk_signal_connect(GTK_OBJECT(cmdListUI), "select-row",
                      GTK_SIGNAL_FUNC(sjog_list_row_selected_callback), NULL);

   /* get the number of items to put in the list */
   length = g_list_length(list);

   if (length == 0)
   {
      fprintf(stderr, "Error: the list contains no items, exiting\n");
      exit(2);
   }

#ifdef DEBUG
   fprintf(stdout, "list contains %d items\n", length);
#endif

   /* loops over the items and add them to the list */
   for (i = 0; i <= length - 1; i++)
   {
      cmd = (sjog_cmd_t *) g_list_nth_data(list, i);
      item[0] = cmd->name;
      gtk_clist_append(GTK_CLIST(cmdListUI), item);
   }

   /* select the 1st item */
   gtk_clist_select_row(GTK_CLIST(cmdListUI), selectedIndex, 0);

   gtk_container_add(GTK_CONTAINER(listWindow), cmdListUI);

}

/* -------------------------------------------------------------------------- */

/*
 * Shows the list window.
 */
void
sjog_list_show()
{

#ifdef DEBUG
   fprintf(stderr, "sjog_list_show\n");
#endif

   if (listWindow == NULL)
   {
      sjog_list_init();
      sjog_list_set_list(list);
   }

   if (!GTK_WIDGET_VISIBLE(listWindow))
   {
      /* change the mode */
      sjog_set_mode(SJOG_MODE_LIST);

      /* show the window */
      sjog_window_set_position(GTK_WINDOW(listWindow));
      gtk_widget_show_all(listWindow);

      /* reset the window hidding timer */
      sjog_list_reset_timer();
   }
}

/* -------------------------------------------------------------------------- */

/*
 * Hides the list window.
 */
void
sjog_list_hide()
{

#ifdef DEBUG
   fprintf(stderr, "sjog_list_hide\n");
#endif

   gtk_timeout_remove(listTimer);
   gtk_widget_hide_all(listWindow);
   /* jump to scroll mode when no window is shown */
   sjog_set_mode(SJOG_MODE_SCROLL);
}

/* -------------------------------------------------------------------------- */

/*
 * Remove the old hidding timer and put a new one.
 */
void
sjog_list_reset_timer()
{

#ifdef DEBUG
   fprintf(stderr, "sjog_list_reset_timer\n");
#endif

   if (listTimer != 0)
      gtk_timeout_remove(listTimer);
   listTimer =
      gtk_timeout_add(SJOG_TIMEOUT, (GtkFunction) sjog_list_timer_callback,
                      NULL);
}

/* -------------------------------------------------------------------------- */

/*
 * Do whatever needed depending on the jog position.
 */
void
sjog_list_check_jog_position(int jog)
{
   switch (jog)
   {
     case JOG_BOTTOM:
        sjog_list_select_next();
        break;
     case JOG_PUSHED:
        sjog_list_exec_cmd();
        break;
     case JOG_TOP:
        sjog_list_select_previous();
        break;
   }
}

/* -------------------------------------------------------------------------- */

/*
 * Called when the jog is pushed while we are in LIST_MODE. Either executes
 * a command or jump to one of the other modes (VOLUME, BRIGHTNESS..)
 */
void
sjog_list_exec_cmd()
{
   sjog_cmd_t *sjog_cmd;     /* sjog_cmd to execute */
   GList *list_element;      /* glib element that contains the cmd */
   gchar *cmd;               /* cmd string to execute */

   /* remove the timer and hide the window */
   sjog_list_hide();

#ifdef DEBUG
   fprintf(stdout, "sjog_list_exec_cmd\n");
#endif

   list_element = g_list_nth(list, selectedIndex);
   sjog_cmd = (sjog_cmd_t *) list_element->data;

#ifdef DEBUG
   fprintf(stdout, "sjog_list_exec_cmd: sjog_cmd->path = %s\n",
           sjog_cmd->path);
#endif

   /* enter brightness mode */
   if (strcmp(sjog_cmd->path, "brightness") == 0)
   {
      sjog_set_mode(SJOG_MODE_BRIGHTNESS);
      sjog_brightness_show();
   }

   /* enter volume mode */
   else if (strcmp(sjog_cmd->path, "volume") == 0)
   {
      sjog_set_mode(SJOG_MODE_VOLUME);
      sjog_volume_show();
   }
   else
   {
      cmd = g_strdup_printf("%s&", sjog_cmd->path);

#ifdef DEBUG
      fprintf(stdout, "sjog_list_exec_cmd: cmd = %s\n", cmd);
#endif

      /* launch the command */
      system(cmd);

      g_free(cmd);
   }
}

/* -------------------------------------------------------------------------- */

/*
 * Selects the next item in the application list.
 */
void
sjog_list_select_next()
{

   /* if we hit bottom, go to top */
   if (selectedIndex == (g_list_length(list) - 1))
      selectedIndex = 0;
   else
      selectedIndex++;

#ifdef DEBUG
   fprintf(stdout, "sjog_list_select_next: %d\n", selectedIndex);
#endif

   gtk_clist_select_row(GTK_CLIST(cmdListUI), selectedIndex, 0);
   sjog_list_reset_timer();

}

/* -------------------------------------------------------------------------- */

/*
 * Selects the previous item in the application list.
 */
void
sjog_list_select_previous()
{

   /* if we hit top, go to bottom */
   if (selectedIndex == 0)
      selectedIndex = g_list_length(list) - 1;
   else
      selectedIndex--;

#ifdef DEBUG
   fprintf(stdout, "sjog_list_select_previous: %d\n", selectedIndex);
#endif

   gtk_clist_select_row(GTK_CLIST(cmdListUI), selectedIndex, 0);
   sjog_list_reset_timer();
}
