/*
    This file is part of the sly ident daemon (slidentd).  slidentd 
    was written by Sean Hunter <sean@uncarved.com> as a minimal 
    RFC1413 (ident) daemon.

    slidentd is copyright (c) 2001 Uncarved Systems Ltd.

    slidentd is free software; you can redistribute it and/or modify
    it under the terms of version 2 of the GNU General Public License 
    as published by the Free Software Foundation.

    slidentd is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    in the file COPYING along with slidentd; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "slid_random.h"
#include "slid_read.h"
#include "slid_log.h"
#include <byte.h>
#include <buffer.h>

static char slid_to_asc(char input);

static int rand_fd;

void
slid_open_rand_device(void)
{
	rand_fd = open(SLIDENTD_RAND_PATH, O_RDONLY);
	if (rand_fd < 0) {
		buffer *log_buffer = slid_get_log_buffer();
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "Unable to open random device ");
		buffer_puts(log_buffer, SLIDENTD_RAND_PATH);
		buffer_puts(log_buffer, " - ");
		buffer_puts(log_buffer, strerror(errno));
		buffer_putsflush(log_buffer, "\n");
		slid_dies();
	}
}

void
slid_close_rand_device(void)
{
	int status;
	status = close(rand_fd);
	if (status != 0) {
		buffer *log_buffer = slid_get_log_buffer();
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer,
			    "Unable to close random device after read - ");
		buffer_puts(log_buffer, strerror(errno));
		buffer_putsflush(log_buffer, "\n");
		slid_dies();
	}
}

void
slid_get_raw(unsigned char *buffer, size_t len)
{
	/* read length bytes of random data into raw buffer */
	slid_readn(rand_fd, buffer, len);
}

char *
slid_alloc_hex_rand_buffer(size_t entropy_bytes)
{
	size_t char_buffer_len;
	char *buf;

	char_buffer_len = (entropy_bytes * 2) + 1;
	buf = (char *) malloc(sizeof (char) * char_buffer_len);
	if (buf == NULL) {
		buffer *log_buffer = slid_get_log_buffer();
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "Unable to allocate ");
		buffer_putlong(log_buffer, (long) char_buffer_len);
		buffer_puts(log_buffer, " bytes for random buffer - ");
		buffer_puts(log_buffer, strerror(errno));
		buffer_putsflush(log_buffer, "\n");
		slid_dies();
	}

	byte_zero(buf, char_buffer_len);
	return (buf);
}

void
slid_free_hex_rand_buffer(char *buffer)
{
	free(buffer);
}

static char
slid_to_asc(char input)
{
	if (input >= 0 && input < 10) {
		return input + '0';
	} else if (input >= 10 && input < 16) {
		return (input - 10) + 'a';
	} else {
		buffer *log_buffer = slid_get_log_buffer();
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "Unable to convert ");
		buffer_putlong(log_buffer, (long) input);
		buffer_putsflush(log_buffer, " - not in range 0 - 15\n");
		slid_dies();
	}
}

void
slid_get_random_hex(char *buf, size_t entropy_bytes)
{
	unsigned char *raw_buffer;
	char *curr;
	size_t raw_buffer_len;
	size_t i;
	char high_nibble;
	char low_nibble;

	raw_buffer_len = entropy_bytes + 1;
	raw_buffer = alloca(raw_buffer_len);
	if (raw_buffer == NULL) {
		buffer *log_buffer = slid_get_log_buffer();
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "Unable to allocate ");
		buffer_putlong(log_buffer, (long) raw_buffer_len);
		buffer_puts(log_buffer, " bytes for raw buffer - ");
		buffer_puts(log_buffer, strerror(errno));
		buffer_putsflush(log_buffer, "\n");
		slid_dies();
	}

	slid_get_raw(raw_buffer, entropy_bytes);

	curr = buf;
	for (i = 0; i < entropy_bytes; i++) {
		high_nibble = raw_buffer[i] >> 4;
		low_nibble = raw_buffer[i] & 0x0f;
		*curr++ = slid_to_asc(high_nibble);
		*curr++ = slid_to_asc(low_nibble);
	}
	*curr = 0;
}
