/***************************************************************************
    smb4kcore  -  The main core class of Smb4K.
                             -------------------
    begin                : Do Apr 8 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KCORE_H
#define SMB4KCORE_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qobject.h>
#include <qmap.h>
#include <qtimer.h>

// application specific includes
#include "smb4kscanner.h"
#include "smb4kmounter.h"
#include "smb4kpasswordhandler.h"
#include "smb4kfileio.h"
#include "smb4kbookmarkhandler.h"
#include "smb4kprint.h"
#include "smb4kshare.h"
#include "smb4ksynchronizer.h"


#define smb4k_core Smb4KCore::smb4kcore()



/**
 * This is the top-level core class. It inherits all other core classes
 * except the containers.
 */

class Smb4KCore : public QObject
{
  Q_OBJECT

  public:
    /**
     * The constructor.
     */
    Smb4KCore( QObject *parent = 0, const char *name = 0 );

    /**
     * The destructor.
     */
    ~Smb4KCore();

    /**
     * Returns the state the scanner is in.
     */
    int scannerState() { return m_scanner_state; }

    /**
     * Returns TRUE, if the scanner is running, otherwise FALSE.
     */
    bool scannerIsRunning() { return m_scanner->isRunning(); }

    /**
     * Returns the state the mounter is in.
     */
    int mounterState() { return m_mounter_state; }

    /**
     * Returns TRUE, if the mounter is running, otherwise FALSE.
     */
    bool mounterIsRunning() { return m_mounter->isRunning(); }

    /**
     * Returns the state the print object is in.
     */
    int printState() { return m_print_state; }

    /**
     * Returns TRUE, if the print object is running, otherwise FALSE.
     */
    bool printIsRunning() { return m_print->isRunning(); }

    /**
     * Returns TRUE if one of the core classes is doing something.
     */
    bool isRunning();

    /**
     * Returns the current state the core class is in.
     */
    int currentState() { return m_current_state; }

    /**
     * Returns a pointer to the scanner object.
     */
    Smb4KScanner *scanner() { return m_scanner; }

    /**
     * Returns a pointer to the mounter object.
     */
    Smb4KMounter *mounter() { return m_mounter; }

    /**
     * Returns a pointer to the bookmark handler object.
     */
    Smb4KBookmarkHandler *bookmarkHandler() { return m_bookmarkHandler; }

    /**
     * Returns a pointer to the file IO object.
     */
    Smb4KFileIO *fileIO() { return m_fileIO; }

    /**
     * Returns a pointer to the password handler object.
     */
    Smb4KPasswordHandler *passwordHandler() { return m_password_handler; }

    /**
     * Returns a pointer to the printer handler object.
     */
    Smb4KPrint *print() { return m_print; }

    /**
     * Returns a pointer to the synchronizer object.
     */
    Smb4KSynchronizer *synchronizer() { return m_synchronizer; }

    /**
     * Aborts any action of the core.
     */
    void abort();

    /**
     * Opens the mount point of a share. This function is used to run the file manager.
     *
     * @param share The share to be opened.
     */
    void open( Smb4KShare *share );

    /**
     * This function returns a static pointer to this class.
     *
     * @returns     a static pointer to this class.
     */
    static Smb4KCore *smb4kcore() { return m_this_class; }

  signals:
    /**
     * This signal is emitted, if one of the core objects
     * starts or stops running.
     */
    void runStateChanged();

  protected slots:
    /**
     * This slot sets the run state of the scanner.
     *
     * @param state         The run state the scanner is in. For definition of the
     *                      run states see smb4kdefs.h header file.
     */
    void slotSetScannerState( int state );

    /**
     * This slot sets the run state of the mounter.
     *
     * @param state         The run state the mounter is in. For definition of the
     *                      run states see smb4kdefs.h header file.
     */
    void slotSetMounterState( int state );

    /**
     * This slot sets the run state of the synchronizer.
     *
     * @param state         The run state the synchronizer is in. For definition of the
     *                      run states see smb4kdefs.h header file.
     */
    void slotSetSynchronizerState( int state );

    /**
     * This slot sets the run state of the print handler.
     *
     * @param state         The run state the print handler is in. For definition of the
     *                      run states see smb4kdefs.h header file.
     */
    void slotSetPrinterHandlerState( int state );

    /**
     * This slot is connected to the KApplication::shutDown() signal. It is invoked
     * when the application is shut down by the KDE logout or by pressing CTRL+Q, etc.
     */
    void slotShutdown();

  private:
    /**
     * The scanner object.
     */
    Smb4KScanner *m_scanner;

    /**
     * The mounter object.
     */
    Smb4KMounter *m_mounter;

    /**
     * The bookmark handler object.
     */
    Smb4KBookmarkHandler *m_bookmarkHandler;

    /**
     * The FileIO object.
     */
    Smb4KFileIO *m_fileIO;

    /**
     * The printer handler object.
     */
    Smb4KPrint *m_print;

    /**
     * The password handler pbject.
     */
    Smb4KPasswordHandler *m_password_handler;

    /**
     * The synchronizer object.
     */
    Smb4KSynchronizer *m_synchronizer;

    /**
     * The state the sanner is in.
     */
    int m_scanner_state;

    /**
     * The state the mounter is in.
     */
    int m_mounter_state;

    /**
     * Holds the current state.
     */
    int m_current_state;

    /**
     * The state the printer handler is in.
     */
    int m_print_state;

    /**
     * The state the synchronizer is in.
     */
    int m_syncer_state;

    /**
     * This function is used to set the current state of the core.
     *
     * @param state         One of the states defined in smb4kdefs.h
     */
    void setCurrentState( int state );

    /**
     * Searches for the needed programs and emits an error
     * if mandatory ones are missing.
     */
    void searchPrograms();

    /**
     * A static pointer to this class.
     */
    static Smb4KCore *m_this_class;
};

#endif
