/***************************************************************************
    smb4kfileio  -  Does file IO operations for Smb4K
                             -------------------
    begin                : Do Jan 1 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KFILEIO_H
#define SMB4KFILEIO_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qobject.h>
#include <qmap.h>
#include <qvaluelist.h>
#include <qdir.h>

// KDE includes
#include <kprocess.h>

// application specific includes
#include "smb4kuser.h"



/**
 * This class handles all IO operations on system files, that
 * are needed by Smb4K.
 *
 * @author Alexander Reinholdt
 */

class Smb4KFileIO : public QObject
{
  Q_OBJECT

  public:
    /**
     * The constructor.
     */
    Smb4KFileIO( QObject *parent = 0, const char *name = 0 );
    /**
     * The destructor
     */
    ~Smb4KFileIO();
    /**
     * Writes the entries needed to gain super user rights to either the super.tab or
     * the sudoers file depending on the input values.
     *
     * @param program         The name of the program that should be used to gain super
     *                        user privileges.
     *
     * @param force           Should be TRUE is the "Force Unmount" config entry will be/is set to TRUE.
     *
     * @param full_use        Should be TRUE if the "Run SUID" config entry will be/is set to TRUE.
     */
    bool writeSuperUserEntries( const QString &program, bool force = false, bool full_use = false );
    /**
     * Removes the entries of a user from the file /etc/super.tab.
     *
     * @return true in case of success and false otherwise.
     */
    bool removeSuperUserEntries( const QString &program = QString::null );
    /**
     * Returns the list of non-system users present on the system.
     */
    const QValueList<Smb4KUser *> getUsers();
    /**
     * Returns the papersize defined on the system. Therefore the
     * /etc/papersize file is read. If this files does not exist,
     * this function will return 'a4' as value, the default value
     * of enscript.
     */
    const QString getPaperSize();
    /**
     * This class holds the contents and the path of a file.
     */
    class FileItem
    {
      public:
        /**
         * The contructor.
         *
         * @param path        The path of the file.
         *
         * @param permissions The permissions the file has.
         *
         * @param owner       The owner of the file.
         *
         * @param group       The group of the file.
         *
         * @param contents    The contents of the file.
         *
         * @param exists      Whether the file exists or not.
         */
        FileItem( const QString &path, const QString &permissions, const QString &owner, const QString &group, const QStringList &contents, bool exists ) : m_path(path), m_permissions( permissions), m_owner( owner ), m_group( group ), m_contents(contents), m_exists(exists) {}
        /**
         * The empty constructor.
         */
        FileItem() : m_path( QString::null ), m_permissions( QString::null ), m_owner( QString::null ), m_group( QString::null ), m_contents( QStringList() ), m_exists( false ) {}
        /**
         * The destructor.
         */
        ~FileItem() {}
        /**
         * Returns the full path of the file.
         */
        QString path() const { return m_path; }
        /**
         * Retruns the contents of the file.
         */
        QStringList contents() const { return m_contents; }
        /**
         * Is TRUE if the file exists and FALSE if not.
         */
        bool exists() const { return m_exists; }
        /**
         * Replaces the old contents of the file item by the
         * new one.
         *
         * @param contents New contents of the file
         */
        void replaceContents( const QStringList &contents ) { m_contents = contents; }
        /**
         * Returns the file name.
         *
         * @returns       The file name.
         */
        QString name() const { return m_path.section( "/", -1, -1 ); }
        /**
         * Returns the permissions of the file in the form 0755 or similar.
         * See the stat(2) manual page for more information.
         *
         * @returns       The permissions of the file.
         */
        QString permissions() const { return m_permissions; }
        /**
         * This function returns the owner of the file that's represented
         * by this file item.
         *
         * @returns       The UID of the owner of the file.
         */
        QString owner() const { return m_owner; }
        /**
         * This function returns the group of the file that's represented
         * by this file item.
         *
         * @returns       The GID of the file.
         */
        QString group() const { return m_group; }

      private:
        QString m_path;
        QString m_permissions;
        QString m_owner;
        QString m_group;
        QStringList m_contents;
        bool m_exists;
    };

  signals:
    /**
     * This signal is emitted, when the writing of the SUID entries to the
     * config files finished. It does not necessarily need to be successful.
     */
    void finished_suid_writing();
    /**
     * This signal is emitted, when something went wrong with the writing of
     * the super user entries.
     */
    void error_suid_writing();

  protected slots:
    /**
     * This slot receives shell program output at Stderr.
     */
    void slotReceivedStderr( KProcess *, char *buf, int len );
    /**
     * This slot receives shell program output at Stderr.
     */
    void slotReceivedStdout( KProcess *, char *buf, int len );
    /**
     * This slot is called, when the process exited.
     */
    void slotProcessExited( KProcess * );
    /**
     * This slot is called by the KApplication::shutDown() signal.
     */
    void slotShutdown();

  private:
    /**
     * Enumeration to specify what is done.
     */
    enum State{ writeSU, removeSU, readSudoers, Idle };
    /**
     * This enumeration determines what to do with the sudoers file.
     */
    enum Sudoers{ Add, Remove };
    /**
     * Holds a value of of the Sudoers enumeration.
     */
    int m_todo_sudoers;
    /**
     * Holds the state value.
     */
    int m_state;
    /**
     * The KProcess object.
     */
    KProcess *m_proc;
    /**
     * Locates a system config file, reads it contents and returns it. In case the
     * config file was not found, it emits and error and returns 0.
     *
     * @param fileName The name of the file.
     *
     * @param strip    Decides whether the white spaces should be stripped. Default is FALSE.
     *
     * @param quiet    If this is set to TRUE, there will be no error emitted if a config file
     *                 could not be read.
     */
    Smb4KFileIO::FileItem readConfigFile( const QString &fileName, bool strip = false, bool quiet = false );
    /**
     * This function writes a file to its destination. It returns TRUE on success
     * and FALSE otherwise.
     *
     * @param item          The FileItem object that represents the file.
     */
    bool writeFile( Smb4KFileIO::FileItem *item );
    /**
     * This function processes the sudoers file, i.e. is adds or removes entries.
     */
    void processSudoers();
    /**
     * This writes the lock file and does some additional tests.
     *
     * @param filename      Takes the name of the system file that should be edited. This parameter
     *                      is not allowed to be empty.
     *
     * @returns             TRUE if the user is allowed to modify system files and FALSE otherwise
     */
    bool write_lock_file( const QString &filename );
    /**
     * Removes the lock file.
     *
     * @returns             TRUE on success and FALSE otherwise
     */
    bool remove_lock_file();
    /**
     * This is the object for the temporary directory.
     */
    QDir m_temp_dir;
    /**
     * This is the bufffer that holds the contents of system files
     * that were read by smb4k_cat.
     */
    QString m_buffer;
    
    /**
     * Enumeration for errors
     */
    enum Error { None, Cat };
    
    /**
     * The error code
     */
    int m_error_code;
};

#endif
