#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <sys/types.h>
#include <libsmbclient.h>
#include "list.h"
#include "common.h"
#include "auth.h"
#include "function.h"
#include "smbctx.h"

typedef struct{
    char		name[12];
    int			count_max;
    int			count;
    LIST		list;
} SMBCTXMANAGER;

char		smb_netbios_name[64]	= "";
int		smb_debug		= 0;
int		ctx_update_interval	= 300;
pthread_mutex_t	m_smbctx		= PTHREAD_MUTEX_INITIALIZER;
SMBCTXMANAGER   ctxman 			= {"SmbCtx", 15, 0,
					    STATIC_LIST_INITIALIZER(ctxman.list)
					  };

int SetSmbNetbiosName(const char *netbios_name){
    if ((netbios_name == NULL) || (*netbios_name == '\0'))  return 0;
    DPRINT(7, "netbios_name=%s\n", netbios_name);
    pthread_mutex_lock(&m_smbctx);
    safecpy(smb_netbios_name, netbios_name);
    pthread_mutex_unlock(&m_smbctx);
    return 1;
}

int SetSmbDebugLevel(int level){
    if ((level < 0) || (level > 10)) return 0;
    DPRINT(7, "level=%d\n", level);
    pthread_mutex_lock(&m_smbctx);
    smb_debug = level;
    pthread_mutex_unlock(&m_smbctx);
    return 1;
}

int GetSmbDebugLevel(void){
    int level;

    pthread_mutex_lock(&m_smbctx);
    level = smb_debug;
    pthread_mutex_unlock(&m_smbctx);
    DPRINT(7, "level=%d\n", level);
    return level;
}

int SetMaxSmbCtxCount(int count){
    if (count < 4) return 0;
    DPRINT(7, "count=%d\n", count);
    pthread_mutex_lock(&m_smbctx);
    ctxman.count_max = count;
    pthread_mutex_unlock(&m_smbctx);
    return 1;
}

int GetMaxSmbCtxCount(void){
    int count;

    pthread_mutex_lock(&m_smbctx);
    count = ctxman.count_max;
    pthread_mutex_unlock(&m_smbctx);
    DPRINT(7, "count=%d\n", count);
    return count;
}

int SetSmbCtxUpdateInterval(int interval){
    if (interval < GetUpdateTimeDelta()) return 0;
    DPRINT(7, "interval=%d\n", interval);
    pthread_mutex_lock(&m_smbctx);
    ctx_update_interval = interval;
    pthread_mutex_unlock(&m_smbctx);
    return 1;
}

int GetSmbCtxUpdateInterval(void){
    int interval;

    pthread_mutex_lock(&m_smbctx);
    interval = ctx_update_interval;
    pthread_mutex_unlock(&m_smbctx);
    DPRINT(7, "interval=%d\n", interval);
    return interval;
}

char* get_context_status_string(void){
    static char	buffer[4096];
    SmbCtx	*smb;
    int		ret;
    size_t	len;
    char	*pos, *ptn;

    memset(buffer, 0, sizeof(buffer));
    len = sizeof(buffer); pos = buffer; ptn = "%s[%d], ";

    *pos++ = '('; len--;
    smb = first_elem(&ctxman.list);
    while(is_valid_elem(&ctxman.list, smb)){
	if (!is_valid_elem(&ctxman.list, next_elem(smb))) ptn = "%s[%d])";
	ret = snprintf(pos, len, ptn, smb->name.string, smb->ref_count);
	if (ret < 0) return "(?error?)";
	if ((size_t)ret >= len){
	    strcpy(buffer + sizeof(buffer) - 5, "...)");
	    return buffer;
	}
	pos += ret; len -= ret;
	smb = next_elem(smb);
    };
    return buffer;
}

SmbCtx* NewSmbCtx(hstring *name){
    SmbCtx	*smb;

    if (name->length >= sizeof(smb->data)) return NULL;
    if (ctxman.count >= ctxman.count_max) return NULL;
    if ((smb = malloc(sizeof(SmbCtx))) == NULL) return NULL;
    memset(smb, 0, sizeof(SmbCtx));
    smb->access_time = time(NULL);
    pthread_mutex_init(&smb->mutex, NULL);

    strncpy(smb->data, name->string, name->length);
    memcpy(&smb->name, name, sizeof(hstring));
    smb->name.string = smb->data;
    
    if ((smb->ctx = smbc_new_context()) == NULL) goto error1;
    smb->ctx->debug = smb_debug;
    smb->ctx->callbacks.auth_fn = smb_auth_fn;

    /* Kerberos authentication by Esben Nielsen */
#if defined(SMB_CTX_FLAG_USE_KERBEROS) && defined(SMB_CTX_FLAG_FALLBACK_AFTER_KERBEROS)
    smb->ctx->flags |= 
	SMB_CTX_FLAG_USE_KERBEROS | SMB_CTX_FLAG_FALLBACK_AFTER_KERBEROS;
#endif

    if (*smb_netbios_name != '\0')
	smb->ctx->netbios_name = strdup(smb_netbios_name);
    if (smbc_init_context(smb->ctx) == NULL) goto error2;
    
    add_to_list_back(&ctxman.list, smb);
    ctxman.count++;
    return smb;
  error2:
    smbc_free_context(smb->ctx, 1);
  error1:
    free(smb);
    return NULL;
}

void DeleteSmbCtx(SmbCtx *smb){
    if (smb->ref_count != 0) goto error;
    if (pthread_mutex_trylock(&smb->mutex) != 0) goto error;
    pthread_mutex_unlock(&smb->mutex);

    remove_from_list(&ctxman.list, smb);
    ctxman.count--;

    pthread_mutex_destroy(&smb->mutex);
    smb->ctx->callbacks.purge_cached_fn(smb->ctx);
    smbc_free_context(smb->ctx, 1);
    free(smb);

    return;
  error:
    DPRINT(0, "WARNING! trying to destroy a busy context!\n");
    return;
}

SmbCtx* find_oldest_free_ctx(void){
    time_t	access_time = time(NULL) + 1;
    SmbCtx	*oldest = NULL, *elem = first_elem(&ctxman.list);
    
    while(is_valid_elem(&ctxman.list, elem)){
	if ((elem->ref_count == 0) && (elem->access_time < access_time)){
	    oldest = elem;
	    access_time = elem->access_time;
	}
	elem = next_elem(elem);
    }
    return oldest;
}

SmbCtx* find_in_ctx_list(hstring *hs){
    SmbCtx	*elem = first_elem(&ctxman.list);

    while(is_valid_elem(&ctxman.list, elem)){
	if ((hstring_casecmp(&elem->name, hs) == 0) && 
	    !(elem->flags & SMBCTX_IS_BAD)) return elem;
	elem = next_elem(elem);
    }
    return NULL;
}

void LockSmbCtx(SmbCtx *smb){
    if (smb->ref_count == 0){
	DPRINT(0, "WARNING! trying to lock an unused context!\n");
	return;
    }
    pthread_mutex_lock(&smb->mutex);
    smb->access_time = time(NULL);
}

void UnlockSmbCtx(SmbCtx *smb){
    if (smb->ref_count == 0){
	DPRINT(0, "WARNING! trying to unlock an unused context!\n");
	return;
    }
    pthread_mutex_unlock(&smb->mutex);
}

void SmbCtxCleanup(SmbCtx *smb){
    smb->flags = 0;
    smb->access_time = 0;
    smb->data[0] = '\0';
    memset(&smb->name, 0, sizeof(smb->name));
    smb->name.string = smb->data;
    smb->ctx->callbacks.purge_cached_fn(smb->ctx);
}

void AllocateSmbCtx(void){
    hstring	hs;

    make_casehstring(&hs, "", 0);
    pthread_mutex_lock(&m_smbctx);
    while(ctxman.count < ctxman.count_max)
	if (NewSmbCtx(&hs) == NULL) break;
    pthread_mutex_unlock(&m_smbctx);
}

void DestroyUnusedCtx(void){
    SmbCtx	*tmp, *elem;

    pthread_mutex_lock(&m_smbctx);
    elem = first_elem(&ctxman.list);
    while(is_valid_elem(&ctxman.list, elem)){
	tmp = elem; elem = next_elem(elem);
	if (tmp->ref_count == 0) DeleteSmbCtx(tmp);
    }
    DPRINT(1, "%s samba contexts still in use: %d\n", ctxman.name, ctxman.count);
    pthread_mutex_unlock(&m_smbctx);
}

void SetSmbCtxFlags(SmbCtx *smb, int flags){
    pthread_mutex_lock(&m_smbctx);
    smb->flags |= flags;
    pthread_mutex_unlock(&m_smbctx);
}

SmbCtx* GetSmbCtx(const char *smbpath, int flags){
    SmbCtx	*smb;
    size_t	len;
    hstring	name;

    if (strncmp(smbpath, "smb://", 6) != 0) return NULL;
    len = filename_end(smbpath + 6) - smbpath;
    if (len >= sizeof(smb->data)){
	len = sizeof(smb->data) - 1;
	flags |= SMBCTX_DO_NOT_UPDATE;
    }

    DPRINT(6, "name='%.*s'\n", len, smbpath);
    make_casehstring(&name, smbpath, len);

    pthread_mutex_lock(&m_smbctx);
    if ((smb = find_in_ctx_list(&name)) != NULL) goto ctx_exist;
    if ((smb = NewSmbCtx(&name)) != NULL) goto ctx_exist;
    if ((smb = find_oldest_free_ctx()) == NULL) goto no_resource;

    SmbCtxCleanup(smb);
    strncpy(smb->data, name.string, name.length); 
    smb->data[name.length] = '\0';
    memcpy(&smb->name, &name, sizeof(name));
    smb->name.string = smb->data;

  ctx_exist:
    smb->flags |= flags;
    smb->ref_count++;
    
  no_resource:
    DPRINT(6, "ctx_total=%d, list=%s\n",
	ctxman.count, get_context_status_string());
    pthread_mutex_unlock(&m_smbctx);
    return smb;
}

void ReleaseSmbCtx(SmbCtx *smb){
    DPRINT(6, "ctx->name=%s[%d]\n", smb->name.string, smb->ref_count);

    if (smb->ref_count == 0){
	DPRINT(0, "WARNING! trying to release an unused context!\n");
	return;
    }

    pthread_mutex_lock(&m_smbctx);
    smb->ref_count--;
    if (smb->ref_count == 0){
	if (smb->flags & (SMBCTX_IS_BAD | SMBCTX_NEEDS_CLEANUP))
	    SmbCtxCleanup(smb);
	if (ctxman.count > ctxman.count_max) DeleteSmbCtx(smb);
    }
    DPRINT(6, "ctx_total=%d, list=%s\n",
	ctxman.count, get_context_status_string());
    pthread_mutex_unlock(&m_smbctx);
    return;
}

void RefreshSmbCtx(SmbCtx *smb){
    DPRINT(6, "%s[%d]\n", smb->name.string, smb->ref_count);
    if (smb->ref_count == 0){
	SmbCtxCleanup(smb);
	DPRINT(6, "ctx_total=%d, list=%s\n",
	    ctxman.count, get_context_status_string());
	return;
    }
    if (smb->flags & (SMBCTX_IS_BAD | SMBCTX_DO_NOT_UPDATE)) return;
    if (pthread_mutex_trylock(&smb->mutex) == 0){
	struct smbc_dirent	*dirent;
	SMBCFILE		*fd;

	smb->access_time = time(NULL);
	smb->ref_count++;
	pthread_mutex_unlock(&m_smbctx);

	if ((fd = smb->ctx->opendir(smb->ctx, smb->name.string)) != NULL){
	    while((dirent = smb->ctx->readdir(smb->ctx, fd)) != NULL){
		(void) dirent->name;
	    }
	    smb->ctx->closedir(smb->ctx, fd);
	}else SetSmbCtxFlags(smb, SMBCTX_IS_BAD);

	pthread_mutex_lock(&m_smbctx);
	smb->ref_count--;
	pthread_mutex_unlock(&smb->mutex);
    }
    return;
}

void RefreshOldSmbCtxs(time_t threashold){
    SmbCtx	*smb;

    pthread_mutex_lock(&m_smbctx);
    smb = first_elem(&ctxman.list);
    while(is_valid_elem(&ctxman.list, smb)){
	if ((smb->access_time != 0) && (smb->access_time <= threashold))
	    RefreshSmbCtx(smb);
	smb = next_elem(smb);
    }
    pthread_mutex_unlock(&m_smbctx);
}
