/*
 * rawplay.cc -- SDL DV Player
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Will build but not run on OSX due to C++ main name mangling getting in the
 * way of SDLmain.h's need to rename name -> SDL_main and then linking in
 * -lSDLmain.  Not a bit deal since smilutils is effectively abandoned.
*/

#include <iostream>
#include <cassert>
using std::cerr;
using std::endl;

#include <SDL.h>
#include <SDL_syswm.h>

#include <PumpViewer.h>
#include <frame.h>
#include <RawDVFileInput.h>
#include <PlayListDVProvider.h>
#include <ExtendedPlayList.h>
#include <DVEncoder.h>
#include <unistd.h>

#if defined(__linux__)
#define HAVE_X11
#endif

int image_count = 0;
int audio_count = 0;

/** Arguments for video and audio initialisation.
*/

class ViewerArguments
{
	public:
		// Initialise the arguments object
		ViewerArguments( ) :
			start_full_screen( false ),
			full_screen( false ),
			mute( false ),
			deinterlaced( false ),
			provider( NULL ),
			pass_through( false ),
			pass_through_on( false ),
			timestamp( false ),
			quality( DV_QUALITY_BEST ),
			frequency( 48000 )
		{
		}
		
		// Full screen by default
		bool IsFullScreen( )
		{
			return full_screen;
		}
		
		// Audio playback attempted
		bool IsMute( )
		{
			return mute;
		}
		
		// Interlace on or off
		bool IsDeinterlaced( )
		{
			return deinterlaced;
		}
		
		bool IsSeekable( )
		{
			return provider != NULL;
		}

		void SetSpeed( int position, double speed )
		{
			if ( IsSeekable( ) )
			{
				mute = speed != 1;
				provider->SetSpeedAndPosition( speed, position );
			}
		}
		
		double GetSpeed( )
		{
			return provider->GetSpeed( );
		}

		void ChangeScene( int position, int scene )
		{
			if ( IsSeekable( ) )
				provider->ChangeScene( position, scene );
		}
		
		void SetEndAction( end_of_playlist action )
		{
			if ( IsSeekable( ) )
				provider->SetEndAction( action );
		}
		
		bool CanPassThrough( )
		{
			return pass_through;
		}
		
		bool SetPassThrough( bool _pass_through_on )
		{
			if ( CanPassThrough( ) )
				return pass_through_on = _pass_through_on;
			else
				return false;
		}
		
		bool IsPassing( )
		{
			return pass_through_on;
		}
		
	protected:
		bool start_full_screen;
		bool full_screen;
		bool mute;
		bool deinterlaced;
		PlayListDVProvider *provider;
		bool pass_through;
		bool pass_through_on;
		bool timestamp;
		int quality;
		int frequency;
};

typedef struct
{
	int width;
	int height;
	int aspect_x;
	int aspect_y;
	int aspect_width;
	int aspect_height;
}
FrameDimensions;

/** The SimpleViewer class provides an unaccellerated video renderer, the audio
	and event handlers. 
*/

class SimpleViewer : 
	public FramePlayer,
	protected ViewerArguments
{
	public:
		// Constructor for the SimpleViewer
		SimpleViewer( ViewerArguments &_args ) :
			ViewerArguments( _args ),
			supplier( NULL ),
			audio_available( 0 ),
			full_screen( false ),
			current_width( 0 ),
			current_height( 0 ),
			alwaysontop( false )
		{
		}
		
		// Destructor for the SimpleViewer
		virtual ~SimpleViewer( )
		{
			delete image;
			delete audio;
		}

		void GetFrameInfo( Frame &frame, FrameDimensions &fd )
		{
			if ( frame.IsWide( ) )
			{
				fd.aspect_x = 16;
				fd.aspect_y = 9;
			}
			else
			{
				fd.aspect_x = 4;
				fd.aspect_y = 3;
			}
			
			fd.width = frame.GetWidth( );
			fd.height = frame.GetHeight( );

			fd.aspect_height = fd.height;
			fd.aspect_width = fd.aspect_height * fd.aspect_x / fd.aspect_y;
		}

		void GetWindowDimensions( SDL_Surface *screen, FrameDimensions &fd )
		{
#ifdef HAVE_X11
			static int last_width = -1;
			static int last_height = -1;
			
			SDL_SysWMinfo wm;
			SDL_VERSION(&wm.version);
			if ( !full_screen && SDL_GetWMInfo( &wm ) == 1 )
			{
				if ( wm.subsystem == SDL_SYSWM_X11 ) 
				{
					wm.info.x11.lock_func( );
					Window window = wm.info.x11.window;
					Display *display = wm.info.x11.display;
					XWindowAttributes attr;
					XGetWindowAttributes(display, window, &attr);

					if ( last_width != attr.width || last_height != attr.height )
					{
						last_width = attr.width;
						last_height = attr.height;

						SDL_SetVideoMode( last_width, last_height, 0, s_flags );
						if ( ( last_height * fd.aspect_x / fd.aspect_y ) < last_width )
						{
							int width2 = last_height * fd.aspect_x / fd.aspect_y;
							SDL_Rect rect = { ( last_width - width2 ) / 2, 0, width2, last_height };
							SDL_SetClipRect( screen, &rect );
						}
						else
						{
							int height2 = last_width * fd.aspect_y / fd.aspect_x;
							SDL_Rect rect = { 0, ( last_height - height2 ) / 2, last_width, height2 };
							SDL_SetClipRect( screen, &rect );
						}
					}
					wm.info.x11.unlock_func( );
				}
			}
			#endif
		}


		void ToggleAlwaysOnTop( void )
		{
   			#ifdef HAVE_X11
			SDL_SysWMinfo wm;
			SDL_VERSION(&wm.version);
			if ( !full_screen && SDL_GetWMInfo( &wm ) == 1 )
			{
				if ( wm.subsystem == SDL_SYSWM_X11 ) 
				{
					//wm.info.x11.lock_func( );
					Window window = wm.info.x11.window;
					Display *display = wm.info.x11.display;

    				char *atom_names[] = {
        				"_NET_WM_STATE",
        				"_NET_WM_STATE_ABOVE",
    				};
    				Atom atoms_return[ 2 ];

    				XInternAtoms( display, atom_names, 2, False, atoms_return );
    				Atom net_wm_state = atoms_return[ 0 ];
    				Atom net_wm_state_above = atoms_return[ 1 ];

        			XEvent ev;

        			alwaysontop = !alwaysontop;

        			ev.type = ClientMessage;
        			ev.xclient.window = window;
        			ev.xclient.message_type = net_wm_state;
        			ev.xclient.format = 32;
        			ev.xclient.data.l[ 0 ] = alwaysontop ? 1 : 0;
        			ev.xclient.data.l[ 1 ] = net_wm_state_above;
        			ev.xclient.data.l[ 2 ] = 0;

        			XSendEvent( display, DefaultRootWindow( display ), True,
                    			SubstructureNotifyMask|SubstructureRedirectMask, &ev );

					XRaiseWindow( display, window );

					//wm.info.x11.unlock_func( );
				}
			}
#endif
    	}

		void GetScreenDimensions( int &width, int &height )
		{
			#ifdef HAVE_X11
			SDL_SysWMinfo wm;
			SDL_VERSION(&wm.version);
			if ( SDL_GetWMInfo( &wm ) == 1 )
			{
				if ( wm.subsystem == SDL_SYSWM_X11 ) 
				{
					wm.info.x11.lock_func( );
					Display *display = wm.info.x11.display;
					if ( display != NULL )
					{
						width = DisplayWidth( display, DefaultScreen( display ) );
						height = DisplayHeight( display, DefaultScreen( display ) );
					}
					wm.info.x11.unlock_func( );
				}
			}
			#endif			
		}

		// Toggle between a full and windowed view
		bool ToggleFullscreen( )
		{
			int screen_width = 800;
			int screen_height = 600;
			
			GetScreenDimensions( screen_width, screen_height );
			
			SDL_Surface *screen = SDL_GetVideoSurface( );
			assert( screen != NULL );
			
			if ( full_screen )
			{
				SDL_SetVideoMode( current_width, current_height, 0, s_flags );
			}
			else
			{
				current_width = screen->w;
				current_height = screen->h;
				SDL_SetVideoMode( screen_width, screen_height, 0, SDL_FULLSCREEN | s_flags );
			}
			
			full_screen = !full_screen;
			SDL_ShowCursor( !full_screen );
			
			return true;
		}

		// Toggle the mute
		bool ToggleMute( )
		{
			mute = !mute;
			return true;
		}
		
		// Initialise the audio
		virtual bool AudioInit( Frame &frame, FrameSupplier *_supplier )
		{
			supplier = _supplier;
			
			SDL_EnableKeyRepeat( SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL );
			
			audio = new int16_t[ DV_AUDIO_MAX_SAMPLES * 16 ];
			buffer = new int16_t[ DV_AUDIO_MAX_SAMPLES * 16 ];	

			SDL_AudioSpec wanted;
			SDL_AudioSpec obtained;
			
			wanted.freq = frequency;
			wanted.format = AUDIO_S16;
			wanted.channels = 2;
			wanted.samples = 512;
			wanted.callback = FillAudio;
			wanted.userdata = (void *)this;
			
			if ( SDL_OpenAudio( &wanted, &obtained ) < 0 ) 
			{
				fprintf( stderr, "Couldn't open audio: %s\n", SDL_GetError() );
				return false;
			}
			
			resampler = new FastAudioResample( obtained.freq );
			
			SDL_PauseAudio( 0 );
			
			return true;
		}
		
		// Fill the audio from the frame supplier
		void FillAudio( Uint8 *stream, int len )
		{
			Frame *ptr = NULL;
			
			while ( audio_available <= len && ( ptr = supplier->GetFrame( ) ) != NULL )
			{
				std::cerr << "In " << setw( 8 ) << audio_count 
						  << " Out " << setw( 8 ) << image_count 
						  << " Dropped " << setw( 8 ) << audio_count - image_count 
						  << "\r";
				audio_count ++;
				resampler->Resample( *ptr );
				memcpy( (uint8_t*)audio + audio_available, (uint8_t*)resampler->output, resampler->size );
				audio_available += resampler->size;
			}

			if ( audio_available >= len )
			{
				if ( !mute )
				{
					int words = len / 2;
					for ( int i = 0; i < words; i ++ )
					{
						( ( uint8_t * ) buffer )[ 2 * i ] = audio[ i ] & 0xff;
						( ( uint8_t * ) buffer )[ 2 * i + 1 ] = ( audio[ i ] & 0xff00 ) >> 8;
					}
				}
				else
				{
					memset( (uint8_t *)buffer, 0, len );
				}
				
				SDL_MixAudio(stream, (uint8_t *)buffer, len, SDL_MIX_MAXVOLUME);
				audio_available -= len;
				memmove( (uint8_t *)audio, (uint8_t *)audio + len, audio_available );
			}
			else
			{
				SDL_MixAudio(stream, (uint8_t *)audio, audio_available, SDL_MIX_MAXVOLUME);
				audio_available = 0;
			}			
		}
		
		// Close the audio
		virtual void AudioClose( )
		{
			SDL_AudioQuit( );
		}
	
		// Initialise the frame displayer
		virtual bool ShowInit( Frame &frame )
		{
			FrameDimensions fd;
			GetFrameInfo( frame, fd );
			
			image = new uint8_t[ 720 * 576 * 3 ];

			s_flags = SDL_HWSURFACE|SDL_ASYNCBLIT|SDL_HWACCEL|SDL_DOUBLEBUF;

			SDL_Surface *screen = SDL_SetVideoMode( fd.width, fd.height, 0, s_flags );
			if ( screen == NULL ) 
			{
				fprintf(stderr, "Unable to set %dx%d video: %s\n", fd.width, fd.height, SDL_GetError());
				return false;
			}

			if ( start_full_screen )
				ToggleFullscreen( );
			
			return true;
		}
		
		// Show a frame
		virtual bool ShowFrame( Frame &frame )
		{
			FrameDimensions fd;
			GetFrameInfo( frame, fd );

			image_count ++;

			frame.decoder->quality = quality;
			frame.ExtractRGB( image );

			if ( IsDeinterlaced( ) )
				frame.GetLowerField( image, 3 );

			uint8_t *p = image;

			SDL_Surface *screen = SDL_GetVideoSurface( );

			assert( screen != NULL );

			if ( SDL_MUSTLOCK(screen) && SDL_LockSurface(screen) < 0 )
				return false;

			int scanlength = screen->pitch/2;

			for ( int y = 0; y < fd.height; y ++ )
			{
				uint16_t *bufp = (uint16_t *)screen->pixels + y * scanlength;
				for ( int x = 0; x < fd.width; x ++ )
				{
					uint32_t color = SDL_MapRGB(screen->format, *p, *( p + 1 ), *( p + 2 ) );
					*bufp ++ = color;
					p += 3;
				}
			}

			if ( SDL_MUSTLOCK(screen) )
				SDL_UnlockSurface(screen);

			SDL_Flip( screen );
				
			return true;
		}
		
		// Close the frame displayer
		virtual void ShowClose( )
		{
		}

		// Handle events
		virtual bool HandleEvents( )
		{
			SDL_Event event;
			while ( SDL_PollEvent( &event ) )
			{
				switch( event.type )
				{
					case SDL_KEYDOWN: 
					{
						if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "f" ) )
							ToggleFullscreen( );
						else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "a" ) )
							ToggleAlwaysOnTop( );
						else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "m" ) )
							ToggleMute( );
						else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "p" ) )
						{
							encoder.SetNewRecording( );
							SetPassThrough( !IsPassing( ) );
						}
						else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "d" ) )
							deinterlaced = !deinterlaced;
						else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "q" ) )
						{
							if ( full_screen )
								ToggleFullscreen( );
							return false;
						}
						else if ( IsSeekable( ) )
						{
							int position = supplier->GetLastFramePosition( );
							if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "1" ) )
								SetSpeed( position, -50 );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "2" ) )
								SetSpeed( position, -25 );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "3" ) )
								SetSpeed( position, -12.5 );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "4" ) )
								SetSpeed( position, -0.5 );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "5" ) )
								SetSpeed( position, 1 );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "6" ) )
								SetSpeed( position, 0.5 );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "7" ) )
								SetSpeed( position, 12.5 );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "8" ) )
								SetSpeed( position, 25 );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "9" ) )
								SetSpeed( position, 50 );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "0" ) )
								SetSpeed( 0, 1 );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "h" ) )
								SetSpeed( position - 1, 0 );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "l" ) )
								SetSpeed( position + 1, 0 );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "j" ) )
								ChangeScene( position, 1 );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "k" ) )
								ChangeScene( position, -1 );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "s" ) )
								SetEndAction( pause_on_end );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "r" ) )
								SetEndAction( loop_on_end );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "t" ) )
								SetEndAction( terminate_on_end );
							else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "space" ) )
							{
								SetSpeed( position, GetSpeed() ? 0.0 : 1.0 );
								mute = GetSpeed( ) != 1.0;
							}
						}
						else if ( !strcmp( SDL_GetKeyName(event.key.keysym.sym), "space" ) )
							mute = supplier->TogglePause( );
						break;
					}
					case SDL_VIDEORESIZE:
						SDL_SetVideoMode( event.resize.w, event.resize.h, 0, s_flags );
						break;
					case SDL_QUIT:
					{
						if ( full_screen )
							ToggleFullscreen( );
						return false;
						break;
					}
					default:
						break;
				}
			}
			return true;
		}
		
		void PassThrough( Frame &frame )
		{
			if ( timestamp )
				encoder.EncodeMetaData( frame );
			if ( IsPassing( ) )
			{
				fwrite( frame.data, frame.GetFrameSize( ), 1, stdout );
				fflush( stdout );
			}
		}
		
	protected:
		FrameSupplier *supplier;
	private:
		DVEncoder encoder;
		uint8_t *image;
		int16_t *audio;
		int16_t *buffer;
		int audio_available;
		AudioResample *resampler;

		static void FillAudio( void *udata, Uint8 *stream, int len )
		{
			SimpleViewer &viewer = *( SimpleViewer * )udata;
			viewer.FillAudio( stream, len );
		}

	protected:
		bool full_screen;
		int s_flags;
		int current_width;
		int current_height;
		bool alwaysontop;
};

/** The AdvancedViewer inherits from SimpleViewer and provides accellerated
	video rendering.
*/

class YUV422Viewer : 
	public SimpleViewer
{
	private:
		SDL_Overlay *overlay;
		uint8_t *image;
	
	public:
		YUV422Viewer( ViewerArguments &args ) :
			SimpleViewer( args ),
			overlay( NULL ),
			image( NULL )
		{
		}
		
		~YUV422Viewer( )
		{
			delete image;
		}
		
		virtual bool ShowInit( Frame &frame )
		{
			image = new uint8_t[ 720 * 576 * 2 ];

			FrameDimensions fd;
			GetFrameInfo( frame, fd );

			s_flags = SDL_HWSURFACE|SDL_ASYNCBLIT|SDL_HWACCEL|SDL_RESIZABLE;
			
			SDL_Surface *screen = SDL_SetVideoMode( fd.aspect_width, fd.aspect_height, 0, s_flags );
			if ( screen == NULL ) 
			{
				fprintf(stderr, "Unable to set %dx%d video: %s\n", fd.width, fd.height, SDL_GetError());
				return false;
			}

			overlay = SDL_CreateYUVOverlay( fd.width, fd.height, SDL_YUY2_OVERLAY, screen );
			if ( overlay == NULL )
			{
				fprintf(stderr, "Unable to create an overlay %dx%d video: %s\n", fd.width, fd.height, SDL_GetError());
				return false;
			}

			SDL_Rect rect = { 0, 0, fd.aspect_width, fd.aspect_height };
			SDL_SetClipRect( screen, &rect );

			if ( start_full_screen )
				ToggleFullscreen( );

			return true;
		}
		
		virtual bool ShowFrame( Frame &frame )
		{
			FrameDimensions fd;
			GetFrameInfo( frame, fd );

			frame.decoder->quality = quality;
			frame.ExtractYUV( image );	

			image_count ++;
			
			if ( IsDeinterlaced( ) )
				frame.GetLowerField( image, 2 );

			SDL_Surface *screen = SDL_GetVideoSurface( );

			GetWindowDimensions( screen, fd );

			if ( SDL_LockYUVOverlay( overlay ) >= 0 )
			{
				memcpy( overlay->pixels[0], image, fd.width * fd.height * 2 );
				SDL_UnlockYUVOverlay( overlay );
				SDL_DisplayYUVOverlay( overlay, &screen->clip_rect );
			}
			
			return true;
		}
		
		virtual void ShowClose( )
		{
			SDL_FreeYUVOverlay( overlay );
		}
};

class YUV420PViewer : 
	public YUV422Viewer
{
	private:
		SDL_Overlay *overlay;
		uint8_t *image;
		uint8_t *output[ 3 ];
	
	public:
		YUV420PViewer( ViewerArguments &args ) :
			YUV422Viewer( args ),
			overlay( NULL ),
			image( NULL )
		{
		}
		
		~YUV420PViewer( )
		{
			delete image;
		}
		
		virtual bool ShowInit( Frame &frame )
		{
			FrameDimensions fd;
			GetFrameInfo( frame, fd );

			image = new uint8_t[ fd.width * fd.height * 2 ];
			output[ 0 ] = new uint8_t[ fd.width * fd.height ];
			output[ 1 ] = new uint8_t[ fd.width * fd.height / 4 ];
			output[ 2 ] = new uint8_t[ fd.width * fd.height / 4 ];

			s_flags = SDL_HWSURFACE|SDL_ASYNCBLIT|SDL_HWACCEL|SDL_RESIZABLE;
			
			SDL_Surface *screen = SDL_SetVideoMode( fd.aspect_width, fd.aspect_height, 0, s_flags );
			if ( screen == NULL ) 
			{
				fprintf(stderr, "Unable to set %dx%d video: %s\n", fd.width, fd.height, SDL_GetError());
				return false;
			}

			overlay = SDL_CreateYUVOverlay( fd.width, fd.height, SDL_YV12_OVERLAY, screen );
			if ( overlay == NULL )
			{
				fprintf(stderr, "Unable to create an overlay %dx%d video: %s\n", fd.width, fd.height, SDL_GetError());
				return false;
			}

			SDL_Rect rect = { 0, 0, fd.aspect_width, fd.aspect_height };
			SDL_SetClipRect( screen, &rect );

			if ( start_full_screen )
				ToggleFullscreen( );

			return true;
		}

		virtual bool ShowFrame( Frame &frame )
		{
			FrameDimensions fd;
			GetFrameInfo( frame, fd );
				
			frame.decoder->quality = quality;
			frame.ExtractYUV420( image, output );	

			image_count ++;
			
			SDL_Surface *screen = SDL_GetVideoSurface( );
			GetWindowDimensions( screen, fd );

			if ( SDL_LockYUVOverlay( overlay ) >= 0 )
			{
				memcpy( overlay->pixels[ 0 ], output[ 0 ], fd.width * fd.height );
				memcpy( overlay->pixels[ 1 ], output[ 2 ], fd.width * fd.height / 4 );
				memcpy( overlay->pixels[ 2 ], output[ 1 ], fd.width * fd.height / 4 );
				SDL_UnlockYUVOverlay( overlay );
				SDL_DisplayYUVOverlay( overlay, &screen->clip_rect );
			}
			
			return true;
		}	
};


class PumpViewerCreator : 
	public ViewerArguments
{
	public:
		PumpViewerCreator( ) :
			ViewerArguments( ),
			xvideo( 3 ),
			blockable( false ),
			buffers( 25 )
		{
		}

		// Determine if the arguments dictate a valid playlist
		bool HasPlayList( )
		{
			return playlist.GetNumFrames( ) != 0;
		}

		// Obtain the play list
		PlayList &GetPlayList( )
		{
			return playlist;
		}

		bool Interpret( int argc, char **argv )
		{
			bool ret_val = true;

			for ( int index = 1; ret_val && index < argc; index ++ )
			{
				if ( !strcmp( argv[ index ], "-s" ) )
					xvideo = 0;
				else if ( !strcmp( argv[ index ], "-f" ) )
					start_full_screen = true;
				else if ( !strcmp( argv[ index ], "-m" ) )
					mute = true;
				else if ( !strcmp( argv[ index ], "-x" ) && index + 1 < argc  )
					xvideo = atoi( argv[ ++ index ] );
				else if ( !strcmp( argv[ index ], "-u" ) && index + 1 < argc  )
					buffers = atoi( argv[ ++ index ] );
				else if ( !strcmp( argv[ index ], "-q" ) && index + 1 < argc  )
					quality = atoi( argv[ ++ index ] );
				else if ( !strcmp( argv[ index ], "-r" ) && index + 1 < argc  )
					frequency = atoi( argv[ ++ index ] );
				else if ( !strcmp( argv[ index ], "-d" ) )
					deinterlaced = true;
				else if ( !strcmp( argv[ index ], "-b" ) )
					blockable = true;
				else if ( !strcmp( argv[ index ], "-t" ) )
					timestamp = true;
				else if ( !strcmp( argv[ index ], "-p" ) )
				{
					if ( !pass_through )
						pass_through = true;
					else
						pass_through_on = true;
				}
				else if ( !strcmp( argv[ index ], "-ao" ) )
				{
					if ( strcmp( argv[ ++ index ], "null" ) )
					{
						setenv( "SDL_AUDIODRIVER", argv[ index ], 1 );
					}
					else
					{
						setenv( "SDL_AUDIODRIVER", "disk", 1 );
						setenv( "SDL_DISKAUDIOFILE", "/dev/null", 1 );
						setenv( "SDL_DISKAUDIODELAY", "10", 1 );
					}
				}
				else if ( !strcmp( argv[ index ], "-vo" ) )
					setenv( "SDL_VIDEODRIVER", argv[ ++ index ], 1 );
				else if ( !strcmp( argv[ index ], "-id" ) )
					setenv( "SDL_WINDOWID", argv[ ++ index ], 1 );
				else if ( !strcmp( argv[ index ], "--help" ) )
					ret_val = false;
				else
					ret_val = playlist.Append( argv[ index ] );
			}

			if ( playlist.GetNumFrames( ) == 0 && isatty( fileno( stdin ) ) )
			{
				cerr << "Input must be from valid files specified on the command line or redirected from a pipe or a file." << endl;
				return false;
			}

			if ( pass_through && isatty( fileno( stdout ) ) )
			{
				cerr << "Passthrough can only be used when stdout is redirected to a pipe or a file." << endl;
				return false;
			}

			return ret_val;
		}

		void Usage( )
		{
			cerr << "Usage rawplay [ options ] [ files ]" << endl;
			cerr << "Where: -s        - simple unaccellerated player" << endl;
			cerr << "       -x type   - force player format (0:rgb,1:yuv422,2:yuv420p,3:optimal)" << endl;
			cerr << "       -f        - play in full screen mode" << endl;
			cerr << "       -m        - audio is initially muted" << endl;
			cerr << "       -d        - deinterlace video" << endl;
			cerr << "       -b        - input stream is blockable" << endl;
			cerr << "       -p        - allow passthrough on stdout" << endl;
			cerr << "       -u frames - number of frame buffers" << endl;
			cerr << "       -q quality- libdv decoding quality" << endl;
			cerr << "       -r freq   - audio playback frequency (default: 48000)" << endl;
			cerr << "       -t        - time stamp on pass through" << endl;
			cerr << "       -vo dev   - SDL video output device" << endl;
			cerr << "       -ao dev   - SDL audio output device" << endl;
			cerr << "       -id id    - window id" << endl;
		}
		
		void Run( )
		{
			DVPumpProvider *pump = GetPump( );
			FramePlayer *player = GetPlayer( );

			if ( pump != NULL && player != NULL )
			{
				pump->ThreadStart( );
		
				PumpViewer viewer( *pump, *player );
				viewer.SetBlockable( blockable );				
				viewer.Thread( );
			
				pump->ThreadStop( );
			}
		
			delete player;
			delete pump;
		}
		
	private:
		int xvideo;
		ExtendedPlayList playlist;
		bool blockable;
		int buffers;
	
		FramePlayer *GetPlayer( )
		{
			switch( xvideo )
			{
				case 1:
					return new YUV422Viewer( *this );
				case 2:
					return new YUV420PViewer( *this );
				case 3:
					#if defined(HAVE_LIBAVCODEC)
						return new YUV420PViewer( *this );
					#else
						return new YUV422Viewer( *this );
					#endif
				default:
					return new SimpleViewer( *this );
			}
		}
		
		DVPumpProvider *GetPump( )
		{
			if ( !HasPlayList( ) )
			{
				RawDVFileInput *pump = new RawDVFileInput( );
				pump->SetPumpSize( buffers );
				pump->SetFile( stdin );
				return pump;
			}
			else
			{
				PlayListDVProvider *pump = new PlayListDVProvider( GetPlayList( ) );
				provider = pump;
				blockable = true;
				pump->SetPumpSize( buffers );
				return pump;
			}
		}
};
	
int main( int argc, char *argv[] )
{
	PumpViewerCreator creator;
	
	if ( !creator.Interpret( argc, argv ) )
	{
		creator.Usage( );
		exit( 0 );
	}
	
    if ( SDL_Init( SDL_INIT_VIDEO | SDL_INIT_AUDIO ) < 0 ) 
	{
        fprintf(stderr, "Unable to init SDL: %s\n", SDL_GetError());
        exit(1);
    }

    atexit( SDL_Quit );

	creator.Run( );

	std::cerr << std::endl;
	
	return 0;
}
