/*
 * PPMPump.cc -- PPM Pump Implementation
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <stdint.h>
#include <pthread.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include "PPMPump.h"

PPMFrame::PPMFrame() :
	image( (uint8_t *)NULL ),
	width( 0 ),
	height( 0 ),
	output( stdout )
{
}

PPMFrame::PPMFrame( int _width, int _height ) :
	width( _width ),
	height( _height )
{
	image = new uint8_t[ _width * _height * 4 ];
}

PPMFrame::PPMFrame( PPMFrame &frame )
{
	uint8_t *_image = frame.GetImage( width, height );
	image = new uint8_t[ width * height * 4 ];
	if ( image != NULL && _image != NULL )
		memcpy( image, _image, width * height * 4 );
}

PPMFrame::~PPMFrame()
{
	delete image;
}

uint8_t *PPMFrame::GetImage( ) const
{
	return image;
}

uint8_t *PPMFrame::GetImage( int &_width, int &_height ) const
{
	_width = width;
	_height = height;
	return image;
}

bool PPMFrame::ReadImage( )
{
	bool ret = false;
	char type[ 4 ];
	int new_width = 0;
	int new_height = 0;
	int maxval = 0;

	if ( ReadHeader( type, new_width, new_height, maxval ) )
	{
		int image_size = new_width * new_height * 4;

		if ( new_width != width || new_height != height )
		{
			width = new_width;
			height = new_height;
			delete image;
			image = new uint8_t[ image_size ];
		}
		if ( image != NULL )
		{
			if ( !strncmp( type, "P4", 2 ) )
			{
				uint8_t *p = image;
				ret = true;
				for ( int i = 0; ret && i < height; i ++ )
				{
					// Pick up most of this row
					for ( int j = 0; ret && j < width / 8; j ++ )
					{
						uint8_t val;
						ret = ReadData( &val, 1 ) == 1;
						for ( int k = 128; k > 0; k = k >> 1 )
						{
							uint8_t pix = ( k & val ) ? 0 : 255;
							*p ++ = pix;
							*p ++ = pix;
							*p ++ = pix;
							*p ++ = 255;
						}
					}
					// Pick up the slops
					if ( width % 8 )
					{
						uint8_t val;
						ret = ReadData( &val, 1 ) == 1;
						for ( int k = 128; k > ( 1 << width % 8 ); k = k >> 1 )
						{
							uint8_t pix = ( k & val ) ? 0 : 255;
							*p ++ = pix;
							*p ++ = pix;
							*p ++ = pix;
							*p ++ = 255;
						}
					}
				}
			}
			else if ( !strncmp( type, "P5", 2 ) )
			{
				uint8_t *p = image;
				uint8_t *end = p + image_size;
				ret = true;
				while ( p < end && ret == true )
				{
					uint8_t val;
					ret = ReadData( &val, 1 ) == 1;
					*p ++ = val;
					*p ++ = val;
					*p ++ = val;
					*p ++ = 255;
				}
			}
			else if ( !strncmp( type, "P6", 2 ) )
			{
				uint8_t *p = image;
				uint8_t *end = p + image_size;
				ret = true;

				int stride = width * 3;
				uint8_t row[ stride ];
				int i;

				while ( p < end && ret == true )
				{
					ret = ReadData( row, stride ) == stride;
					for ( i = 0; i < stride; )
					{
						*p ++ = row[ i ++ ];
						*p ++ = row[ i ++ ];
						*p ++ = row[ i ++ ];
						*p ++ = 255;
					}
				}
			}
			else if ( !strncmp( type, "P8", 2 ) )
			{
				ret = ReadData( image, image_size ) == image_size;
			}
		}
	}
	return ret;
}

bool PPMFrame::WriteImage( bool alpha )
{
	bool ret = false;
	char temp[ 132 ];
	if ( alpha )
	{
		sprintf( temp, "P8\n%d %d\n255\n", width, height );
		if ( image != NULL && WriteData( (uint8_t *)temp, strlen( temp ) ) )
			ret = WriteData( image, width * height * 4 ) == ( width * height * 4 );
	}
	else
	{
		sprintf( temp, "P6\n%d %d\n255\n", width, height );
		if ( image != NULL && WriteData( (uint8_t *)temp, strlen( temp ) ) )
		{
			uint8_t *p = image;
			uint8_t *end = p + width * height * 4;
			//uint8_t r, g, b, a;
			//double value;

			int stride = width * 3;
			uint8_t row[ stride ];
			int i;

			ret = true;

			while ( p < end && ret == true )
			{
				for ( i = 0; i < stride; )
				{
					//r = *p ++;
					//g = *p ++;
					//b = *p ++;
					//a = *p ++;
					//value = (double)a / 255.0;
					//row[ i ++ ] = (uint8_t)( r * value );
					//row[ i ++ ] = (uint8_t)( g * value );
					//row[ i ++ ] = (uint8_t)( b * value );
					row[ i ++ ] = *p ++;
					row[ i ++ ] = *p ++;
					row[ i ++ ] = *p ++;
					p ++;
				}
				ret = WriteData( row, stride ) == stride;
			}
		}
	}

	Flush( );

	return ret;
}

int PPMFrame::ReadData( uint8_t *data, int length )
{
	return fread( data, 1, length, stdin );
}

int PPMFrame::WriteData( uint8_t *data, int length )
{
	return fwrite( data, 1, length, output );
}

bool PPMFrame::Flush( )
{
	return fflush( stdout ) == 0;
}

int PPMFrame::ReadNumber( )
{
	bool eof = false;
	int value = 0;
	uint8_t c = '\0';

	do 
	{
		eof = ReadData( &c, 1 ) == 0;

		while( !eof && !isdigit( c ) && c != '#' )
			eof = ReadData( &c, 1 ) == 0;

		if ( c == '#' )
			while( !eof && c != '\n' )
				eof = ReadData( &c, 1 ) == 0;
	}
	while ( !isdigit( c ) && !eof );

	while( isdigit( c ) && !eof )
	{
		value = value * 10 + ( c - '0' );
		eof = ReadData( &c, 1 ) == 0;
	}

	return value;
}

bool PPMFrame::ReadHeader( char *type, int &_width, int &_height, int &_maxval )
{
	int maxval;
	
	if ( ReadData( ( uint8_t * )type, 2 ) == 2 && 
		 ( !strncmp( type, "P4", 2 ) ||
		   !strncmp( type, "P5", 2 ) ||
		   !strncmp( type, "P8", 2 ) ||
		   !strncmp( type, "P6", 2 ) ) )
	{
		_width = ReadNumber( );
		_height = ReadNumber( );
		if ( strncmp( type, "P4", 2 ) )
			_maxval = ReadNumber( );
		return _width != 0 && _height != 0;
	}

	return false;
}

bool PPMFrame::GetPixel( uint8_t rgb[ 4 ], int x, int y )
{
	if ( x >= 0 && y >= 0 && x < width && y < height )
	{
		uint8_t *p = image + ( y * width + x ) * 4;
		rgb[ 0 ] = *p ++;
		rgb[ 1 ] = *p ++;
		rgb[ 2 ] = *p ++;
		rgb[ 3 ] = *p ++;
		return true;
	}
	return false;
}

bool PPMFrame::SetPixel( uint8_t rgb[ 4 ], int x, int y )
{
	if ( x >= 0 && y >= 0 && x < width && y < height )
	{
		uint8_t *p = image + ( y * width + x ) * 4;
		*p ++ = rgb[ 0 ];
		*p ++ = rgb[ 1 ];
		*p ++ = rgb[ 2 ];
		*p ++ = rgb[ 3 ];
		return true;
	}
	return false;
}

void PPMFrame::FillArea( int x, int y, int _width, int _height, uint8_t rgb[ 4 ] )
{
	int upper_x = x + _width;
	int upper_y = y + _height;
	for ( int j = y; j < upper_y; j ++ )
		for ( int i = x; i < upper_x; i ++ )
			SetPixel( rgb, i, j );
}

bool PPMFrame::Scale( int _width, int _height, int quality )
{
	if ( image == NULL )
	{
		width = _width;
		height = _height;
		image = new uint8_t[ width * height * 4 ];
	}

	if ( width != _width || height != _height )
	{
		GdkInterpType scaler = GDK_INTERP_HYPER;

		switch( quality )
		{
			case 0:
				scaler = GDK_INTERP_NEAREST;
				break;
			case 1:
				scaler = GDK_INTERP_TILES;
				break;
			case 2:
				scaler = GDK_INTERP_BILINEAR;
				break;
		};

		GdkPixbuf *input = gdk_pixbuf_new_from_data( image, GDK_COLORSPACE_RGB, TRUE, 8, 
													 width, height, width * 4, 
													 (GdkPixbufDestroyNotify)NULL, (gpointer)NULL );
		GdkPixbuf *scaled = gdk_pixbuf_scale_simple( input, _width, _height, scaler );

		delete image;
		image = new uint8_t[ _width * _height * 4 ];
		width = _width;
		height = _height;

    	uint8_t *p_src, *p_dest;

    	int stride = gdk_pixbuf_get_rowstride( scaled );

    	p_dest = image;
    	p_src = gdk_pixbuf_get_pixels( scaled );

    	for ( int i = 0; i < height; i++ )
    	{
           	memcpy( p_dest, p_src, width * 4 );
       		p_src += stride;
       		p_dest += width * 4;
    	}

		gdk_pixbuf_unref( scaled );
		gdk_pixbuf_unref( input );
	}

	return true;
}

bool PPMFrame::Load( string filename )
{
	GError *error = NULL;
	GdkPixbuf *scaled = gdk_pixbuf_new_from_file( filename.c_str( ), &error );

	if ( scaled != NULL )
	{
		delete image;

		if ( !gdk_pixbuf_get_has_alpha( scaled ) )
		{
			GdkPixbuf *alpha = gdk_pixbuf_add_alpha( scaled, FALSE, 0, 0, 0 );
			gdk_pixbuf_unref( scaled );
			scaled = alpha;
		}

		width = gdk_pixbuf_get_width( scaled );
		height = gdk_pixbuf_get_height( scaled );
		image = new uint8_t[ width * height * 4 ];

   		uint8_t *p_src, *p_dest;

   		int stride = gdk_pixbuf_get_rowstride( scaled );

   		p_dest = image;
   		p_src = gdk_pixbuf_get_pixels( scaled );

    	for ( int i = 0; i < height; i++ )
    	{
           	memcpy( p_dest, p_src, width * 4 );
       		p_src += stride;
       		p_dest += width * 4;
    	}

		gdk_pixbuf_unref( scaled );

		return true;
	}

	return false;
}

bool PPMFrame::Overlay( PPMFrame &frame, int x, int y, int w, int h, double weight )
{
	int x_start = 0;
	int x_end = w;

	if ( x < 0 )
	{
		x_start = - x;
		x_end += x_start;
	}

	frame.Scale( w, h );

   	int stride = w * 4;

	uint8_t *lower = image;
	uint8_t *upper = image + width * height * 4;

   	uint8_t *p_dest = image + ( y * width + x ) * 4;
   	uint8_t *p_src = frame.GetImage( );

   	for ( int i = 0; i < h; i++ )
   	{
		uint8_t *p = p_src;
		uint8_t *q = p_dest;
		uint8_t *o = p_dest;

		for ( int j = 0; j < w; j ++ )
		{
			if ( q >= lower && q < upper && j >= x_start && j < x_end )
			{
				uint8_t r = *p ++;
				uint8_t g = *p ++;
				uint8_t b = *p ++;
				uint8_t a = *p ++;
				double value = weight * (double)a / 255.0;
				*o ++ = (uint8_t)( r * value + *q++ * ( 1 - value ) );
				*o ++ = (uint8_t)( g * value + *q++ * ( 1 - value ) );
				*o ++ = (uint8_t)( b * value + *q++ * ( 1 - value ) );
				*o ++ = (uint8_t)( a * value + *q++ * ( 1 - value ) );
			}
			else
			{
				p += 4;
				o += 4;
				q += 4;
			}
		}

   		p_src += stride;
		p_dest += width * 4;
   	}

	return true;
}

bool PPMFrame::Overlay( string filename, int x, int y, int w, int h, double weight )
{
	PPMFrame frame;
	frame.Load( filename );
	return Overlay( frame, x, y, w, h, weight );
}

bool PPMFrame::Copy( PPMFrame &frame )
{
	int _width;
	int _height;
	uint8_t *_image = frame.GetImage( _width, _height );
	if ( _width != width || _height != height )
	{
		delete image;
		image = new uint8_t[ _width * _height * 4 ];
		width = _width;
		height = _height;
	}
	memcpy( image, _image, width * height * 4 );
	return true;
}
