/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* $Id: idb.c,v 1.1.1.1 2005/11/04 07:19:34 tkitame Exp $ 
 *
 * Copyright (c) 2005 VA Linux Systems Japan, K.K. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#include "idb.h"
#include "idbdata.h"

static inline int endian_little()
{
  static union {
    unsigned int i;
    char c[sizeof(unsigned int)];
  } d = {i:1};

  return d.c[0];
}

static inline void swap4(void *s)
{
  char tmp;
  if (endian_little()) {
    char *p = s;
    tmp = p[0]; p[0] = p[3]; p[3] = tmp;
    tmp = p[1]; p[1] = p[2]; p[2] = tmp;
  }
}

static void (*idb_errfunc)(int error, char *fmt, ...);
void idb_set_errfunc(void (*func)(int, char *, ...))
{
  idb_errfunc = func;
}

#define idb_err(dbenv,ret,fmt...) { \
    if (idb_errfunc)  { idb_errfunc(ret,fmt); \
    } else if (dbenv) { (dbenv)->err(dbenv,ret,fmt); \
    } else            { fprintf(stderr,fmt); \
                        if (ret) {fprintf(stderr, ": %s",db_strerror(ret));} \
                        fprintf(stderr,"\n"); \
   } \
}

int idb_close(idb_t *idb)
{
  if (idb->sdb != NULL) {
    idb->sdb->close(idb->sdb, 0);
    idb->sdb = NULL;
  }
  if (idb->pdb!= NULL) {
    idb->pdb->close(idb->pdb, 0);
    idb->pdb = NULL;
  }
  if (idb->dbenv != NULL) {
    idb->dbenv->close(idb->dbenv, 0);
    idb->dbenv = NULL;
  }
  return 0;
}

static int idb_set_skey(DB *sdbp, const DBT *pkey, const DBT *pdata, DBT *skey)
{
  memset(skey, 0, sizeof(DBT));
  skey->data = &((idb_data_t *)pdata->data)->xtime;
  skey->size = sizeof ((idb_data_t *)pdata->data)->xtime;
  return 0;
}

int idb_open(idb_t *idb, char *progname, char *dbhome, char *dbname, char *mode, int perm)
{
  int ret;
  int dbenv_mode = 0;
  int db_mode = DB_RDONLY;

  if (mode) {
    if (*mode == 'w') {
      dbenv_mode = 0;
      db_mode = 0;
    } else if (*mode == 'c') {
      dbenv_mode = DB_CREATE;
      db_mode = DB_CREATE;
    }
  }
  memset(idb, 0, sizeof(idb_t));
  if ((ret = db_env_create(&idb->dbenv, 0)) != 0) {
    idb->dbenv = NULL;
    idb_err(idb->dbenv, ret, "idb_open: db_env_create()");
    return (ret);
  }

  idb->dbenv->set_errfile(idb->dbenv, stderr);
  idb->dbenv->set_errpfx(idb->dbenv, progname);

  if ((ret = idb->dbenv->open(idb->dbenv, dbhome,
			      dbenv_mode|DB_INIT_CDB|DB_INIT_MPOOL|DB_THREAD,
			      perm))!=0) {
    idb_err(idb->dbenv, ret, "idb_open: DBENV->open(%s)", dbhome);
    goto err;
  }
  // primary database
  if ((ret = db_create(&idb->pdb, idb->dbenv, 0)) != 0) {
    idb->pdb = NULL;
    idb_err(idb->dbenv, ret, "idb_open: db_create(primary)");
    goto err;
  }
  if ((ret = idb->pdb->open(idb->pdb, NULL, dbname, "primary",
			    DB_BTREE, db_mode|DB_THREAD, perm)) != 0) {
    idb_err(idb->dbenv, ret, "idb_open: DB->open(%s,primary)", dbname);
    goto err;
  }

  // secondary index
  if ((ret = db_create(&idb->sdb, idb->dbenv, 0)) != 0) {
    idb->sdb = NULL;
    idb_err(idb->dbenv, ret, "idb_open: db_create(secondary)");
    goto err;
  }
  if ((ret = idb->sdb->set_flags(idb->sdb,
				 DB_DUP|DB_DUPSORT)) != 0) {
    idb_err(idb->dbenv, ret, "idb_open: DB->set_flags(%s,secondary)", dbname);
    goto err;
  }
  if ((ret = idb->sdb->open(idb->sdb, NULL, dbname, "secondary",
			    DB_BTREE, db_mode|DB_THREAD, perm)) != 0) {
    idb_err(idb->dbenv, ret, "idb_open: DB->open(%s,secondary)", dbname);
    goto err;
  }

  if ((ret = idb->pdb->associate(idb->pdb, NULL, idb->sdb,
				 idb_set_skey,0)) != 0) {
    idb_err(idb->dbenv, ret, "idb_open: DB->associate(%s)", dbname);
    goto err;
  }

  return 0;
 err:
  idb_close(idb);
  return ret;
}

int idb_get(idb_t *idb, idb_data_t *d, unsigned int ip)
{
  int ret;
  DBT key, data;

  memset(&key, 0, sizeof(key));
  memset(&data, 0, sizeof(data));
  
  //swap4(&ip);
  key.data = &ip;
  key.size = sizeof(ip);
  data.data = d;
  data.ulen = sizeof(*d);
  data.flags = DB_DBT_USERMEM;

  if ((ret = idb->pdb->get(idb->pdb, NULL, &key, &data, 0)) != 0) {
    if (ret != DB_NOTFOUND) {
      idb_err(idb->dbenv, ret, "idb_get: DB->get(%08x)", ip);
    }
    return ret;
  }
  //swap4(&d->ip);
  swap4(&d->xtime);
  return 0;
}

int idb_put(idb_t *idb, idb_data_t *d)
{
  int ret;
  DBT key, data;
  memset(&key, 0, sizeof(key));
  memset(&data, 0, sizeof(data));

  //swap4(&d->ip);
  swap4(&d->xtime);

  key.data = &d->ip; key.size = sizeof(d->ip);
  data.data = d; data.size = sizeof(*d);

  ret = idb->pdb->put(idb->pdb, NULL, &key, &data, 0);

  //swap4(&d->ip);
  swap4(&d->xtime);

  if (ret) {
    idb_err(idb->dbenv, ret, "idb_put: DB->put(%08x)", d->ip);
  }
  return ret;
}

int idb_del(idb_t *idb, unsigned int ip)
{
  int ret;
  DBT key;

  memset(&key, 0, sizeof(key));
  
  //swap4(&ip);
  key.data = &ip;
  key.size = sizeof(ip);

  ret = idb->pdb->del(idb->pdb, NULL, &key, 0);
  if (ret && ret != DB_NOTFOUND) {
    idb_err(idb->dbenv, ret, "idb_del: DB->del(%08x)", ip);
  }
  return ret;
}

int idb_modify(idb_t *idb, idb_data_t *d, unsigned int ip,
	       idb_modify_callback_t callback)
{
  int ret, ret2;
  DBT key, data;
  DBC *dbcp;

  memset(&key, 0, sizeof(key));
  memset(&data, 0, sizeof(data));
  
  //swap4(&ip);
  key.data = &ip;
  key.size = sizeof(ip);
  data.data = d;
  data.ulen = sizeof(*d);
  data.flags = DB_DBT_USERMEM;

  if ((ret = idb->pdb->cursor(idb->pdb, NULL, &dbcp, DB_WRITECURSOR)) != 0) {
    idb_err(idb->dbenv, ret, "idb_modify: DB->cursor()");
    return ret;
  }
  ret = dbcp->c_get(dbcp, &key, &data, DB_SET);
  if (ret != 0) {
    if (ret != DB_NOTFOUND) {
      idb_err(idb->dbenv, ret, "idb_modify: DBcursor->c_get(%08x)", ip);
      return ret;
    }
    memset(d, 0, sizeof(d));
    d->ip = ip;
  } else {
    //swap4(&d->ip);
    swap4(&d->xtime);
  }
  if (callback(d) == 0) {
    if ((ret = dbcp->c_put(dbcp, &key, &data, DB_CURRENT)) != 0) {
      idb_err(idb->dbenv, ret, "idb_modify: DBcursor->c_put(%08x)", ip);
    }
  }
  if ((ret2 = dbcp->c_close(dbcp)) != 0) {
    idb_err(idb->dbenv, ret2, "idb_modify: DB->cursor->c_close()");
    ret = ret2;
  }
  return ret;
}

int idb_expire(idb_t *idb, unsigned int t)
{
  DBC *dbcp;
  DBT key, data;
  int ret = 0;
  int ret2 = 0;
  unsigned int dt;

  for (; ; ) {
    if ((ret = idb->sdb->cursor(idb->sdb, NULL, &dbcp, 0)) != 0) {
      idb_err(idb->dbenv, ret, "idb_expire: DB->cursor()");
      break;
    }
    memset(&key, 0, sizeof(key));
    memset(&data, 0, sizeof(data));

    if ((ret = dbcp->c_get(dbcp, &key, &data, DB_NEXT)) != 0) {
      if (ret != DB_NOTFOUND) {
	idb_err(idb->dbenv, ret, "idb_expire: DBcursor->c_get(DB_NEXT)");
      }
    } else {
      memcpy(&dt, key.data, 4);
    }
    if ((ret2 = dbcp->c_close(dbcp)) != 0) {
      idb_err(idb->dbenv, ret2, "idb_expire: DB->cursor->c_close()");
      ret = ret2;
    }
    if (ret != 0) {
      break;
    }
    swap4(&dt);
    if (dt > t) {
      break;
    }
    swap4(&dt);
    memset(&key, 0, sizeof(key));
    key.data = &dt;
    key.size = sizeof(dt);
    ret = idb->sdb->del(idb->sdb, NULL, &key, 0);
    if (ret && ret != DB_NOTFOUND) {
      idb_err(idb->dbenv, ret, "idb_expire: DB->del(%08x)", dt);
      break;
    }
  }
  if (ret == DB_NOTFOUND) {
    ret = 0;
  }
  return ret;
}

int idb_traverse(idb_t *idb, idb_traverse_callback_t callback)
{
  int ret;
  int ret2;
  DBT key, data;
  DBC *dbcp;
  idb_data_t d;

  if ((ret = idb->pdb->cursor(idb->pdb, NULL, &dbcp, 0)) != 0) {
    idb_err(idb->dbenv, ret, "idb_traverse: DB->cursor()");
    return ret;
  }
  memset(&key, 0, sizeof(key));
  memset(&data, 0, sizeof(data));
  data.data = &d;
  data.ulen = sizeof(d);
  data.flags = DB_DBT_USERMEM;

  while ((ret = dbcp->c_get(dbcp, &key, &data, DB_NEXT)) == 0) {
    //swap4(&d.ip);
    swap4(&d.xtime);
    // callback returns:
    //    0: normal
    //   -1: error or quit
    if ((ret = callback(&d)) != 0) {
      break;
    }
  }
  // DB_NOTFOUND is not an error condition
  if (ret == DB_NOTFOUND) {
    ret = 0;
  }
  // assumes bdb does not use -1 as return value
  if (ret != 0 && ret != -1) {
    idb_err(idb->dbenv, ret, "idb_traverse: DBcursor->c_get(DB_NEXT)");
  }
  if ((ret2 = dbcp->c_close(dbcp)) != 0) {
    idb_err(idb->dbenv, ret2, "idb_traverse: DB->cursor->c_close()");
    ret = ret2;
  }
  return ret;
}
