/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* $Id: smtpguard-manager.c,v 1.1.1.1 2005/11/04 07:19:35 tkitame Exp $ 
 *
 * Copyright (c) 2005 VA Linux Systems Japan, K.K. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/un.h>
#include <unistd.h>
#include <signal.h>
#include <time.h>

#include "smtpguard-daemon.h"

#define PROGNAME "smtpguard-manager"

int	main_loop;
int	disp_type;

struct ent_ip {
	unsigned int	ip;
	struct ent_ip	*next;
};

struct ent_ip	geip;
struct ent_ip	*now;


//
// print usage
static void
usage (void)
{
	fprintf(stderr, "Usage: %s [OPTION]... [COMMAND]\n", PROGNAME);
	fprintf(stderr, "OPTION:\n");
	fprintf(stderr, "  -h dbhome     directory where the database is stored\n");
	fprintf(stderr, "                (default: %s)\n", DEFAULT_DBHOME);
	fprintf(stderr, "  -n dbname     main database file\n");
	fprintf(stderr, "                (default: %s)\n", DEFAULT_DBNAME);
	fprintf(stderr, "  -c            use csv format for output (default)\n");
	fprintf(stderr, "  -p            use verbose format for output\n");
	fprintf(stderr, "COMMAND:\n");
	fprintf(stderr, "  delete [all | ip ...]\n");
	fprintf(stderr, "                delete all data or specified record\n");
	fprintf(stderr, "  print [all | ip ...]\n");
	fprintf(stderr, "                print all data or specified record\n");
	fprintf(stderr, "  set [ip expire-time SC FC TC P]\n");
	fprintf(stderr, "                modify db data\n");
	fprintf(stderr, "NOTE:\n");
	fprintf(stderr, "     ip                : xxx.xxx.xxx.xxx\n");
	exit(1);
}


//
// get ip for entry data
int get_ip(idb_data_t *d)
{
	struct ent_ip	*neip;

	if (!main_loop) return -1;

	now->ip = d->ip;
	neip = malloc(sizeof(struct ent_ip));
	if (neip == NULL) {
		perror("malloc");
		return -1;
	}
	neip->next = NULL;
	now->next = neip;
	now = neip;

	return 0;
}


//
// free ip list
void free_ent_ip(struct ent_ip *eip)
{
	if (!eip) return;
	free_ent_ip(eip->next);
	free(eip);
}


//
// print entry
int print_entry(idb_data_t *d)
{
	struct in_addr	inp;
	char	*ip;

	if (!main_loop) return 1;

	inp.s_addr = d->ip;
	ip = inet_ntoa(inp);

	if (disp_type) {
		fprintf(stdout, "IP: %s\n", ip);
		fprintf(stdout, "DELETE AFTER: %lu\n", d->xtime - time(NULL));
		fprintf(stdout, "TIME DEL: %u  REG: %u  MOD: %u\n", d->xtime, d->rtime, d->mtime);
		fprintf(stdout, "TOTAL SESSIONS: %u\n", d->SC);
		fprintf(stdout, "TOTAL MAILFROM: %u\n", d->FC);
		fprintf(stdout, "TOTAL RCPTTO:   %u\n", d->TC);
		fprintf(stdout, "POINT:          %u\n", d->P);
	}
	else {
		fprintf(stdout, "%s,%u,%u,%u,%u,%u,%u,%u\n",
				ip, d->xtime, d->rtime, d->mtime,
				d->SC, d->FC, d->TC, d->P);
	}

	return 0;
}

int get_num(char *p, unsigned int *val)
{
  char *r;
  errno = 0;
  *val = strtoul(p, &r, 0);
  if (errno) {
    perror(p);
    return -1;
  } else if (*r) {
    fprintf(stderr, "bad number: %s\n", p);
    return -1;
  }
  return 0;
}

//
// set data to DB
int set_data(idb_t *idb, int num, char **target)
{
	int	ret;
	idb_data_t	d;
	struct in_addr	inp;
	unsigned int xt;

	if (num < 6) {
		fprintf(stderr, "There are few variables.\n");
		return -1;
	}

	if (!inet_aton(target[0], &inp)) {
		fprintf(stderr, "illegal ip address: %s\n", target[0]);
		return -1;
	}
	memset(&d, 0, sizeof(idb_data_t));
	d.ip = inp.s_addr;
	d.rtime = time(NULL);
	d.mtime = d.rtime;

	if (get_num(target[1], &xt)) return -1;
	d.xtime = d.rtime + xt;

	if (get_num(target[2], &d.SC)) return -1;
	if (get_num(target[3], &d.FC)) return -1;
	if (get_num(target[4], &d.TC)) return -1;
	if (get_num(target[5], &d.P)) return -1;

	print_entry(&d);
	// put data to DB
	if((ret = update_db(idb, &d, 1)) != 0) {
		fprintf(stderr, "database error: put: %d\n", ret);
		return 1;
	}

	return 0;
}


//
// print data from DB
int print_data(idb_t *idb, int num, char **target)
{
	int	i;
	int	ret;
	idb_data_t	d;
	struct in_addr	inp;
	struct ent_ip	*cur;

	for (i = 0; i < num; i++) {
		if (!strcmp(target[i], "all")) {
			now = &geip;
			if ((ret = idb_traverse(idb, get_ip)) != 0) {
				return 1;
			}

			for (cur = &geip; cur->next != NULL; cur = cur->next) {
				if ((ret = idb_get(idb, &d, cur->ip)) != 0 && ret != DB_NOTFOUND) {
					return -1;
				}
				else if (ret == DB_NOTFOUND) {
					continue;
				}
				print_entry(&d);
			}
			free_ent_ip(geip.next);
			
			return 0;
		}

		if (!inet_aton(target[i], &inp)) {
			fprintf(stderr, "illegal ip address: %s\n", target[i]);
			return -1;
		}
		if ((ret = idb_get(idb, &d, inp.s_addr)) != 0 && ret != DB_NOTFOUND) {
			return -1;
		}
		else if (ret == DB_NOTFOUND) {
			fprintf(stdout, "%s not found.\n", target[i]);
			continue;
		}

		print_entry(&d);
	}

	return 0;
}


//
// delete data on DB
int delete_data(idb_t *idb, int num, char **target)
{
	int	i;
	int	ret;
	idb_data_t	d;
	struct in_addr	inp;
	struct ent_ip	*cur;

	for (i = 0; i < num; i++) {
		if (!strcmp(target[i], "all")) {
			now = &geip;
			if ((ret = idb_traverse(idb, get_ip)) != 0) {
				return 1;
			}

			for (cur = &geip; cur->next != NULL; cur = cur->next) {
				inp.s_addr = cur->ip;
				fprintf(stdout, "delete %s...\n", inet_ntoa(inp));
				if((ret = idb_del(idb, cur->ip)) != 0 && ret != DB_NOTFOUND) {
					return -1;
				}
			}
			free_ent_ip(geip.next);
			
			return 0;
		}

		if (!inet_aton(target[i], &inp)) {
			fprintf(stderr, "illegal ip address: %s\n", target[i]);
			return -1;
		}
		if ((ret = idb_get(idb, &d, inp.s_addr)) != 0 && ret != DB_NOTFOUND) {
			return -1;
		}
		else if (ret == DB_NOTFOUND) {
			fprintf(stdout, "%s not found.\n", target[i]);
			return 0;
		}
		fprintf(stdout, "delete %s...\n", target[i]);
		if((ret = idb_del(idb, inp.s_addr)) != 0 && ret != DB_NOTFOUND) {
			return -1;
		}
	}

	return 0;
}


//
// signal handler
void sigterm_catch(int signo)
{
	sg_debug(1, "catch SIGTERM\n");
	main_loop = 0;
	return;
}


//
// main program
int main(int argc, char **argv)
{
	int	ret;
	int	num;
	char 	c;
	char 	*dbhome = DEFAULT_DBHOME;
	char 	*dbname = DEFAULT_DBNAME;
	char	*kind;
	char	**target;
	idb_t		idb;

	main_loop = 1;
	disp_type = 0;
	spam_errmsg[0] = 0;

	sg_set_strerror(db_strerror);
	idb_set_errfunc(sg_err);

	if (getenv(SG_DBHOME)) {
		dbhome = getenv(SG_DBHOME);
	}
	if (getenv(SG_DBNAME)) {
		dbname = getenv(SG_DBNAME);
	}

	opterr = 0;
	while ((c = getopt(argc, argv, "ch:n:t:p")) != -1) {
		switch(c) {
			case 'c':
				disp_type = 0;
				break;
			case 'h':
				dbhome = strdup(optarg);
				if (dbhome == NULL) {
					sg_err(errno, "strdup for dbhome");
					exit(1);
				}
				break;
			case 'n':
				dbname = strdup(optarg);
				if (dbname == NULL) {
					sg_err(errno, "strdup for dbname");
					exit(1);
				}
				break;
			case 'p':
				disp_type = 1;
				break;
			default:
				break;
		}
	}

	if (optind + 1 >= argc) {
		fprintf(stderr, "%s: illegal sub command.\n", PROGNAME);
		usage();
	}

	kind = argv[optind];
	target = &argv[optind + 1];
	num = argc - optind - 1;
	if (strcmp(kind, "delete") && strcmp(kind, "print") && strcmp(kind, "set")) {
		fprintf(stderr, "%s: illegal sub command.\n", PROGNAME);
		usage();
	}

	if (dbhome == NULL || dbname == NULL) {
		fprintf(stderr, "DB parameter isn't set up.\n");
		exit(1);
	}

	// signal handler
	signal(SIGINT, sigterm_catch);
	signal(SIGTERM, sigterm_catch);
	signal(SIGPIPE, sigterm_catch);

	// open DB
	if ((ret = idb_open(&idb, PROGNAME, dbhome, dbname, "w", 0600)) != 0) {
		exit(1);
	}

	ret = 0;
	if (!strcmp(kind, "delete")) {
		ret = delete_data(&idb, num, target);
	}
	else if (!strcmp(kind, "print")) {
		ret = print_data(&idb, num, target);
	}
	else if (!strcmp(kind, "set")) {
		ret = set_data(&idb, num, target);
		if (ret && ret == -1) {
			usage();
		}
	}

	// close DB
	if (idb_close(&idb) != 0) {
		exit(1);
	}

	return ret;
}
