/*==================================================================
 * uif_menutbar.c - Menu and tool bar interface routines
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include <stdio.h>
#include <gtk/gtk.h>
#include "uif_menutbar.h"
#include "uif_sfont.h"
#include "uif_sample.h"
#include "uif_sfgen.h"
#include "uif_pianospan.h"
#include "uif_piano.h"
#include "uif_pref.h"
#include "uif_sftree.h"
#include "uif_help.h"
#include "uiface.h"
#include "wavetable.h"
#include "pixmap.h"
#include "sequencer.h"
#include "smurfjam.h"
#include "midi.h"
#include "util.h"
#include "i18n.h"

static GtkWidget *tbar_seq_togbtn;
static GtkWidget *tbar_gen_view_togbtn;
static GtkWidget *tbar_gen_ctrl_togbtn;
static GtkWidget *tbar_sam_view_togbtn;
static GtkWidget *tbar_piano_togbtn;
static GtkWidget *tbar_velocity_togbtn;

static GtkWidget *tbar_seq_on_pm;
static GtkWidget *tbar_seq_off_pm;
static gboolean tbar_seq_pixmap_on = FALSE;	/* T, F = seq_on_pm, seq_off_pm */

/*** menu functions ***/

void
sfmenu_view_log (void)
{
  log_view (NULL);
}

/* restarts the wavetable and midi drivers (possibly changing to another one
   selected by the user in preferences) */
void
sfmenu_restart_sequencer (void)
{
  if (midi_active) midi_close ();
  if (seq_active) seq_close ();
  if (wtbl_active) wtbl_close ();

  wtbl_init_from_config ();
  seq_init_from_config ();
  midi_init_from_config ();

  /* initialize all virtual keyboard (piano) controls */
  uipiano_set_all_controls ();

  tbar_update_seq_togbtn ();	/* update button to state of sequencer */
}

void
sfmenu_cb_find (void)
{
  sftree_set_selection (NULL);	/* clear selection so find goes global */
  uisf_find ();
  sftree_unset_selection ();	/* re-enable selection */
}

void
sfmenu_set_filemenu_save_sensitive (gboolean state)
{
  GtkWidget *widg;

  widg = gtk_object_get_data (GTK_OBJECT (ui_main_window), "save");
  gtk_widget_set_sensitive (widg, state);

  widg = gtk_object_get_data (GTK_OBJECT (ui_main_window), "save_as");
  gtk_widget_set_sensitive (widg, state);
}

void
sfmenu_cb_save (void)
{
  if (!uisf_selected_uisfont)
    {
      util_quick_popup (_("Nothing selected in sound font tree"), NULL);
      return;
    }

  uisf_save_sfont (FALSE);
}

void
sfmenu_cb_save_as (void)
{
  if (!uisf_selected_uisfont)
    {
      util_quick_popup (_("Nothing selected in sound font tree"), NULL);
      return;
    }

  uisf_save_sfont (TRUE);
}

void
sfmenu_cb_gen_view (GtkWidget * widg)
{
  if (GTK_CHECK_MENU_ITEM (widg)->active)
    {
      sfgen_set_win (SFGEN_VIEW);
      tbar_set_lowpane_togbtn (SFGEN_VIEW);
    }
}

void
sfmenu_cb_gen_ctrl (GtkWidget * widg)
{
  if (GTK_CHECK_MENU_ITEM (widg)->active)
    {
      sfgen_set_win (SFGEN_CTRL);
      tbar_set_lowpane_togbtn (SFGEN_CTRL);
    }
}

void
sfmenu_cb_sam_view (GtkWidget * widg)
{
  if (GTK_CHECK_MENU_ITEM (widg)->active)
    {
      sfgen_set_win (SFGEN_SAMVIEW);
      tbar_set_lowpane_togbtn (SFGEN_SAMVIEW);
    }
}

void
sfmenu_cb_piano_mode (GtkWidget * widg)
{
  if (GTK_CHECK_MENU_ITEM (widg)->active)
    {
      pianospan_set_mode (PIANOSPAN_PIANO);
      pianospan_update ();
      tbar_set_piano_mode_togbtn (PIANOSPAN_PIANO);
    }
}

void
sfmenu_cb_velocity_mode (GtkWidget * widg)
{
  if (GTK_CHECK_MENU_ITEM (widg)->active)
    {
      pianospan_set_mode (PIANOSPAN_VELOCITY);
      pianospan_update ();
      tbar_set_piano_mode_togbtn (PIANOSPAN_VELOCITY);
    }
}

void
sfmenu_set_lowpane_radio_item (gint mode)
{
  GtkWidget *mnuitem;

  switch (mode)
    {
    case SFGEN_VIEW:
      mnuitem = gtk_object_get_data (GTK_OBJECT (ui_main_window),
				     "MNUgenview");
      break;
    case SFGEN_CTRL:
      mnuitem = gtk_object_get_data (GTK_OBJECT (ui_main_window),
				     "MNUgenctrl");
      break;
    case SFGEN_SAMVIEW:
      mnuitem = gtk_object_get_data (GTK_OBJECT (ui_main_window),
				     "MNUsamview");
      break;
    }
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mnuitem), TRUE);
}

void
sfmenu_set_piano_mode_radio_item (gint mode)
{
  GtkWidget *mnuitem;

  switch (mode)
    {
    case PIANOSPAN_PIANO:
      mnuitem = gtk_object_get_data (GTK_OBJECT (ui_main_window),
				     "MNUpiano");
      break;
    case PIANOSPAN_VELOCITY:
      mnuitem = gtk_object_get_data (GTK_OBJECT (ui_main_window),
				     "MNUvelocity");
      break;
    }
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mnuitem), TRUE);
}

void
sfmenu_update_control_buttons (void)
{
  GtkWidget *mnuitem;

  mnuitem = gtk_object_get_data (GTK_OBJECT (ui_main_window), "MNUwtblCache");
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mnuitem),
				  wtbl_cache_samples);

  mnuitem = gtk_object_get_data (GTK_OBJECT (ui_main_window),"MNUpianoFollows");
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mnuitem),
				  uisf_piano_follows);

  mnuitem = gtk_object_get_data(GTK_OBJECT(ui_main_window),"MNUautoTmpAudible");
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mnuitem),
				  uisf_auto_temp_audible);

  mnuitem = gtk_object_get_data (GTK_OBJECT (ui_main_window), "MNUbankRefresh");
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mnuitem),
				  wtbl_recursive_bank_refresh);
  /* set bank refresh item to sensitive only if sample caching active */
  gtk_widget_set_sensitive (mnuitem, wtbl_cache_samples);
}

void
sfmenu_wavetable_caching (GtkWidget *menuitem)
{
  GtkWidget *widg;

  wtbl_cache_samples = GTK_CHECK_MENU_ITEM (menuitem)->active;

  /* set bank refresh item to sensitive only if sample caching active */
  widg = gtk_object_get_data (GTK_OBJECT (ui_main_window), "MNUbankRefresh");
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (widg),
				  wtbl_cache_samples ?
				  wtbl_recursive_bank_refresh : FALSE);
  gtk_widget_set_sensitive (widg, wtbl_cache_samples);
}

void
sfmenu_piano_follows (GtkWidget *menuitem)
{
  uisf_piano_follows = GTK_CHECK_MENU_ITEM (menuitem)->active;
}

void
sfmenu_auto_temp_audible (GtkWidget *menuitem)
{
  uisf_auto_temp_audible = GTK_CHECK_MENU_ITEM (menuitem)->active;
}

void
sfmenu_bank_refresh (GtkWidget *menuitem)
{
  wtbl_recursive_bank_refresh = GTK_CHECK_MENU_ITEM (menuitem)->active;
}

/* not available YET, still in dream phase */
#if 0
void
sfmenu_start_jam_server (void)
{
  jam_server (44002);
}

void
sfmenu_start_jam_client (void)
{
  jam_client ("127.0.0.1", 44002);
}
#endif

/* Toolbar functions */

void
tbar_create (void)
{
  GtkWidget *handlebox;
  GtkWidget *toolbar;
  GtkWidget *icon;

  handlebox = gtk_object_get_data (GTK_OBJECT (ui_main_window), "HNDLtbar");

  toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
  gtk_toolbar_set_space_size (GTK_TOOLBAR (toolbar), 10);
  gtk_container_set_border_width (GTK_CONTAINER (toolbar), 5);
  gtk_toolbar_set_tooltips (GTK_TOOLBAR (toolbar), TRUE);
  gtk_container_add (GTK_CONTAINER (handlebox), toolbar);

  gtk_widget_show (toolbar);

  gtk_object_set_data (GTK_OBJECT (ui_main_window), "toolbar1", toolbar);

  /* Sequencer on/off toggle button (pixmap toggles too) */

  tbar_seq_on_pm = util_create_pixmap (seq_on_xpm);
  gtk_widget_show (tbar_seq_on_pm);
  gtk_widget_ref (tbar_seq_on_pm);	/* reference inactive pixmap widget */

  tbar_seq_off_pm = util_create_pixmap (seq_off_xpm);
  gtk_widget_show (tbar_seq_off_pm);

  tbar_seq_togbtn = gtk_toggle_button_new ();
  gtk_container_add (GTK_CONTAINER (tbar_seq_togbtn), tbar_seq_off_pm);
  gtk_widget_show (tbar_seq_togbtn);
  gtk_signal_connect (GTK_OBJECT (tbar_seq_togbtn), "toggled",
    (GtkSignalFunc)tbar_cb_seq_toggle, NULL);

  gtk_toolbar_append_widget (GTK_TOOLBAR (toolbar), tbar_seq_togbtn,
    _("Toggle sequencer on/off"), NULL);


  gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));


  /* lower pane toggle buttons */

  icon = util_create_pixmap (gen_view_xpm);
  tbar_gen_view_togbtn = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
    GTK_TOOLBAR_CHILD_RADIOBUTTON, NULL, _("Gen View"),
    _("View generator values"), NULL,
    icon, NULL, NULL);
  gtk_widget_show (tbar_gen_view_togbtn);

  icon = util_create_pixmap (gen_ctrl_xpm);
  tbar_gen_ctrl_togbtn = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
    GTK_TOOLBAR_CHILD_RADIOBUTTON, tbar_gen_view_togbtn, _("Gen Control"),
    _("Set generator values"), NULL,
    icon, NULL, NULL);
  gtk_widget_show (tbar_gen_ctrl_togbtn);

  icon = util_create_pixmap (samview_xpm);
  tbar_sam_view_togbtn = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
    GTK_TOOLBAR_CHILD_RADIOBUTTON, tbar_gen_ctrl_togbtn, _("Sample View"),
    _("Set sample loop parameters"), NULL,
    icon, NULL, NULL);
  gtk_widget_show (tbar_sam_view_togbtn);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (tbar_gen_view_togbtn),
				TRUE);

  gtk_signal_connect (GTK_OBJECT (tbar_gen_view_togbtn), "toggled",
    (GtkSignalFunc)tbar_cb_gen_set_win, GINT_TO_POINTER(SFGEN_VIEW));
  gtk_signal_connect (GTK_OBJECT (tbar_gen_ctrl_togbtn), "toggled",
    (GtkSignalFunc)tbar_cb_gen_set_win, GINT_TO_POINTER(SFGEN_CTRL));
  gtk_signal_connect (GTK_OBJECT (tbar_sam_view_togbtn), "toggled",
    (GtkSignalFunc)tbar_cb_gen_set_win, GINT_TO_POINTER(SFGEN_SAMVIEW));


  gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));


  /* piano/velocity toggle buttons */

  icon = util_create_pixmap (piano_xpm);
  tbar_piano_togbtn = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
    GTK_TOOLBAR_CHILD_RADIOBUTTON, NULL, _("Piano"), _("Piano mode"),
    NULL, icon, NULL, NULL);
  gtk_widget_show (tbar_piano_togbtn);

  icon = util_create_pixmap (velocity_xpm);
  tbar_velocity_togbtn = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
    GTK_TOOLBAR_CHILD_RADIOBUTTON, tbar_piano_togbtn, _("Velocity"),
    _("Velocity mode"), NULL, icon, NULL, NULL);
  gtk_widget_show (tbar_velocity_togbtn);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (tbar_piano_togbtn), TRUE);

  gtk_signal_connect (GTK_OBJECT (tbar_piano_togbtn), "toggled",
    (GtkSignalFunc)tbar_cb_pianospan_set_mode,
    GINT_TO_POINTER (PIANOSPAN_PIANO));
  gtk_signal_connect (GTK_OBJECT (tbar_velocity_togbtn), "toggled",
    (GtkSignalFunc)tbar_cb_pianospan_set_mode,
    GINT_TO_POINTER (PIANOSPAN_VELOCITY));

  tbar_update_seq_togbtn ();

  uipiano_create_controls ();
}

void
tbar_update_seq_togbtn (void)
{
  tbar_set_seq_toggle (tbar_seq_togbtn, seq_active);
}

void
tbar_set_seq_toggle (GtkWidget * btn, gboolean active)
{
  GtkWidget *old_pm, *new_pm;

  if (tbar_seq_pixmap_on != active)	/* change pixmap? */
    {
      if (tbar_seq_pixmap_on)		/* change from "on" state to "off" */
	{
	  old_pm = tbar_seq_on_pm;
	  new_pm = tbar_seq_off_pm;
	}
      else				/* change from "off" state to "on" */
	{
	  old_pm = tbar_seq_off_pm;
	  new_pm = tbar_seq_on_pm;
	}
      gtk_widget_ref (old_pm);		/* ref so doesn't get destroyed */
      gtk_container_remove (GTK_CONTAINER (tbar_seq_togbtn), old_pm);
      gtk_container_add (GTK_CONTAINER (tbar_seq_togbtn), new_pm);
      gtk_widget_unref (new_pm);	/* unref bogus reference */

      tbar_seq_pixmap_on = active;
    }

  gtk_signal_handler_block_by_func (GTK_OBJECT (btn),
    GTK_SIGNAL_FUNC (tbar_cb_seq_toggle), NULL);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (btn), active);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (btn),
    GTK_SIGNAL_FUNC (tbar_cb_seq_toggle), NULL);
}

gboolean
tbar_cb_seq_toggle (GtkWidget * btn)
{
  if (seq_active)
    {
      midi_close ();
      seq_close ();
      wtbl_close ();
    }
  else
    {
      if (!wtbl_init ()) return (FALSE);
      if (!seq_init ()) return (FALSE);
      if (!midi_init ()) return (FALSE);

      /* initialize all virtual keyboard (piano) controls */
      uipiano_set_all_controls ();
    }

  /* set the button to the current state of the sequencer */
  tbar_set_seq_toggle (btn, seq_active);

  return (FALSE);
}

gboolean
tbar_cb_gen_set_win (GtkWidget * rad, gint wintype)
{
  if (!GTK_TOGGLE_BUTTON (rad)->active)
    return (TRUE);

  sfgen_set_win (wintype);

  /* update the view menu radio buttons */
  sfmenu_set_lowpane_radio_item (wintype);

  return (FALSE);
}

void
tbar_cb_pianospan_set_mode (GtkWidget * rad, gint mode)
{
  if (!GTK_TOGGLE_BUTTON (rad)->active)
    return;

  pianospan_set_mode (mode);
  pianospan_update ();


  /* update the view menu radio buttons */
  sfmenu_set_piano_mode_radio_item (mode);
}

/* sets which lower pane toggle button is active on toolbar */
void
tbar_set_lowpane_togbtn (gint mode)
{
  GtkWidget *btn;

  switch (mode)
    {
    case SFGEN_VIEW:
      btn = tbar_gen_view_togbtn;
      break;
    case SFGEN_CTRL:
      btn = tbar_gen_ctrl_togbtn;
      break;
    case SFGEN_SAMVIEW:
      btn = tbar_sam_view_togbtn;
      break;
    }
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (btn), TRUE);
}

/* sets piano or velocity button to active on toolbar */
void
tbar_set_piano_mode_togbtn (gint mode)
{
  GtkWidget *btn;

  switch (mode)
    {
    case PIANOSPAN_PIANO:
      btn = tbar_piano_togbtn;
      break;
    case PIANOSPAN_VELOCITY:
      btn = tbar_velocity_togbtn;
      break;
    }
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (btn), TRUE);
}
