/*==================================================================
 * uif_piano.c - Piano user interface routines
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include <config.h>

#include <stdio.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "uif_piano.h"
#include "widgets/piano.h"
#include "smurfcfg.h"
#include "sequencer.h"
#include "wavetable.h"
#include "uiface.h"
#include "util.h"
#include "i18n.h"

static gint uipiano_parse_smurfcfg_octkeys (gchar *str, guint **keys);
static void uipiano_set_bank_dummy (gint chan, gint bank);
static void uipiano_set_preset_dummy (gint chan, gint preset);
static void uipiano_cb_note_on (Piano * piano, guint keynum);
static void uipiano_cb_note_off (Piano * piano, guint keynum);
static void uipiano_cb_control_selection_done (GtkWidget * menu);
static void uipiano_cb_control_value_changed (GtkWidget * adj,
					      GtkWidget *opmenu);
static void uipiano_update_spin_button (gint ctrlindex);

static void uipiano_chan_value_changed (GtkAdjustment *adj, GtkWidget *spb);
static void uipiano_bank_value_changed (GtkAdjustment *adj, GtkWidget *spb);
static void uipiano_pset_value_changed (GtkAdjustment *adj, GtkWidget *spb);

GtkWidget *uipiano_widg;
static gboolean uipiano_actvkeys[128];	/* active piano widget keys */
static gint uipiano_adj_changed_handler;   /* ctrl changed signal handler id */

gint uipiano_chan = 0;	/* midi channel of virtual keyboard (starts with 0) */

/* default key table */
guint uipiano_default_keytable[] =
{
  /* lower keyboard */
  GDK_z, GDK_s, GDK_x, GDK_d, GDK_c, GDK_v, GDK_g, GDK_b, GDK_h, GDK_n, GDK_j,
  GDK_m, GDK_comma, GDK_l, GDK_period, GDK_semicolon, GDK_slash,

  /* upper keyboard */
  GDK_q, GDK_2, GDK_w, GDK_3, GDK_e, GDK_r, GDK_5, GDK_t, GDK_6, GDK_y, GDK_7,
  GDK_u, GDK_i, GDK_9, GDK_o, GDK_0, GDK_p, GDK_bracketleft, GDK_equal,
  GDK_bracketright
};

/* piano keytable stored here */
guint uipiano_keytable[UIPIANO_TOTAL_NUMKEYS];

typedef struct _MidiCtrlParms {
    gchar *name;	/* name of MIDI control */
    gint lo;		/* lowest value */
    gint hi;		/* highest value */
    gint def;		/* default */
    void (*func)(gint ch, gint val);	/* function for setting this control */
}MidiCtrlParms;

/* --- ! Update if more controls are added to uipiano_ctrl_parms ! --- */
enum { UIMIDI_VOLUME, UIMIDI_VELOCITY_ON, UIMIDI_VELOCITY_OFF, UIMIDI_REVERB,
       UIMIDI_CHORUS, UIMIDI_OCTAVE, UIMIDI_BEND_RANGE, UIMIDI_BANK,
       UIMIDI_PSET, UIMIDI_CTRL_COUNT };

static const MidiCtrlParms uipiano_ctrl_parms[] = {
  { N_("Volume"), 0, 127, 100, seq_main_volume },
  { N_("Velocity On"), 0, 127, 127, NULL },
  { N_("Velocity Off"), 0, 127, 127, NULL },
  { N_("Reverb"), 0, 127,   0, seq_reverb },
  { N_("Chorus"), 0, 127,   0, seq_chorus },
  { N_("Octave"), 0,  10,   3, NULL },
  { N_("Bend Range"), 0, 127, 2, seq_pitch_bend_range },
  { NULL, 0, 127, 0, uipiano_set_bank_dummy }, /* Bank control */
  { NULL, 0, 127, 0, uipiano_set_preset_dummy } /* Preset control */
};

/* 16 arrays (one for each MIDI channel) of integer MIDI control values,
 [16][UIMIDI_CTRL_COUNT] */
static gint (*uipiano_ctrl_vals)[16][UIMIDI_CTRL_COUNT];


void
uipiano_create (void)
{
  gint i, c;
  gchar *s;
  guint *keyvals;
  guint *srcvals;

  /* initialize piano key table low octave */
  s = smurfcfg_get_val (SMURFCFG_PIANO_LOWOCTKEYS)->v_string;
  c = uipiano_parse_smurfcfg_octkeys (s, &keyvals);

  if (c == UIPIANO_LOW_NUMKEYS)
    srcvals = keyvals;  /* if smurf config variable contains valid # of keys */
  else
    srcvals = uipiano_default_keytable;  /* smurfcfg not valid, use default */

  for (i = 0; i < UIPIANO_LOW_NUMKEYS; i++)  /* copy keys to key table */
    uipiano_keytable[i] = srcvals[i];

  if (c) g_free (keyvals);


  /* initialize piano key table hi octave */
  s = smurfcfg_get_val (SMURFCFG_PIANO_HIOCTKEYS)->v_string;
  c = uipiano_parse_smurfcfg_octkeys (s, &keyvals);

  if (c == UIPIANO_HI_NUMKEYS)
    srcvals = keyvals;  /* if smurf config variable contains valid # of keys */
  else   /* smurfcfg not valid, use default */
    srcvals = &uipiano_default_keytable[UIPIANO_LOW_NUMKEYS];

  for (i = 0; i < UIPIANO_HI_NUMKEYS; i++)  /* copy keys to key table */
    uipiano_keytable[i + UIPIANO_LOW_NUMKEYS] = srcvals[i];

  if (c) g_free (keyvals);


  /* create the piano widget */
  for (i = 0; i < 128; i++)
    uipiano_actvkeys[i] = FALSE;
  uipiano_widg = piano_new ((gboolean *) & uipiano_actvkeys);
  gtk_signal_connect (GTK_OBJECT (uipiano_widg), "note-on",
    GTK_SIGNAL_FUNC (uipiano_cb_note_on), NULL);
  gtk_signal_connect (GTK_OBJECT (uipiano_widg), "note-off",
    GTK_SIGNAL_FUNC (uipiano_cb_note_off), NULL);
}

/* parses a smurfcfg octave keys string (key names seperated by commas)
   INPUT: str is the string to parse, keys gets a pointer to an array of
   unsigned integer key values which should be free'd when no longer needed
   RETURNS: # of key names parsed or 0 on error
*/
static gint
uipiano_parse_smurfcfg_octkeys (gchar *str, guint **keys)
{
  gchar **keynames;
  guint *keyvals;
  gint i;

  /* allocate for array of maximum number of expected keys */
  keyvals = g_malloc (SMURFCFG_PIANO_HIOCTKEYS * sizeof (guint));

  keynames = g_strsplit (str, ",", SMURFCFG_PIANO_HIOCTKEYS);

  i = 0;
  while (keynames[i])
    {
      keyvals[i] = gdk_keyval_from_name (keynames[i]);
      if (keyvals[i] == GDK_VoidSymbol)  /* invalid keyname? */
	{
	  g_free (keyvals);
	  g_strfreev (keynames);
	  return (0);
	}
      i++;
    }

  g_strfreev (keynames);

  if (i)
    *keys = keyvals;
  else
    g_free (keyvals);

  return (i);
}

/* composes a smurfcfg key name string
   INPUT: keyvals is an array of unsigned integer key values,
     keycount is the # of keys
   RET: pointer to key names string or NULL (string should be free'd)
*/
gchar *
uipiano_encode_smurfcfg_octkeys (guint *keyvals, gint keycount)
{
  gchar **keynames;
  gchar *s;
  gint i;

  /* allocate for array of key name strings */
  keynames = g_malloc ((keycount + 1) * sizeof (gchar *));

  for (i = 0; i < keycount; i++)  /* loop over each key */
    {
      keynames[i] = gdk_keyval_name (keyvals[i]);  /* get key name */
      if (!keynames[i])  /* if no key name for key, fail */
	{
	  g_free (keynames);
	  return (NULL);
	}
    }

  keynames[keycount] = NULL;

  s = g_strjoinv (",", keynames);  /* create the comma delimited key string */

  g_free (keynames);

  return (s);
}

void
uipiano_set_chan (gint chan)
{
  GtkWidget *widg;
  GtkAdjustment *radj;
  gint ndx;

  uipiano_chan = chan - 1;	/* uipiano_chan starts at 0 */

  /* update current control value for new channel */
  widg = gtk_object_get_data (GTK_OBJECT (ui_main_window), "OPTctrl");
  UTIL_OPMENU_INDEX (ndx, widg);
  uipiano_update_spin_button (ndx);

  /* update Bank and Preset spin buttons */
  widg = gtk_object_get_data (GTK_OBJECT (ui_main_window), "SPBbank");
  radj = gtk_spin_button_get_adjustment (GTK_SPIN_BUTTON (widg));
  gtk_signal_handler_block_by_func (GTK_OBJECT (radj),
				    uipiano_bank_value_changed,widg);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg),
			     (float)((*uipiano_ctrl_vals)
			     [uipiano_chan][UIMIDI_BANK]));
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (radj),
				      uipiano_bank_value_changed, widg);

  widg = gtk_object_get_data (GTK_OBJECT (ui_main_window), "SPBpset");
  radj = gtk_spin_button_get_adjustment (GTK_SPIN_BUTTON (widg));
  gtk_signal_handler_block_by_func (GTK_OBJECT (radj),
				    uipiano_pset_value_changed,widg);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg),
			     (float)((*uipiano_ctrl_vals)
			     [uipiano_chan][UIMIDI_PSET]));
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (radj),
				      uipiano_pset_value_changed, widg);
}

void
uipiano_set_bank (gint bank)
{
  GtkWidget *widg;
  GtkAdjustment *adj;

  seq_set_bank (uipiano_chan, bank);

  (*uipiano_ctrl_vals)[uipiano_chan][UIMIDI_BANK] = bank;

  widg = gtk_object_get_data (GTK_OBJECT (ui_main_window), "SPBbank");
  adj = gtk_spin_button_get_adjustment (GTK_SPIN_BUTTON (widg));
  gtk_signal_handler_block_by_func (GTK_OBJECT (adj),
				    uipiano_bank_value_changed, widg);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)bank);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (adj),
				      uipiano_bank_value_changed, widg);
}

void
uipiano_set_preset (gint preset)
{
  GtkWidget *widg;
  GtkAdjustment *adj;

  seq_set_preset (uipiano_chan, preset);

  (*uipiano_ctrl_vals)[uipiano_chan][UIMIDI_PSET] = preset;

  widg = gtk_object_get_data (GTK_OBJECT (ui_main_window), "SPBpset");
  adj = gtk_spin_button_get_adjustment (GTK_SPIN_BUTTON (widg));
  gtk_signal_handler_block_by_func (GTK_OBJECT (adj),
				    uipiano_pset_value_changed, widg);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)preset);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (adj),
				      uipiano_pset_value_changed, widg);
}

/* refresh current bank and preset controls
   hack for OSS AWE driver, blasted thing resets bank:preset to 0:0
   after some wavetable operations (not sure which) */
void
uipiano_refresh_bank_preset (void)
{
  seq_set_bank (uipiano_chan, (*uipiano_ctrl_vals)[uipiano_chan][UIMIDI_BANK]);
  seq_set_preset (uipiano_chan,(*uipiano_ctrl_vals)[uipiano_chan][UIMIDI_PSET]);
}

/* dummy routines to put in uipiano_ctrl_parms.func[] */
static void
uipiano_set_bank_dummy (gint chan, gint bank)
{
  uipiano_set_bank (bank);
}

static void
uipiano_set_preset_dummy (gint chan, gint preset)
{
  uipiano_set_preset (preset);
}

/* this routine handles the computer keyboard control of the piano, to get
around the inability to turn off key auto repeat (except for the entire X
session). It does this by checking for a subsequent press event (in the GDK
event queue) after a release key event is received, if the press event matches
the same key that was just released, then both events are ignored. Notice the
nasty hack to get around my inability to get gdk_event_peek() to work,
which comes into play if the second event is actually unrelated to the expected
auto repeat key press event */
gboolean
uipiano_cb_key_event (GtkWidget * widg, GdkEventKey * event, gint press)
{
  GdkEvent *pevent;
  guint8 i;
  guint8 key;
  guint8 note;

  if (event->state & (GDK_CONTROL_MASK | GDK_MOD1_MASK))
    return (FALSE);

  key = gdk_keyval_to_lower (event->keyval);
  for (i = 0; i < UIPIANO_TOTAL_NUMKEYS; i++)
    {
      if (key == uipiano_keytable[i])
	{
	  note = i;
	  if (i >= UIPIANO_LOW_NUMKEYS)
	    note = note - UIPIANO_LOW_NUMKEYS + 12;
	  note += (*uipiano_ctrl_vals)[uipiano_chan][UIMIDI_OCTAVE] * 12;
	  if (note > 127)
	    break;
	  if (press)
	    {
	      seq_note_on (uipiano_chan, note,
			  (*uipiano_ctrl_vals)[uipiano_chan]
			   [UIMIDI_VELOCITY_ON]);
	      piano_note_on (PIANO (uipiano_widg), note);
	    }
	  else
	    {			/* release event */
	      if (gdk_events_pending ())
		{
		  /* agghh! gdk_event_peek doesn't work! */
		  pevent = gdk_event_get ();
		  /* check if event is not actually what we expected */
		  if (pevent && (pevent->type != GDK_KEY_PRESS
		      || ((GdkEventKey *) (pevent))->keyval != event->keyval))
		    {
		      gdk_event_put (pevent);	/* put event back, oops */
		      gdk_event_free (pevent);
		    }
		  else if (pevent)
		    {
		      gdk_event_free (pevent);
		      break;	/* release is an auto repeat event */
		    }
		}
	      seq_note_off (uipiano_chan, note,
			    (*uipiano_ctrl_vals)[uipiano_chan]
			    [UIMIDI_VELOCITY_OFF]);
	      piano_note_off (PIANO (uipiano_widg), note);
	    }		/* release event */
	  break;
	}
    }
  return (FALSE);
}

static void
uipiano_cb_note_on (Piano * piano, guint keynum)
{
  seq_note_on (uipiano_chan, keynum,
	       (*uipiano_ctrl_vals)[uipiano_chan][UIMIDI_VELOCITY_ON]);
  piano_note_on (PIANO (uipiano_widg), keynum);
}

static void
uipiano_cb_note_off (Piano * piano, guint keynum)
{
  seq_note_off (uipiano_chan, keynum,
		(*uipiano_ctrl_vals)[uipiano_chan][UIMIDI_VELOCITY_OFF]);
  piano_note_off (PIANO (uipiano_widg), keynum);
}

void
uipiano_create_controls (void)
{
  GtkWidget *opt;
  GtkWidget *menu;
  GtkWidget *item;
  GtkWidget *spbtn;
  GtkWidget *hscale;
  GtkWidget *widg;
  GtkObject *adj;
  GtkAdjustment *radj;
  gint i, chan;

  widg = gtk_object_get_data (GTK_OBJECT (ui_main_window), "SPBchan");
  radj = gtk_spin_button_get_adjustment (GTK_SPIN_BUTTON (widg));
  gtk_signal_connect (GTK_OBJECT (radj), "value-changed",
		      uipiano_chan_value_changed, widg);
  widg = gtk_object_get_data (GTK_OBJECT (ui_main_window), "SPBbank");
  radj = gtk_spin_button_get_adjustment (GTK_SPIN_BUTTON (widg));
  gtk_signal_connect (GTK_OBJECT (radj), "value-changed",
		      uipiano_bank_value_changed, widg);
  widg = gtk_object_get_data (GTK_OBJECT (ui_main_window), "SPBpset");
  radj = gtk_spin_button_get_adjustment (GTK_SPIN_BUTTON (widg));
  gtk_signal_connect (GTK_OBJECT (radj), "value-changed",
		      uipiano_pset_value_changed, widg);

  opt = gtk_object_get_data (GTK_OBJECT (ui_main_window), "OPTctrl");

  gtk_option_menu_remove_menu (GTK_OPTION_MENU (opt));

  menu = gtk_menu_new ();

  uipiano_ctrl_vals = g_malloc (16 * UIMIDI_CTRL_COUNT * sizeof (gint));

  /* set to default values for controls in all MIDI channels */
  for (chan = 0; chan < 16; chan++)
    for (i = 0; i < UIMIDI_CTRL_COUNT; i++)
      (*uipiano_ctrl_vals)[chan][i] = uipiano_ctrl_parms[i].def;

  for (i = 0; i < UIMIDI_CTRL_COUNT; i++)
    {
      /* set control */
      if (uipiano_ctrl_parms[i].func)
	(*uipiano_ctrl_parms[i].func)(uipiano_chan, (*uipiano_ctrl_vals)[0][i]);

      /* create menu item */
      if (uipiano_ctrl_parms[i].name)
	{
	  item = gtk_menu_item_new_with_label (_(uipiano_ctrl_parms[i].name));
	  gtk_widget_show (item);
	  gtk_menu_append (GTK_MENU (menu), item);
	}
    }

  gtk_option_menu_set_menu (GTK_OPTION_MENU (opt), menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (opt), 0);

  gtk_signal_connect (GTK_OBJECT (menu), "selection-done",
		      (GtkSignalFunc)uipiano_cb_control_selection_done, NULL);

  adj = gtk_adjustment_new (100.0, 0.0, 127.0, 1.0, 10.0, 0.0);

  spbtn = gtk_object_get_data (GTK_OBJECT (ui_main_window), "SPBctrl");
  gtk_spin_button_set_adjustment (GTK_SPIN_BUTTON (spbtn),
				  GTK_ADJUSTMENT (adj));

  uipiano_adj_changed_handler =
    gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
			(GtkSignalFunc)uipiano_cb_control_value_changed, opt);

  hscale = gtk_object_get_data (GTK_OBJECT (ui_main_window), "HSCctrl");
  gtk_range_set_adjustment (GTK_RANGE (hscale), GTK_ADJUSTMENT (adj));

  uipiano_update_spin_button (0);
}

void
uipiano_set_all_controls (void)
{
  gint i;

  for (i = 0; i < UIMIDI_CTRL_COUNT; i++)
    if (uipiano_ctrl_parms[i].func)
      (*uipiano_ctrl_parms[i].func)(uipiano_chan,
				    (*uipiano_ctrl_vals)[uipiano_chan][i]);
}

static void
uipiano_cb_control_selection_done (GtkWidget * menu)
{
  GtkWidget *mnuitem;
  gint index;

  mnuitem = gtk_menu_get_active (GTK_MENU (menu));
  index = g_list_index (GTK_MENU_SHELL (menu)->children, mnuitem);

  uipiano_update_spin_button (index);
}

static void
uipiano_cb_control_value_changed (GtkWidget * adj, GtkWidget *opmenu)
{
  gint ndx;
  gint val;

  UTIL_OPMENU_INDEX (ndx, opmenu);

  val = (gint)(GTK_ADJUSTMENT (adj)->value + 0.5);

  (*uipiano_ctrl_vals)[uipiano_chan][ndx] = val;

  if (uipiano_ctrl_parms[ndx].func)
    (*uipiano_ctrl_parms[ndx].func)(uipiano_chan, val);
}

static void
uipiano_update_spin_button (gint ctrlindex)
{
  GtkWidget *widg;
  GtkAdjustment *adj;

  widg = gtk_object_get_data (GTK_OBJECT (ui_main_window), "SPBctrl");
  adj = gtk_spin_button_get_adjustment (GTK_SPIN_BUTTON (widg));

  adj->lower = uipiano_ctrl_parms[ctrlindex].lo;
  adj->upper = uipiano_ctrl_parms[ctrlindex].hi;
  adj->value = (*uipiano_ctrl_vals)[uipiano_chan][ctrlindex];

  gtk_signal_handler_block (GTK_OBJECT (adj), uipiano_adj_changed_handler);

  gtk_adjustment_changed (adj);
  gtk_adjustment_value_changed (adj);

  gtk_signal_handler_unblock (GTK_OBJECT (adj),
			      uipiano_adj_changed_handler);
}

static void
uipiano_chan_value_changed (GtkAdjustment *adj, GtkWidget *spb)
{
  gint val;

  val = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spb));
  uipiano_set_chan (val);
}

static void
uipiano_bank_value_changed (GtkAdjustment *adj, GtkWidget *spb)
{
  gint val;

  val = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spb));
  uipiano_set_bank (val);
}

static void
uipiano_pset_value_changed (GtkAdjustment *adj, GtkWidget *spb)
{
  gint val;

  val = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spb));
  uipiano_set_preset (val);
}
