/*==================================================================
 * uif_sample.c - User interface digital sample routines
 *
 * Multiple file selection ideas shamelessly taken from XMMS
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include "uif_sample.h"
#include "uif_sftree.h"
#include "uif_sfont.h"
#include "glade_interface.h"
#include "sfont.h"
#include "sample.h"
#include "sfdofunc.h"
#include "sfundo.h"
#include "uiface.h"
#include "wavetable.h"
#include "util.h"
#include "smurfcfg.h"
#include "i18n.h"

static gchar *uisam_sample_load_path = NULL;	/* path to last loaded sample */
static gchar *uisam_sample_export_path = NULL;	/* path of last exported sample */
static gint uisam_sample_export_type = -1;	/* last export file type or -1 */

static void uisam_cb_sample_file_dialog_okay (GtkWidget * samwin);
static void uisam_cb_newsamwin_okay (GtkWidget * btn, GtkWidget * newsamwin);
static void uisam_cb_export_sample_ok (GtkWidget * expwin);

void
uisam_load_sample (void)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  UISFont *uisf;
  GtkWidget *samwin;
  GtkWidget *btn;
  GtkWidget *box;
  GtkWidget *lbl;
  GtkTooltips *ttips;
  GTokenValue *val;

  if (!(itemid = sftree_get_selection_single ())) return;
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;
  uisf = SFTREE_UPFIND_UISF (node);

  if (util_activate_unique_dialog ("loadsam", uisf->sf->itemid)) return;

  samwin = gtk_file_selection_new (_("Load Sample"));

  util_register_unique_dialog (samwin, "loadsam", uisf->sf->itemid);

  /* to allow multiple file selection */
  gtk_clist_set_selection_mode (GTK_CLIST(GTK_FILE_SELECTION(samwin)->
					  file_list), GTK_SELECTION_EXTENDED);

  box = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (box);
  gtk_box_pack_start (GTK_BOX (GTK_FILE_SELECTION (samwin)->main_vbox), box,
    FALSE, FALSE, 0);

  btn = gtk_check_button_new_with_label (_("Load as RAW sample"));
  gtk_widget_show (btn);
  gtk_box_pack_start (GTK_BOX (box), btn, FALSE, FALSE, 0);
  gtk_object_set_data (GTK_OBJECT (samwin), "rawbtn", btn);

  ttips = gtk_tooltips_new ();
  gtk_tooltips_set_tip (ttips, btn,
    _("Attempt to load sample as raw data"), NULL);

  /* someday.. multiple sample loading */
#if 0
  btn = gtk_button_new_with_label (_("Add selected files"));
  gtk_widget_show (btn);
  gtk_box_pack_start (GTK_BOX (box), btn, FALSE, FALSE, 0);
  gtk_signal_connect_object (GTK_OBJECT (btn), "clicked",
			     (GtkSignalFunc) uisam_cb_sample_file_dialog_okay,
			     GTK_OBJECT (samwin));
#endif

  /* if sample load path not set, use default sample path from smurf.cfg */
  if (!uisam_sample_load_path)
    {
      val = smurfcfg_get_val (SMURFCFG_DEF_SAMPLE_PATH);
      uisam_sample_load_path = g_strdup (val->v_string);
    }

  if (strlen (uisam_sample_load_path))
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (samwin),
      uisam_sample_load_path);

  gtk_object_set_data (GTK_OBJECT (samwin), "itemid", GINT_TO_POINTER (itemid));

  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (samwin)->
      ok_button), "clicked", (GtkSignalFunc) uisam_cb_sample_file_dialog_okay,
    GTK_OBJECT (samwin));

  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (samwin)->
      cancel_button), "clicked", (GtkSignalFunc) gtk_widget_destroy,
    GTK_OBJECT (samwin));

  /* (Ripped from xmms code)
   * This is ugly.
   * Change the Cancel buttons caption to Close.
   */
  lbl = gtk_label_new (_("Close"));
  gtk_container_remove (GTK_CONTAINER (GTK_FILE_SELECTION (samwin)->
      cancel_button),
    gtk_container_children (GTK_CONTAINER (GTK_FILE_SELECTION (samwin)->
	cancel_button))->data);
  gtk_container_add (GTK_CONTAINER (GTK_FILE_SELECTION (samwin)->
      cancel_button), lbl);
  gtk_widget_show (lbl);

  gtk_widget_show (samwin);
}

static void
uisam_cb_sample_file_dialog_okay (GtkWidget * fsamwin)
{
  AudioFileRef *afref;
  gchar *fname;
  gchar *samname;
  GtkWidget *newsamwin;
  GtkWidget *btn;
  GtkWidget *lbl;
  GtkWidget *entry;
  GtkWidget *box;
  gboolean rawsam;
  gchar *s;
  gint i;

  fname = gtk_file_selection_get_filename (GTK_FILE_SELECTION (fsamwin));

  btn = gtk_object_get_data (GTK_OBJECT (fsamwin), "rawbtn");
  rawsam = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (btn));

  /* open and check sample format */
  if (!rawsam)
    {
      if (!(afref = sam_open_file (fname, TRUE, NULL)))
	return;
      if (!sam_check_file (afref))
	return;
    }
  else
    afref = NULL;

  newsamwin = create_newsamwin ();

  if (rawsam)
    {
      gtk_object_set_data (GTK_OBJECT (newsamwin), "fname", fname);
      box = gtk_object_get_data (GTK_OBJECT (newsamwin), "VBXrawsam");
      gtk_widget_show (box);
    }
  else
    {
      /* Audio sample information string, example: "16 bit, mono 44100 Hz, 836 samples" */
      s = g_strdup_printf (_("%d bit, %s %d Hz, %d samples"),
	afref->info.width,
	(afref->info.channels != 2) ? _("mono") : _("stereo"),
	afref->info.rate, afref->info.frames);

      lbl = gtk_object_get_data (GTK_OBJECT (newsamwin), "LBLsaminfo");
      gtk_label_set_text (GTK_LABEL (lbl), s);
      gtk_widget_show (lbl);
      g_free (s);
    }

  /* update sample load path */
  g_free (uisam_sample_load_path);
  s = g_dirname (fname);
  uisam_sample_load_path = g_strconcat (s, "/", NULL);
  g_free (s);

  /* close audio sample if newsamwin is destroyed */
  if (afref)
    gtk_object_set_data_full (GTK_OBJECT (newsamwin), "afref", afref,
      (GtkDestroyNotify) sam_close_file);

  /* make sure newsamwin gets destroyed if sample load file selection does */
  gtk_signal_connect_object_while_alive (GTK_OBJECT (fsamwin), "destroy",
    (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT (newsamwin));

  gtk_object_set_data (GTK_OBJECT (newsamwin), "fsamwin", fsamwin);

  fname = g_basename (fname);	/* file name portion of sample path */
  s = strrchr (fname, '.');	/* look for '.' to exclude file extension */
  if (s) i = (gint) (s - fname); /* if found, then get index of it */
  else i = strlen (fname);
  if (i > 20) i = 20;		/* clip sample name to max length */
  samname = g_strndup (fname, i);

  entry = gtk_object_get_data (GTK_OBJECT (newsamwin), "ENname");
  gtk_entry_set_text (GTK_ENTRY (entry), samname);	/* set initial name */

  /* Select the initial name for easy editing */
  gtk_entry_select_region (GTK_ENTRY (entry), 0, strlen (samname));
  g_free (samname);

  btn = gtk_object_get_data (GTK_OBJECT (newsamwin), "BTNok");
  gtk_signal_connect (GTK_OBJECT (btn), "clicked",
    (GtkSignalFunc) uisam_cb_newsamwin_okay, newsamwin);

  gtk_widget_show (newsamwin);
}

static void
uisam_cb_newsamwin_okay (GtkWidget * btn, GtkWidget * newsamwin)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  GtkWidget *entry, *fsamwin;
  UISFont *uisf;
  SFSample *sam;
  GSList *p;
  AudioFileRef *afref;
  gchar *samname, *osamname;
  gboolean raw = FALSE, stereo = FALSE;
  gchar *s, *s2;

  fsamwin = gtk_object_get_data (GTK_OBJECT (newsamwin), "fsamwin");
  afref = gtk_object_get_data (GTK_OBJECT (newsamwin), "afref");

  itemid = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (fsamwin),"itemid"));
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid)))
    {
      gtk_widget_destroy (newsamwin);
      return;
    }
  uisf = SFTREE_UPFIND_UISF (node);

  if (!afref)
    raw = TRUE;

  if (raw)
    {				/* user requested raw sample load? */
      GtkWidget *widg;
      AudioInfo afmt;
      gchar *fname;
      gint i;

      /* fetch the sample rate */
      widg = gtk_object_get_data (GTK_OBJECT (newsamwin), "OPrate");
      UTIL_OPMENU_INDEX (i, widg);
      switch (i)
	{
	case 0:
	  afmt.rate = 44100;
	  break;
	case 1:
	  afmt.rate = 22050;
	  break;
	case 2:
	  afmt.rate = 11025;
	  break;
	default:		/* custom rate */
	  widg = gtk_object_get_data (GTK_OBJECT (newsamwin), "ENcustom");
	  afmt.rate = atoi (gtk_entry_get_text (GTK_ENTRY (widg)));
	  break;
	}

      /* fetch the bit width */
      widg = gtk_object_get_data (GTK_OBJECT (newsamwin), "OPwidth");
      UTIL_OPMENU_INDEX (i, widg);
      if (i != 1)
	afmt.width = 16;
      else
	afmt.width = 8;

      /* fetch the # of channels (mono/stereo) */
      widg = gtk_object_get_data (GTK_OBJECT (newsamwin), "OPchan");
      UTIL_OPMENU_INDEX (i, widg);
      if (i != 1)
	afmt.channels = 1;
      else
	afmt.channels = 2;

      /* fetch the endianness of the data (little/big) */
      widg = gtk_object_get_data (GTK_OBJECT (newsamwin), "OPendian");
      UTIL_OPMENU_INDEX (i, widg);
      afmt.lendian = (i == 0);

      /* fetch signed/unsigned selection */
      widg = gtk_object_get_data (GTK_OBJECT (newsamwin), "OPsign");
      UTIL_OPMENU_INDEX (i, widg);
      afmt.signd = (i == 0);

      fname = gtk_object_get_data (GTK_OBJECT (newsamwin), "fname");
      if (!(afref = sam_open_file (fname, TRUE, &afmt)))
	return;

      if (!sam_check_file (afref))
	{
	  sam_close_file (afref);
	  return;
	}
    }

  /* fetch the sample name from the dialog entry */
  entry = gtk_object_get_data (GTK_OBJECT (newsamwin), "ENname");
  samname = gtk_entry_get_text (GTK_ENTRY (entry));

  if (!strlen (samname))
    {				/* must have something for sample name */
      if (raw) sam_close_file (afref);
      util_quick_popup (_("Sample name is required"), NULL);
      return;
    }

  if (afref->info.channels == 2)
    stereo = TRUE;

  if (!stereo)
    {
      if (sfont_find_sample (uisf->sf, samname, NULL))
	{
	  if (raw) sam_close_file (afref);
	  util_quick_popup (_("A sample already exists with that name"), NULL);
	  return;
	}
    }
  else
    {				/* stereo sample, compose each channels name */
      s = smurfcfg_get_val (SMURFCFG_SAM_CH2_POSTFIX)->v_string;
      if (strlen (s) > 10) s = "_R";
      if (strlen (samname) > 20 - strlen (s))
	{
	  s2 = g_strndup (samname, 20 - strlen (s));
	  osamname = g_strconcat (s2, s, NULL);
	  g_free (s2);
	}
      else osamname = g_strconcat (samname, s, NULL);

      s = smurfcfg_get_val (SMURFCFG_SAM_CH1_POSTFIX)->v_string;
      if (strlen (s) > 10) s = "_L";
      if (strlen (samname) > 20 - strlen (s))
	{
	  s2 = g_strndup (samname, 20 - strlen (s));
	  samname = g_strconcat (s2, s, NULL);
	  g_free (s2);
	}
      else samname = g_strconcat (samname, s, NULL);

      /* check for name conflicts */
      if (sfont_find_sample (uisf->sf, samname, NULL)
	|| sfont_find_sample (uisf->sf, osamname, NULL))
	{
	  if (raw) sam_close_file (afref);
	  g_free (samname);
	  g_free (osamname);
	  util_quick_popup (
	    _("Stereo sample name(s) conflict with existing sample"), NULL);
	  return;
	}
    }

  if (!(sam = sam_load_file_as_sample (afref, uisf->sf)))
    {
      if (raw)
	sam_close_file (afref);
      if (stereo)
	{
	  g_free (samname);
	  g_free (osamname);
	}
      return;
    }

  sfont_set_namestr (uisf->sf, sam->name, samname);
  p = g_slist_find (uisf->sf->sample, sam);

  node = sftree_add_sample (p, uisf->nodes, SFTREE_NODE_APPEND);

  sfdo_group (_("New sample"));
  dofunc_noitem_save (SFTREE_NODE_REF (node)->itemid);

  if (stereo)
    {
      p = g_slist_next (p);
      sam = (SFSample *) (p->data);
      sfont_set_namestr (uisf->sf, sam->name, osamname);
      node = sftree_add_sample (g_slist_find (uisf->sf->sample, sam),
				uisf->nodes, SFTREE_NODE_APPEND);

      dofunc_noitem_save (SFTREE_NODE_REF (node)->itemid);

      g_free (samname);
      g_free (osamname);
    }

  sfdo_done ();

  gtk_widget_destroy (newsamwin);
}

void
uisam_export_sample (void)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  SFTreeRef *ref;
  GtkWidget *expwin;
  GtkWidget *box;
  GtkWidget *lbl;
  GtkWidget *opt;
  GtkWidget *menu;
  GtkWidget *item;
  GTokenValue *val;

  if (!(itemid = sftree_get_selection_single ())) return;
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;

  if (util_activate_unique_dialog ("expsam", itemid)) return;

  expwin = gtk_file_selection_new (_("Export Sample"));

  util_register_unique_dialog (expwin, "expsam", itemid);

  /* if export path not set, use default sample path from smurf.cfg */
  if (!uisam_sample_export_path)
    {
      val = smurfcfg_get_val (SMURFCFG_DEF_SAMPLE_PATH);
      uisam_sample_export_path = g_strdup (val->v_string);
    }

  if (strlen (uisam_sample_export_path))
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (expwin),
      uisam_sample_export_path);

  ref = SFTREE_NODE_REF (node);
  gtk_object_set_data (GTK_OBJECT (expwin), "itemid", GINT_TO_POINTER (itemid));

  box = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (box);
  gtk_box_pack_start (GTK_BOX (GTK_FILE_SELECTION (expwin)->main_vbox), box,
    FALSE, FALSE, 0);

  lbl = gtk_label_new (_("File Type:"));
  gtk_widget_show (lbl);
  gtk_box_pack_start (GTK_BOX (box), lbl, FALSE, FALSE, 2);

  opt = gtk_option_menu_new ();
  menu = gtk_menu_new ();

  /* "WAV" the audio file type */
  item = gtk_menu_item_new_with_label (_("WAV"));
  gtk_object_set_data (GTK_OBJECT (item), "val",
    GINT_TO_POINTER (AUDIO_TYPE_WAV));
  gtk_widget_show (item);
  gtk_menu_append (GTK_MENU (menu), item);

  /* "AIFF" the audio file type */
  item = gtk_menu_item_new_with_label (_("AIFF"));
  gtk_object_set_data (GTK_OBJECT (item), "val",
    GINT_TO_POINTER (AUDIO_TYPE_AIFF));
  gtk_widget_show (item);
  gtk_menu_append (GTK_MENU (menu), item);

  /* "AU" the audio file type */
  item = gtk_menu_item_new_with_label (_("AU"));
  gtk_object_set_data (GTK_OBJECT (item), "val",
    GINT_TO_POINTER (AUDIO_TYPE_AU));
  gtk_widget_show (item);
  gtk_menu_append (GTK_MENU (menu), item);

  gtk_option_menu_set_menu (GTK_OPTION_MENU (opt), menu);

  if (uisam_sample_export_type != -1)
    gtk_option_menu_set_history (GTK_OPTION_MENU (opt),
      uisam_sample_export_type);

  gtk_widget_show (opt);
  gtk_object_set_data (GTK_OBJECT (expwin), "filefmt", opt);
  gtk_box_pack_start (GTK_BOX (box), opt, FALSE, FALSE, 2);

  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (expwin)->
      ok_button), "clicked", (GtkSignalFunc) uisam_cb_export_sample_ok,
    GTK_OBJECT (expwin));

  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (expwin)->
      cancel_button), "clicked", (GtkSignalFunc) gtk_widget_destroy,
    GTK_OBJECT (expwin));

  gtk_widget_show (expwin);
}

static void
uisam_cb_export_sample_ok (GtkWidget * expwin)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  SFTreeRef *ref;
  UISFont *uisf;
  SFSample *sam;
  GtkWidget *opt;
  GtkWidget *menu;
  GtkWidget *actv;
  gint filetype, ndx;
  gchar *fname, *s;

  itemid =
    GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (expwin), "itemid"));
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;

  uisf = SFTREE_UPFIND_UISF (node);

  ref = SFTREE_NODE_REF (node);
  sam = (SFSample *) (((GSList *) (ref->dptr))->data);

  opt = gtk_object_get_data (GTK_OBJECT (expwin), "filefmt");

  menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (opt));
  actv = gtk_menu_get_active (GTK_MENU (menu));
  filetype = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (actv), "val"));
  ndx = g_list_index (GTK_MENU_SHELL (menu)->children, actv);

  fname = gtk_file_selection_get_filename (GTK_FILE_SELECTION (expwin));

  if (!sam_export_sample (sam, uisf->sf, fname, filetype))
    return;

  /* update sample export path */
  g_free (uisam_sample_export_path);
  s = g_dirname (fname);
  uisam_sample_export_path = g_strconcat (s, "/", NULL);
  g_free (s);

  uisam_sample_export_type = ndx;

  gtk_widget_destroy (expwin);
}
