/*
 * ------------------------------------------------------------------------
 * clock.c
 *      Emulation of clock trees 
 *
 * (C) 2006 Jochen Karrer 
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <stdarg.h>
#include <clock.h>
/*
 * -------------------------------------------------------------
 * Clock trace is currently not allowed to remove itself
 * -------------------------------------------------------------
 */
void 
Clock_Set(Clock_t *clock,Frequency_t hz) 
{
	ClockTrace_t *trace;
	if(clock->hz == hz) {
		return;
	}
	clock->hz = hz;
	for(trace=clock->traceHead;trace;trace=trace->next) {
		if(!trace->isactive && trace->proc) {
			trace->proc(clock,hz,trace->clientData);
		}	
	}
}

ClockTrace_t *
Clock_Trace(Clock_t *clock,ClockTraceProc *proc,void *traceData) 
{
	ClockTrace_t *trace = malloc(sizeof(ClockTrace_t));	
	if(!trace) {
		fprintf(stderr,"Out of memory allocating ClockTrace_t\n");
		exit(1);
	}
	trace->isactive=0;
	trace->proc = proc;
	trace->clientData = traceData;
	trace->next = clock->traceHead;
	clock->traceHead = trace;
	return trace;
};

void
Clock_Untrace(Clock_t *clock,ClockTrace_t *trace) 
{
	ClockTrace_t *cursor,*prev;
	for(prev=NULL,cursor = clock->traceHead; cursor;prev=cursor,cursor=cursor->next) {
		if(trace == cursor) {
			break;
		}
	}
	if(!cursor) {
		fprintf(stderr,"Warning: can not remove nonexisting clock trace\n");
		return;
	}
	if(prev) {
		prev->next=cursor->next;
		return;
	} else {
		clock->traceHead = cursor->next;
	}
}

/*
 * ---------------------------------------------------
 * Registration in namespace is currently missing
 * ---------------------------------------------------
 */
Clock_t * 
Clock_New(const char *format,...)
{
        char name[512];
        va_list ap;
        Clock_t *clock;
        va_start (ap, format);
        vsnprintf(name,512,format,ap);
        va_end(ap);
	clock = malloc(sizeof(Clock_t));
	if(!clock) {
		fprintf(stderr,"Out of memory allocating clock\n");
		exit(1);
	}
	memset(clock,0,sizeof(Clock_t));
	clock->name = strdup(name);
	return clock;
}

static void 
clock_calculate(Clock_t *master,Frequency_t hz,void *clientData) 
{
	Clock_t *slave = (Clock_t *)clientData;
	slave->hz = master->hz * slave->derivation_ratio; 
}

#if 0
static void
clock_removeslave(Clock_t *master,Clock_t *slave) 
{
}

static void
clock_addslave(Clock_t *master,Clock_t *slave) 
{
}
#endif

/*
 * -----------------------------------------------------
 * Decouple derived clock from its master 
 * -----------------------------------------------------
 */
void
Clock_Decouple(Clock_t *slave)  
{
	Clock_t *master = slave->master;
	Clock_t *cursor,*prev;
	if(!master)
		return;
	Clock_Untrace(slave->master,slave->master_trace);
	slave->master = NULL;
	slave->master_trace = NULL;
	slave->hz = 0;
	for(prev=0,cursor=master->first_slave;cursor;prev=cursor,cursor=cursor->next_slave) {
		if(cursor == slave) {
			if(prev) {
				prev->next_slave = cursor->next_slave;
			} else {
				master->first_slave = cursor->next_slave;
			}
			return;
		}
	}
	fprintf(stderr,"Clock decouple: cursor not found in slave list !\n");
}

/*
 * -------------------------------------------------------------------
 * Make a clock to be derived from another clock by a fraction
 * -------------------------------------------------------------------
 */
void
Clock_MakeDerived(Clock_t *slave,Clock_t *master,uint64_t nom,uint64_t denom) 
{	
	double ratio = (double)nom/(double)denom;
	/* Clear already existing derivation */
	if(!slave) {
		fprintf(stderr,"Emulator bug: access to nonexisting clock\n");
		exit(1);
	}
	if(slave->master_trace) {
		if(slave->master != master) {
			fprintf(stderr,"Warning, clock is already derived\n");
		}
		Clock_Decouple(slave);
	}
	slave->master=master;
	slave->master_trace = Clock_Trace(master,clock_calculate,slave);
	slave->derivation_ratio = ratio;
	slave->derivation_nom = nom;
	slave->derivation_denom = denom;
	slave->derivation_phase = 0; /* Not implemented */
	/* update clock */
	Clock_Set(slave,master->hz * slave->derivation_ratio);
	clock_calculate(master,master->hz,slave);
	slave->next_slave = master->first_slave;
	master->first_slave = slave;
}

static void
_Clock_DumpTree(Clock_t *clock,int indent) 
{
	Clock_t *slave;
	int i;
	for(i=0;i<indent;i++) {
		fprintf(stderr," ");
	}
	fprintf(stderr,"%s: %fkHz\n",clock->name,clock->hz/1000);
	for(slave=clock->first_slave;slave;slave=slave->next_slave) 
	{
		_Clock_DumpTree(slave,indent+3);
	}
}

void
Clock_DumpTree(Clock_t *top) 
{
	_Clock_DumpTree(top,0);	
}
