/*
 * -------------------------------------------------------------------------
 *
 * Remote Frame buffer protocol server (vncserver)
 *
 * (C) 2006 Jochen Karrer
 *
 * State:
 *	Works with RAW encoding.
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <fio.h>
#include <rfbserver.h>
#include <byteorder.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <unistd.h>
#include <sys/mman.h>
#include <byteorder.h>
#include <configfile.h>
#include <fbdisplay.h>
#include <keyboard.h>

#if 0
#define dbgprintf(x...) { fprintf(stderr,x); }
#else
#define dbgprintf(x...)
#endif


#define IBUFSIZE 8192 

#define CONSTAT_NEW		(0)
#define	CONSTAT_PROTO_WAIT	(1)
#define	CONSTAT_SHARED_WAIT	(2)
#define CONSTAT_IDLE		(3)

/* 
 * -----------------------------------------------------------------------
 * PixelFormat 
 *	Pixelformat is used to store the default
 *	pixelformat of the server and the required
 *	pixelformat for the rfb connection
 * ----------------------------------------------------------------------
 */

typedef struct PixelFormat {
	uint8_t bits_per_pixel;
	uint8_t bypp;
	uint8_t depth;
	uint8_t big_endian_flag;
	uint8_t true_color_flag;
	uint16_t red_max;
	uint8_t red_bits; 	/* redundant, calculated from red_max */
	uint16_t blue_max;
	uint8_t blue_bits; 	/* redundant, calculated from blue_max */
	uint16_t green_max;
	uint8_t green_bits;	/* redundant, calculated from green_max */
	uint8_t red_shift; 
	uint8_t green_shift;
	uint8_t blue_shift;
	uint8_t padding[3];

} PixelFormat;

#define MAX_NAME_LEN (80)

typedef struct FrameBufferInfo {
	int fb_size;	/* Number of bytes allocated in *framebuffer */
	char *framebuffer;
	unsigned int fb_width;	
	unsigned int fb_height;
	unsigned int fb_linebytes;
	PixelFormat pixfmt;
	uint32_t name_length;
	char name_string[MAX_NAME_LEN];
} FrameBufferInfo;

typedef struct UpdateRequest {
	uint16_t x;
	uint16_t y;
	uint16_t width;
	uint16_t height;
	//struct UpdateRequest *next;
} UpdateRequest;

/*
 * -----------------------------------------------------------------------------
 * RfbConnection structure contains the state information 
 * for one client. It is inserted in a linked list of clients 
 * with list head in RfbServer 
 * ------------------------------------------------------------------------------
 */
typedef struct RfbConnection {
	int protoversion; /* major in high 16 bit minor in lower 16 Bit */
	int sock_fd;
	int state;
	int current_encoding;
	FIO_FileHandler ifh;
	PixelFormat pixfmt;
	FrameBufferInfo *fbi; /* points to fbi of RfbServer */
	int ifh_is_active;
	struct RfbConnection *next;
	uint8_t ibuf[IBUFSIZE];
	int	ibuf_wp;
	int 	ibuf_expected;
	/* to lazy for a list currently */
	UpdateRequest update_request; 
	RfbServer *rfbserv;
} RfbConnection;

struct RfbServer {
	FbDisplay display;
	Keyboard keyboard;
	FIO_TcpServer tcpserv;
	pid_t viewerpid; 
	int sock_fd;
	RfbConnection *con_head;
	/* Servers native FBI (window info & Pixelformat */
	FrameBufferInfo fbi;
};

int
onecount(uint16_t value) {
	uint32_t val=value;
	int ones=0;
	while(val) {
		if(val&1) {
			ones++;
		}
		val>>=1;
	}
	return ones;
}

/*
 * ----------------------------------------------------------------
 * Update the bit count fields when the maxval has changed 
 * ----------------------------------------------------------------
 */
static void
pixfmt_update_bits(PixelFormat *pixfmt) 
{
	pixfmt->red_bits = onecount(pixfmt->red_max);		
	pixfmt->green_bits = onecount(pixfmt->green_max);		
	pixfmt->blue_bits = onecount(pixfmt->blue_max);		
}
/*
 * --------------------------------------------------------------------------
 * rfbsrv_disconnect
 * 	Close connection to client
 * --------------------------------------------------------------------------
 */
static void 
rfbsrv_disconnect(RfbConnection *rcon) 
{
	char c;
	RfbConnection *cursor, *prev;
	RfbServer *rfbserv = rcon->rfbserv;
	if(rcon->ifh_is_active) {
		FIO_RemoveFileHandler(&rcon->ifh);	
		rcon->ifh_is_active = 0;
	}
	shutdown(rcon->sock_fd,SHUT_RDWR);
        while(read(rcon->sock_fd,&c,1)>0) {
		// nothing
        }
	if(close(rcon->sock_fd)<0) {
		perror("close fd failed");
	} else {
		rcon->sock_fd=-1;
	}
	for(prev=NULL,cursor=rfbserv->con_head;cursor;prev=cursor,cursor=cursor->next) {
		if(cursor == rcon) {
			if(prev) {
				prev->next = cursor->next;
			} else {
				rfbserv->con_head = cursor->next;
			}	
		} 
	}
}

/*
 * ----------------------------------------------------------
 * rfbcon_send
 *	send data to a client
 *
 *	Warning: If called more than once per fileevent
 *		 check return value, because on error
 *		 connection might be closed and may not
 *		 exist any more on next call.
 * ----------------------------------------------------------
 */
static int 
rfbcon_send(RfbConnection *rcon,char *buf,int count) 
{
	int result;
	int counter = count;
        fcntl(rcon->sock_fd,F_SETFL,0);
        while(counter) {
		int wsize = counter;
#if 0
		if(wsize>10)
			wsize=10;
#endif
                result = write(rcon->sock_fd,buf,wsize);
                if(result<=0) {
			fprintf(stderr,"Connection to vncviewer broken, closing\n");
			/* dangerous, maybe delayed is better */ 
			rfbsrv_disconnect(rcon);
                        return result;
                }
                counter-=result; buf+=result;
        }
        fcntl(rcon->sock_fd,F_SETFL,O_NONBLOCK);
	return count;
};

static int 
Msg_ProtocolVersion(RfbConnection *rcon)  
{
	char *msg = "RFB 003.003\n";
	return rfbcon_send(rcon,msg,strlen(msg));
}
/*
 * ------------------------------------------------------------
 * Check the reply if protocol version is compatible
 * ------------------------------------------------------------
 */
static int 
CheckProtocolVersion(RfbConnection *rcon)  
{
	int major,minor;
	uint32_t protoversion;
	rcon->ibuf[12] = 0;
	sscanf((char*)rcon->ibuf,"RFB %03d.%03d",&major,&minor);
	protoversion = major << 16 | (minor & 0xffff);
	if(protoversion != rcon->protoversion) {
		/* Compatibility check missing here */
		rcon->protoversion = protoversion;
	}	
	dbgprintf("protocol version reply: \"%s\"\n",rcon->ibuf);
	return 0;
}
/*
 * ----------------------------------------------------------
 * Currently no authentication is used.
 * Need DES implementation for adding VNC authentication
 * ----------------------------------------------------------
 */
static int
Msg_Auth(RfbConnection *rcon) 
{
	char msg[] = { 0,0,0,1}; /* no authentication required */
	return rfbcon_send(rcon,msg,4);
}
static int 
Msg_ServerInitialisation(RfbConnection *rcon) 
{
	RfbServer *rfbserv = rcon->rfbserv;
	char *msg = malloc(sizeof(FrameBufferInfo));
	char *p = msg;
	FrameBufferInfo *fbi = &rfbserv->fbi;
	PixelFormat *pixfmt = &fbi->pixfmt;
	fbi->name_length = strlen(fbi->name_string);	
	*(uint16_t*) p = htons(fbi->fb_width); 		p+=2;
	*(uint16_t*) p = htons(fbi->fb_height); 	p+=2;
	*(uint8_t*) p = pixfmt->bits_per_pixel; 	p+=1;
	*(uint8_t*) p = pixfmt->depth; 			p+=1;
	*(uint8_t*) p = pixfmt->big_endian_flag;	p+=1;
	*(uint8_t*) p = pixfmt->true_color_flag;	p+=1;
	*(uint16_t*) p = htons(pixfmt->red_max);	p+=2;
	*(uint16_t*) p = htons(pixfmt->green_max); 	p+=2;
	*(uint16_t*) p = htons(pixfmt->blue_max);	p+=2;
	*(uint8_t*) p = pixfmt->red_shift;		p+=1;
	*(uint8_t*) p = pixfmt->green_shift;		p+=1;
	*(uint8_t*) p = pixfmt->blue_shift;		p+=1;
	/* padding */
	p+=3;

	*(uint32_t*) p = htonl(fbi->name_length);	p+=4;
	memcpy(p,fbi->name_string,fbi->name_length);	
	p+= fbi->name_length;
	return rfbcon_send(rcon,msg,p-msg);
}

/*
 * -------------------------------------------------------------------
 * The following messages might be received by the rfbserver
 * -------------------------------------------------------------------
 */
#define CLNT_SET_PIXEL_FORMAT	(0)
#define CLNT_SET_ENCODINGS	(2)
#define		ENC_RAW			(0)
#define		ENC_COPYRECT		(1)
#define		ENC_RRE			(2)
#define		ENC_CORRE		(4)
#define		ENC_HEXTILE		(5)
#define		ENC_ZLIB		(6)	
#define		ENC_TIGHT		(7)
#define		ENC_ZLIBHEX		(8)
#define		ENC_ZRLE		(16)
#define		ENC_CURSOR_PSEUDO	(-239)
#define		ENC_DESKTOP_SIZE	(-223)
#define CLNT_FB_UPDATE_REQ	(3)
#define	CLNT_KEY_EVENT		(4)
#define CLNT_POINTER_EVENT	(5)
#define CLNT_CLIENT_CUT_TEXT	(6)

/*
 * ---------------------------------------------------
 * The following messages are sent by the server
 * ---------------------------------------------------
 */
#define SRV_FB_UPDATE			(0)
#define SRV_SET_COLOUR_MAP_ENTRIES	(1)
#define SRV_BELL			(2)
#define	SRV_CUT_TEXT			(3)

#define read16be(addr) (ntohs(*(uint16_t*)(addr)))
#define read32be(addr) (ntohl(*(uint32_t*)(addr)))
#define write16be(addr,value) (*(uint16_t*)(addr) = htons(value))
#define write32be(addr,value) (*(uint32_t*)(addr) = htonl(value))

#define write16(addr,value) (*(uint16_t*)(addr) = (value))
#define write32(addr,value) (*(uint32_t*)(addr) = (value))


static inline void 
srv_fb_update_raw(RfbConnection *rcon,UpdateRequest *udrq) 
{
	FrameBufferInfo *fbi = rcon->fbi;
	PixelFormat *pixf = &rcon->pixfmt;
	PixelFormat *fbpixf = &fbi->pixfmt;
	int con_bypp = pixf->bits_per_pixel >> 3;
	int fb_bypp = fbi->pixfmt.bits_per_pixel >> 3;
	int size = udrq->width * udrq->height * con_bypp;
	char *reply = alloca(size + 16);
	char *data;
	int x,y;
	int red,green,blue;
	reply[0] = SRV_FB_UPDATE;
	reply[1] = 0;
	write16be(reply+2,1);
	write16be(reply+4,udrq->x);
	write16be(reply+6,udrq->y);
	write16be(reply+8,udrq->width);
	write16be(reply+10,udrq->height);
	write32(reply+12,ENC_RAW);
	data = reply+16;
	int swap =  (pixf->big_endian_flag != fbpixf->big_endian_flag);
	for(y = udrq->y; y < (udrq->y+udrq->height);y++) {
		int ofs = (y * fbi->fb_width + udrq->x) * fb_bypp;
		for(x=udrq->x; x < (udrq->x + udrq->width);x++) {
			uint32_t pixval = *(uint16_t*)(&fbi->framebuffer[ofs]); 
			red = (pixval >> fbpixf->red_shift) & fbpixf->red_max;
			green = (pixval >> fbpixf->green_shift) & fbpixf->green_max;
			blue = (pixval >> fbpixf->blue_shift) & fbpixf->blue_max;
			if(fbpixf->red_bits > pixf->red_bits) {
				red = red >> (fbpixf->red_bits - pixf->red_bits); 
			} else {
				red = red << (pixf->red_bits - fbpixf->red_bits); 
			}
			if(fbpixf->green_bits > pixf->green_bits) {
				green = green >> (fbpixf->green_bits - pixf->green_bits);
			} else {
				green = green << (pixf->green_bits - fbpixf->green_bits);
			}
			if(fbpixf->blue_bits > pixf->blue_bits) {
				blue = blue >> (fbpixf->blue_bits - pixf->blue_bits);
			} else {
				blue = blue << (pixf->blue_bits - fbpixf->blue_bits);
			}
			pixval = (red << pixf->red_shift) | (green <<  pixf->green_shift) |
				(blue << pixf->blue_shift);
			switch(con_bypp) {
				case 1:
					data[0]=pixval;
					break;
				case 2:
					if(swap) {
						write16(data,swap16(pixval));
					} else {
						//fprintf(stderr,"%d(%d).%d(%d).%d(%d) %04x\n",red,green,blue,pixval);
						write16(data,pixval);
					}
					break;
				case 4:
					if(swap) {
						write32(data,swap32(pixval));
					} else {
						write32(data,pixval);
						//fprintf(stderr,"Send pixval %08x\n",pixval);
					}
					break;
				default:
					fprintf(stderr,"%d bytes per pixel not implemented\n",con_bypp);
			}
			ofs += fb_bypp;
			data += con_bypp;
		}
	}
	rfbcon_send(rcon,reply,size+16);
	return;
}

static inline void 
srv_fb_update(RfbConnection *rcon,UpdateRequest *udrq) 
{
	if(rcon->state != CONSTAT_IDLE) {
		return; 
	}
	switch(rcon->current_encoding) {
		case ENC_RAW:
			srv_fb_update_raw(rcon,udrq);
			break;
		case ENC_HEXTILE:
		default:
			fprintf(stderr,"encoding %d not implemented\n",rcon->current_encoding);
	}	
}
#if 0
static void 
srv_set_colour_map_entries() 
{

}
static void 
srv_bell() 
{

}
static void 
src_cut_text() 
{

}
#endif
static void
clnt_set_pixel_format(RfbConnection *rcon,uint8_t *data,int len) 
{
	
	PixelFormat *pixf = &rcon->pixfmt;
	
	pixf->bits_per_pixel = data[4];
	pixf->depth = data[5];
	pixf->big_endian_flag = !!data[6];
	pixf->true_color_flag = data[7];
	pixf->red_max = read16be(data+8);
	pixf->green_max = read16be(data+10);
	pixf->blue_max = read16be(data+12);
	pixf->red_shift = data[14];
	pixf->green_shift = data[15];
	pixf->blue_shift = data[16];
	pixfmt_update_bits(pixf);

#if 1
	dbgprintf("got set pixelformat message\n");
	dbgprintf("msgtype %02x\n",data[0]);
	dbgprintf("bpp    %d\n",data[4]);
	dbgprintf("depth  %d\n",data[5]);
	dbgprintf("bigend  %d\n",data[6]);
	dbgprintf("truecol  %d\n",data[7]);
	dbgprintf("redmax  %d\n",read16be(data+8));
	dbgprintf("greenmax  %d\n",read16be(data+10));
	dbgprintf("bluemax  %d\n",read16be(data+12));
	dbgprintf("redshift  %d\n",data[14]);
	dbgprintf("greenshift  %d\n",data[15]);
	dbgprintf("blueshift  %d\n",data[16]);
#endif
}
static void
clnt_set_encodings(RfbConnection *rcon,uint8_t *data,int len) 
{
	int32_t *encodings = (int32_t *)(data+4);
	int32_t enc;
	int n_encs = read16be(data+2);
	int i;
	for(i=0;i < n_encs;i++) {
		enc = ntohl(encodings[i]);
		dbgprintf("enc %d\n",enc);
		switch(enc) {
			case ENC_RAW:	
			case ENC_COPYRECT:
			case ENC_RRE:
			case ENC_CORRE:
			case ENC_HEXTILE:
			case ENC_ZLIB:
			case ENC_TIGHT:
			case ENC_ZLIBHEX:
			case ENC_ZRLE:
			case ENC_CURSOR_PSEUDO:
			case ENC_DESKTOP_SIZE:
			default:
				dbgprintf("Encoding %d not implemented\n",enc);
			break;
		}
	}
}

static void 
clnt_fb_update_req(RfbConnection *rcon,uint8_t *data,int len) 
{
	uint8_t incremental;
	UpdateRequest udrq;
	incremental = data[1];
	udrq.x=read16be(data+2);
	udrq.y=read16be(data+4);
	udrq.width = read16be(data+6);
	udrq.height = read16be(data+8);
	if(!incremental) {
		dbgprintf("UDRQ x %d y %d w %d h %d\n",udrq.x,udrq.y,udrq.width,udrq.height);
		srv_fb_update(rcon,&udrq) ;
	}
}

static void 
clnt_key_event(RfbConnection *rcon,uint8_t *data,int len) 
{
	struct KeyEvent kev;
	RfbServer *rfbserv = rcon->rfbserv;
	kev.down = data[1];
	kev.key = read32be(data+4);
	Keyboard_SendEvent(&rfbserv->keyboard,&kev);
	//fprintf(stderr,"Got key event %08x\n",key);
}

static void
clnt_pointer_event(RfbConnection *rcon,uint8_t *data,int len) 
{
	uint8_t button_mask;
	uint16_t x,y;
	button_mask = data[1];
	x = read16be(data+2);
	y = read16be(data+4);
	/* do something */
}

static void
clnt_client_cut_text(RfbConnection *rcon,uint8_t *data,int len)
{
	uint32_t length;	
	length = read32be(data+4);
	/* do something */
}

/*
 * -------------------------------------------------------------------
 * decode_message
 * 	The message decoder becomes active as soon as Connection 
 * 	initialization sequence is done 
 * -------------------------------------------------------------------
 */
static int 
decode_message(RfbConnection *rcon,uint8_t *data,int len) 
{
	int opcode; 
	int required_len;
	if(len < 4) {
		return 4;
	}
	opcode = data[0];
	switch(opcode) {
		case CLNT_SET_PIXEL_FORMAT:
			required_len = 20; 
			if(len<required_len) {
				return required_len;
			}
			clnt_set_pixel_format(rcon,rcon->ibuf,rcon->ibuf_wp);
			break;

		case CLNT_SET_ENCODINGS:
			required_len = 4 + (ntohs(*(uint16_t*)(data+2)) << 2);
			if(len<required_len) {
				return required_len;
			}
			clnt_set_encodings(rcon,rcon->ibuf,rcon->ibuf_wp);
			break;

		case CLNT_FB_UPDATE_REQ:
			required_len = 10;
			if(len<required_len) {
				return required_len;
			}
			clnt_fb_update_req(rcon,rcon->ibuf,rcon->ibuf_wp);
			break;

		case CLNT_KEY_EVENT:
			required_len = 8;
			if(len<required_len) {
				return required_len;
			}
			clnt_key_event(rcon,rcon->ibuf,rcon->ibuf_wp);
			break;

		case CLNT_POINTER_EVENT:
			required_len = 6;			
			if(len<required_len) {
				return required_len;
			}
			clnt_pointer_event(rcon,rcon->ibuf,rcon->ibuf_wp);
			break;
	
		case CLNT_CLIENT_CUT_TEXT:
			if(len >= 8) {
				required_len = 8+ntohl(*(uint32_t*)(data+4));
			} else {
				required_len = 8;
			}
			if(len<required_len) {
				return required_len;
			}
			clnt_client_cut_text(rcon,rcon->ibuf,rcon->ibuf_wp);
			break;
		default:
			fprintf(stderr,"RFBServer got invalid opcode %02x from client\n",opcode);
			break;
	}
	return 0;
}

static int
rfbcon_handle_message(RfbConnection *rcon) 
{
	int result;
	switch(rcon->state) {
		case CONSTAT_PROTO_WAIT:
			if(CheckProtocolVersion(rcon)<0) {
				//close
			}
			Msg_Auth(rcon);
			rcon->state = CONSTAT_SHARED_WAIT;
			rcon->ibuf_expected = 1;
			return 0;
			break;

		case CONSTAT_SHARED_WAIT:
			// handle shared flag is missing here (close other cons)
			Msg_ServerInitialisation(rcon);
			rcon->state = CONSTAT_IDLE;
			rcon->ibuf_expected = 4;
			break;

		/* This is the normal operating state */ 
		case CONSTAT_IDLE:
			result = decode_message(rcon,rcon->ibuf,rcon->ibuf_wp);
			if(result > rcon->ibuf_wp) {
				rcon->ibuf_expected = result;
				return result;
			} else {
				rcon->ibuf_expected = 4;
			}
			break;
		default:
			fprintf(stderr,"RFBCon Invalid state %d\n",rcon->state);
			break;
	}
	return 0;
}
/*
 * -----------------------------------------------------------------
 * RFB input event handler
 *	read data from socket and assemble into an input buffer.
 *	check if inputbuffer is a complete message
 * -----------------------------------------------------------------
 */
static int
rfbcon_input(void *clientData, int mask) 
{
	RfbConnection *rcon = (RfbConnection *) clientData;
	int count = IBUFSIZE - rcon->ibuf_wp;
	if(count == 0) {
		fprintf(stderr,"rfbserver: input buffer overflow. \n");
		/* Maybe it would be better to close connection here */
		rcon->ibuf_wp = 0; count = IBUFSIZE;
	}
	if(count > (rcon->ibuf_expected - rcon->ibuf_wp)) {
		count = (rcon->ibuf_expected - rcon->ibuf_wp);
	}
	count = read(rcon->sock_fd,rcon->ibuf+rcon->ibuf_wp,count);
	if(count < 0) {
		if((errno == -EAGAIN) || (errno == -EWOULDBLOCK)) {
			return 0;
		} else {
			perror("error reading from socket");
			rfbsrv_disconnect(rcon);
			return 0; 
		}
	} else if (count == 0) {
		dbgprintf("socket: end of file\n");
		rfbsrv_disconnect(rcon);
		return 0;
	} else {
		rcon->ibuf_wp += count;
		/* check if buffer is a complete message */
		if(rcon->ibuf_wp >= rcon->ibuf_expected) {
			int result;
			result = rfbcon_handle_message(rcon);
			if(result == 0) {
				rcon->ibuf_wp = 0;
			}
		}
	}
	return 0;
}
/*
 * --------------------------------------------------------------------------
 * The socket connect event handler
 * --------------------------------------------------------------------------
 */
static void
rfbsrv_accept(int fd,char *host,unsigned short port, void *clientData) 
{
	RfbServer *rfbserv = (RfbServer *) clientData;   
	int on;
	RfbConnection *rcon;
	rcon = malloc(sizeof(RfbConnection));
	if(!rcon) {
		fprintf(stderr,"Out of memory allocating rfb connection\n");
		close(fd);
		return;
	}
	memset(rcon,0,sizeof(RfbConnection));
	rcon->current_encoding = -1; /* Hope this doesnt exist */
	rcon->sock_fd = fd;
	rcon->rfbserv = rfbserv;
	rcon->next = rfbserv->con_head;
	rcon->fbi = &rfbserv->fbi;
	/* Default Pixel format is framebuffers pixelformat */
	memcpy(&rcon->pixfmt,&rfbserv->fbi.pixfmt,sizeof(PixelFormat));
	rfbserv->con_head = rcon;
	rcon->current_encoding = ENC_RAW;
	fcntl(rcon->sock_fd,F_SETFL,O_NONBLOCK);
	if(setsockopt(rcon->sock_fd,IPPROTO_TCP,TCP_NODELAY,(void *)&on,sizeof(on)) < 0) {
		perror("Error setting sockopts");
        }
        FIO_AddFileHandler(&rcon->ifh,rcon->sock_fd,FIO_READABLE,rfbcon_input,rcon);
	rcon->ifh_is_active = 1;
	Msg_ProtocolVersion(rcon);
	rcon->ibuf_expected = 12; /* Expecting Protocol version reply */
	rcon->state = CONSTAT_PROTO_WAIT;
	rcon->protoversion = 0x00030003;
	return;	
}

/*
 * -----------------------------------------------------------------------
 * rfbserv_set_fbformat
 *	Set the framebuffer format. Called by a user of the display 
 *	(For example a LCD controller emulator) to tell the rfbserver
 *	about memory layout.   
 * -----------------------------------------------------------------------
 */

static void 
rfbserv_set_fbformat(struct FbDisplay *fbdisp,FbFormat *fbf) 
{
	RfbServer *rfbserv = fbdisp->owner;
	FrameBufferInfo *fbi = &rfbserv->fbi;	
	PixelFormat *pixf = &fbi->pixfmt;

	pixf->red_max = (1<<fbf->red_bits) - 1;
	pixf->red_bits = fbf->red_bits;
	pixf->red_shift = fbf->red_shift;

	pixf->green_max = (1<<fbf->green_bits) - 1;
	pixf->green_bits = fbf->green_bits;
	pixf->green_shift = fbf->green_shift;

	pixf->blue_max = (1<<fbf->blue_bits) - 1;
	pixf->blue_bits = fbf->blue_bits;
	pixf->blue_shift = fbf->blue_shift;

}

/*
 * ----------------------------------------------------------------
 * rfbserv_update_display
 *	Called for example from a LCD controller emulator
 *	when it detects some change in framebuffer memory
 * ----------------------------------------------------------------
 */
static int 
rfbserv_update_display(struct FbDisplay *fbdisp,FbUpdateRequest *fbudrq) 
{
	RfbServer *rfbserv = fbdisp->owner;
	RfbConnection *rcon;
	UpdateRequest udrq;
	FrameBufferInfo *fbi = &rfbserv->fbi;
	unsigned int start = fbudrq->offset;
	unsigned int count = fbudrq->count;
	if(start > fbi->fb_size) {
		return -1;
	} 
	if(start + count > fbi->fb_size) {
		count = fbi->fb_size - start;
	}
	memcpy(fbi->framebuffer+start,fbudrq->fbdata,count);
	//fprintf(stderr,"RFBserver received update request\n");
	udrq.x = 0;
	udrq.y = ((start / fbi->fb_width)>>1);
	udrq.width = fbi->fb_width; 
	udrq.height = (count+fbi->fb_linebytes-1)/(fbi->fb_linebytes); 
	if(udrq.height+udrq.y > fbi->fb_height) {
		udrq.height =  fbi->fb_height - udrq.y;	
	}
	for(rcon = rfbserv->con_head;rcon;rcon=rcon->next) {
		srv_fb_update(rcon,&udrq); 
	}
	return -1;
}

void
RfbServer_New(const char *name,FbDisplay **display,Keyboard **keyboard,void **mouse)
{
        int fd;
        RfbServer *rfbserv;
	FrameBufferInfo *fbi;
	PixelFormat *pixf;
	uint32_t width,height;
	uint32_t port;
	char *host=Config_ReadVar(name,"host");
	char *startcmd = Config_ReadVar(name,"start");
	*display = NULL;
	*keyboard = NULL;
	if(Config_ReadUInt32(&port,name,"port")<0) {
       		return; 
        }
	if(Config_ReadUInt32(&width,name,"width")<0) {
       		return; 
        }
	if(Config_ReadUInt32(&height,name,"height")<0) {
       		return; 
        }
	if(!host) {
		host="127.0.0.1";
	}
	/* Sanity check for width/height is missing here */

        rfbserv=malloc(sizeof(RfbServer));
        if(!rfbserv)
                return;
	memset(rfbserv,0,sizeof(RfbServer));
	fbi = &rfbserv->fbi;	

	fbi->fb_width = width;	
	fbi->fb_height = height;
	fbi->fb_linebytes = fbi->fb_width * 2;
	fbi->fb_size = fbi->fb_width*fbi->fb_height*2;
	fbi->framebuffer = malloc(fbi->fb_size);
	if(!fbi->framebuffer) {
		fprintf(stderr,"Out of memory allocating framebuffer\n");
		exit(1);
	}
	sprintf(fbi->name_string,"%s %s","softgun",name);

	pixf = &fbi->pixfmt;
	pixf->bits_per_pixel = 16;
	pixf->bypp = 2; 
	pixf->depth = 12;
	pixf->big_endian_flag = 0;
	pixf->true_color_flag = 1;
	pixf->red_max = 15;
	pixf->blue_max = 15;
	pixf->green_max = 15;
	pixf->red_shift = 8;
	pixf->green_shift = 4;
	pixf->blue_shift = 0;
	pixfmt_update_bits(pixf);

        if((fd=FIO_InitTcpServer(&rfbserv->tcpserv,rfbsrv_accept,rfbserv,host,port))<0) {
                free(rfbserv);
                return;
        }
        fprintf(stderr,"RFB Server Listening on host \"%s\" port %d\n",host,port);
	rfbserv->display.owner = rfbserv;
	rfbserv->display.fbUpdateRequest = rfbserv_update_display;
	rfbserv->display.setFbFormat = rfbserv_set_fbformat;
	*keyboard = &rfbserv->keyboard;
        *display =  &rfbserv->display;
	if(startcmd && !(rfbserv->viewerpid = fork())) {
		int maxfd = getdtablesize();
		int fd;
    		for (fd = 0; fd < maxfd; fd++ ) {
        		close( fd );
    		}
		if(open("/dev/null",O_RDWR)!=0) {
			/* Error message does not work because stderr is closed */
			/* perror("VNC-viewer fork: Can not open /dev/null");   */
			exit(1);
		}
		dup(0);
		dup(0);
		system(startcmd);	
		exit(0);
	}
}
