/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#include "player.h"
#include "admin.h"
#include "other.h"
#include "util.h"
#include "audio.h"
#include "volume.h"
#include "list.h"

struct admin_funcs funcs[] = {
  {"next", admin_next},
  {"prev", admin_prev},
  {"pause", admin_pause},
  {"quit", admin_quit},
  {"set_volume", admin_set_volume},
  {"set_balance", admin_set_balance},
  {"get_volume", admin_get_volume},
  {"get_balance", admin_get_balance},
  {"get_playlist", admin_get_playlist},
  {"add_playlist", admin_add_playlist},
  {"del_playlist", admin_del_playlist},
  {"move_playlist", admin_move_playlist},
  {"play_playlist", admin_play_playlist},
  {"this", admin_this},
  {NULL, NULL}
};

char *admin_get_line (int);
void *admin_thread (char *);
void admin_check_cmd (char *, int);

/* Parse the parameter */
admin_data *
admin_parameter (char *txt)
{
  int len;
  int ch;
  admin_data *data;

  if (!txt)
    return NULL;

  if (!(data = (admin_data *) malloc (sizeof (admin_data))))
    fatal (_("Error: memory."));
  memset (data, 0, sizeof (data));

  data->socket = NULL;
  data->type = SOCKET_UNIX;
  data->interface = NULL;
  data->port = ADMIN_PORT;
  data->listen = ADMIN_LISTEN;

  ch = 0;
  len = strlen (txt);

  while (ch < len)
    {
      switch (*(txt + ch))
	{

	case 's':
	  if (!strncmp (txt + ch, "socket=", 7))
	    {
	      ch += 7;
	      data->socket = audio_parse_str (txt, &ch, len);
	    }
	  else
	    {
	      msg_error (_("Error in audiodevice parameter: %s"), txt);
	      free (data);
	      return NULL;
	    }

	  break;

	case 'i':
	  if (!strncmp (txt + ch, "interface=", 10))
	    {
	      ch += 10;
	      data->interface = audio_parse_str (txt, &ch, len);
	    }
	  else
	    {
	      msg_error (_("Error in audiodevice parameter: %s"), txt);
	      free (data);
	      return NULL;
	    }

	  break;

	case 'l':
	  if (!strncmp (txt + ch, "listen=", 7))
	    {
	      ch += 7;
	      data->listen = audio_parse_int (txt, &ch, len);
	    }
	  else
	    {
	      msg_error (_("Error in audiodevice parameter: %s"), txt);
	      free (data);
	      return NULL;
	    }

	  break;

	case 'p':
	  if (!strncmp (txt + ch, "port=", 5))
	    {
	      ch += 5;
	      data->port = audio_parse_int (txt, &ch, len);
	    }
	  else
	    {
	      msg_error (_("Error in audiodevice parameter: %s"), txt);
	      free (data);
	      return NULL;
	    }

	  break;

	case 't':
	  if (!strncmp (txt + ch, "type=", 5))
	    {
	      char *buf;
	      ch += 5;

	      buf = audio_parse_str (txt, &ch, len);

	      if (*buf == 't' || *buf == 'T')
		data->type = SOCKET_TCP;
	      else
		data->type = SOCKET_UNIX;

	      free (buf);
	    }
	  else
	    {
	      msg_error (_("Error in audiodevice parameter: %s"), txt);
	      free (data);
	      return NULL;
	    }

	  break;

	default:
	  msg_error (_("Error in audiodevice parameter: %s"), txt);
	  free (data);
	  return NULL;
	}

      ch++;
    }

  if (data->type == SOCKET_UNIX)
    {
      if (data->interface)
	{
	  msg ("Interface is not important for unix socket: Ignored.");
	  free (data->interface);
	  data->interface = NULL;
	}
    }
  else
    {
      if (data->socket)
	{
	  msg ("Socket file is not important for tcp socket: Ignored.");
	  free (data->socket);
	  data->socket = NULL;
	}
    }

  return data;
}

/* Open the admin pthread */
int
admin_start (void)
{
  int fd;
  char *sock = NULL;

  if (!play->admin)
    fatal (_("Internal error."));

  msg ("Admin socket actived");

  /* If socket TCP... */
  if (play->admin->type == SOCKET_TCP)
    {
      int yes = 1;
      struct sockaddr_in sock;
      struct hostent *hp;
      struct linger lin;

      memset ((void *) &sock, 0, sizeof (sock));
      sock.sin_family = AF_INET;
      sock.sin_port = htons (play->admin->port);

      if (!play->admin->interface)
	sock.sin_addr.s_addr = htonl (INADDR_ANY);
      else
	{

	  if (!(hp = gethostbyname (play->admin->interface)))
	    {
	      msg_error (_("Binding error: %s."), play->admin->interface);
	      return 1;
	    }

	  sock.sin_family = hp->h_addrtype;
	  memcpy ((caddr_t) & sock.sin_addr, hp->h_addr, hp->h_length);
	}

      if ((fd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
	{
	  msg_error (_("Socket error."));
	  return 1;
	}

      lin.l_onoff = 1;
      lin.l_linger = 100;
      if (setsockopt
	  (fd, SOL_SOCKET, SO_LINGER, &lin, sizeof (struct linger)) < 0)
	{
	  msg_error (_("Linger error."));
	  return 1;
	}

      if (setsockopt (fd, SOL_SOCKET, SO_REUSEADDR, &yes, sizeof (int)))
	{
	  msg_error (_("Setsockopt error."));
	  return 1;
	}

      if (bind (fd, (struct sockaddr *) &sock, sizeof (sock)) < 0)
	{
	  msg_error (_("Binding error."));
	  return 1;
	}

      if (listen (fd, play->admin->listen) < 0)
	{
	  msg_error (_("Listen error."));
	  return 1;
	}

      play->admin->fd = fd;
    }

  /* UNIX socket */
  else
    {
      struct sockaddr_un saddr;
      int d = 0;
      struct stat st;

      if ((fd = socket (AF_UNIX, SOCK_STREAM, 0)) < 0)
	{
	  msg_error (_("Socket error."));
	  return 1;
	}

      saddr.sun_family = AF_UNIX;

      if (play->admin->socket)
	{
	  strncpy (saddr.sun_path, play->admin->socket, 108);
	}
      else
	{
	  snprintf (saddr.sun_path, 108, "%s/somaplayer-%d.sock",
		    get_tmp_dir (), getpid ());

	  if (!lstat (saddr.sun_path, &st))
	    {
	      d = 1;

	      while (d < 1000)
		{
		  snprintf (saddr.sun_path, 108, "%s/somaplayer-%d.%.2d.sock",
			    get_tmp_dir (), getpid (), d);

		  if (lstat (saddr.sun_path, &st))
		    {
		      d = 0;
		      break;
		    }

		  d++;
		}
	    }

	  if (d)
	    {
	      msg_error (_("You have 1000 somaplayer runs?"));
	      return 1;
	    }

          if(!(play->admin->socket=strdup(saddr.sun_path))) fatal(_("Error: memory."));
	}

      if (bind (fd, (struct sockaddr *) &saddr, sizeof (saddr)) < 0)
	{
	  msg_error (_("Binding error."));
	  return 1;
	}

      if (listen (fd, play->admin->listen) < 0)
	{
	  msg_error (_("Listen error."));
	  return 1;
	}

      play->admin->fd = fd;
      sock = strdup (saddr.sun_path);
    }

  if (pthread_create (&play->admin->th, NULL, admin_thread, sock))
    fatal (_("Thread error."));

  return 0;
}

/* This thread get data from the socket and exec the functions */
void *
admin_thread (char *sock)
{
  fd_set set;
  struct timeval tv;
  char *buf;
  int c;

  while (!events.quit
#ifdef ENABLE_GTK
	 && play->admin_active
#endif
    )
    {
      FD_ZERO (&set);
      FD_SET (play->admin->fd, &set);

      tv.tv_sec = 0;
      tv.tv_usec = 100000;

      if (play->admin->type == SOCKET_UNIX)
	{
	  struct sockaddr_un incoming;
	  size_t size_un = sizeof (struct sockaddr_un);

	  if ((select (play->admin->fd + 1, &set, NULL, NULL, &tv) <= 0)
	      ||
	      ((c =
		accept (play->admin->fd, (struct sockaddr *) &incoming,
			&size_un)) == -1))
	    continue;
	}
      else
	{
	  struct sockaddr_in incoming;
	  size_t size_in = sizeof (struct sockaddr_in);

	  if ((select (play->admin->fd + 1, &set, NULL, NULL, &tv) <= 0)
	      ||
	      ((c =
		accept (play->admin->fd, (struct sockaddr *) &incoming,
			&size_in)) == -1))
	    continue;

	}

      buf = admin_get_line (c);

      if (buf)
	admin_check_cmd (buf, c);

      close (c);
    }

  if (sock)
    {
      unlink (sock);
      free (sock);
    }

  close (play->admin->fd);

  pthread_exit (NULL);
}

/* The get line function. Return only 1 line */
char *
admin_get_line (int fd)
{
  static char buf[SIZE_BUFFER];
  int i = 0;

  while (i < SIZE_BUFFER)
    {
      if (read (fd, buf + i, 1) < 0)
	return NULL;
      if (*(buf + i) == '\n')
	{
	  *(buf + i) = 0;

	  if (i)
	    return buf;

	  else
	    return NULL;
	}

      i++;
    }

  buf[SIZE_BUFFER - 1] = 0;
  return buf;
}

/* Find the command in the table list */
void
admin_check_cmd (char *buf, int fd)
{
  struct admin_funcs *f = funcs;

  while (f->text)
    {

      if (!strncmp (f->text, buf, strlen (f->text)))
	{
	  if (buf[strlen (f->text)] == ' ' || buf[strlen (f->text)] == 0)
	    {
	      f->func (buf, fd);
	      break;
	    }
	}

      f++;
    }
}

/* Functions: */
void
admin_quit (char *dummy, int fd)
{
  events.quit = 1;
}

void
admin_pause (char *dummy, int fd)
{
  if (events.pause)
    {
      events.pause = 0;
      pthread_cond_signal (&play->p_pop);
    }
  else
    events.pause = 1;
}

void
admin_next (char *dummy, int fd)
{
  events.skip = 1;
}

void
admin_prev (char *dummy, int fd)
{
  events.next = -1;
  events.skip = 1;
}

void
admin_set_volume (char *dummy, int fd)
{
  int i;
  int len = strlen (dummy);
  int volume = VOLUME_MAX;

  for (i = 0; i < len; i++)
    {
      if (dummy[i] == ' ' && i < len - 1)
	{
	  volume = atoi (dummy + i);
	  break;
	}
    }


  if (volume < VOLUME_MIN)
    volume = VOLUME_MIN;
  if (volume > VOLUME_MAX)
    volume = VOLUME_MAX;
  play->volume = volume;
}

void
admin_set_balance (char *dummy, int fd)
{
  int i;
  int len = strlen (dummy);
  int balance = 0;

  for (i = 0; i < len; i++)
    {
      if (dummy[i] == ' ' && i < len - 1)
	balance = atoi (&dummy[i]);
    }

  if (balance < -VOLUME_MAX)
    balance = -VOLUME_MAX;
  if (balance > VOLUME_MAX)
    balance = VOLUME_MAX;

  play->balance = balance;
}

void
admin_get_volume (char *dummy, int fd)
{
  char buf[10];
  int len;

  len = snprintf (buf, 10, "%d\n", play->volume);
  write (fd, buf, len);
}

void
admin_get_balance (char *dummy, int fd)
{
  char buf[10];
  int len;

  len = snprintf (buf, 10, "%d\n", play->balance);
  write (fd, buf, len);
}

void
admin_this (char *dummy, int fd)
{
  char buf[1024];
  int len;
  char *file = playing_file ();

  if (file)
    {
      len = snprintf (buf, 1024, "%s\n", file);
      write (fd, buf, len);
    }
  else
    write (fd, "\n", 1);
}

void
admin_get_playlist (char *dummy, int fd)
{
  list *tmp;

  pthread_mutex_lock (&play->m_list);

  tmp = play->first;

  while (tmp)
    {
      write (fd, tmp->name, strlen (tmp->name));
      write (fd, "\n", 1);

      tmp = tmp->next;
    }

  pthread_mutex_unlock (&play->m_list);
}

void
admin_add_playlist (char *dummy, int fd)
{
  int i;
  int len = strlen (dummy);

  for (i = 0; i < len; i++)
    {
      if (dummy[i] == ' ' && i < len - 1)
	break;
    }

  if (i != len)
    list_add (dummy + i, NULL);
}

void
admin_del_playlist (char *dummy, int fd)
{
  int i;
  int len = strlen (dummy);
  int n = -1;

  for (i = 0; i < len; i++)
    {
      if (dummy[i] == ' ' && i < len - 1)
	{
	  n = atoi (dummy + i);
	  break;
	}
    }

  if (n > -1)
    list_remove (n);
}

void
admin_move_playlist (char *dummy, int fd)
{
  list *what, *where;
  int what_id, where_id;
  int flag = 0;
  int i;
  int len = strlen (dummy);

  what_id = 0;
  where_id = 0;

  for (i = 0; i < len; i++)
    {
      if (dummy[i] == ' ' && i < len - 1)
	{
	  if (!flag)
	    {
	      what_id = atoi (dummy + i);
	      flag++;
	    }
	  else
	    {
	      where_id = atoi (dummy + i);
	      flag++;
	      break;
	    }
	}
    }

  if (flag != 2 || what_id == where_id)
    return;

  pthread_mutex_lock (&play->m_list);

  what = list_get (what_id);
  where = list_get (where_id);

  pthread_mutex_unlock (&play->m_list);

  if (!what)
    return;

  list_move (what, where);
}

void
admin_play_playlist (char *dummy, int fd)
{
  list *what;
  int what_id;
  int i;
  int len = strlen (dummy);

  what_id = 0;

  for (i = 0; i < len; i++)
    {
      if (dummy[i] == ' ' && i < len - 1)
	{
	  what_id = atoi (dummy + i);
	  break;
	}
    }

  pthread_mutex_lock (&play->m_list);

  what = list_get (what_id);

  pthread_mutex_unlock (&play->m_list);

  play->playlist_selected = what;
  events.skip = 1;

  if (events.stop)
    events.stop = 0;
  if (events.pause)
    events.pause = 0;

  pthread_cond_signal (&play->p_pop);
}


/* EOF */
