<?php
/**
 * $Horde: passwd/main.php,v 1.1.1.1.2.14 2005/02/01 21:31:20 ericr Exp $
 *
 * Copyright 2002-2005 Eric Jon Rostetter <eric.rostetter@physics.utexas.edu>
 *
 * See the enclosed file LICENSE for license information (BSD). If you
 * did not receive this file, see http://www.horde.org/bsdl.php.
 */

@define('PASSWD_BASE', dirname(__FILE__));
include_once PASSWD_BASE . '/lib/base.php';
include_once PASSWD_BASE . '/config/conf.php';
include_once PASSWD_BASE . '/config/backends.php';

// Use a do-while to allow easy breaking if an error is found.
do {
    // Has the user submitted the form yet?
    $submit = Horde::getFormData('submit', false);
    if ( !$submit ) {
        // No so we don't need to do anything in this loop.
        break;
    }

    // Get the backend details.
    $backend_key = Horde::getFormData('backend', false);
    $driver = $backends[$backend_key]['driver'];
    $params = $backends[$backend_key]['params'];
    $passwordPolicy = array_key_exists('password policy',
        $backends[$backend_key])
        ? $backends[$backend_key]['password policy']
        : array();

    // Get the username.
    $userid = Horde::getFormData('userid');

    // Check for users that can not change their passwords.
    if (in_array($userid, $conf['user']['refused'])) {
       Horde::raiseMessage(sprintf(_("You can't change password for user %s"),
                           $userid), HORDE_ERROR);
       break;
    }

    // We must be passed the old (current) password, or its an error.
    $oldpassword = Horde::getFormData('oldpassword', false);
    if (!$oldpassword) {
        Horde::raiseMessage(_("You must give your old password"),HORDE_WARNING);
        break;
    }

    // See if they entered the new password and verified it.
    $newpassword0 = Horde::getFormData('newpassword0', false);
    $newpassword1 = Horde::getFormData('newpassword1', false);
    if (!$newpassword0) {
        Horde::raiseMessage(_("You must give your new password"),HORDE_WARNING);
        break;
    }
    if (!$newpassword1) {
        Horde::raiseMessage(_("You must verify your new password"),
                            HORDE_WARNING);
        break;
    }
    if ($newpassword0 != $newpassword1) {
        Horde::raiseMessage( _("Your new passwords didn't match"),HORDE_WARNING);
        break;
    }

    if ($newpassword0 == $oldpassword) {
        Horde::raiseMessage(_("Your new password must be different from your current password"), HORDE_WARNING);
        break;
    }

    // Check max/min lengths if specified in the backend config.
    if (array_key_exists('minLength', $passwordPolicy) &&
        strlen($newpassword0) < $passwordPolicy['minLength']) {
        Horde::raiseMessage(sprintf(_("Your new password must be at least %d characters long!"), $passwordPolicy['minLength']), HORDE_WARNING);
        break;
    }
    if (array_key_exists('maxLength', $passwordPolicy) &&
        strlen($newpassword0) > $passwordPolicy['maxLength']) {
        Horde::raiseMessage(sprintf(_("Your new password is too long; passwords may not be more than %d characters long!"), $passwordPolicy['maxLength']), HORDE_WARNING);
        break;
    }

    // Disect the password in a localised way if the ctype extensions
    // are available, otherwise in a non-localized way
    $alpha = $alnum = $num = $upper = $lower = $space = 0;
    for ($i = 0; $i < strlen($newpassword0); $i++) {
        $char = substr($newpassword0, $i, 1);
	if (extension_loaded("ctype")) {
            if (ctype_lower($char)) {
                $lower++; $alpha++; $alnum++;
            } elseif (ctype_upper($char)) {
                $upper++; $alpha++; $alnum++;
            } elseif (ctype_digit($char)) {
                $num++; $alnum++;
            } elseif (ctype_space($char)) {
                $space++;
            }
        } else {
            if (preg_match('/[[:lower:]]/', $char)) {
                $lower++; $alpha++; $alnum++;
            } elseif (preg_match('/[[:upper:]]/', $char)) {
                $upper++; $alpha++; $alnum++;
            } elseif (preg_match('/[[:digit:]]/', $char)) {
                $num++; $alnum++;
            } elseif ($char === ' ') {
                $space++;
            }
        }
    }

    // Check reamaining password policy options.
    if (array_key_exists('minUpper', $passwordPolicy) &&
        $passwordPolicy['minUpper'] > $upper) {
        Horde::raiseMessage(sprintf(_("Your new password must contain at least %d uppercase characters."), $passwordPolicy['minUpper']), HORDE_WARNING);
        break;
    }
    if (array_key_exists('minLower', $passwordPolicy) &&
        $passwordPolicy['minLower'] > $lower) {
        Horde::raiseMessage(sprintf(_("Your new password must contain at least %d lowercase characters."), $passwordPolicy['minLower']), HORDE_WARNING);
        break;
    }
    if (array_key_exists('minNumeric', $passwordPolicy) &&
        $passwordPolicy['minNumeric'] > $num) {
        Horde::raiseMessage(sprintf(_("Your new password must contain at least %d numeric characters."), $passwordPolicy['minNumeric']), HORDE_WARNING);
        break;
    }
    if (array_key_exists('minAlpha', $passwordPolicy) &&
        $passwordPolicy['minAlpha'] > $alpha) {
        Horde::raiseMessage(sprintf(_("Your new password must contain at least %d alphabetic characters."), $passwordPolicy['minAlpha']), HORDE_WARNING);
        break;
    }
   if (array_key_exists('minAlphaNum', $passwordPolicy) &&
        $passwordPolicy['minAlphaNum'] > $alnum) {
        Horde::raiseMessage(sprintf(_("Your new password must contain at least %d alphanumeric characters."), $passwordPolicy['minAlphaNum']), HORDE_WARNING);        break;
    }
    if (array_key_exists('maxSpace', $passwordPolicy) &&
        $passwordPolicy['maxSpace'] < $space) {
        if ($passwordPolicy['maxSpace'] > 0) {
            Horde::raiseMessage(sprintf(_("Your new password must contain less than %d whitespace characters."), $passwordPolicy['maxSpace']), HORDE_WARNING);
        } else {
            Horde::raiseMessage(_("Your new password must not contain whitespace characters."), HORDE_WARNING);
        }
        break;
    }

    // Do some simple strength tests, if enabled in the config file
    if ($conf['password']['strengthtests']) {
        // Check for new==old, pass==user, simple reverse strings, etc.
        if ( (strcasecmp($newpassword0, $userid) == 0) ||
            (strcasecmp($newpassword0, strrev($userid)) == 0) ||
            (strcasecmp($newpassword0, $oldpassword) == 0) ||
            (strcasecmp($newpassword0, strrev($oldpassword)) == 0) ) {
          Horde::raiseMessage(_("Your new password is too simple to guess!  Not changed!"), HORDE_WARNING);
            break;
        }
        // Check for percentages similarity also.  This will catch very simple
        // things like "password" -> "password2" or "xpasssword"...
        @similar_text($newpassword0, $oldpassword, $percent1);
        @similar_text($newpassword0, $userid, $percent2);
        if ( ($percent1 > 80) || ($percent2 > 80) ) {
           Horde::raiseMessage(_("Your new password is too simple to guess!  Not changed!"), HORDE_WARNING);
            break;
        }
    }

    /* Create a Password_Driver instance. */
    require_once PASSWD_BASE . "/lib/Driver.php";
    $daemon = &Passwd_Driver::factory($driver, $params);

    if ($daemon === false) {
       Horde::raiseMessage(_("Password module is not properly configured"), HORDE_ERROR);
    } else {
        $backend_userid = $userid;

        if ($conf['hooks']['username']) {
            include_once HORDE_BASE . '/config/hooks.php';
            if (function_exists('_passwd_hook_username')) {
                $backend_userid = call_user_func('_passwd_hook_username',
                                                 $userid);
            }
        }
        $res = $daemon->change_password($backend_userid, $oldpassword,
                                        $newpassword0);

        if (!PEAR::isError($res)) {
            Horde::raiseMessage(_("Password changed!"), HORDE_SUCCESS);
        } else {
            Horde::raiseMessage( _("Failure in changing password : ") .
                    $res->getMessage(), HORDE_ERROR);
        }
    }
} while (0);

/* Choose the prefered backend from config/backends.php. */
foreach ($backends as $key => $curBackend) {
    if (!isset($backend_key) && substr($key, 0, 1) != '_') {
        $backend_key = $key;
    }
    if (Passwd::isPreferredBackend($curBackend)) {
        $backend_key = $key;
        break;
    }
}

/* Build the <select> widget for the servers list. */
if ($conf['backend']['backend_list'] == 'shown') {
    $backends_list = '';

    foreach ($backends as $key => $curBackend) {
        $sel = ($key == $backend_key) ? ' selected="selected"' : '';
        $backends_list .= "<option value=\"$key\"$sel>";
        $backends_list .= $curBackend['name'] . '</option>';
    }
}

// Extract userid to be shown in the username field.
if (empty($userid)) {
    if ($conf['hooks']['default_username']) {
        include_once HORDE_BASE . '/config/hooks.php';
        if (function_exists('_passwd_hook_default_username')) {
            $userid = call_user_func('_passwd_hook_default_username',
                                     Auth::getAuth());
        }
    } else {
        $splitted  = split("@", Auth::getAuth());
        $userid = @$splitted[0];
    }
}

$title = _("Change Password");
require $registry->getTemplatePath() . '/common-header.inc';
require_once $registry->getTemplatePath('horde') . '/javascript/open_help_win.js';
require $registry->getTemplatePath() . '/main/javascript.inc';
require $registry->getFileRoot() . '/status.php';
require $registry->getTemplatePath() . '/main/main.inc';
require PASSWD_BASE . '/templates/common-footer.inc';
