/*
 * soup-env.h: SOAP environment
 *
 * Authors:
 *      Rodrigo Moya (rodrigo@ximian.com)
 *
 * Copyright (C) 2001, Ximian, Inc.
 */

#include "soup-env.h"

struct _SoupEnv {
	gchar *address;
	GHashTable *request_headers;
	GHashTable *response_headers;
	SoupFault *fault;
};

/* g_hash_table_foreach() callback to add a header to a GList */
static void
add_header_to_list (gpointer key, gpointer value, gpointer data)
{
	GList **list = (GList **) data;
	*list = g_list_append(*list, key);
}

/* g_hash_table_foreach() callback to destroy a SOAP header */
static void
free_header (gpointer key, gpointer value, gpointer data)
{
	gchar *header;

	header = (gchar *) value;
	g_free((gpointer) key);
	g_free((gpointer) header);
}

SoupEnv *
soup_env_new (void)
{
	SoupEnv *env;

	env = g_new0(SoupEnv, 1);
	env->request_headers = g_hash_table_new(g_str_hash, g_str_equal);
	env->response_headers = g_hash_table_new(g_str_hash, g_str_equal);

	return env;
}

void 
soup_env_set_address (SoupEnv *env,
		      const gchar *address)
{
	g_return_if_fail (env != NULL);

	if (env->address != NULL)
		g_free (env->address);

	env->address = g_strdup (address);
}

const gchar *
soup_env_get_address (SoupEnv *env)
{
	g_return_val_if_fail (env != NULL, NULL);

	return env->address;
}

void
soup_env_set_request_header (SoupEnv *env,
			     const gchar *key,
			     const gchar *value)
{
	gchar *content;

	g_return_if_fail(env != NULL);
	g_return_if_fail(key != NULL);

	content = (gchar *) g_hash_table_lookup(env->request_headers,
						(gconstpointer) key);
	if (content) {
		/* key already exists, so remove it */
		g_hash_table_remove(env->request_headers,
				    (gconstpointer) key);
		g_free((gpointer) content);
	}

	/* add the header to the list */
	g_hash_table_insert(env->request_headers,
			    (gpointer) g_strdup(key),
			    (gpointer) g_strdup(value));
}

const gchar *
soup_env_get_request_header (SoupEnv *env, const gchar *key)
{
	gchar *header;

	g_return_val_if_fail(env != NULL, NULL);
	g_return_val_if_fail(key != NULL, NULL);

	header = (gchar *) g_hash_table_lookup(env->request_headers,
					       (gconstpointer) key);

	return (const gchar *) header;
}

GList *
soup_env_get_request_header_list (SoupEnv *env)
{
	GList *retval = NULL;

	g_return_val_if_fail(env != NULL, NULL);

	g_hash_table_foreach(env->request_headers, add_header_to_list, &retval);
	return retval;
}

void
soup_env_set_response_header (SoupEnv *env,
			      const gchar *key,
			      const gchar *value)
{
	gchar *content;

	g_return_if_fail(env != NULL);
	g_return_if_fail(key != NULL);

	content = (gchar *) g_hash_table_lookup(env->response_headers,
						(gconstpointer) key);
	if (content) {
		/* key already exists, so remove it */
		g_hash_table_remove(env->response_headers,
				    (gconstpointer) key);
		g_free((gpointer) content);
	}

	/* add the header to the list */
	g_hash_table_insert(env->response_headers,
			    (gpointer) g_strdup(key),
			    (gpointer) g_strdup(value));
}

const gchar *
soup_env_get_response_header (SoupEnv *env, const gchar *key)
{
	gchar *header;

	g_return_val_if_fail(env != NULL, NULL);
	g_return_val_if_fail(key != NULL, NULL);

	header = (gchar *) g_hash_table_lookup(env->response_headers,
					       (gconstpointer) key);

	return (const gchar *) header;
}

GList *
soup_env_get_response_header_list (SoupEnv *env)
{
	GList *retval = NULL;

	g_return_val_if_fail(env != NULL, NULL);

	g_hash_table_foreach(env->response_headers, add_header_to_list, &retval);
	return retval;
}

SoupFault *
soup_env_get_fault (SoupEnv *env)
{
	g_return_val_if_fail(env != NULL, NULL);
	return env->fault;
}

void
soup_env_set_fault (SoupEnv *env, SoupFault *fault)
{
	g_return_if_fail(env != NULL);
	g_return_if_fail(fault != NULL);

	if (env->fault)
		soup_fault_free(fault);

	env->fault = soup_fault_new(soup_fault_get_code(fault),
				    soup_fault_get_string(fault),
				    soup_fault_get_actor(fault),
				    soup_fault_get_detail(fault));
}

void
soup_env_clear_request_headers (SoupEnv *env)
{
	g_return_if_fail(env != NULL);

	g_hash_table_foreach(env->request_headers, free_header, NULL);
	g_hash_table_destroy(env->request_headers);
	env->request_headers = g_hash_table_new(g_str_hash, g_str_equal);
}

void
soup_env_clear_response_headers (SoupEnv *env)
{
	g_return_if_fail(env != NULL);

	g_hash_table_foreach(env->response_headers, free_header, NULL);
	g_hash_table_destroy(env->response_headers);
	env->response_headers = g_hash_table_new(g_str_hash, g_str_equal);
}

void
soup_env_clear_fault (SoupEnv *env)
{
	g_return_if_fail(env != NULL);

	if (env->fault)
		soup_fault_free(env->fault);
	env->fault = NULL;
}

void
soup_env_clear_all (SoupEnv *env)
{
	g_return_if_fail(env != NULL);

	soup_env_clear_request_headers(env);
	soup_env_clear_response_headers(env);
	soup_env_clear_fault(env);
}

void
soup_env_free (SoupEnv *env)
{
	g_return_if_fail(env != NULL);

	g_hash_table_foreach(env->request_headers, (GHFunc) free_header, NULL);
	g_hash_table_destroy(env->request_headers);
	g_hash_table_foreach(env->response_headers, (GHFunc) free_header, NULL);
	g_hash_table_destroy(env->response_headers);

	if (env->fault)
		soup_fault_free(env->fault);

	g_free((gpointer) env);
}
