/* speakup.c
   review functions for the speakup screen review package.
   originally written by: Kirk Reiser and Andy Berdan.

  extensively modified by David Borowski.

    Copyright (C ) 1998  Kirk Reiser.
    Copyright (C ) 2003  David Borowski.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option ) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#define __KERNEL_SYSCALLS__

#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/vt.h>
#include <linux/tty.h>
#include <linux/mm.h> /* __get_free_page( ) and friends */
#include <linux/vt_kern.h>
#if (LINUX_VERSION_CODE < 132419)
#include <linux/console_struct.h>
#include <linux/kbd_ll.h>
#include <asm/keyboard.h>
#endif
#include <linux/ctype.h>
#include <linux/selection.h>
#include <asm/uaccess.h> /* copy_from|to|user( ) and others */
#include <linux/unistd.h>

#include "../console_macros.h"	/* for x, y, attr and pos macros */
#include <linux/keyboard.h>	/* for KT_SHIFT */
#include <linux/kbd_kern.h> /* for vc_kbd_* and friends */
#include <linux/vt_kern.h>
#include <linux/input.h>
#include <linux/kmod.h>
#include <linux/speakup.h>

#include "cvsversion.h"
#include "spk_priv.h"
#include <linux/bootmem.h>	/* for alloc_bootmem */

#define SPEAKUP_VERSION "Speakup v-2.00" CVSVERSION
#define MAX_DELAY ( (500 * HZ ) / 1000 )
#define KEY_MAP_VER 119
#define MINECHOCHAR SPACE

/* these are globals from the kernel code */
extern void *kmalloc (size_t, int );
extern void kfree (const void * );
extern struct kbd_struct * kbd;
extern int fg_console;
extern short punc_masks[];

static special_func special_handler = NULL;
special_func help_handler = NULL;

static int errno;
int synth_file_inuse = 0;
short pitch_shift = 0, synth_flags = 0;
static char buf[256];
short attrib_bleep = 0, bleeps = 0,  bleep_time = 1;
short no_intr = 0, spell_delay = 0;
short key_echo = 0, cursor_timeout = 120, say_word_ctl = 0;
short say_ctrl = 0, bell_pos = 0;
short punc_mask = 0, punc_level = 0, reading_punc = 0;
char str_caps_start[MAXVARLEN+1] = "\0", str_caps_stop[MAXVARLEN+1] = "\0";
bits_data punc_info[] = {
	{ "none", "", 0 },
	{ "some", "/$%&@", SOME },
	{ "most", "$%&#()=+*/@^<>|\\", MOST },
	{ "all", "!\"#$%&'()*+,-./:;<=>?@[\\]^_`{|}~", PUNC },
	{ "delimiters", "", B_WDLM },
	{ "repeats", "()", CH_RPT },
	{ "extended numeric", "", B_EXNUM },
	{ "symbols", "", B_SYM },
	{ 0, 0 }
};
char mark_cut_flag = 0;
u_short mark_x = 0, mark_y = 0;
static char synth_name[10] = CONFIG_SPEAKUP_DEFAULT;
#define MAX_KEY 160
u_char *our_keys[MAX_KEY], *shift_table;
static u_char key_buf[600];
static u_char key_defaults[] = {
#include "speakupmap.h"
};

#if (LINUX_VERSION_CODE < 132419)
extern struct tty_struct *tty;
typedef void (*k_handler_fn)(unsigned char value, char up_flag);
#define KBD_PROTO u_char value, char up_flag
#define KBD_ARGS value, up_flag
#else
struct tty_struct *tty;
#define key_handler k_handler
typedef void (*k_handler_fn)(struct vc_data *vc, unsigned char value,
                            char up_flag, struct pt_regs *regs);
#define KBD_PROTO struct vc_data *vc, u_char value, char up_flag, struct pt_regs *regs
#define KBD_ARGS vc, value, up_flag, regs
#endif
extern k_handler_fn key_handler[16];
static k_handler_fn do_shift, do_spec, do_latin, do_cursor;
EXPORT_SYMBOL( help_handler );
EXPORT_SYMBOL( special_handler );
EXPORT_SYMBOL( our_keys );

static void speakup_date (int currcons );
static void spkup_write (const char *in_buf, int count );
int set_mask_bits( const char *input, const int which, const int how );

char str_ctl[] = "control-";
char *colors[] = {
	"black", "blue", "green", "cyan", "red", "magenta", "yellow", "white",
	"grey"
};

char *phonetic[] = {
	"alpha", "beta", "charley", "delta", "echo", "fox", "gamma", "hotel",
	"india", "juleiet", "keelo", "leema", "mike", "november", "oscar",
	"papa",
	"quebec", "romeo", "seeara", "tango", "uniform", "victer", "wiskey",
	"x ray", "yankee", "zooloo"
};

// array of 256 char pointers (one for each ASCII character description )
// initialized to default_chars and user selectable via /proc/speakup/characters
char *characters[256];

char *default_chars[256] = {
	"null", "^a", "^b", "^c", "^d", "^e", "^f", "^g",
	"^h", "^i", "^j", "^k", "^l", "^m", "^n", "^o",
	"^p", "^q", "^r", "^s", "^t", "^u", "^v", "^w",
	"^x", "^y", "^z", NULL, NULL, NULL, NULL, NULL,
	"space", "bang!", "quote", "number", "dollar", "percent", "and",
	"tick",
	"left paren", "right paren", "star", "plus", "comma", "dash", "dot",
	"slash",
	"zero", "one", "two", "three", "four", "five", "six", "seven",
	"eight", "nine",
	"colon", "semmy", "less", "equals", "greater", "question", "at",
	"eigh", "b", "c", "d", "e", "f", "g",
	"h", "i", "j", "k", "l", "m", "n", "o",
	"p", "q", "r", "s", "t", "u", "v", "w", "x",
	"y", "zehd", "left bracket", "backslash", "right bracket", "caret",
	"line",
	"accent", NULL, NULL, NULL, NULL, NULL, NULL, NULL,
	NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
	NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
	NULL, NULL, NULL, "left brace", "bar", "right brace", "tihlduh",
	"delta", "see cedilla", "u oomlout", "e acute", /* 128 */
	"eigh circumflex", "eigh oomlout", "eigh grave", "eigh ring", /* 132 */
	"see cedilla", "e circumflex", "e oomlout", "e grave", /* 136 */
	"i oomlout", "i circumflex", "i grave", "eigh oomlout", /* 140 */
	"eigh ring", "e acute", "eigh e dipthong", "eigh e dipthong", /* 144 */
	"o circumflex", "o oomlout", "o grave", "u circumflex", /* 148 */
	"u grave", "y oomlout", "o oomlout", "u oomlout", /* 152 */
	"cents", "pounds", "yen", "peseta", /* 156 */
	"florin", "eigh acute", "i acute", "o acute", /* 160 */
	"u acute", "n tilde", "n tilde", "feminine ordinal", /* 164 */
	"masculin ordinal", "inverted question", "reversed not", "not", /* 168 */
	"half", "quarter", "inverted bang", "much less than", /* 172 */
	"much greater than", "dark shading", "medium shading", /* 176 */
	"light shading", "verticle line", "left tee", /* 179 */
	"double left tee", "left double tee", "double top right", /* 182 */
	"top double right", "double left double tee", /* 185 */
	"double vertical line", "double top double right", /* 187 */
	"double bottom double right", "double bottom right", /* 189 */
	"bottom double right", "top right", "left bottom", /* 191 */
	"up tee", "tee down", "tee right", "horizontal line", /* 194 */
	"cross bars", "tee double right", "double tee right", /* 198 */
	"double left double bottom", "double left double top", /* 201 */
	"double up double tee", "double tee double down", /* 203 */
	"double tee double right", "double horizontal line", /* 205 */
	"double cross bars", "up double tee", "double up tee", /* 207 */
	"double tee down", "tee double down", /* 210 */
	"double left bottom", "left double bottom", /* 212 */
	"double left top", "left double top", /* 214 */
	"double vertical cross", "double horizontal cross", /* 216 */
	"bottom right", "left top", "solid square", /* 218 */
	"solid lower half", "solid left half", "solid right half", /* 221 */
	"solid upper half", "alpha", "beta", "gamma", /* 224 */
	"pie", "sigma", "sigma", "mu", /* 228 */
	"tou", "phigh", "thayta", "ohmega", /* 232 */
	"delta", "infinity", "phigh", "epsilaun", /* 236 */
"intersection", "identical to", "plus or minus", "equal grater than", /* 240 */
	"less than equal", "upper integral", "lower integral", /* 244 */
		"divided by", "almost equal", "degrees", /* 247 */
	"centre dot", "bullet", "square root", /* 250 */
	"power", "squared", "black square", "white space" /* 252 */
};

u_short spk_chartab[256] = {
 B_CTL, B_CTL, B_CTL, B_CTL, B_CTL, B_CTL, B_CTL, B_CTL, /* 0-7 */
 B_CTL, B_CTL, A_CTL, B_CTL, B_CTL, B_CTL, B_CTL, B_CTL, /* 8-15 */
 B_CTL, B_CTL, B_CTL, B_CTL, B_CTL, B_CTL, B_CTL, B_CTL, /*16-23 */
 B_CTL, B_CTL, B_CTL, B_CTL, B_CTL, B_CTL, B_CTL, B_CTL, /* 24-31 */
WDLM, A_PUNC, PUNC, PUNC, PUNC, PUNC, PUNC, A_PUNC, /*  !"#$%&' */
PUNC, PUNC, PUNC, PUNC, A_PUNC, A_PUNC, A_PUNC, PUNC, /* ( )*+, -./ */
NUM, NUM, NUM, NUM, NUM, NUM, NUM, NUM, /* 01234567 */
NUM, NUM, A_PUNC, PUNC, PUNC, PUNC, PUNC, A_PUNC, /* 89:;<=>? */
PUNC, A_CAP, A_CAP, A_CAP, A_CAP, A_CAP, A_CAP, A_CAP, /* @ABCDEFG */
A_CAP, A_CAP, A_CAP, A_CAP, A_CAP, A_CAP, A_CAP, A_CAP, /* HIJKLMNO */
A_CAP, A_CAP, A_CAP, A_CAP, A_CAP, A_CAP, A_CAP, A_CAP, /* PQRSTUVW */
A_CAP, A_CAP, A_CAP, PUNC, PUNC, PUNC, PUNC, PUNC, /* XYZ[\]^_ */
PUNC, ALPHA, ALPHA, ALPHA, ALPHA, ALPHA, ALPHA, ALPHA, /* `abcdefg */
ALPHA, ALPHA, ALPHA, ALPHA, ALPHA, ALPHA, ALPHA, ALPHA, /* hijklmno */
ALPHA, ALPHA, ALPHA, ALPHA, ALPHA, ALPHA, ALPHA, ALPHA, /* pqrstuvw */
ALPHA, ALPHA, ALPHA, PUNC, PUNC, PUNC, PUNC, 0, /* xyz{|}~ */
B_CAPSYM, B_CAPSYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, /* 128-135 */
B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_CAPSYM, /* 136-143 */
B_CAPSYM, B_CAPSYM, B_SYM, B_CAPSYM, B_SYM, B_SYM, B_SYM, B_SYM, /* 144-151 */
B_SYM, B_SYM, B_CAPSYM, B_CAPSYM, B_SYM, B_SYM, B_SYM, B_SYM, /* 152-159 */
B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_CAPSYM, B_SYM, /* 160-167 */
B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, /* 168-175 */
B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, /* 176-183 */
B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, /* 184-191 */
B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, /* 192-199 */
B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, /* 200-207 */
B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, /* 208-215 */
B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, /* 216-223 */
B_SYM, B_SYM, B_SYM, B_CAPSYM, B_SYM, B_CAPSYM, B_SYM, B_SYM, /* 224-231 */
B_SYM, B_CAPSYM, B_CAPSYM, B_CAPSYM, B_SYM, B_SYM, B_SYM, B_SYM, /* 232-239 */
B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, /* 240-247 */
B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM, B_SYM /* 248-255 */
};

int spk_keydown = 0;
static u_char spk_lastkey = 0, spk_close_press = 0, keymap_flags = 0;
static u_char last_keycode = 0, this_speakup_key = 0;
static u_long last_spk_jiffy = 0;

spk_t *speakup_console[MAX_NR_CONSOLES];

int spk_setup (char *str )
{
	int ints[4];
	str = get_options (str, ARRAY_SIZE (ints ), ints );
	if (ints[0] > 0 && ints[1] >= 0 )
		synth_port_forced = ints[1];
	return 1;
}

int spk_ser_setup (char *str )
{
	int lookup[4] = { 0x3f8, 0x2f8, 0x3e8, 0x2e8 };
	int ints[4];
	str = get_options (str, ARRAY_SIZE (ints ), ints );
	if (ints[0] > 0 && ints[1] >= 0 )
		synth_port_forced = lookup[ints[1]];
	return 1;
}

int spk_synth_setup (char *str )
{
	size_t len = MIN (strlen (str ), 9 );
	memcpy (synth_name, str, len );
	synth_name[len] = '\0';
	return 1;
}

__setup ("speakup_port=", spk_setup );
__setup ("speakup_ser=", spk_ser_setup );
__setup ("speakup_synth=", spk_synth_setup );

char *
strlwr (char *s )
{
	char *p;
	for (p = s; *p; p++ ) {
		if (*p >= CAP_A && *p <= CAP_Z ) *p |= 32;
	}
	return s;
}

static void
bleep (u_short val )
{
	static short vals[] = { 350, 370, 392, 414, 440, 466, 491, 523,
554, 587, 619, 659 };
	short freq;
	int time = bleep_time;
	freq = vals[val%12];
	if (val > 11 )
		freq *= (1<<(val/12 ) );
	kd_mksound (freq, time );
}

void
speakup_shut_up (int currcons )
{
	if (spk_killed ) return;
	spk_shut_up |= 0x01;
	spk_parked &= 0xfe;
	speakup_date (currcons );
	if (synth == NULL ) return;
	do_flush( );
}

void
speech_kill (int currcons )
{
	char val = synth->is_alive ( );
	if (val == 0 ) return;
	/* re-enables synth, if disabled */
	if (val == 2 || spk_killed ) {	/* dead */
		spk_shut_up &= ~0x40;
		synth_write_msg ("Eyem a Lighve!" );
	} else {
		synth_write_msg ("You killed speak up!" );
		spk_shut_up |= 0x40;
	}
}

static void
speakup_off (int currcons )
{
	if (spk_shut_up & 0x80 ) {
		spk_shut_up &= 0x7f;
		synth_write_msg ("hey. That's better!" );
	} else {
		spk_shut_up |= 0x80;
		synth_write_msg ("You turned me off!" );
	}
	speakup_date (currcons );
}

static void
speakup_parked (int currcons )
{
	if (spk_parked & 0x80 ) {
		spk_parked = 0;
		synth_write_msg ("unparked!" );
	} else {
		spk_parked |= 0x80;
		synth_write_msg ("parked!" );
	}
}

static void
speakup_cut (int currcons )
{
	static const char err_buf[] = "set selection failed";
	int ret;
	u_char args[6*sizeof(short )];
	u_short *arg;
	if (!mark_cut_flag ) {
		mark_cut_flag = 1;
		mark_x = spk_x;
		mark_y = spk_y;
		synth_write_msg ("mark" );
		return;
	}
	arg = (u_short * )args + 1;
	arg[0] = mark_x + 1;
	arg[1] = mark_y + 1;
	arg[2] = (u_short )spk_x + 1;
	arg[3] = (u_short )spk_y + 1;
	arg[4] = 0;	/* char-by-char selection */
	mark_cut_flag = 0;
	synth_write_msg ("cut" );
	clear_selection( );
#if (LINUX_VERSION_CODE < 132419)
	ret = set_selection ((unsigned long) args+sizeof(short)-1, tty, 0 );
#else
	ret = set_selection ((const struct tiocl_selection *) arg, tty, 0 );
#endif
	switch (ret ) {
	case 0:
		break; /* no error */
	case -EFAULT :
		pr_warn( "%sEFAULT\n", err_buf );
		break;
	case -EINVAL :
		pr_warn( "%sEINVAL\n", err_buf );
		break;
	case -ENOMEM :
		pr_warn( "%sENOMEM\n", err_buf );
		break;
	}
}

static void
speakup_paste (int currcons )
{
	if (mark_cut_flag ) {
		mark_cut_flag = 0;
		synth_write_msg ("mark, cleared" );
	} else {
		synth_write_msg ("paste" );
		paste_selection (tty );
	}
}

static void
say_attributes (int currcons )
{
	int fg= spk_attr&0x0f, bg = spk_attr>>4;
	if (fg > 8 ) {
		synth_write_string("bright " );
		fg -= 8;
	}
	synth_write_string(colors[fg] );
	if (bg > 7 ) {
		synth_write_string(" on blinking " );
		bg -= 8;
	} else
		synth_write_string(" on " );
	synth_write_msg(colors[bg] );
}

static char *blank_msg = "blank";
static char *edges[] = { "top, ", "bottom, ", "left, ", "right, ", "" };
enum { edge_top = 1, edge_bottom, edge_left, edge_right, edge_quiet };

static void
announce_edge (int currcons, int msg_id )
{
	if (bleeps&1 )
	bleep (spk_y );
	if (bleeps&2 )
		synth_write_msg (edges[msg_id-1] );
}

static void
speak_char( u_char ch )
{
	char *cp = characters[ ch];
	synth_buffer_add( SPACE );
	if (IS_CHAR(ch, B_CAP ) ) {
		pitch_shift++;
		synth_write_string(str_caps_start );
		synth_write_string(cp );
		synth_write_string(str_caps_stop );
	} else {
		if (*cp == '^' ) {
			synth_write_string(str_ctl );
			cp++;
		}
		synth_write_string(cp );
	}
	synth_buffer_add( SPACE );
}

static void
say_char (int currcons )
{
	u_short ch;
	spk_old_attr = spk_attr;
	ch = scr_readw ((u_short * ) spk_pos );
	spk_attr = ((ch & 0xff00 ) >> 8 );
	if (spk_attr != spk_old_attr ) {
		if (  attrib_bleep&1 ) bleep (spk_y );
		if (  attrib_bleep&2 ) say_attributes( currcons );
	}
	speak_char( ch&0xff );
}

static void
say_phonetic_char (int currcons )
{
	u_short ch;
	spk_old_attr = spk_attr;
	ch = scr_readw ((u_short * ) spk_pos );
	spk_attr = ((ch & 0xff00 ) >> 8 );
		if ( IS_CHAR(ch, B_ALPHA ) ) {
		ch &= 0x1f;
		synth_write_msg(phonetic[--ch] );
	} else {
		if ( IS_CHAR(ch, B_NUM ) )
		synth_write_string( "number " );
		speak_char( ch );
	}
}

static void
say_prev_char (int currcons )
{
	spk_parked |= 0x01;
	if (spk_x == 0 ) {
		announce_edge(currcons, edge_left );
		return;
	}
	spk_x--;
	spk_pos -= 2;
	say_char (currcons );
}

static void
say_next_char (int currcons )
{
	spk_parked |= 0x01;
	if (spk_x == video_num_columns - 1 ) {
		announce_edge(currcons, edge_right );
		return;
	}
	spk_x++;
	spk_pos += 2;
	say_char (currcons );
}

/* get_word - will first check to see if the character under the
   reading cursor is a space and if say_word_ctl is true it will
   return the word space.  If say_word_ctl is not set it will check to
   see if there is a word starting on the next position to the right
   and return that word if it exists.  If it does not exist it will
   move left to the beginning of any previous word on the line or the
   beginning off the line whichever comes first.. */

static u_long
get_word (int currcons )
{
	u_long cnt = 0, tmpx = spk_x, tmp_pos = spk_pos;
	char ch;
	u_short attr_ch;
	spk_old_attr = spk_attr;
	ch = (char ) scr_readw ((u_short * ) tmp_pos );

/* decided to take out the sayword if on a space (mis-information */
	if ( say_word_ctl && ch == SPACE ) {
		*buf = '\0';
		synth_write_msg( "space" );
		return 0;
	} else if ((tmpx < video_num_columns-2 )
		   && (ch == SPACE || IS_WDLM(ch ))
		   && ((char) scr_readw ((u_short * ) tmp_pos+1 ) > SPACE)) {
	  tmp_pos += 2;
	  tmpx++;
	} else
	while (tmpx > 0 ) {
	  if (((ch = (char ) scr_readw ((u_short * ) tmp_pos-1 )) == SPACE
	       || IS_WDLM(ch ))
	      && ((char) scr_readw ((u_short * ) tmp_pos ) > SPACE))
	    break;
	  tmp_pos -= 2;
	  tmpx--;
	}
	attr_ch = scr_readw ((u_short * ) tmp_pos );
	spk_attr = attr_ch >> 8;
	buf[cnt++] = attr_ch&0xff;
	while (tmpx < video_num_columns-1 ) {
	  tmp_pos += 2;
	  tmpx++;
	  ch = (char ) scr_readw ((u_short * ) tmp_pos );
	  if ((ch == SPACE ) 
	      || (IS_WDLM(buf[cnt-1] ) && ( ch > SPACE )))
	    break;
	  buf[cnt++] = ch;
	}
	buf[cnt] = '\0';
	return cnt;
}

static void
say_word (int currcons )
{
	u_long cnt = get_word(currcons );
	u_short saved_punc_mask = punc_mask;
	if ( cnt == 0 ) return;
	punc_mask = PUNC;
	buf[cnt++] = SPACE;
	spkup_write (buf, cnt );
	punc_mask = saved_punc_mask;
}

static void
say_prev_word (int currcons )
{
	char ch;
	u_short edge_said = 0, last_state = 0, state = 0;
	spk_parked |= 0x01;
	if (spk_x == 0 ) {
		if ( spk_y == 0 ) {
			announce_edge(currcons, edge_top );
			return;
		}
		spk_y--;
		spk_x = video_num_columns;
		edge_said = edge_quiet;
	}
	while ( 1 ) {
		if (spk_x == 0 ) {
			if (spk_y == 0 ) {
				edge_said = edge_top;
				break;
			}
			if ( edge_said != edge_quiet ) edge_said = edge_left;
			if ( state > 0 ) break;
			spk_y--;
			spk_x = video_num_columns-1;
		} else spk_x--;
			spk_pos -= 2;
		ch = (char ) scr_readw ((u_short * ) spk_pos );
		if ( ch == SPACE ) state = 0;
		else if (IS_WDLM(ch ) ) state = 1;
		else state = 2;
		if (state < last_state ) {
			spk_pos += 2;
			spk_x++;
			break;
		}
		last_state = state;
	}
	if ( spk_x == 0 && edge_said == edge_quiet )
		edge_said = edge_left;
	if ( edge_said > 0 && edge_said < edge_quiet )
		announce_edge( currcons, edge_said );
	say_word (currcons );
}

static void
say_next_word (int currcons )
{
	char ch;
	u_short edge_said = 0, last_state = 2, state = 0;
	spk_parked |= 0x01;
	if ( spk_x == video_num_columns - 1 && spk_y == video_num_lines-1 ) {
		announce_edge(currcons, edge_bottom );
		return;
	}
	while ( 1 ) {
		ch = (char ) scr_readw ((u_short * ) spk_pos );
		if ( ch == SPACE ) state = 0;
		else if (IS_WDLM(ch ) ) state = 1;
		else state = 2;
		if ( state > last_state ) break;
		if (spk_x >= video_num_columns-1 ) {
			if (spk_y == video_num_lines-1 ) {
				edge_said = edge_bottom;
				break;
			}
			state = 0;
			spk_y++;
			spk_x = 0;
			edge_said = edge_right;
		} else spk_x++;
			spk_pos += 2;
		last_state = state;
	}
	if ( edge_said > 0 )
		announce_edge( currcons, edge_said );
	say_word (currcons );
}

static void
spell_word (int currcons )
{
	static char *delay_str[] = { " ", ", ", ". ", ". . ", ". . . " };
	char *cp = buf, *str_cap= str_caps_stop;
	char *cp1, *last_cap = str_caps_stop;
	u_char ch;
	if ( !get_word(currcons ) ) return;
	while ((ch = (u_char )*cp ) ) {
		if ( cp != buf )
			synth_write_string (delay_str[spell_delay] );
		if (IS_CHAR(ch, B_CAP ) ) {
			str_cap = str_caps_start;
			if ( *str_caps_stop ) pitch_shift++;
		else last_cap = str_caps_stop; /* synth has no pitch */
		} else str_cap = str_caps_stop;
		if ( str_cap !=last_cap ) {
			synth_write_string( str_cap );
			last_cap = str_cap;
		}
		if ( this_speakup_key == SPELL_PHONETIC && ( IS_CHAR( ch, B_ALPHA ) ) ) {
			ch &= 31;
			cp1 = phonetic[--ch];
		} else {
			cp1 = characters[ch];
			if (*cp1 == '^' ) {
				synth_write_string(str_ctl );
				cp1++;
			}
		}
		synth_write_string(cp1 );
	cp++;
	}
	if ( str_cap != str_caps_stop )
		synth_write_string( str_caps_stop );
}

static int
get_line (int currcons )
{
	u_long tmp = spk_pos - (spk_x * 2 );
	int i = 0;
	spk_old_attr = spk_attr;
	spk_attr = (u_char ) (scr_readw ((u_short * ) spk_pos ) >> 8 );
	for (i = 0; i < video_num_columns; i++ ) {
		buf[i] = (u_char ) scr_readw ((u_short * ) tmp );
		tmp += 2;
	}
	for (--i; i >= 0; i-- )
		if (buf[i] != SPACE ) break;
	return ++i;
}

static void
say_line (int currcons )
{
	int i = get_line( currcons );
	char *cp;
	char num_buf[8];
	u_short saved_punc_mask = punc_mask;
	if (i == 0 ) {
		synth_write_msg (blank_msg );
		return;
	}
	buf[i++] = '\n';
	if ( this_speakup_key == SAY_LINE_INDENT ) {
		for ( cp = buf; *cp == SPACE; cp++ );
		sprintf( num_buf, "%d, ", ( cp-buf )+1 );
		synth_write_string( num_buf );
	}
	punc_mask = punc_masks[reading_punc];
	spkup_write (buf, i );
	punc_mask = saved_punc_mask;
}

static void
say_prev_line (int currcons )
{
	spk_parked |= 0x01;
	if (spk_y == 0 ) {
		announce_edge (currcons, edge_top );
		return;
	}
	spk_y--;
	spk_pos -= video_size_row;
	say_line (currcons );
}

static void
say_next_line (int currcons )
{
	spk_parked |= 0x01;
	if (spk_y == video_num_lines - 1 ) {
		announce_edge (currcons, edge_bottom );
		return;
	}
	spk_y++;
	spk_pos += video_size_row;
	say_line (currcons );
}

static int
say_from_to (int currcons, u_long from, u_long to, int read_punc )
{
	int i = 0;
	u_short saved_punc_mask = punc_mask;
	spk_old_attr = spk_attr;
	spk_attr = (u_char ) (scr_readw ((u_short * ) from ) >> 8 );
	while (from < to ) {
		buf[i++] = (char ) scr_readw ((u_short * ) from );
		from += 2;
		if ( i > 250 ) break;
	}
	for (--i; i >= 0; i-- )
		if (buf[i] != SPACE ) break;
	buf[++i] = SPACE;
	if ( i < 1 ) return i;
	if ( read_punc ) punc_mask = punc_info[reading_punc].mask;
	spkup_write (buf, i );
	if ( read_punc ) punc_mask = saved_punc_mask;
	return i;
}

static void
say_line_from_to (int currcons, u_long from, u_long to, int read_punc )
{
	u_long start = origin+(spk_y*video_size_row );
	u_long end = start+( to * 2 );
	start += from*2;
	if ( say_from_to( currcons, start, end, read_punc ) <= 0 )
		synth_write_msg (blank_msg );
}

static void
say_screen_from_to (int currcons, u_long from, u_long to )
{
	u_long start = origin, end;
	if ( from > 0 ) start += from * video_size_row;
	if ( to > video_num_lines ) to = video_num_lines;
	end = origin + ( to * video_size_row);
	for ( from = start; from < end; from = to ) {
		to = from + video_size_row;
		say_from_to( currcons, from, to, 1 );
	}
}

static void
say_screen (int currcons )
{
	say_screen_from_to( currcons, 0, video_num_lines );
}

static void
speakup_win_say (int currcons )
{
	u_long start, end, from, to;
	if ( win_start < 2 ) {
		synth_write_msg( "no window" );
		return;
	}
	start = origin + ( win_top * video_size_row );
	end = origin + ( win_bottom * video_size_row );
	while ( start <= end ) {
		from = start + ( win_left * 2 );
	to = start + ( win_right * 2 );
	say_from_to( currcons, from, to, 1 );
		start += video_size_row;
	}
}

static void
top_edge (int currcons )
{
	spk_parked |= 0x01;
	spk_pos = origin + 2 * spk_x;
	spk_y = 0;
	say_line (currcons );
}

static void
bottom_edge (int currcons )
{
	spk_parked |= 0x01;
	spk_pos += (video_num_lines - spk_y - 1 ) * video_size_row;
	spk_y = video_num_lines - 1;
	say_line (currcons );
}

static void
left_edge (int currcons )
{
	spk_parked |= 0x01;
	spk_pos -= spk_x * 2;
	spk_x = 0;
	say_char (currcons );
}

static void
right_edge (int currcons )
{
	spk_parked |= 0x01;
	spk_pos += (video_num_columns - spk_x - 1 ) * 2;
	spk_x = video_num_columns - 1;
	say_char (currcons );
}

static void
say_first_char (int currcons )
{
	int i, len = get_line( currcons );
	u_char ch;
	spk_parked |= 0x01;
	if ( len == 0 ) {
		synth_write_msg( blank_msg );
		return;
	}
	for ( i = 0; i < len; i++ ) if ( buf[i] != SPACE ) break;
	ch = buf[i];
	spk_pos -= ( spk_x-i ) * 2;
	spk_x = i;
	sprintf (buf, "%d, ", ++i );
	synth_write_string (buf );
	speak_char( ch );
}

static void
say_last_char (int currcons )
{
	int len = get_line( currcons );
	u_char ch;
	spk_parked |= 0x01;
	if ( len == 0 ) {
		synth_write_msg( blank_msg );
		return;
	}
	ch = buf[--len];
	spk_pos -= ( spk_x-len ) * 2;
	spk_x = len;
	sprintf (buf, "%d, ", ++len );
	synth_write_string (buf );
	speak_char( ch );
}

static void
say_position (int currcons )
{
	sprintf (buf, "line %ld, col %ld, t t y %d\n", spk_y + 1,
		     spk_x + 1, currcons + 1 );
	synth_write_string (buf );
}

// Added by brianb
static void
say_char_num (int currcons )
{
	u_short ch = scr_readw ((u_short * ) spk_pos );
	ch &= 0x0ff;
	sprintf (buf, "hex %02x, decimal %d", ch, ch );
	synth_write_msg (buf );
}

/* these are stub functions to keep keyboard.c happy. */

static void
say_from_top (int currcons )
{
	say_screen_from_to (currcons, 0, spk_y );
}

static void
say_to_bottom (int currcons )
{
	say_screen_from_to (currcons, spk_y, video_num_lines );
}

static void
say_from_left (int currcons )
{
	say_line_from_to (currcons, 0, spk_x, 1 );
}

static void
say_to_right (int currcons )
{
	say_line_from_to (currcons, spk_x, video_num_columns, 1 );
}

/* end of stub functions. */

static void
spkup_write (const char *in_buf, int count )
{
	static int rep_count = 0;
	static u_char ch = '\0', old_ch = '\0';
	static u_short char_type = 0, last_type = 0;
	static u_char *exn_ptr = NULL;
	int in_count = count;
	char rpt_buf[32];
	spk_keydown = 0;
	while ( count-- ) {
		ch = (u_char )*in_buf++;
		char_type = spk_chartab[ch];
		if (ch == old_ch && !(char_type&B_NUM ) ) {
			if (++rep_count > 2 ) continue;
		} else {
			if ( (last_type&CH_RPT) && rep_count > 2 ) {
				sprintf (rpt_buf, " times %d . ", ++rep_count );
				synth_write_string (rpt_buf );
			}
			rep_count = 0;
		}
		if ( !( char_type&B_NUM ) )
				exn_ptr = NULL;
		if (ch == spk_lastkey ) {
			rep_count = 0;
			if ( key_echo == 1 && ch >= MINECHOCHAR )
				speak_char( ch );
		} else if ( ( char_type&B_ALPHA ) ) {
			if ( (synth_flags&SF_DEC) && (last_type&PUNC) )
				synth_buffer_add ( SPACE );
			synth_write( &ch, 1 );
		} else if ( ( char_type&B_NUM ) ) {
			rep_count = 0;
			if ( (last_type&B_EXNUM) && synth_buff_in == exn_ptr+1 ) {
				synth_buff_in--;
				synth_buffer_add( old_ch );
				exn_ptr = NULL;
			}
			synth_write( &ch, 1 );
		} else if ( (char_type&punc_mask) ) {
			speak_char( ch );
			char_type &= ~PUNC; /* for dec nospell processing */
		} else if ( ( char_type&SYNTH_OK ) ) {
/* these are usually puncts like . and , which synth needs for expression.
 * suppress multiple to get rid of long pausesand clear repeat count so if
 *someone has repeats on you don't get nothing repeated count */
			if ( ch != old_ch )
				synth_write( &ch, 1 );
			else rep_count = 0;
		} else {
			if ( ( char_type&B_EXNUM ) )
					exn_ptr = (u_char *)synth_buff_in;
/* send space and record position, if next is num overwrite space */
			if ( old_ch != ch ) synth_buffer_add ( SPACE );
			else rep_count = 0;
		}
		old_ch = ch;
		last_type = char_type;
	}
	spk_lastkey = 0;
	if (in_count > 2 && rep_count > 2 ) {
		if ( (last_type&CH_RPT) ) {
			sprintf (rpt_buf, " repeated %d . ", ++rep_count );
			synth_write_string (rpt_buf );
		}
		rep_count = 0;
	}
}

static char *ctl_key_ids[] = {
	"shift", "altgr", "control", "ault", "l shift", "speakup",
"l control", "r control"
};
#define NUM_CTL_LABELS 8

static void
handle_shift( KBD_PROTO )
{
	int currcons = fg_console;
	(*do_shift)( KBD_ARGS );
	if ( synth == NULL || up_flag || spk_killed ) return;
	spk_shut_up &= 0xfe;
	do_flush( );
	if ( say_ctrl && value < NUM_CTL_LABELS )
		synth_write_string ( ctl_key_ids[value] );
}

static void
handle_latin( KBD_PROTO )
{
	int currcons = fg_console;
	(*do_latin)( KBD_ARGS );
	if ( up_flag ) {
		spk_lastkey = spk_keydown = 0;
		return;
	}
	if ( synth == NULL || spk_killed ) return;
	spk_shut_up &= 0xfe;
	spk_lastkey = value;
	spk_keydown++;
	spk_parked &= 0xfe;
	if ( key_echo == 2 && value >= MINECHOCHAR )
		speak_char( value );
}

static int
set_key_info( u_char *key_info, u_char *k_buffer )
{
	int i = 0,  states, key_data_len;
	u_char *cp = key_info, *cp1 = k_buffer;
	u_char ch, version, num_keys;
	version = *cp++;
	if ( version != KEY_MAP_VER ) return -1;
	num_keys = *cp;
	states = (int)cp[1];
	key_data_len = ( states+1 ) * ( num_keys+1 );
	if ( key_data_len+SHIFT_TBL_SIZE+4 >= sizeof(key_buf ) ) return -2;
	memset( k_buffer, 0, SHIFT_TBL_SIZE );
	memset( our_keys, 0, sizeof( our_keys ) );
	shift_table = k_buffer;
	our_keys[0] = shift_table;
	cp1 += SHIFT_TBL_SIZE;
	memcpy( cp1, cp, key_data_len+3 );
/* get num_keys, states and data*/
	cp1 += 2; /* now pointing at shift states */
	for ( i = 1; i <= states; i++ ) {
		ch = *cp1++;
		if ( ch >= SHIFT_TBL_SIZE ) return -3;
		shift_table[ch] = i;
	}
	keymap_flags = *cp1++;
	while ( ( ch = *cp1 ) ) {
		if ( ch >= MAX_KEY ) return -4;
		our_keys[ch] = cp1;
		cp1 += states+1;
	}
	return 0;
}

num_var spk_num_vars[] = { /* bell must be first to set high limit */
	{ BELL_POS, 0, 0, 0, 0, 0, 0, 0 },
	{ SPELL_DELAY, 0, 0, 0, 5, 0, 0, 0 },
	{ ATTRIB_BLEEP, 0, 1, 0, 3, 0, 0, 0 },
	{ BLEEPS, 0, 3, 0, 3, 0, 0, 0 },
	{ BLEEP_TIME, 0, 4, 1, 20, 0, 0, 0 },
	{ PUNC_LEVEL, 0, 1, 0, 4, 0, 0, 0 },
	{ READING_PUNC, 0, 1, 0, 4, 0, 0, 0 },
	{ CURSOR_TIME, 0, 120, 50, 600, 0, 0, 0 },
	{ SAY_CONTROL, TOGGLE_0 },
	{ SAY_WORD_CTL, TOGGLE_0 },
	{ NO_INTERRUPT, TOGGLE_0 },
	{ KEY_ECHO, 0, 1, 0, 2, 0, 0, 0 },
	V_LAST_NUM
};

static int cursor_track = 1;
static char *cursor_msgs[] = { "cursoring off", "cursoring on", 
	"attribute cursor" };
#define MAXCURSORTRACK 1
/* increase when we add more cursor modes */
/* attribute cursor code coming soon */

static void
toggle_cursoring( int currcons )
{
        cursor_track++;
	if ( cursor_track > MAXCURSORTRACK )
		cursor_track = 0;
	synth_write_msg (cursor_msgs[cursor_track] );
}

static void
reset_default_chars (void )
{
	int i;
	if (default_chars[(int )'a'] == NULL ) { /* lowers are null first time */
		for (i = (int )'a'; default_chars[i] == NULL; i++ )
			default_chars[i] = default_chars[i-32];
	} else { /* free any non-default */
		for (i = 0; i < 256; i++ ) {
			if (characters[i] != default_chars[i] )
				kfree (characters[i] );
		}
	}
		memcpy( characters, default_chars, sizeof( default_chars ) );
}

static void
handle_cursor( KBD_PROTO );
static void
handle_spec( KBD_PROTO );
static void
cursor_done(u_long data );
declare_timer( cursor_timer );

void __init speakup_open (int currcons, spk_t *first_console )
{
	int i;
	num_var *n_var;
	reset_default_chars ( );
	memset( speakup_console, 0, sizeof( speakup_console ) );
	if ( first_console == NULL ) return;
	memset( first_console, 0, spk_size );
	speakup_console[currcons] = first_console;
	speakup_date( currcons);
	pr_info ("%s: initialized\n", SPEAKUP_VERSION );
	init_timer (&cursor_timer );
#if (LINUX_VERSION_CODE >= 132419)
	cursor_timer.entry.prev=NULL;
#endif
	cursor_timer.function = cursor_done;
	init_sleeper ( synth_sleeping_list );
	strlwr (synth_name );
	synth_init ( synth_name );
 	spk_num_vars[0].high = video_num_columns;
	for ( n_var = spk_num_vars; n_var->var_id >= 0; n_var++ )
		speakup_register_var( n_var );
	for (i = 1; punc_info[i].mask != 0; i++ )
		set_mask_bits( 0, i, 2 );
	do_latin = key_handler[KT_LATIN];
	key_handler[KT_LATIN] = handle_latin;
	do_spec = key_handler[KT_SPEC];
	key_handler[KT_SPEC] = handle_spec;
	do_cursor = key_handler[KT_CUR];
	key_handler[KT_CUR] = handle_cursor;
	do_shift = key_handler[KT_SHIFT];
	key_handler[KT_SHIFT] = handle_shift;
	set_key_info( key_defaults, key_buf );
}

#ifdef CONFIG_PROC_FS

// speakup /proc interface code

/* Usage:
cat /proc/speakup/version

cat /proc/speakup/characters > foo
less /proc/speakup/characters
vi /proc/speakup/characters

cat foo > /proc/speakup/characters
cat > /proc/speakup/characters
echo 39 apostrophe > /proc/speakup/characters
echo 87 w > /proc/speakup/characters
echo 119 w > /proc/speakup/characters
echo defaults > /proc/speakup/characters
echo reset > /proc/speakup/characters
*/

// keymap handlers

static int
keys_read_proc (PROC_READ_PROTOTYPE )
{
	char *cp = page;
	int i, n, num_keys, nstates;
	u_char *cp1 = key_buf + SHIFT_TBL_SIZE, ch;
	num_keys = (int)(*cp1);
	nstates = (int)cp1[1];
	cp += sprintf( cp, "%d, %d, %d,\n", KEY_MAP_VER,  num_keys, nstates );
	cp1 += 2; /* now pointing at shift states */
/* dump num_keys+1 as first row is shift states + flags,
   each subsequent row is key + states */
	for ( n = 0; n <= num_keys; n++ ) {
		for ( i = 0; i <= nstates; i++ ) {
			ch = *cp1++;
			cp += sprintf( cp, "%d,", (int)ch );
			*cp++ = ( i < nstates ) ? SPACE : '\n';
		}
	}
	cp += sprintf( cp, "0, %d\n", KEY_MAP_VER );
	*start = 0;
	*eof = 1;
	return (int)(cp-page);
}

static char *
s2uchar ( char *start, char *dest )
{
	int val = 0;
	while ( *start && *start <= SPACE ) start++;
	while ( *start >= '0' && *start <= '9' ) {
		val *= 10;
		val += ( *start ) - '0';
		start++;
	}
	if ( *start == ',' ) start++;
	*dest = (u_char)val;
	return start;
}

static int
keys_write_proc (PROC_WRITE_PROTOTYPE )
{
	int i, ret = count;
	char *in_buff, *cp;
	u_char *cp1;
	if (count < 1 || count > 1800 )
		return -EINVAL;
	in_buff = ( char * ) __get_free_page ( GFP_KERNEL );
	if ( !in_buff ) return -ENOMEM;
	if (copy_from_user (in_buff, buffer, count ) ) {
		free_page ( ( unsigned long ) in_buff );
		return -EFAULT;
	}
	if (in_buff[count - 1] == '\n' ) count--;
	in_buff[count] = '\0';
	if ( count == 1 && *in_buff == 'd' ) {
		free_page ( ( unsigned long ) in_buff );
		set_key_info( key_defaults, key_buf );
		return ret;
	}
	cp = in_buff;
	cp1 = (u_char *)in_buff;
	for ( i = 0; i < 3; i++ ) {
		cp = s2uchar( cp, cp1 );
		cp1++;
	}
	i = (int)cp1[-2]+1;
	i *= (int)cp1[-1]+1;
	i+= 2; /* 0 and last map ver */
	if ( cp1[-3] != KEY_MAP_VER || cp1[-1] > 10 || 
			i+SHIFT_TBL_SIZE+4 >= sizeof(key_buf ) ) {
pr_warn( "i %d %d %d %d\n", i, (int)cp1[-3], (int)cp1[-2], (int)cp1[-1] );
		free_page ( ( unsigned long ) in_buff );
		return -EINVAL;
	}
	while ( --i >= 0 ) {
		cp = s2uchar( cp, cp1 );
		cp1++;
		if ( !(*cp) ) break;
	}
	if ( i != 0 || cp1[-1] != KEY_MAP_VER || cp1[-2] != 0 ) {
		ret = -EINVAL;
pr_warn( "end %d %d %d %d\n", i, (int)cp1[-3], (int)cp1[-2], (int)cp1[-1] );
	} else {
		if ( set_key_info( in_buff, key_buf ) ) {
			set_key_info( key_defaults, key_buf );
		ret = -EINVAL;
pr_warn( "set key failed\n" );
		}
	}
	free_page ( ( unsigned long ) in_buff );
	return ret;
}

// this is the handler for /proc/speakup/version
static int
version_read_proc (PROC_READ_PROTOTYPE )
{
	int len = sprintf (page, "%s\n", SPEAKUP_VERSION );
	if ( synth != NULL )
		len += sprintf( page+len, "synth %s version %s\n",
			synth->name, synth->version );
	*start = 0;
	*eof = 1;
	return len;
}

// this is the read handler for /proc/speakup/characters
static int
chars_read_proc (PROC_READ_PROTOTYPE )
{
	int i, len = 0;
	off_t begin = 0;
	char *cp;
	for (i = 0; i < 256; i++ ) {
		cp = (characters[i] ) ? characters[i] : "NULL";
		len += sprintf (page + len, "%d\t%s\n", i, cp );
		if (len + begin > off + count )
			break;
		if (len + begin < off ) {
			begin += len;
			len = 0;
		}
	}
	if (i >= 256 )
		*eof = 1;
	if (off >= len + begin )
		return 0;
	*start = page + (off - begin );
	return ((count < begin + len - off ) ? count : begin + len - off );
}

static volatile int chars_timer_active = 0;	// indicates when timer is set
static declare_timer( chars_timer );

static inline void
chars_stop_timer (void )
{
	if (chars_timer_active )
		stop_timer ( chars_timer );
}

static int strings, rejects, updates;

static void
show_char_results (u_long data )
{
	int len;
	char buf[80];
	chars_stop_timer ( );
	len = sprintf (buf, " updated %d of %d character descriptions\n",
		       updates, strings );
	if (rejects )
		sprintf (buf + (len-1), " with %d reject%s\n",
			 rejects, rejects > 1 ? "s" : "" );
	printk( buf );
}

/* this is the write handler for /proc/speakup/silent */
static int
silent_write_proc (PROC_WRITE_PROTOTYPE )
{
	int currcons = fg_console;
	char ch = 0, shut;
	if (count > 0 || count < 3 ) {
		get_user (ch, buffer );
		if ( ch == '\n' ) ch = '0';
	}
	if ( ch < '0' || ch > '7' ) {
		pr_warn ( "silent value not in range (0,7)\n" );
		return count;
	}
	if ( (ch&2) ) {
		shut = 1;
		do_flush( );
	} else shut = 0;
	if ( (ch&4) ) shut |= 0x40;
	if ( (ch&1) )
		spk_shut_up |= shut;
		else spk_shut_up &= ~shut;
	return count;
}

// this is the write handler for /proc/speakup/characters
static int
chars_write_proc (PROC_WRITE_PROTOTYPE )
{
#define max_desc_len 72
	static int cnt = 0, state = 0;
	static char desc[max_desc_len + 1];
	static u_long jiff_last = 0;
	short i = 0, num;
	int len;
	char ch, *cp, *p_new;
	// reset certain vars if enough time has elapsed since last called
	if (jiffies - jiff_last > 10 ) {
		cnt = state = strings = rejects = updates = 0;
	}
	jiff_last = jiffies;
get_more:
	desc[cnt] = '\0';
	state = 0;
	for (; i < count && state < 2; i++ ) {
		get_user (ch, buffer + i );
		if ( ch == '\n' ) {
			desc[cnt] = '\0';
			state = 2;
		} else if (cnt < max_desc_len )
			desc[cnt++] = ch;
	}
	if (state < 2 ) return count;
	cp = desc;
	while ( *cp && *cp <= SPACE ) cp++;
	if ((!cnt ) || strchr ("dDrR", *cp ) ) {
		reset_default_chars ( );
		pr_info( "character descriptions reset to defaults\n" );
		cnt = 0;
		return count;
	}
	cnt = 0;
	if (*cp == '#' ) goto get_more;
	num = -1;
	cp = speakup_s2i(cp, &num );
	while ( *cp && *cp <= SPACE ) cp++;
	if (num < 0 || num > 255 ) {	// not in range
		rejects++;
		strings++;
		goto get_more;
	}
	if (num >= 27 && num <= 31 ) goto get_more;
	if (!strcmp(cp, characters[num] ) ) {
		strings++;
		goto get_more;
	}
	len = strlen(cp );
	if (characters[num] == default_chars[num] )
		p_new = (char * ) kmalloc (sizeof (char ) * len+1, GFP_KERNEL );
	else if ( strlen(characters[num] ) >= len )
		p_new = characters[num];
	else {
		kfree(characters[num] );
		characters[num] = default_chars[num];
		p_new = (char * ) kmalloc (sizeof (char ) * len+1, GFP_KERNEL );
	}
	if (!p_new ) return -ENOMEM;
	strcpy ( p_new, cp );
	characters[num] = p_new;
	updates++;
	strings++;
	if (i < count ) goto get_more;
	chars_stop_timer ( );
	init_timer (&chars_timer );
	chars_timer.function = show_char_results;
	chars_timer.expires = jiffies + 5;
		start_timer (chars_timer );
	chars_timer_active++;
	return count;
}

static int
bits_read_proc (PROC_READ_PROTOTYPE )
{
	int i;
	var_header *p_header = (var_header * )data;
	proc_var *var = p_header->data;
	bits_data *pb = &punc_info[var->value];
	short mask = pb->mask;
	char *cp = page;
	*start = 0;
	*eof = 1;
	for ( i = 33; i < 128; i++ ) {
		if ( !(spk_chartab[i]&mask ) ) continue;
		*cp++ = (char )i;
	}
	*cp++ = '\n';
	return cp-page;
}

/* set_mask_bits sets or clears the punc/delim/repeat bits,
 * if input is null uses the defaults.
 * values for how: 0 clears bits of chars supplied,
 * 1 clears allk, 2 sets bits for chars */

int
set_mask_bits( const char *input, const int which, const int how )
{
	u_char *cp;
	short mask = punc_info[which].mask;
	if ( how&1 ) {
		for ( cp = (u_char * )punc_info[3].value; *cp; cp++ )
			spk_chartab[*cp] &= ~mask;
	}
	cp = (u_char * )input;
	if ( cp == 0 ) cp = punc_info[which].value;
	else {
		for ( ; *cp; cp++ ) {
			if ( *cp < SPACE ) break;
			if ( mask < PUNC ) {
				if ( !(spk_chartab[*cp]&PUNC) ) break;
			} else if ( (spk_chartab[*cp]&B_NUM) ) break;
		}
		if ( *cp ) return -EINVAL;
		cp = (u_char * )input;
	}
	if ( how&2 ) {
		for ( ; *cp; cp++ )
			if ( *cp > SPACE ) spk_chartab[*cp] |= mask;
	} else {
		for ( ; *cp; cp++ )
			if ( *cp > SPACE ) spk_chartab[*cp] &= ~mask;
	}
	return 0;
}

static bits_data *pb_edit = NULL;

static int edit_bits (int currcons, u_char type, u_char ch, u_short key )
{
	short mask = pb_edit->mask, ch_type = spk_chartab[ch];
	if ( type != KT_LATIN || (ch_type&B_NUM ) || ch < SPACE ) return -1;
	if ( ch == SPACE ) {
		synth_write_msg( "edit done" );
		special_handler = NULL;
		return 1;
	}
	if ( mask < PUNC && !(ch_type&PUNC) ) return -1;
	spk_chartab[ch] ^= mask;
	speak_char( ch );
	synth_write_msg( (spk_chartab[ch]&mask ) ? " on" : " off" );
	return 1;
}

static int
bits_write_proc (PROC_WRITE_PROTOTYPE )
{
	var_header *p_header = (var_header * )data;
	proc_var *var = p_header->data;
	int ret = count;
	char punc_buf[100];
	if (count < 1 || count > 99 )
		return -EINVAL;
	if (copy_from_user (punc_buf, buffer, count ) )
		return -EFAULT;
	if (punc_buf[count - 1] == '\n' )
		count--;
	punc_buf[count] = '\0';
	if ( *punc_buf == 'd' || *punc_buf == 'r' )
		count = set_mask_bits( 0, var->value, 3 );
	else
		count = set_mask_bits( punc_buf, var->value, 3 );
	if ( count < 0 ) return count;
	return ret;
}

// this is the read handler for /proc/speakup/synth
static int
synth_read_proc (PROC_READ_PROTOTYPE )
{
	int len;
	if ( synth == NULL ) strcpy( synth_name, "none" );
	else strcpy( synth_name, synth->name );
	len = sprintf (page, "%s\n", synth_name );
	*start = 0;
	*eof = 1;
	return len;
}

// this is the write handler for /proc/speakup/synth
static int
synth_write_proc (PROC_WRITE_PROTOTYPE )
{
	int ret = count;
	char new_synth_name[10];
	const char *old_name = ( synth != NULL ) ? synth->name : "none";
	if (count < 2 || count > 9 )
		return -EINVAL;
	if (copy_from_user (new_synth_name, buffer, count ) )
		return -EFAULT;
	if (new_synth_name[count - 1] == '\n' )
		count--;
	new_synth_name[count] = '\0';
	strlwr (new_synth_name );
	if (!strcmp (new_synth_name, old_name ) ) {
		pr_warn ( "%s already in use\n", new_synth_name );
		return ret;
	}
	if ( synth_init( new_synth_name ) == 0 ) return ret;
	pr_warn( "failed to init synth %s\n", new_synth_name );
	return -ENODEV;
}

proc_var spk_proc_vars[] = {
	 { VERSION, version_read_proc, 0, 0 },
	 { SILENT, 0, silent_write_proc, 0 },
	 { CHARS, chars_read_proc, chars_write_proc, 0 },
	 { SYNTH, synth_read_proc, synth_write_proc, 0 },
	 { KEYMAP, keys_read_proc, keys_write_proc, 0 },
	 { PUNC_SOME, bits_read_proc, bits_write_proc, 1 },
	 { PUNC_MOST, bits_read_proc, bits_write_proc, 2 },
	 { PUNC_ALL, bits_read_proc, 0, 3 },
	 { DELIM, bits_read_proc, bits_write_proc, 4 },
	 { REPEATS, bits_read_proc, bits_write_proc, 5 },
	 { EXNUMBER, bits_read_proc, bits_write_proc, 6 },
	{ -1, 0, 0, 0 }
};

#endif // CONFIG_PROC_FS

#ifdef CONFIG_SPEAKUP

void __init
speakup_init (int currcons )
{
	spk_t *first_console = (spk_t *) alloc_bootmem (spk_size+1 );
	speakup_open( currcons, first_console );
}

#endif

void
speakup_allocate (int currcons )
{
	if ( speakup_console[currcons] == NULL ) {
		speakup_console[currcons] = (spk_t *) kmalloc (spk_size + 1,
			GFP_KERNEL );
		if ( speakup_console[currcons] == NULL ) return;
		memset( speakup_console[currcons], 0, spk_size );
		speakup_date( currcons);
	} else if ( !spk_parked ) speakup_date( currcons);
}

static void
speakup_date (int currcons )
{
	spk_x = spk_cx = x;
	spk_y = spk_cy = y;
	spk_pos = spk_cp = pos;
	spk_old_attr = spk_attr;
	spk_attr = ((scr_readw ((u_short * ) spk_pos ) & 0xff00 ) >> 8 );
}

static u_char is_cursor = 0;
static u_long old_cursor_pos, old_cursor_x, old_cursor_y;
static int cursor_con;
volatile int cursor_timer_active = 0;

void
cursor_stop_timer(void )
{
  if (!cursor_timer_active ) return;
		stop_timer ( cursor_timer );
	cursor_timer_active = 0;
}

static void
handle_cursor( KBD_PROTO )
{
	int currcons = fg_console;
	(*do_cursor)( KBD_ARGS );
	spk_parked &= 0xfe;
	if ( synth == NULL || up_flag || spk_shut_up || cursor_track == 0 )
	  return;
	spk_shut_up &= 0xfe;
	if ( no_intr ) do_flush( );
/* the key press flushes if !no_inter but we want to flush on cursor
 * moves regardless of no_inter state */
	is_cursor = value+1;
	old_cursor_pos = pos;
	old_cursor_x = x;
	old_cursor_y = y;
	cursor_con = currcons;
	cursor_stop_timer( );
	cursor_timer.expires = jiffies + cursor_timeout;
		start_timer (cursor_timer );
	cursor_timer_active++;
}

static void
cursor_done (u_long data )
{
	int currcons = cursor_con;
	cursor_stop_timer( );
	if (cursor_con != fg_console ) {
		is_cursor = 0;
		return;
	}
	speakup_date (currcons );
	if ( win_enabled ) {
		if ( x >= win_left && x <= win_right &&
		y >= win_top && y <= win_bottom ) {
			spk_keydown = is_cursor = 0;
			return;
		}
	}
	if ( is_cursor == 1 || is_cursor == 4 )
		say_line_from_to (currcons, 0, video_num_columns, 0 );
	else say_char ( currcons );
	spk_keydown = is_cursor = 0;
}

/* These functions are the interface to speakup from the actual kernel code. */

void
speakup_bs (int currcons )
{
	if (!spk_parked )
		speakup_date (currcons );
	if ( spk_shut_up || synth == NULL ) return;
	if ( currcons == fg_console  && spk_keydown ) {
		spk_keydown = 0;
		if (!is_cursor ) say_char (currcons );
	}
}

void
speakup_con_write (int currcons, const char *str, int len )
{
	if (spk_shut_up || (currcons != fg_console ) )
		return;
	if (bell_pos && spk_keydown && (x == bell_pos - 1 ) )
		bleep(3 );
	if (synth == NULL || is_cursor ) return;
	if ( win_enabled ) {
		if ( x >= win_left && x <= win_right &&
		y >= win_top && y <= win_bottom ) return;
	}
	spkup_write (str, len );
}

void
speakup_con_update (int currcons )
{
	if ( speakup_console[currcons] == NULL || spk_parked )
		return;
	speakup_date (currcons );
}

static void
handle_spec( KBD_PROTO )
{
	int currcons = fg_console, on_off = 2;
	char *label;
static const char *lock_status[] = { " off", " on", "" };
	(*do_spec)( KBD_ARGS );
	if ( synth == NULL || up_flag || spk_killed ) return;
	spk_shut_up &= 0xfe;
	if ( no_intr ) do_flush( );
	switch (value ) {
		case KVAL( K_CAPS ):
			label = "caps lock";
			on_off =  (vc_kbd_led(kbd , VC_CAPSLOCK ) );
			break;
		case KVAL( K_NUM ):
			label = "num lock";
			on_off = (vc_kbd_led(kbd , VC_NUMLOCK ) );
			break;
		case KVAL( K_HOLD ):
			label = "scroll lock";
			on_off = (vc_kbd_led(kbd , VC_SCROLLOCK ) );
			break;
	default:
		spk_parked &= 0xfe;
		return;
	}
	synth_write_string ( label );
	synth_write_msg ( lock_status[on_off] );
}

static int
inc_dec_var( u_char value )
{
	var_header *p_header;
	num_var *var_data;
	char num_buf[32];
	char *cp = num_buf, *pn;
	int var_id = (int)value - VAR_START;
	int how = (var_id&1) ? E_INC : E_DEC;
	var_id = var_id/2+FIRST_SET_VAR;
	p_header = get_var_header( var_id );
	if ( p_header == NULL ) return -1;
	if ( p_header->var_type != VAR_NUM ) return -1;
	var_data = p_header->data;
	if ( set_num_var( 1, p_header, how ) != 0 )
		return -1;
	if ( !spk_close_press ) {
		for ( pn = p_header->name; *pn; pn++ ) {
			if ( *pn == '_' ) *cp = SPACE;
			else *cp++ = *pn;
		}
	}
	sprintf( cp, " %d ", (int)var_data->value );
	synth_write_string( num_buf );
	return 0;
}

static void
speakup_win_set (int currcons )
{
	char info[40];
	if ( win_start > 1 ) {
		synth_write_msg( "window already set, clear then reset" );
		return;
	}
	if ( spk_x < win_left || spk_y < win_top ) {
		synth_write_msg( "error end before start" );
		return;
	}
	if ( win_start && spk_x == win_left && spk_y == win_top ) {
		win_left = 0;
		win_right = video_num_columns-1;
		win_bottom = spk_y;
		sprintf( info, "window is line %d", (int)win_top+1 );
	} else {
		if ( !win_start ) {
			win_top = spk_y;
			win_left = spk_x;
		} else {
			win_bottom = spk_y;
			win_right = spk_x;
		}
		sprintf( info, "%s at line %d, column %d",
			(win_start) ? "end" : "start",
			(int)spk_y+1, (int)spk_x+1 );
	}
	synth_write_msg( info );
	win_start++;
}

static void
speakup_win_clear (int currcons )
{
	win_top = win_bottom = 0;
	win_left = win_right = 0;
	win_start = 0;
	synth_write_msg( "window cleared" );
}

static void
speakup_win_enable (int currcons )
{
	if ( win_start < 2 ) {
		synth_write_msg( "no window" );
		return;
	}
	win_enabled ^= 1;
	if ( win_enabled ) synth_write_msg( "window silenced" );
	else synth_write_msg( "window silence disabled" );
}

static void
speakup_bits (int currcons )
{
	int val = this_speakup_key - ( FIRST_EDIT_BITS - 1 );
	if ( special_handler != NULL || val < 1 || val > 6 ) {
		synth_write_msg( "error" );
		return;
	}
	pb_edit = &punc_info[val];
	sprintf( buf, "edit  %s, press space when done", pb_edit->name );
	synth_write_msg( buf );
	special_handler = edit_bits;
}

static int handle_goto (int currcons, u_char type, u_char ch, u_short key )
{
	static u_char *goto_buf = "\0\0\0\0\0\0";
	static int num = 0;
	short maxlen, go_pos;
	char *cp;
	if ( type == KT_SPKUP && ch == SPEAKUP_GOTO ) goto do_goto;
	if ( type == KT_LATIN && ch == '\n' ) goto do_goto;
	if ( type != 0 ) goto oops;
	if (ch == 8 ) {
		if ( num == 0 ) return -1;
		ch = goto_buf[--num];
		goto_buf[num] = '\0';
		spkup_write( &ch, 1 );
		return 1;
}
	if ( ch < '+' || ch > 'y' ) goto oops;
	goto_buf[num++] = ch;
	goto_buf[num] = '\0';
	spkup_write( &ch, 1 );
	maxlen = ( *goto_buf >= '0' ) ? 3 : 4;
	if ((ch == '+' || ch == '-' ) && num == 1 ) return 1;
	if (ch >= '0' && ch <= '9' && num < maxlen ) return 1;
	if ( num < maxlen-1 || num > maxlen ) goto oops;
	if ( ch < 'x' || ch > 'y' ) {
oops:
		if (!spk_killed )
			synth_write_msg (" goto canceled" );
		goto_buf[num = 0] = '\0';
		special_handler = NULL;
		return 1;
	}
	cp = speakup_s2i (goto_buf, &go_pos );
	goto_pos = (u_long)go_pos;
	if (*cp == 'x' ) {
		if (*goto_buf < '0' ) goto_pos += spk_x;
		else goto_pos--;
		if (goto_pos < 0 ) goto_pos = 0;
		if (goto_pos >= video_num_columns )
			goto_pos = video_num_columns-1;
		goto_x = 1;
	} else {
		if (*goto_buf < '0' ) goto_pos += spk_y;
		else goto_pos--;
		if (goto_pos < 0 ) goto_pos = 0;
	if (goto_pos >= video_num_lines ) goto_pos = video_num_lines-1;
		goto_x = 0;
	}
		goto_buf[num = 0] = '\0';
do_goto:
	special_handler = NULL;
	spk_parked |= 0x01;
	if ( goto_x ) {
		spk_pos -= spk_x * 2;
		spk_x = goto_pos;
		spk_pos += goto_pos * 2;
		say_word( currcons );
	} else {
		spk_y = goto_pos;
		spk_pos = origin + ( goto_pos * video_size_row );
		say_line( currcons );
	}
	return 1;
}

static void
speakup_goto (int currcons )
{
	if ( special_handler != NULL ) {
		synth_write_msg( "error" );
		return;
	}
	synth_write_msg( "go to?" );
	special_handler = handle_goto;
	return;
}

static void
load_help ( void *dummy )
{
	request_module( "speakup_keyhelp" );
	if ( help_handler ) {
		(*help_handler)(0, KT_SPKUP, SPEAKUP_HELP, 0 );
	} else synth_write_string( "help module not found" );
}

#if (LINUX_VERSION_CODE >= 132419)
static DECLARE_WORK(ld_help, load_help, NULL);
#define schedule_help schedule_work
#else
static struct tq_struct ld_help = { routine: load_help, };
#define schedule_help schedule_task
#endif

static void
speakup_help (int currcons )
{
	if ( help_handler == NULL ) {
/* we can't call request_module from this context so schedule it*/
/* **** note kernel hangs and my wrath will be on you */
		schedule_help (&ld_help);
		return;
	}
	(*help_handler)(currcons, KT_SPKUP, SPEAKUP_HELP, 0 );
}

static void
do_nothing (int currcons )
{
	return; /* flush done in do_spkup */
}
static u_char key_speakup = 0, spk_key_locked = 0;

static void
speakup_lock (int currcons )
{
	if ( !spk_key_locked )
		spk_key_locked = key_speakup = 16;
	else spk_key_locked = key_speakup = 0;
}

typedef void (*spkup_hand )(int );
spkup_hand spkup_handler[] = { /* must be ordered same as defines in speakup.h */
	do_nothing, speakup_goto, speech_kill, speakup_shut_up,
	speakup_cut, speakup_paste, say_first_char, say_last_char,
	say_char, say_prev_char, say_next_char,
	say_word, say_prev_word, say_next_word,
	say_line, say_prev_line, say_next_line,
	top_edge, bottom_edge, left_edge, right_edge,
	        spell_word, spell_word, say_screen,
	say_position, say_attributes,
	speakup_off, speakup_parked, say_line, // this is for indent
	say_from_top, say_to_bottom,
	say_from_left, say_to_right,
	say_char_num, speakup_bits, speakup_bits, say_phonetic_char,
	speakup_bits, speakup_bits, speakup_bits,
	speakup_win_set, speakup_win_clear, speakup_win_enable, speakup_win_say,
	speakup_lock, speakup_help, toggle_cursoring, NULL
};

void do_spkup( int currcons,u_char value )
{
	if (spk_killed && value != SPEECH_KILL ) return;
	spk_keydown = 0;
	spk_lastkey = 0;
	spk_shut_up &= 0xfe;
	this_speakup_key = value;
	if (value < SPKUP_MAX_FUNC && spkup_handler[value] ) {
		do_flush( );
		(*spkup_handler[value] )(currcons );
	} else {
		if ( inc_dec_var( value ) < 0 )
			bleep( 9 );
	}
}

	static const char *pad_chars = "0123456789+-*/\015,.?()";

int
#if (LINUX_VERSION_CODE < 132419)
speakup_key ( int shift_state, u_char keycode, u_short keysym, u_char up_flag )
#else
speakup_key (struct vc_data *vc, int shift_state, int keycode, u_short keysym, int up_flag, struct pt_regs *regs )
#endif
{
	int currcons = fg_console;
	u_char *key_info;
	u_char type = KTYP( keysym ), value = KVAL( keysym ), new_key = 0;
	u_char shift_info, offset;
#if (LINUX_VERSION_CODE >= 132419)
	tty = vc_cons[currcons].d->vc_tty;
#endif
	if ( synth == NULL ) return 0;
	if ( type >= 0xf0 ) type -= 0xf0;
	if ( type == KT_PAD && (vc_kbd_led(kbd , VC_NUMLOCK ) ) ) {
		if ( up_flag ) {
			spk_keydown = 0;
			return 0;
		}
		value = spk_lastkey = pad_chars[value];
		spk_keydown++;
		spk_parked &= 0xfe;
		goto no_map;
	}
	if ( keycode >= MAX_KEY ) goto no_map;
	if ( ( key_info = our_keys[keycode] ) == 0 ) goto no_map;
	shift_info = ( shift_state&0x0f ) + key_speakup;
	offset = shift_table[shift_info];
	if ( offset && ( new_key = key_info[offset] ) ) {
		if ( new_key == SPK_KEY ) {
			if ( !spk_key_locked )
				key_speakup = ( up_flag ) ? 0 : 16;
			if ( up_flag || spk_killed ) return 1;
			spk_shut_up &= 0xfe;
			do_flush( );
			return 1;
		}
		if ( up_flag ) return 1;
		if ( last_keycode == keycode && last_spk_jiffy+MAX_DELAY > jiffies ) {
			spk_close_press = 1;
			offset = shift_table[shift_info+32];
/* double press? */
			if ( offset && key_info[offset] )
				new_key = key_info[offset];
		}
		last_keycode = keycode;
		last_spk_jiffy = jiffies;
		type = KT_SPKUP;
		value = new_key;
	}
no_map:
	if ( type == KT_SPKUP && special_handler == NULL ) {
		do_spkup( currcons, new_key );
		spk_close_press = 0;
		return 1;
	}
	if ( up_flag || spk_killed || type == KT_SHIFT ) return 0;
	spk_shut_up &= 0xfe;
	if (!no_intr ) do_flush( );
	if ( special_handler ) {
		int status;
		if ( type == KT_SPEC && value == 1 ) {
			value = '\n';
			type = KT_LATIN;
		} else if ( type == KT_LETTER ) type = KT_LATIN;
		else if ( value == 0x7f ) value = 8; /* make del = backspace */
		status = (*special_handler)(currcons, type, value, keycode );
		spk_close_press = 0;
		if ( status < 0 ) bleep( 9 );
		return status;
	}
	last_keycode = 0;
	return 0;
}

#ifdef MODULE

extern void proc_speakup_init( void );
extern void proc_speakup_remove( void );
extern void speakup_set_addresses ( spk_con_func, spk_con_func, spk_write_func, spk_con_func, spk_key_func);

static void __exit mod_speakup_exit( void )
{
	int i;
	key_handler[KT_LATIN] = do_latin;
	key_handler[KT_SPEC] = do_spec;
	key_handler[KT_CUR] = do_cursor;
	key_handler[KT_SHIFT] = do_shift;
	speakup_set_addresses( NULL, NULL, NULL, NULL, NULL );
	synth_release( );
	proc_speakup_remove( );
	for (i = 0; i < 256; i++ ) {
		if (characters[i] != default_chars[i] )
			kfree (characters[i] );
	}
	for ( i = 0; speakup_console[i]; i++) {
	  kfree( speakup_console[i] );
	  speakup_console[i] = NULL;
	}
}

static int __init mod_speakup_init( void )
{
int i;
	spk_t *first_console = (spk_t *) kmalloc (spk_size + 1, GFP_KERNEL );
	speakup_open( fg_console, first_console );
for ( i = 0; vc_cons[i].d; i++)
  speakup_allocate(i);
	speakup_set_addresses( speakup_allocate, speakup_bs,
		speakup_con_write, speakup_con_update, speakup_key );
	proc_speakup_init( );
	return 0;
}

module_init( mod_speakup_init );
module_exit( mod_speakup_exit );

#endif
