#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <pthread.h>
#include <glib.h>
#include "specimen.h"
#include "maths.h"
#include "ticks.h"
#include "patch.h"
#include "sample.h"
#include "adsr.h"
#include "lfo.h"
#include "driver.h"		/* for DRIVER_DEFAULT_SAMPLERATE */

/* the minimum envelope release value we'll allow (to prevent clicks) */
const float PATCH_MIN_RELEASE = 0.05;

/* how much time to wait before actually releasing legato patches;
 * this is to make sure that noteons immediately following noteoffs
 * stay "connected" */
const float PATCH_LEGATO_LAG = 0.05;

/* how much to decrease v->declick_vol by each tick; calculated to
 * take PATCH_MIN_RELEASE seconds (the initial value below is
 * bogus) */
static float declick_dec = -0.1;

/* how many ticks legato releases lag; calculated to take
 * PATCH_LEGATO_LAG seconds (init value is bogus) */
static int legato_lag = 20;

/* in certain places, we consider values with an absolute value less
 * than or equal to this to be equivalent to zero */
const float ALMOST_ZERO = 0.000001;

/* what sample rate we think the audio interface is running at */
static float samplerate = DRIVER_DEFAULT_SAMPLERATE;

/* type for currently playing notes (voices) */
typedef struct _PatchVoice
{
    gboolean active;		/* whether this voices is playing or not */
    Tick     ticks;		/* at what time this voice was activated */
    int      offset;		/* how many ticks should pass before this voice is used */
    int      relset;		/* how many ticks should pass before we release this voice (negative if N/A) */
    gboolean released;		/* whether or not we've been released or not */
    float    declick_vol;	/* current volume being applied to declick phase */
    int      dir;		/* current direction (forward == 1, reverse == -1) */
    int      note;		/* the note that activated us */
    double   pitch;		/* what pitch ratio to play at */
    double   pitch_step;	/* how much to increment pitch by each porta_tick */
    int      porta_ticks;	/* how many ticks to increment pitch for */
    int      posi;		/* integer sample index */
    guint32  posf;		/* fractional sample index */
    int      stepi;		/* integer step amount */
    guint32  stepf;		/* fractional step amount */
    float    vel;		/* velocity; volume of this voice */

    /* legato stuff */
    int      retrig_ticks;	/* how many ticks until we retrigger */
    int      retrig_note;	/* the note we will retrigger at */
    float    retrig_vel;	/* the velocity we will retrigger at */

    ADSR     vol;		/* volume envelope */
    ADSR     pan;		/* panning envelope */
    ADSR     ffreq;       	/* filter frequency envelope */
    ADSR     freso;		/* filter resonance envelope */
    ADSR     pitchenv;		/* pitch envelope */

    LFO      vol_lfo;		/* volume lfo */
    LFO      pan_lfo;		/* panning lfo */
    LFO      ffreq_lfo;		/* filter frequency lfo */
    LFO      freso_lfo;		/* filter resonance lfo */
    LFO      pitch_lfo;		/* pitch lfo */

    float    fll;		/* lowpass filter buffer, left */
    float    flr;		/* lowpass filter buffer, right */
    float    fbl;		/* bandpass filter buffer, left */
    float    fbr;		/* bandpass filter buffer, right */
}
    PatchVoice;

/* type for modulatable parameters */
typedef struct _PatchParam
{
    float      val;		/* value of this parameter */

    ADSRParams env;		/* envelope parameters */
    float      env_amt;		/* amount of envelope we add [-1.0, 1.0] */
    gboolean   env_on;		/* whether we are using this envelope or not */

    LFOParams  lfo_params;	/* lfo-params for this parameter */
    float      lfo_amt;		/* amount of lfo we add [-1.0, 1.0] */
    gboolean   lfo_on;		/* whether we are using this lfo or not */
    gboolean   use_global_lfo;	/* whether to use global_lfo or voice's lfo */
    LFO        global_lfo;	/* optionally used global lfo */
    float*     lfo_tab;		/* used for storing global_lfo values */
     
    float      vel_amt;		/* amount of velocity to use [0.0, 1.0] */
}
    PatchParam;

/* type for array of instruments (called patches) */
typedef struct _Patch
{
    int      active;		/* whether this instance of Patch is in use or not */
    Sample*  sample;		/* sample data */
    int      display_index;	/* order in which this Patch to be displayed */
    char     name[PATCH_MAX_NAME];
    int      channel;		/* midi channel to listen on */
    int      note;		/* midi note to listen on */
    int      range;		/* whether to listen to a range of notes or not */
    int      lower_note;	/* lowest note in range */
    int      upper_note;	/* highest note in range */
    int      cut;		/* cut signal this patch emits */
    int      cut_by;		/* what cut signals stop this patch */
    int      sample_start;	/* the first frame to play */
    int      sample_stop;	/* the last frame to play */
    int      loop_start;	/* the first frame to loop at */
    int      loop_stop;		/* the last frame to loop at */
    gboolean porta;		/* whether portamento is being used or not */
    float    porta_secs;	/* length of portamento slides in seconds */
    int      pitch_steps;	/* range of pitch.val in halfsteps */
    float    pitch_bend;	/* pitch bending factor */
    gboolean mono;		/* whether patch is monophonic or not */
    gboolean legato;		/* whether patch is played legato or not */

    PatchPlayMode play_mode;	/* how this patch is to be played */
    PatchParam    vol;		/* volume:                  [0.0, 1.0] */
    PatchParam    pan;		/* panning:                [-1.0, 1.0] */
    PatchParam    ffreq;	/* filter cutoff frequency: [0.0, 1.0] */
    PatchParam    freso;	/* filter resonance:        [0.0, 1.0] */
    PatchParam    pitch;	/* pitch scaling:           [0.0, 1.0] */

    /* the following parameters handle the special case of "amount"
     * when modulating pitch by LFO and/or ADSR */
    double lfo_pitch_max;
    double lfo_pitch_min;
    double env_pitch_max;
    double env_pitch_min;
     
    /* each patch is responsible for its own voices */
    PatchVoice voices[PATCH_VOICE_COUNT];
    int        last_note;	/* the last MIDI note value that played us */
     
    /* used exclusively by patch_lock functions to ensure that
     * patch_render ignores this patch */
    pthread_mutex_t mutex;
}
    Patch;

/* array of all patches  */
static Patch patches[PATCH_COUNT];

/*****************************************************************************/
/************************ PRIVATE GENERAL HELPER FUNCTIONS********************/
/*****************************************************************************/

/* verifies that a given id refers to a valid patch */
inline static int isok (int id)
{
    if (id < 0 || id >= PATCH_COUNT || !patches[id].active)
	return 0;

    return 1;
}

/* a utility function which finds the PatchParam* associated with
 * a particular PatchParamType */
static int patch_get_param (PatchParam** p, int id, PatchParamType param)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    switch (param)
    {
    case PATCH_PARAM_VOLUME:
	*p = &patches[id].vol;
	break;
    case PATCH_PARAM_PANNING:
	*p = &patches[id].pan;
	break;
    case PATCH_PARAM_CUTOFF:
	*p = &patches[id].ffreq;
	break;
    case PATCH_PARAM_RESONANCE:
	*p = &patches[id].freso;
	break;
    case PATCH_PARAM_PITCH:
	*p = &patches[id].pitch;
	break;
    default:
	return PATCH_PARAM_INVALID;
	break;
    }

    return 0;
}

/* locks a patch so that it will be ignored by patch_render() */
inline static void patch_lock (int id)
{
    g_assert (id >= 0 && id < PATCH_COUNT);
     
    pthread_mutex_lock (&patches[id].mutex);
}

/* same as above, but returns immediately with EBUSY if mutex is already held */
inline static int patch_trylock (int id)
{
    g_assert (id >= 0 && id < PATCH_COUNT);

    return pthread_mutex_trylock (&patches[id].mutex);
}

/* unlocks a patch after use */
inline static void patch_unlock (int id)
{
    g_assert (id >= 0 && id < PATCH_COUNT);

    pthread_mutex_unlock (&patches[id].mutex);
}

/* triggers all global LFOs if they are used with amounts greater than 0 */
inline static void patch_trigger_global_lfos ( )
{
    int i;

    debug ("retriggering...\n")
	for (i = 0; i < PATCH_COUNT; i++)
	{
	    lfo_trigger (&patches[i].vol.global_lfo, &patches[i].vol.lfo_params);
	    lfo_trigger (&patches[i].pan.global_lfo, &patches[i].pan.lfo_params);
	    lfo_trigger (&patches[i].ffreq.global_lfo, &patches[i].ffreq.lfo_params);
	    lfo_trigger (&patches[i].freso.global_lfo, &patches[i].freso.lfo_params);
	    lfo_trigger (&patches[i].pitch.global_lfo, &patches[i].pitch.lfo_params);
	}
    debug ("done\n");
}     
/*****************************************************************************/
/************************ UTILITY FUNCTIONS **********************************/
/*****************************************************************************/

/* returns the number of patches currently active */
int patch_count ( )
{
    int id, count;

    for (id = count = 0; id < PATCH_COUNT; id++)
	if (patches[id].active)
	    count++;

    return count;
}

/* returns assigned patch id on success, negative value on failure */
int patch_create (const char *name)
{
    int id, i;
    ADSRParams defadsr;
    LFOParams deflfo;
    PatchVoice defvoice;

    /* find an unoccupied patch id */
    for (id = 0; patches[id].active; id++)
	if (id == PATCH_COUNT)
	    return PATCH_LIMIT;

    patch_lock (id);
    patches[id].active = 1;

    debug ("Creating patch %s (%d).\n", name, id);

    /* name */
    g_strlcpy (patches[id].name, name, PATCH_MAX_NAME);
     
    /* default values */
    patches[id].channel = 0;
    patches[id].note = 60;
    patches[id].range = 0;
    patches[id].lower_note = 60;
    patches[id].upper_note = 60;
    patches[id].play_mode = PATCH_PLAY_FORWARD | PATCH_PLAY_SINGLESHOT;
    patches[id].cut = 0;
    patches[id].cut_by = 0;
    patches[id].sample_start = 0;
    patches[id].sample_stop = 0;
    patches[id].loop_start = 0;
    patches[id].loop_stop = 0;
    patches[id].porta = FALSE;
    patches[id].mono = FALSE;
    patches[id].legato = FALSE;
    patches[id].porta_secs = 0.05;
    patches[id].pitch_steps = 2;
    patches[id].pitch_bend = 0;
    patches[id].lfo_pitch_max = 1.0;
    patches[id].lfo_pitch_min = 1.0;
    patches[id].env_pitch_max = 1.0;
    patches[id].env_pitch_min = 1.0;
     
    /* default adsr params */
    defadsr.delay   = 0.0;
    defadsr.attack  = 0.0;
    defadsr.hold    = 0.0;
    defadsr.decay   = 0.0;
    defadsr.sustain = 1.0;
    defadsr.release = PATCH_MIN_RELEASE;

    /* default lfo params */
    deflfo.positive = FALSE;
    deflfo.shape = LFO_SHAPE_SINE;
    deflfo.freq = 1.0;
    deflfo.sync_beats = 1.0;
    deflfo.sync = FALSE;
    deflfo.delay = 0.0;
    deflfo.attack = 0.0;

    /* volume */
    patches[id].vol.val = 1.0;
    patches[id].vol.env = defadsr;
    patches[id].vol.env_amt = 0.0; /* this parameter is actually ignored for volume */
    patches[id].vol.env_on = FALSE;
    patches[id].vol.vel_amt = 1.0;
    patches[id].vol.lfo_params = deflfo;
    patches[id].vol.lfo_amt = 0.0;
    patches[id].vol.lfo_on = FALSE;
    patches[id].vol.use_global_lfo = FALSE;
    lfo_prepare (&patches[id].vol.global_lfo);

    /* panning */
    patches[id].pan.val = 0.0;
    patches[id].pan.env = defadsr;
    patches[id].pan.env_amt = 0.0;
    patches[id].pan.env_on = FALSE;
    patches[id].pan.vel_amt = 0.0;
    patches[id].pan.lfo_params = deflfo;
    patches[id].pan.lfo_amt = 0.0;
    patches[id].pan.lfo_on = FALSE;
    patches[id].pan.use_global_lfo = FALSE;
    lfo_prepare (&patches[id].pan.global_lfo);
     
    /* cutoff */
    patches[id].ffreq.val = 1.0;
    patches[id].ffreq.env = defadsr;
    patches[id].ffreq.env_amt = 0.0;
    patches[id].ffreq.env_on = FALSE;
    patches[id].ffreq.vel_amt = 0.0;
    patches[id].ffreq.lfo_params = deflfo;
    patches[id].ffreq.lfo_amt = 0.0;
    patches[id].ffreq.lfo_on = FALSE;
    patches[id].ffreq.use_global_lfo = FALSE;
    lfo_prepare (&patches[id].ffreq.global_lfo);

    /* resonance */
    patches[id].freso.val = 0.0;
    patches[id].freso.env = defadsr;
    patches[id].freso.env_amt = 0.0;
    patches[id].freso.env_on = FALSE;
    patches[id].freso.vel_amt = 0.0;
    patches[id].freso.lfo_params = deflfo;
    patches[id].freso.lfo_amt = 0.0;
    patches[id].freso.lfo_on = FALSE;
    patches[id].freso.use_global_lfo = FALSE;
    lfo_prepare (&patches[id].freso.global_lfo);

    /* pitch */
    patches[id].pitch.val = 0.0;
    patches[id].pitch.env = defadsr;
    patches[id].pitch.env_amt = 0.0;
    patches[id].pitch.env_on = FALSE;
    patches[id].pitch.vel_amt = 0.0;
    patches[id].pitch.lfo_params = deflfo;
    patches[id].pitch.lfo_amt = 0.0;
    patches[id].pitch.lfo_on = FALSE;
    patches[id].pitch.use_global_lfo = FALSE;
    lfo_prepare (&patches[id].pitch.global_lfo);

    /* default voice */
    defvoice.active = FALSE;
    defvoice.offset = 0;
    defvoice.note = 0;
    defvoice.posi = 0;
    defvoice.posf = 0;
    defvoice.stepi = 0;
    defvoice.stepf = 0;
    defvoice.vel = 0;

    adsr_init (&defvoice.vol);
    adsr_init (&defvoice.pan);
    adsr_init (&defvoice.ffreq);
    adsr_init (&defvoice.freso);
    adsr_init (&defvoice.pitchenv);

    lfo_prepare (&defvoice.vol_lfo);
    lfo_prepare (&defvoice.pan_lfo);
    lfo_prepare (&defvoice.ffreq_lfo);
    lfo_prepare (&defvoice.freso_lfo);
    lfo_prepare (&defvoice.pitch_lfo);

    defvoice.fll = 0;
    defvoice.flr = 0;
    defvoice.fbl = 0;
    defvoice.fbr = 0;
     
    /* initialize voices */
    for (i = 0; i < PATCH_VOICE_COUNT; i++)
    {
	patches[id].voices[i] = defvoice;
	patches[id].last_note = 60;
    }
     
    /* set display_index to next unique value */
    patches[id].display_index = 0;
    for (i = 0; i < PATCH_COUNT; i++)
    {
	if (i == id)
	    continue;
	if (patches[i].active
	    && patches[i].display_index >= patches[id].display_index)
	{
	    patches[id].display_index = patches[i].display_index + 1;
	}
    }

    patch_unlock (id);
    return id;
}

/* destroy a single patch with given id */
int patch_destroy (int id)
{
    int index;

    if (!isok (id))
	return PATCH_ID_INVALID;
    debug ("Removing patch: %d\n", id);

    patch_lock (id);

    patches[id].active = 0;
    sample_free_file (patches[id].sample);

    patch_unlock (id);

    /* every active patch with a display_index greater than this
     * patch's needs to have it's value decremented so that we
     * preservere continuity; no locking necessary because the
     * display_index is not thread-shared data */
    index = patches[id].display_index;
    for (id = 0; id < PATCH_COUNT; id++)
    {
	if (patches[id].active && patches[id].display_index > index)
	    patches[id].display_index--;
    }

    return 0;
}

/* destroy all patches */
void patch_destroy_all ( )
{
    int id;

    for (id = 0; id < PATCH_COUNT; id++)
	patch_destroy (id);

    return;
}

/* place all patch ids, sorted in ascending order by display index,
   into array 'id' and return number of patches */
int patch_dump (int **dump)
{
    int i, j, id, count, tmp;

    *dump = NULL;

    /* determine number of patches */
    count = patch_count ( );

    if (count == 0)
	return count;

    /* allocate dump */
    *dump = malloc (sizeof (int) * count);
    if (*dump == NULL)
	return PATCH_ALLOC_FAIL;

    /* place active patches into dump array */
    for (id = i = 0; id < PATCH_COUNT; id++)
	if (patches[id].active)
	    (*dump)[i++] = id;

    /* sort dump array by display_index in ascending order */
    for (i = 0; i < count; i++)
    {
	for (j = i; j < count; j++)
	{
	    if (patches[(*dump)[j]].display_index <
		patches[(*dump)[i]].display_index)
	    {
		tmp = (*dump)[i];
		(*dump)[i] = (*dump)[j];
		(*dump)[j] = tmp;
	    }
	}
    }

    return count;
}

int patch_duplicate (int target)
{
    int id, i;
    Sample* oldsam;
    float* vol;
    float* pan;
    float* cut;
    float* res;
    float* pitch;
     
    if (target < 0 || target > PATCH_COUNT || !patches[target].active)
	return PATCH_ID_INVALID;

    /* find an unoccupied patch id */
    for (id = 0; patches[id].active; id++)
	if (id == PATCH_COUNT)
	    return PATCH_LIMIT;

    debug ("Creating patch (%d) from patch %s (%d).\n", id,
	   patches[target].name, target);

    patch_lock (id);

    /* we have to store and restore our pointers to allocated memory,
     * because the assignment below will overwrite them </kludge?> */
    vol = patches[id].vol.lfo_tab;
    pan = patches[id].pan.lfo_tab;
    cut = patches[id].ffreq.lfo_tab;
    res = patches[id].freso.lfo_tab;
    pitch = patches[id].pitch.lfo_tab;
    oldsam = patches[id].sample;
     
    patches[id] = patches[target];

    patches[id].vol.lfo_tab = vol;
    patches[id].pan.lfo_tab = pan;
    patches[id].ffreq.lfo_tab = cut;
    patches[id].freso.lfo_tab = res;
    patches[id].pitch.lfo_tab = pitch;
    patches[id].sample = oldsam;

    /* this is residual paranoia, I think */
    patches[id].sample->sp = NULL;

    /* load same soundfile as target patch */
    if (patches[target].sample->sp != NULL)
    {
	patch_sample_load (id, sample_get_file (patches[target].sample));
    }

    /* set display_index to next unique value */
    patches[id].display_index = 0;
    for (i = 0; i < PATCH_COUNT; i++)
    {
	if (i == id)
	    continue;
	if (patches[i].active
	    && patches[i].display_index >= patches[id].display_index)
	{
	    patches[id].display_index = patches[i].display_index + 1;
	}
    }
    debug ("chosen display: %d\n", patches[id].display_index);

    patch_unlock (id);
    return id;
}

/* stop all currently playing voices in given patch */
int patch_flush (int id)
{
    int i;
     
    if (!isok(id))
	return PATCH_ID_INVALID;

    patch_lock (id);
    if (patches[id].sample->sp == NULL)
    {
	patch_unlock (id);
	return 0;
    }

    for (i = 0; i < PATCH_VOICE_COUNT; i++)
	patches[id].voices[i].active = FALSE;
	  
    patch_unlock (id);
    return 0;
}

/* stop all voices for all patches */
void patch_flush_all ( )
{
    int i;

    for (i = 0; i < PATCH_COUNT; i++)
    {
	patch_flush (i);
    }
}    

/* constructor */
void patch_init ( )
{
    int i;
     
    debug ("initializing...\n");
    for (i = 0; i < PATCH_COUNT; i++)
    {
	pthread_mutex_init (&patches[i].mutex, NULL);
	patches[i].sample = sample_new ( );
	patches[i].vol.lfo_tab = NULL;
	patches[i].pan.lfo_tab = NULL;
	patches[i].ffreq.lfo_tab = NULL;
	patches[i].freso.lfo_tab = NULL;
	patches[i].pitch.lfo_tab = NULL;
    }
    debug ("done\n");
}

/* returns error message associated with error code */
const char *patch_strerror (int error)
{
    switch (error)
    {
    case PATCH_PARAM_INVALID:
	return "patch parameter is invalid";
	break;
    case PATCH_ID_INVALID:
	return "patch id is invalid";
	break;
    case PATCH_ALLOC_FAIL:
	return "failed to allocate space for patch";
	break;
    case PATCH_NOTE_INVALID:
	return "specified note is invalid";
	break;
    case PATCH_PAN_INVALID:
	return "specified panning is invalid";
	break;
    case PATCH_CHANNEL_INVALID:
	return "specified channel is invalid";
	break;
    case PATCH_VOL_INVALID:
	return "specified volume is invalid";
	break;
    case PATCH_PLAY_MODE_INVALID:
	return "specified patch play mode is invalid";
	break;
    case PATCH_LIMIT:
	return "maximum patch count reached, can't create another";
	break;
    case PATCH_SAMPLE_INDEX_INVALID:
	return "specified sample is invalid";
	break;
    default:
	return "unknown error";
    }
}

/* loads a sample file for a patch */
int patch_sample_load (int id, const char *name)
{
    int val;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if (name == NULL)
    {
	debug ("Refusing to load null sample for patch %d\n", id);
	return PATCH_PARAM_INVALID;
    }

    debug ("Loading sample %s for patch %d\n", name, id);
    patch_flush (id);

    /* we lock *after* we call patch_flush because patch_flush does
     * its own locking */
    patch_lock (id);
    val = sample_load_file (patches[id].sample, name, samplerate);

    /* set the sample/loop start/stop point appropriately */
    patches[id].sample_start = 0;
    patches[id].sample_stop = patches[id].sample->frames - 1;
    patches[id].loop_start = 0;
    patches[id].loop_stop = patches[id].sample->frames - 1;

    patch_unlock (id);
    return val;
}

/* unloads a patch's sample */
void patch_sample_unload (int id)
{
    if (!isok(id))
	return;
     
    debug ("Unloading sample for patch %d\n", id);
    patch_lock (id);

    sample_free_file (patches[id].sample);

    patches[id].sample_start = 0;
    patches[id].sample_stop = 0;
    patches[id].loop_start = 0;
    patches[id].loop_stop = 0;

    patch_unlock (id);
}

/* sets our buffersize and reallocates our lfo_tab; this function
 * doesn't need to do any locking because we have a guarantee that
 * mixing will stop when the buffersize changes */
void patch_set_buffersize (int nframes)
{
    int i;
    Patch* p;
     
    debug ("setting buffersize to %d\n", nframes);
    for (i = 0; i < PATCH_COUNT; i++)
    {
	p = &patches[i];

	p->vol.lfo_tab = g_renew (float, p->vol.lfo_tab, nframes);
	p->pan.lfo_tab = g_renew (float, p->pan.lfo_tab, nframes);
	p->ffreq.lfo_tab = g_renew (float, p->ffreq.lfo_tab, nframes);
	p->freso.lfo_tab = g_renew (float, p->freso.lfo_tab, nframes);
	p->pitch.lfo_tab = g_renew (float, p->pitch.lfo_tab, nframes);
    }
}

/* sets our samplerate and resamples if necessary; this function
 * doesn't need to do any locking because we have a guarantee that
 * mixing will stop when the samplerate changes */
void patch_set_samplerate (int rate)
{
    int id;
    char *name;
    int oldrate = samplerate;

    samplerate = rate;

    debug ("changing samplerate to %d\n", rate);
    if (samplerate != oldrate)
    {	 
	for (id = 0; id < PATCH_COUNT; id++)
	{
	    if (!patches[id].active)
		continue;
	       
	    name = patch_get_sample_name (id);
	    patch_sample_load (id, name);
	    g_free (name);
	}
    }

    declick_dec = 1.0 / (PATCH_MIN_RELEASE * rate);
    legato_lag = PATCH_LEGATO_LAG * rate;
    debug("declick_dec = %f\n", declick_dec);
    debug("legato_lag = %d\n", legato_lag);
     
    patch_trigger_global_lfos ( );
}

/* destructor */
void patch_shutdown ( )
{
    int i;
     
    debug ("shutting down...\n");

    for (i = 0; i < PATCH_COUNT; i++)
    {
	sample_free (patches[i].sample);
	g_free (patches[i].vol.lfo_tab);
	g_free (patches[i].pan.lfo_tab);
	g_free (patches[i].ffreq.lfo_tab);
	g_free (patches[i].freso.lfo_tab);
	g_free (patches[i].pitch.lfo_tab);
    }
     
    debug ("done\n");
}

/* re-sync all global lfos to new tempo */
void patch_sync (float bpm)
{
    lfo_set_tempo (bpm);
    patch_trigger_global_lfos ( );
}

/*****************************************************************************/
/******************* PLAYBACK AND RENDERING FUNCTIONS ************************/
/*****************************************************************************/

/* a helper function to release all voices matching a given criteria
 * (if note is a negative value, all active voices will be released) */
inline static void patch_release_patch (Patch* p, int note, int offset)
{
    int i;
    PatchVoice* v;

    for (i = 0; i < PATCH_VOICE_COUNT; i++)
    {
	if (p->voices[i].active && (p->voices[i].note == note || note < 0))
	{
	    v = &p->voices[i];

	    /* we don't really release here, that's the job of
	     * advance( ); we just tell it *when* to release */
	    v->relset = offset;

	    if (p->mono && p->legato)
		v->relset += legato_lag;
	}
    }
}

/* a helper function to cut all patches whose cut_by value matches the
 * cut value for the given patch */
inline static void patch_cut_patch (Patch* p, int offset)
{
    int i;

    /* a cut value of zero is ignored so that the user has a way of *not* using
     * cuts*/
    if (p->cut == 0)
	return;

    for (i = 0; i < PATCH_COUNT; i++)
    {
	if (patches[i].active && patches[i].cut_by == p->cut)
	{
	    patch_release_patch (&patches[i], -69, offset);
	}
    }
}

/* a helper function to prepare a voice's pitch information */
inline static void prepare_pitch(Patch* p, PatchVoice* v, int note)
{
    if (p->porta && (p->porta_secs > 0.0)
	&& (p->last_note != note))
    {
	/* we calculate the pitch here because we can't be certain
	 * what the current value of the last voice's pitch is */
	v->pitch = pow (2, (p->last_note - p->note) / 12.0);
	  
	v->porta_ticks = ticks_secs_to_ticks (p->porta_secs);

	/* calculate the value to be added to pitch each tick by
	 * subtracting the target pitch from the initial pitch and
	 * dividing by porta_ticks */
	v->pitch_step = ((pow (2, (note - p->note)
			       / 12.0))
			 - v->pitch)
	    / v->porta_ticks;
    }
    else
    {
	v->pitch = pow (2, (note - p->note) / 12.0);
	v->porta_ticks = 0;
    }

    /* store the note that called us as the last note that was played
     * (it's important that we do this *after* we use the
     * p->last_note variable, otherwise we'll be comparing ourself
     * against ourself) */
    p->last_note = note;

    v->pitch *= pow (2, (p->pitch.val * p->pitch_steps) / 12.0); /* scale to base pitch */
    v->stepi = v->pitch;
    v->stepf = (v->pitch - v->stepi) * (0xFFFFFFFFU); /* maximum value of guint32 */
}

/* a helper function to activate a voice for a patch */
inline static void patch_trigger_patch (Patch* p, int note, float vel,
					Tick ticks, int offset)
{
    int i;
    PatchVoice* v;
    int index;			/* the index we ended up settling on */
    int gzr = 0;		/* index of the oldest running patch */
    int empty = -1;		/* index of the first empty patch we encountered */
    Tick oldest = ticks;	/* time of the oldest running patch */

    /* we don't activate voices if we don't have a sample */
    if (p->sample->sp == NULL)
	return;

    if (p->mono && p->legato)
    {
	/* find a voice to retrigger */
	for (i = 0; i < PATCH_VOICE_COUNT; ++i)
	{
	    if (p->voices[i].ticks <= oldest)
	    {
		oldest = p->voices[i].ticks;
		gzr = i;
	    }

	    if (!p->voices[i].active)
		empty = i;
	    else if (!p->voices[i].released)
		break;
	}

	/* if we couldn't find a voice to take over, use the first
	 * empty one we encountered; failing that, we'll take the
	 * oldest running */
	if (i == PATCH_VOICE_COUNT)
	{
	    if (empty < 0)
		index = gzr;
	    else
		index = empty;
	}
	else
	{
	    p->voices[i].ticks = ticks;
	    p->voices[i].retrig_ticks = offset;
	    p->voices[i].retrig_note = note;
	    p->voices[i].retrig_vel = vel;
	    p->voices[i].relset = -1; /* cancel any pending release */

	    return;
	}
    }
    else
    {
	/* find a free voice slot and determine the oldest running voice */
	for (i = 0; i < PATCH_VOICE_COUNT; ++i)
	{
	    if (p->voices[i].ticks <= oldest)
	    {
		oldest = p->voices[i].ticks;
		gzr = i;
	    }
	  
	    if (!p->voices[i].active)
	    {
		break;
	    }
	}

	/* take the oldest running voice's slot if we couldn't find an
	 * empty one */
	if (i == PATCH_VOICE_COUNT)
	{
	    index = gzr;
	}
	else
	{
	    index = i;
	}
    }

    v = &p->voices[index];
    
    /* shutdown any running voices if monophonic */
    if (p->mono)
	patch_release_patch(p, -1, offset);

    /* fill in our voice */
    v->ticks = ticks;
    v->offset = offset;
    v->relset = -1;		/* N/A at this time */
    v->released = FALSE;
    v->declick_vol = 1.0;
    v->note = note;
    v->fll = 0;
    v->fbl = 0;
    v->flr = 0;
    v->fbr = 0;
    v->retrig_ticks = -1;	/* N/A for non-legato patches */
    v->retrig_note = -1;	/* N/A for non-legato patches */
    v->vel = vel;

    /* setup direction */
    if (!(p->mono && p->legato && v->active))
    {
	if (p->play_mode & PATCH_PLAY_REVERSE)
	{
	    v->posi = p->sample_stop;
	    v->posf = 0;
	    v->dir = -1;
	}
	else
	{
	    v->posi = p->sample_start;
	    v->posf = 0;
	    v->dir = 1;
	}
    }

    prepare_pitch(p, v, note);

    /* setup envelopes */
    adsr_set_params (&v->vol, &p->vol.env);
    adsr_set_params (&v->pan, &p->pan.env);
    adsr_set_params (&v->ffreq, &p->ffreq.env);
    adsr_set_params (&v->freso, &p->freso.env);
    adsr_set_params (&v->pitchenv, &p->pitch.env);
     
    /* trigger envelopes */
    adsr_trigger (&v->vol);
    adsr_trigger (&v->pan);
    adsr_trigger (&v->ffreq);
    adsr_trigger (&v->freso);
    adsr_trigger (&v->pitchenv);

    /* trigger local lfos */
    lfo_trigger (&v->vol_lfo, &p->vol.lfo_params);
    lfo_trigger (&v->pan_lfo, &p->pan.lfo_params);
    lfo_trigger (&v->ffreq_lfo, &p->ffreq.lfo_params);
    lfo_trigger (&v->freso_lfo, &p->freso.lfo_params);
    lfo_trigger (&v->pitch_lfo, &p->pitch.lfo_params);

    /* mark our territory */
    v->active = TRUE;
}

/* a helper routine to determine the pitch-scaled sample values to use for a frame */
inline static void pitchscale (Patch * p, PatchVoice * v, float *l,
			       float *r)
{
    int y0, y1, y2, y3;

    /* determine sample indices */
    if ((y0 = (v->posi - 1) * 2) < 0)
	y0 = 0;
    y1 = v->posi * 2;
    if ((y2 = (v->posi + 1) * 2) >= p->sample->frames * 2)
	y2 = 0;
    if ((y3 = (v->posi + 2) * 2) >= p->sample->frames * 2)
	y3 = 0;

    /* interpolate */
    *l = cerp (p->sample->sp[y0],
	       p->sample->sp[y1],
	       p->sample->sp[y2], p->sample->sp[y3], v->posf >> 24);
    *r = cerp (p->sample->sp[y0 + 1],
	       p->sample->sp[y1 + 1],
	       p->sample->sp[y2 + 1], p->sample->sp[y3 + 1], v->posf >> 24);
}

/* a helper routine to setup panning of two samples in a frame  */
inline static void pan (Patch * p, PatchVoice * v, int index, float *l, float *r)
{
    float pan;

    /* get pan value */
    pan = p->pan.val;

    /* add lfo value */
    if (p->pan.lfo_on != 0)
    {
	if (p->pan.use_global_lfo)
	{
	    pan += p->pan.lfo_tab[index] * p->pan.lfo_amt;
	}
	else
	{
	    pan += lfo_tick (&v->pan_lfo) * p->pan.lfo_amt;
	}
    }

    /* add adsr value */
    if (p->pan.env_on)
    {
	pan += (adsr_tick (&v->pan) * p->pan.env_amt);
    }

    /* scale to velocity */
    pan = lerp (pan, pan * v->vel, p->pan.vel_amt);

    /* determine values after modulation */
    if (pan > 1.0)
	pan = 1.0;
    else if (pan < -1.0)
	pan = -1.0;

    if (pan < 0.0)
    {				/* panned left */
	*l += *r * -pan;
	*r *= 1 + pan;
    }
    else if (pan > 0.0)
    {				/* panned right */
	*r += *l * pan;
	*l *= 1 - pan;
    }
}

/* a helper routine to apply filters to a frame */
inline static void filter (Patch* p, PatchVoice* v, int index,  float* l,
			   float* r)
{
    float ffreq, freso;

    /* get filter cutoff frequency */
    ffreq = p->ffreq.val;

    /* add current lfo value */
    if (p->ffreq.lfo_on)
    {
	if (p->ffreq.use_global_lfo)
	{
	    ffreq += p->ffreq.lfo_tab[index] * p->ffreq.lfo_amt;
	}
	else
	{
	    ffreq += lfo_tick (&v->ffreq_lfo) * p->ffreq.lfo_amt;
	}
    }

    /* add envelope value */
    if (p->ffreq.env_on)
    {
	ffreq += (adsr_tick (&v->ffreq) * p->ffreq.env_amt);
    }

    /* scale to velocity */
    ffreq = lerp (ffreq, ffreq * v->vel, p->ffreq.vel_amt);
     
    /* clip */
    if (ffreq > 1.0)
	ffreq = 1.0;
    else if (ffreq < 0.0)
	ffreq = 0.0;

    /* get filter resonant frequency */
    freso = p->freso.val;

    /* add current lfo value */
    if (p->freso.lfo_on)
    {
	if (p->freso.use_global_lfo)
	{
	    freso += p->freso.lfo_tab[index] * p->freso.lfo_amt;
	}
	else
	{
	    freso += lfo_tick (&v->freso_lfo) * p->freso.lfo_amt;
	}
    }

    
    /* add envelope value */
    if (p->freso.env_on)
    {
	freso += (adsr_tick (&v->freso) * p->freso.env_amt);
    }

    /* scale to velocity */
    freso = lerp (freso, freso * v->vel, p->freso.vel_amt);
     
    /* clip */
    if (freso > 1.0)
	freso = 1.0;
    else if (freso < 0.0)
	freso = 0.0;

    /* left */
    v->fbl = freso * v->fbl + ffreq * (*l - v->fll);
    v->fll += ffreq * v->fbl;
    *l = v->fll;

    /* right */
    v->fbr = freso * v->fbr + ffreq * (*r - v->flr);
    v->flr += ffreq * v->fbr;
    *r = v->flr;
}

/* a helper routine to adjust the volume of a frame */
inline static int gain (Patch* p, PatchVoice* v, int index, float* l, float* r)
{
    float vol;

    /* first, we use our set value as a base */
    vol = p->vol.val;

    /* now we add whatever the lfo value is supposed to be */
    if (p->vol.lfo_on)
    {
	if (p->vol.use_global_lfo)
	{
	    vol += p->vol.lfo_tab[index] * p->vol.lfo_amt;
	}
	else
	{
	    vol += lfo_tick (&v->vol_lfo) * p->vol.lfo_amt;
	}
    }

    /* envelope */
    if (p->vol.env_on)
    {
	/* we want our envelope to "override" the above parameters, so we
	 * apply it after them */
	vol *= adsr_tick (&v->vol);
    }
    else if (v->released)
    {
	vol *= v->declick_vol;
	v->declick_vol -= declick_dec;
    }

    /* velocity should be the last parameter considered because it
     * has the most "importance" */
    vol = lerp (vol, vol * v->vel, p->vol.vel_amt);
    

    /* clip */
    if (vol > 1.0)
	vol = 1.0;
    else if (vol < 0.0)
	vol = 0.0;

    /* adjust volume */
    *l *= vol;
    *r *= vol;


    /* check to see if we've finished a release */
    if (v->released && vol < ALMOST_ZERO)
	return -1;

    return 0;
}

/* a helper routine to advance to the next frame while properly
 * accounting for the different possible play modes (negative value
 * returned if we are out of samples after doing our work) */
inline static int advance (Patch* p, PatchVoice* v, int index)
{
    double pitch;
    double scale;
    guint32 next_posf;
    gboolean recalc = FALSE;	/* whether we need to recalculate our pos/step vars */
     
    /* portamento */
    if (p->porta && v->porta_ticks)
    {
	recalc = TRUE;
	v->pitch += v->pitch_step;
	--(v->porta_ticks);
    }

    /* base pitch value */
    pitch = v->pitch;

    if( p->pitch_bend )
    {
	recalc = TRUE;
	pitch *= p->pitch_bend;
    }
     
    /* pitch lfo value */
    if (p->pitch.lfo_on)
    {
	recalc = TRUE;

	if (p->pitch.use_global_lfo)
	{
	    scale = p->pitch.lfo_tab[index];
	}
	else
	{
	    scale = lfo_tick (&v->pitch_lfo);
	}

	/* we don't multiply against p->pitch.lfo_amount because the
	 * "amount" variable has already been expressed in the
	 * values of lfo_pitch_max and lfo_pitch_min (the same logic
	 * applies when handling the envelopes below) */
	if (scale >= 0.0)
	{
	    pitch *= lerp (1.0, p->lfo_pitch_max, scale);
	}
	else
	{
	    pitch *= lerp (1.0, p->lfo_pitch_min, -scale);
	}
    }

    /* pitch envelope value */
    if (p->pitch.env_on)
    {
	recalc = TRUE;
	scale = adsr_tick (&v->pitchenv);
	if (scale >= 0.0)
	{
	    pitch *= lerp (1.0, p->env_pitch_max, scale);
	}
	else
	{
	    pitch *= lerp (1.0, p->env_pitch_min, -scale);
	}
    }

    /* scale to velocity */
    if (p->pitch.vel_amt > ALMOST_ZERO)
    {
	recalc = TRUE;
	pitch = lerp (pitch, pitch * v->vel, p->pitch.vel_amt);
    }

    if (recalc)
    {
	v->stepi = pitch;
	v->stepf = (pitch - v->stepi) * (0xFFFFFFFFU);
    }
     
    /* advance our position indices */
    if (v->dir > 0)
    {
	next_posf = v->posf + v->stepf;
	if (next_posf < v->posf)	/* unsigned int wraps around */
	    v->posi++;
	v->posf = next_posf;
	v->posi += v->stepi;
    }
    else
    {
	next_posf = v->posf + v->stepf;
	if (next_posf < v->posf)	/* unsigned int wraps around */
	    v->posi--;
	v->posf = next_posf;
	v->posi -= v->stepi;
    }

    /* adjust our indices according to our play mode */
    if (p->play_mode & PATCH_PLAY_LOOP)
    {
	if (p->play_mode & PATCH_PLAY_PINGPONG)
	{
	    if ((v->dir > 0) && (v->posi > p->loop_stop))
	    {
		v->posi = p->loop_stop;
		v->dir = -1;
	    }
	    else if ((v->dir < 0) && (v->posi < p->loop_start))
	    {
		v->posi = p->loop_start;
		v->dir = 1;
	    }
	}
	else
	{
	    if ((v->dir > 0) && (v->posi > p->loop_stop))
	    {
		v->posi = p->loop_start;
	    }
	    else if ((v->dir < 0) && (v->posi < p->loop_start))
	    {
		v->posi = p->loop_stop;
	    }
	}
    }
    else
    {
	if (((v->dir > 0) && (v->posi > p->sample_stop))
	    || ((v->dir < 0) && (v->posi < p->sample_start)))
	{

	    /* we need to let our caller know that they are out of
	     * samples */
	    return -1;
	}
    }

    /* check to see if it's time to release */
    if (v->relset >= 0 && !v->released)
    {
	if (v->relset == 0)
	{
	    adsr_release (&v->vol);
	    adsr_release (&v->pan);
	    adsr_release (&v->ffreq);
	    adsr_release (&v->freso);
	    adsr_release (&v->pitchenv);	       
	    v->released = TRUE;
	}
	else
	{
	    --v->relset;
	}
    }

    /* otherwise, check to see if we should retrigger */
    else if (p->mono && p->legato && v->retrig_ticks >= 0)
    {
	if (v->retrig_ticks == 0)
	{
	    prepare_pitch(p, v, v->retrig_note);
	    v->note = v->retrig_note;
	    v->vel = v->retrig_vel;
	}

	--v->retrig_ticks;
    }

    return 0;
}

inline static void patch_update_lfo_tab (float* tab, LFO* lfo, int nframes)
{
    register int i;

    for (i = 0; i < nframes; i++)
    {
	tab[i] = lfo_tick (lfo);
    }
}

/* a helper rountine to render all active voices of a given patch into buf */
inline static void patch_render_patch (Patch* p, float* buf, int nframes)
{
    register int i;
    register int j;
    PatchVoice* v;
    float l, r;
    gboolean done;

    /* update global LFO tables if applicable */
    if (p->vol.use_global_lfo)
	patch_update_lfo_tab (p->vol.lfo_tab, &p->vol.global_lfo, nframes);

    if (p->pan.use_global_lfo)
	patch_update_lfo_tab (p->pan.lfo_tab, &p->pan.global_lfo, nframes);

    if (p->ffreq.use_global_lfo)
	patch_update_lfo_tab (p->ffreq.lfo_tab, &p->ffreq.global_lfo, nframes);

    if (p->freso.use_global_lfo)
	patch_update_lfo_tab (p->freso.lfo_tab, &p->freso.global_lfo, nframes);

    if (p->pitch.use_global_lfo)
	patch_update_lfo_tab (p->pitch.lfo_tab, &p->pitch.global_lfo, nframes);

    for (i = 0; i < PATCH_VOICE_COUNT; i++)
    {
	if (p->voices[i].active == FALSE)
	    continue;

	/* sanity check */
	if (p->voices[i].posi < p->sample->frames)
	{
	    v = &p->voices[i];
	}
	else
	{
	    p->voices[i].active = FALSE;
	    continue;
	}

	done = FALSE;
	for (j = v->offset; j < nframes && done == FALSE; j++)
	{
	    /* process samples */
	    pitchscale (p, v,    &l, &r);
	    pan        (p, v, j, &l, &r);
	    filter     (p, v, j, &l, &r);

	    /* adjust volume and stop rendering if we finished
	     * a release */
	    if (gain (p, v, j, &l, &r) < 0)
		done = TRUE;

	    buf[j * 2] += l;
	    buf[j * 2 + 1] += r;

	    /* advance our position and stop rendering if we
	     * run out of samples */
	    if (advance (p, v, j) < 0)
		done = TRUE;
	}

	/* check to see if it's time to stop rendering */
	if (done)
	    v->active = FALSE;

	/* decrease the voices offset if applicable */
	if (v->offset > 0)
	{
	    if (v->offset > nframes)
	    {
		v->offset -= nframes;
	    }
	    else
	    {
		v->offset = 0;
	    }
	}

	/* overflows bad, OVERFLOWS BAD! */
	if (v->active && (v->posi < 0 || v->posi >= p->sample->frames))
	{
	    debug ("overflow! NO! BAD CODE! DIE DIE DIE!\n");
	    debug ("v->posi == %d, p->sample.frames == %d\n", v->posi, p->sample->frames);
	    v->active = 0;
	}
    }
}

/* deactivate all active patches matching given criteria */
void patch_release (int chan, int note, int offset)
{
    int i;

    for (i = 0; i < PATCH_COUNT; i++)
    {
	if (patches[i].active
	    && !(patches[i].play_mode & PATCH_PLAY_SINGLESHOT)
	    && patches[i].channel == chan
	    && (patches[i].note == note
		|| (patches[i].range > 0
		    && note >= patches[i].lower_note
		    && note <= patches[i].upper_note)))
	{

	    patch_release_patch (&patches[i], note, offset);
	}
    }

    return;
}

/* deactivate a single patch with a given id */
void patch_release_with_id (int id, int note, int offset)
{
    if (id < 0 || id >= PATCH_COUNT)
	return;
    if (!patches[id].active)
	return;
    if (patches[id].play_mode & PATCH_PLAY_SINGLESHOT)
	return;

    patch_release_patch (&patches[id], note, offset);
    return;
}

/* render nframes of all active patches into buf */
void patch_render (float *buf, int nframes)
{
    int i;

    /* render patches */
    for (i = 0; i < PATCH_COUNT; i++)
    {
	if (patches[i].active)
	{
	    if (patch_trylock (i) != 0)
	    {
		continue;
	    }
	    if (patches[i].sample->sp == NULL)
	    {
		patch_unlock (i);
		continue;
	    }
	    patch_render_patch (&patches[i], buf, nframes);
	    patch_unlock (i);
	}
    }
}

/* triggers all patches matching criteria */
void patch_trigger (int chan, int note, float vel, Tick ticks,
		    int offset)
{
    static int idp[PATCH_COUNT];	/* holds all patches to be activated */
    int i, j;

    /* We gather up all of the patches that need to be activated here
     * so that we can run their cuts and then trigger them without
     * having to find them twice.  We have to make sure that we do
     * cutting before triggering, otherwise patches which listen on
     * the same note and have the same cut/cut_by values will end up
     * stepping over each other before they both are heard.
     */

    for (i = j = 0; i < PATCH_COUNT; i++)
    {
	if (patches[i].active
	    && patches[i].channel == chan
	    && (patches[i].note == note
		|| (patches[i].range
		    && note >= patches[i].lower_note
		    && note <= patches[i].upper_note)))
	{

	    idp[j++] = i;
	}
    }

    /* do cuts */
    for (i = 0; i < j; i++)
	patch_cut_patch (&patches[idp[i]], offset);

    /* do triggers */
    for (i = 0; i < j; i++)
	patch_trigger_patch (&patches[idp[i]], note, vel, ticks, offset);
}

/* activate a single patch with given id */
void patch_trigger_with_id (int id, int note, float vel, Tick ticks,
			    int offset)
{
    if (id < 0 || id >= PATCH_COUNT)
	return;
    if (!patches[id].active)
	return;

    patch_cut_patch (&patches[id], offset);
    patch_trigger_patch (&patches[id], note, vel, ticks, offset);
    return;
}

static void patch_control_patch(Patch* p, ControlParamType param, float value)
{
    switch( param )
    {
    case CONTROL_PARAM_VOLUME:
	p->vol.val = value;
	break;
    case CONTROL_PARAM_PANNING:
	p->pan.val = value;
	break;
    case CONTROL_PARAM_CUTOFF:
	p->ffreq.val = value;
	break;
    case CONTROL_PARAM_RESONANCE:
	p->freso.val = value;
	break;
    case CONTROL_PARAM_PITCH:
	p->pitch_bend = pow(2, value);
	break;
    case CONTROL_PARAM_PORTAMENTO:
	p->porta = (value < 0.5) ? 0 : 1;
	break;
    case CONTROL_PARAM_PORTAMENTO_TIME:
	p->porta_secs = value;
	break;
    }
}

void patch_control(int chan, ControlParamType param, float value)
{
    int i;

    for (i = 0; i < PATCH_COUNT; i++)
    {
	if (patches[i].active && patches[i].channel == chan)
	{
	    patch_control_patch (&patches[i], param, value);
	}
    }

    return;
}

/*****************************************************************************/
/*************************** ENVELOPE SETTERS ********************************/
/*****************************************************************************/

/* sets whether the envelope is used or not */
int patch_set_env_on (int id, PatchParamType param, gboolean state)
{
    PatchParam* p;
    int err;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    p->env_on = state;
    return 0;
}

/* sets how much the envelope affects a param */
int patch_set_env_amount (int id, PatchParamType param, float amt)
{
    PatchParam* p;
    int err;

    if (param == PATCH_PARAM_VOLUME)
	return PATCH_PARAM_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    if (amt < -1.0 || amt > 1.0)
	return PATCH_PARAM_INVALID;

    p->env_amt = amt;

    /* pitch is a special case; we set these helper variables so that
     * we can pitch scale properly in advance() */
    if (param == PATCH_PARAM_PITCH)
    {
	patches[id].env_pitch_max = pow (2, (amt * PATCH_MAX_PITCH_STEPS) / 12.0);
	patches[id].env_pitch_min = pow (2, -(amt * PATCH_MAX_PITCH_STEPS) / 12.0);
    }

    return 0;
}

/* sets the delay length in seconds */
int patch_set_env_delay (int id, PatchParamType param, float secs)
{
    PatchParam* p;
    int err;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    if (secs < 0.0)
	return PATCH_PARAM_INVALID;

    p->env.delay = secs;
    return 0;
}

/* sets the attack length in seconds */
int patch_set_env_attack (int id, PatchParamType param, float secs)
{
    PatchParam* p;
    int err;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    if (secs < 0.0)
	return PATCH_PARAM_INVALID;

    p->env.attack = secs;
    return 0;
}

/* sets the hold length in seconds */
int patch_set_env_hold (int id, PatchParamType param, float secs)
{
    PatchParam* p;
    int err;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    if (secs < 0.0)
	return PATCH_PARAM_INVALID;

    p->env.hold = secs;
    return 0;
}

/* sets the decay length in seconds */
int patch_set_env_decay (int id, PatchParamType param, float secs)
{
    PatchParam* p;
    int err;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    if (secs < 0.0)
	return PATCH_PARAM_INVALID;

    p->env.decay = secs;
    return 0;
}

/* sets the sustain level */
int patch_set_env_sustain (int id, PatchParamType param, float level)
{
    PatchParam* p;
    int err;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    if (level < 0.0 || level > 1.0)
	return PATCH_PARAM_INVALID;

    p->env.sustain = level;
    return 0;
}

/* sets the release length in seconds */
int patch_set_env_release (int id, PatchParamType param, float secs)
{
    PatchParam* p;
    int err;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    if (secs < 0.0)
	return PATCH_PARAM_INVALID;

    if (secs < PATCH_MIN_RELEASE)
	secs = PATCH_MIN_RELEASE;

    p->env.release = secs;
    return 0;
}

/*****************************************************************************/
/*************************** ENVELOPE GETTERS ********************************/
/*****************************************************************************/

/* returns whether the envelope is being used or not */
int patch_get_env_on (int id, PatchParamType param, gboolean* val)
{
    PatchParam* p;
    int err;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->env_on;
    return 0;
}

/* places the amount the envelope is used into val */
int patch_get_env_amount (int id, PatchParamType param, float* val)
{
    PatchParam* p;
    int err;

    if (param == PATCH_PARAM_VOLUME)
	return PATCH_PARAM_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->env_amt;
    return 0;
}

/* places the delay length in seconds into val */
int patch_get_env_delay (int id, PatchParamType param, float* val)
{
    PatchParam* p;
    int err;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->env.delay;
    return 0;
}

/* places the attack length in seconds into val */
int patch_get_env_attack (int id, PatchParamType param, float* val)
{
    PatchParam* p;
    int err;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->env.attack;
    return 0;
}

/* places the hold length in seconds into val */
int patch_get_env_hold (int id, PatchParamType param, float* val)
{
    PatchParam* p;
    int err;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->env.hold;
    return 0;
}

/* places the decay length in seconds into val */
int patch_get_env_decay (int id, PatchParamType param, float* val)
{
    PatchParam* p;
    int err;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->env.decay;
    return 0;
}

/* places the sustain level into val */
int patch_get_env_sustain (int id, PatchParamType param, float* val)
{
    PatchParam* p;
    int err;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->env.sustain;
    return 0;
}

/* places the release length in seconds into val */
int patch_get_env_release (int id, PatchParamType param, float* val)
{
    PatchParam* p;
    int err;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->env.release;

    /* we hide the fact that we have a min release value from the
     * outside world (they're on a need-to-know basis) */
    if (*val <= PATCH_MIN_RELEASE)
	*val = 0;
    return 0;
}

/************************************************************************/
/*************************** LFO SETTERS ********************************/
/************************************************************************/

/* set whether the lfo is used or not */
int patch_set_lfo_on (int id, PatchParamType param, gboolean state)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    p->lfo_on = state;
    return 0;
}

/* set the amount of the lfo to add to param */
int patch_set_lfo_amount (int id, PatchParamType param, float amt)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    if (amt < -1.0 || amt > 1.0)
	return PATCH_PARAM_INVALID;

    p->lfo_amt = amt;

    /* pitch is a special case; we set these helper variables so that
     * we can pitch scale properly in advance() */
    if (param == PATCH_PARAM_PITCH)
    {
	patches[id].lfo_pitch_max = pow (2, (amt * PATCH_MAX_PITCH_STEPS) / 12.0);
	patches[id].lfo_pitch_min = pow (2, -(amt * PATCH_MAX_PITCH_STEPS) / 12.0);
    }
	  
    return 0;
}

/* set the attack time of the param's LFO */
int patch_set_lfo_attack (int id, PatchParamType param, float secs)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    if (secs < 0.0)
	return PATCH_PARAM_INVALID;

    p->lfo_params.attack = secs;
    lfo_trigger (&p->global_lfo, &p->lfo_params);
    return 0;
}

/* set the period length of the param's lfo in beats */
int patch_set_lfo_beats (int id, PatchParamType param, float beats)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    if (beats < 0.0)
	return PATCH_PARAM_INVALID;

    p->lfo_params.sync_beats = beats;
    lfo_trigger (&p->global_lfo, &p->lfo_params);
    return 0;
}

/* set the delay time of the param's LFO */
int patch_set_lfo_delay (int id, PatchParamType param, float secs)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    if (secs < 0.0)
	return PATCH_PARAM_INVALID;

    p->lfo_params.delay = secs;
    lfo_trigger (&p->global_lfo, &p->lfo_params);
    return 0;
}

/* set the frequency of the param's lfo */
int patch_set_lfo_freq (int id, PatchParamType param, float freq)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    if (freq < 0.0)
	return PATCH_PARAM_INVALID;

    p->lfo_params.freq = freq;
    lfo_trigger (&p->global_lfo, &p->lfo_params);
    return 0;
}

/* set whether to use the param's global lfo or not */
int patch_set_lfo_global (int id, PatchParamType param, gboolean state)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    p->use_global_lfo = state;
    return 0;
}

/* set whether to constrain the param's LFOs to positive values or not */
int patch_set_lfo_positive (int id, PatchParamType param, gboolean state)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    p->lfo_params.positive = state;
    lfo_trigger (&p->global_lfo, &p->lfo_params);
    return 0;
}

/* set the param's lfo shape */
int patch_set_lfo_shape (int id, PatchParamType param, LFOShape shape)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    p->lfo_params.shape = shape;
    lfo_trigger (&p->global_lfo, &p->lfo_params);
    return 0;
}

/* set whether to the param's lfo should sync to tempo or not */
int patch_set_lfo_sync (int id, PatchParamType param, gboolean state)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    p->lfo_params.sync = state;
    lfo_trigger (&p->global_lfo, &p->lfo_params);
    return 0;
}

/************************************************************************/
/*************************** LFO GETTERS ********************************/
/************************************************************************/

/* get whether the lfo is used or not */
int patch_get_lfo_on (int id, PatchParamType param, gboolean* val)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->lfo_on;
    return 0;
}

/* get the amount of the lfo that is added to param */
int patch_get_lfo_amount (int id, PatchParamType param, float* val)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->lfo_amt;
    return 0;
}

/* get the attack time of the param's LFO */
int patch_get_lfo_attack (int id, PatchParamType param, float* val)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->lfo_params.attack;
    return 0;
}

/* get the param's lfo period length in beats */
int patch_get_lfo_beats (int id, PatchParamType param, float* val)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->lfo_params.sync_beats;
    return 0;
}

/* get the delay time of the param's LFO */
int patch_get_lfo_delay (int id, PatchParamType param, float* val)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->lfo_params.delay;
    return 0;
}

/* get the param's lfo frequency */
int patch_get_lfo_freq (int id, PatchParamType param, float* val)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->lfo_params.freq;
    return 0;
}

/* get whether the global lfo is used or not */
int patch_get_lfo_global (int id, PatchParamType param, gboolean* val)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->use_global_lfo;
    return 0;
}

/* get whether the param's lfo is constrained to positive values or not */
int patch_get_lfo_positive (int id, PatchParamType param, gboolean* val)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->lfo_params.positive;
    return 0;
}

/* get param's lfo shape */
int patch_get_lfo_shape (int id, PatchParamType param, LFOShape* val)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->lfo_params.shape;
    return 0;
}

/* get whether param's lfo is tempo synced or not */
int patch_get_lfo_sync (int id, PatchParamType param, gboolean* val)
{
    PatchParam* p;
    int err;

    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->lfo_params.sync;
    return 0;
}

/*****************************************************************************/
/*************************** VELOCITY FUNCTIONS ******************************/
/*****************************************************************************/

int patch_set_vel_amount (int id, PatchParamType param, float amt)
{
    PatchParam* p;
    int err;
     
    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    if (amt < 0.0 || amt > 1.0)
	return PATCH_PARAM_INVALID;

    p->vel_amt = amt;
    return 0;
}
     
int patch_get_vel_amount (int id, PatchParamType param, float* val)
{
    PatchParam* p;
    int err;
     
    if (!isok (id))
	return PATCH_ID_INVALID;

    if ((err = patch_get_param (&p, id, param)) < 0)
	return err;

    *val = p->vel_amt;
    return 0;
}

/*****************************************************************************/
/*************************** PARAMETER SETTERS *******************************/
/*****************************************************************************/

/* sets channel patch listens on */
int patch_set_channel (int id, int channel)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    if (channel < 0 || channel > 15)
	return PATCH_CHANNEL_INVALID;

    patches[id].channel = channel;
    return 0;
}

/* sets the cut signal this patch emits when activated */
int patch_set_cut (int id, int cut)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    patches[id].cut = cut;
    return 0;
}

/* sets the cut signal that terminates this patch if active */
int patch_set_cut_by (int id, int cut_by)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    patches[id].cut_by = cut_by;
    return 0;
}

/* sets filter cutoff frequency */
int patch_set_cutoff (int id, float freq)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    if (freq < 0.0 || freq > 1.0)
	return PATCH_PARAM_INVALID;

    patches[id].ffreq.val = freq;
    return 0;
}

/* set whether this patch should be played legato or not */
int patch_set_legato(int id, gboolean val)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    patches[id].legato = val;
    return 0;
}
    

/* sets the start loop point */
int patch_set_loop_start (int id, int start)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    if (patches[id].sample->sp == NULL)
	return 0;

    if (start < patches[id].sample_start)
	start = patches[id].sample_start;
    else if (start > patches[id].sample_stop)
	start = patches[id].sample_stop;

    patches[id].loop_start = start;
    if (start > patches[id].loop_stop)
	patches[id].loop_stop = start;

    return 0;
}

/* sets the stopping loop point */
int patch_set_loop_stop (int id, int stop)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    if (patches[id].sample->sp == NULL)
	return 0;

    if (stop > patches[id].sample_stop)
	stop = patches[id].sample_stop;
    else if (stop < patches[id].sample_start)
	stop = patches[id].sample_start;

    patches[id].loop_stop = stop;
    if (stop < patches[id].loop_start)
	patches[id].loop_start = stop;

    return 0;
}

/* sets the lower note of a patch's range */
int patch_set_lower_note (int id, int note)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    if (note < 0 || note > 127)
	return PATCH_NOTE_INVALID;

    patches[id].lower_note = note;
    return 0;
}

/* set whether the patch is monophonic or not */
int patch_set_monophonic(int id, gboolean val)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    patches[id].mono = val;
    return 0;
}

/* sets the name */
int patch_set_name (int id, const char *name)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    strncpy (patches[id].name, name, PATCH_MAX_NAME);
    return 0;
}

/* sets the root note */
int patch_set_note (int id, int note)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    if (note < 0 || note > 127)
	return PATCH_NOTE_INVALID;

    patches[id].note = note;
    return 0;
}

/* sets the panorama */
int patch_set_panning (int id, float pan)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    if (pan < -1.0 || pan > 1.0)
	return PATCH_PAN_INVALID;

    patches[id].pan.val = pan;
    return 0;
}

/* set the pitch */
int patch_set_pitch (int id, float pitch)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    if (pitch < -1.0 || pitch > 1.0)
	return PATCH_PARAM_INVALID;

    patches[id].pitch.val = pitch;
    return 0;
}

/* set the pitch range */
int patch_set_pitch_steps (int id, int steps)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    if (steps < -PATCH_MAX_PITCH_STEPS
	|| steps > PATCH_MAX_PITCH_STEPS)
	return PATCH_PARAM_INVALID;

    patches[id].pitch_steps = steps;
    return 0;
}

/* sets the play mode */
int patch_set_play_mode (int id, PatchPlayMode mode)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    /* verify direction */
    if (mode & PATCH_PLAY_FORWARD)
    {
	if (mode & PATCH_PLAY_REVERSE)
	{
	    return PATCH_PLAY_MODE_INVALID;
	}
    }
    else if (mode & PATCH_PLAY_REVERSE)
    {
	if (mode & PATCH_PLAY_FORWARD)
	{
	    return PATCH_PLAY_MODE_INVALID;
	}
    }
    else
    {
	return PATCH_PLAY_MODE_INVALID;
    }

    /* verify duration */
    if (mode & PATCH_PLAY_SINGLESHOT)
    {
	if ((mode & PATCH_PLAY_TRIM) || (mode & PATCH_PLAY_LOOP))
	{
	    return PATCH_PLAY_MODE_INVALID;
	}
    }
    else if (mode & PATCH_PLAY_TRIM)
    {
	if ((mode & PATCH_PLAY_SINGLESHOT) || (mode & PATCH_PLAY_LOOP))
	{
	    return PATCH_PLAY_MODE_INVALID;
	}
    }
    else if (mode & PATCH_PLAY_LOOP)
    {
	if ((mode & PATCH_PLAY_SINGLESHOT) || (mode & PATCH_PLAY_TRIM))
	{
	    return PATCH_PLAY_MODE_INVALID;
	}
    }

    /* make sure pingpong isn't frivolously set (just for style
     * points) */
    if ((mode & PATCH_PLAY_PINGPONG) && !(mode && PATCH_PLAY_LOOP))
    {
	return PATCH_PLAY_MODE_INVALID;
    }

    patches[id].play_mode = mode;
    return 0;
}

/* set whether portamento is being used or not */
int patch_set_portamento (int id, gboolean val)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    patches[id].porta = val;
    return 0;
}

/* set length of portamento slides in seconds */
int patch_set_portamento_time (int id, float secs)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    if (secs < 0.0)
	return PATCH_PARAM_INVALID;

    patches[id].porta_secs = secs;
    return 0;
}

/* set patch to listen to a range of notes if non-zero */
int patch_set_range (int id, int range)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
     
    patches[id].range = range;
    return 0;
}

/* set the filter's resonance */
int patch_set_resonance (int id, float reso)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    if (reso < 0.0 || reso > 1.0)
	return PATCH_PARAM_INVALID;

    patches[id].freso.val = reso;
    return 0;
}

/* set the point within the sample to begin playing */
int patch_set_sample_start (int id, int start)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    if (patches[id].sample->sp == NULL)
	return 0;

    if (start < 0)
    {
	debug ("refusing to set negative sample start point\n");
	return PATCH_PARAM_INVALID;
    }

    if (start > patches[id].sample_stop)
    {
	debug ("refusing to set incongruous sample start point\n");
	return PATCH_PARAM_INVALID;
    }

    patches[id].sample_start = start;
    return 0;
}

/* set the point within the sample to stop playing */
int patch_set_sample_stop (int id, int stop)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    if (patches[id].sample->sp == NULL)
	return 0;

    if (stop >= patches[id].sample->frames)
    {
	debug ("refusing to set sample stop greater than sample frame count\n");
	return PATCH_PARAM_INVALID;
    }

    if (stop < patches[id].sample_start)
    {
	debug ("refusing to set incongruous sample stop point\n");
	return PATCH_PARAM_INVALID;
    }

    patches[id].sample_stop = stop;
    return 0;
}

/* set the upper note of a patch's range */
int patch_set_upper_note (int id, int note)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    if (note < 0 || note > 127)
	return PATCH_NOTE_INVALID;

    patches[id].upper_note = note;
    return 0;
}

/* set the volume */
int patch_set_volume (int id, float vol)
{

    if (!isok (id))
	return PATCH_ID_INVALID;

    if (vol < 0 || vol > 1.0)
	return PATCH_VOL_INVALID;

    patches[id].vol.val = vol;
    return 0;
}

/*****************************************************************************/
/*************************** PARAMETER GETTERS********************************/
/*****************************************************************************/

/* get the channel the patch listens on */
int patch_get_channel (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].channel;
}

/* get the cut signal */
int patch_get_cut (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].cut;
}

/* get the cut-by signal */
int patch_get_cut_by (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].cut_by;
}

/* get the filter cutoff value */
float patch_get_cutoff (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].ffreq.val;
}

/* get the display index */
int patch_get_display_index (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].display_index;
}

/* get the number of frame in the sample */
int patch_get_frames (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    if (patches[id].sample->sp == NULL)
	return 0;

    return patches[id].sample->frames;
}

/* get whether this patch is played legato or not */
gboolean patch_get_legato(int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].legato;
}   

/* get the starting loop point */
int patch_get_loop_start (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].loop_start;
}

/* get the stopping loop point */
int patch_get_loop_stop (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].loop_stop;
}

/* get the lower note */
int patch_get_lower_note (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].lower_note;
}

/* get whether this patch is monophonic or not */
gboolean patch_get_monophonic(int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].mono;
}

/* get the name */
char *patch_get_name (int id)
{
    char *name;

    if (id < 0 || id >= PATCH_COUNT || !patches[id].active)
	name = strdup ("\0");
    else
	name = strdup (patches[id].name);

    return name;
}

/* get the root note */
int patch_get_note (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].note;
}

/* get the panorama */
float patch_get_panning (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].pan.val;
}

/* get the pitch */
float patch_get_pitch (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    return patches[id].pitch.val;
}

/* get the pitch range */
int patch_get_pitch_steps (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    return patches[id].pitch_steps;
}

/* get the play mode */
PatchPlayMode patch_get_play_mode (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].play_mode;
}

/* get whether portamento is used or not */
gboolean patch_get_portamento (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    return patches[id].porta;
}

/* get length of portamento slides in seconds */
float patch_get_portamento_time (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;

    return patches[id].porta_secs;
}

/* get whether a range of notes is used or not */
int patch_get_range (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].range;
}

/* get the filter's resonance amount */
float patch_get_resonance (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].freso.val;
}

/* get a pointer to the sample data */
const float *patch_get_sample (int id)
{
    if (id < 0 || id >= PATCH_COUNT || !patches[id].active)
	return NULL;

    return patches[id].sample->sp;
}

/* get the name of the sample file */
char *patch_get_sample_name (int id)
{
    char *name;

    if (id < 0 || id >= PATCH_COUNT || !patches[id].active)
	name = strdup ("\0");
    else
	name = strdup (sample_get_file (patches[id].sample));
    return name;
}

/* get the starting playback point */
int patch_get_sample_start (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].sample_start;
}

/* get the stopping playback point */
int patch_get_sample_stop (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].sample_stop;
}

/* get the upper note */
int patch_get_upper_note (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].upper_note;
}

/* get the volume */
float patch_get_volume (int id)
{
    if (!isok (id))
	return PATCH_ID_INVALID;
    return patches[id].vol.val;
}
