#!/usr/local/bin/perl
#
# This object represents an acknowledgment.  When a problem happens, you can
# associate one of these objects with the host to let operators or other staff
# know that you are aware of the problem, and what you are doing about it.
#
#   host          - string FQDN of the machine its associated with
#   start         - time() that the ack was created
#   end           - time() that the ack will be over
#   contact       - arbitrary contact info
#   services      - arbitrary service names (sep by commas is standard)
#   message       - arbitrary message
#
# + new()         - constructor (sets instance vars to arguments passed in)
# + gets/sets()   - magical set/get functions (autoloaded based on func name)
# + display()     - output format and view
#
# History:
# (1) Cleaned up (Ed July 30, 1997);

package Spong::Ack;

# Constructor.  This expects all of its instance vars to be passed in when the
# object is created.  See above for specific information on the variables.
# This object is dependent on the AckList object for loading the information
# about it from the database.

sub new {
   my( $class, $host, $start, $end, $contact, $services, $message ) = @_;
   my $self = {};

   $self->{'host'}     = $host;
   $self->{'start'}    = $start;
   $self->{'end'}      = $end;
   $self->{'contact'}  = $contact;
   $self->{'services'} = $services;
   $self->{'message'}  = $message;

   bless $self;
   return $self;
}


# Get/Set methods, nothing fancy here...

sub host { my $var = 'host';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub start { my $var = 'start';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub end { my $var = 'end';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub contact { my $var = 'contact';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub services { my $var = 'services';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub message { my $var = 'message';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }


# Display summary.  Does both text and html, doesn't make any calls to 
# sub-objects or other helper objects, just spits out the data that it has.
#
# brief:      Just shows name of services and time ack is over.
# standard:   Shows detailed record of service information (contact, message)
# full:       (same as standard)

sub display {
   my( $self, $type, $view ) = @_;

   if( $type eq "text" ) { return $self->display_text( $view ); }
   if( $type eq "html" ) { return $self->display_html( $view ); }
}

sub display_text {
   my( $self, $format ) = @_;
   
   if( $format eq "brief" ) {
      print $self->host(), "-", $self->services(), "-", $self->end();
   } elsif( $format eq "standard" ) {
      my( $d1, $m1, $y1 ) = (localtime( $self->end()))[3,4,5];
      my( $d2, $m2, $y2 ) = (localtime())[3,4,5];
      my $shost = $main::HOSTS{$self->host()}->{'display_name'};
      my $date;
      my $message = $self->message();

      $shost = (split( /\./, $self->host() ))[0] unless $shost;
      print substr( $shost, 0, 12 ), " "x(14-length(substr($shost, 0, 12)));

      print substr( $self->services(), 0, 7 );
      print " "x(9-length(substr( $self->services(), 0, 7 )));

      if( $d1 == $d2 && $m1 == $m2 && $y1 == $y2 ) {
	 $date = POSIX::strftime( $main::TIMEFMTNOSEC, localtime($self->end()) );
      } else {
	 $date = POSIX::strftime( $main::DATEFMT, localtime($self->end()) );
      }
      print $date, " "x(12-length($date));

      $message =~ s/[\n]*$//;
      print substr( $message, 0, 45 );
   } elsif( $format eq "full" ) {
      $message = $self->message();

      print "Service(s): ", $self->services(), "\n";
      print "Ack. Until: ", scalar localtime( $self->end() ), "\n";
      print "Ack. By:    ", $self->contact(), "\n";

      if( $message ne "" ) {
	 $message =~ s/[\n]*$/\n/; print "Message:    $message"; }
   } elsif( $format eq "special" ) {
      print $self->host(), ":", $self->services(), ":", $self->end();
   }
}

sub display_html {
   my( $self, $format ) = @_;
   
   if( $format eq "brief" ) {
      print "<table border=0 cellspacing=1 cellpadding=1>\n";
      print "<tr><td align=center valign=top>";
      print $self->services();
      print "</td><td align=left valign=top>";
      print scalar localtime( $self->end() );
      print "</td></tr>\n";
      print "</table>\n";

   } elsif( $format eq "standard" || $format eq "full" ) {
      my $message = $self->message();
      my $id = $self->host() . "-" . $self->services() . "-" . $self->end();

      print "<table width=100% border=0 cellspacing=1 cellpadding=1><tr>\n";
      print "<td colspan=2>";
      print "<a href=\"" . $main::WWWACK . "/delete/$id\">Delete</a></td>";
      print "</tr><td nowrap width=15% align=left>Service(s): </td>\n"; 
      print "<td width=85% align=left>", $self->services(), "</td></tr>\n";

      print "<tr><td nowrap width=15% align=left>Ack. Until: </td>\n";
      print "<td width=85% align=left>", scalar localtime( $self->end() );
      print "</td></tr>\n";

      print "<tr><td nowrap width=15% align=left>Ack. By: </td>\n";
      print "<td width=85% align=left>", $self->contact(), "</td></tr>\n";

      if( $message ne "" ) {
	 print "<tr><td nowrap width=15% align=left>Message: </td>\n";
	 print "<td width=85% align=left>$message</td></tr>\n";
      }
      print "</table>\n";
   }
}

1;
