#!/usr/local/bin/perl
#
# This object represents a list of history events that happen on a given host
# or a set of hosts.  HistoryLists are a little unique in that they don't have
# to be associated with one specific host, they can contain events from a
# number of hosts, and they can be populated from an empty container.
#
#   host          - string FQDN of the machine its associated with
#   events        - list of history objects.
#
# + new()         - constructor (sets instance vars to arguments passed in)
# + gets/sets()   - magical set/get functions (autoloaded based on func name)
# + display()     - output format and view
#
# + add_history() - used to add history objects to the list
#
# History:
# (1) Cleaned up (Ed July 31, 1997);

use Spong::History;

package Spong::HistoryList;

# Constructor.  A simple constructor that will load all the history events
# associated with a particular host (if a hostname is passed in), otherwise
# it just sets up an empty object that expects to be filled by someone else.
# If there are no History events associated with a host, then undef is returned

sub new {
   my( $class, $host ) = @_;
   my $self = {};
   my( @list );

   # If we are given a host name, then look up the events for that host, 
   # otherwise, just return a shell that will be filled later.

   if( $host ) {
      return undef if ! -d "$main::SPONGDB/$host/history";

      # Go through the current history, and build history objects for each
      # item in the history.

      open( FILE, "$main::SPONGDB/$host/history/current" );
      while( <FILE> ) { 
	 chomp;
	 if( /^status (\d+) (\S+) (green|yellow|red|purple|clear) (.*)$/ ) {
	    push( @list, 
		  Spong::History->new( $host, "status", $1, $2, $3, $4 ) );
	 } elsif( /^ack (\d+) (\S+) (.*)$/ ) {
	    push( @list, 
		  Spong::History->new( $host, "ack", $1, $2, "blue", $3 ));
	 } elsif( /^event (\d+) (\S+) (\S+) (.*)$/ ) {
	    push( @list, 
                  Spong::History->new( $host, "event", $1, $2, $3, $4 ) );
         }
      }
      close( FILE );
   }

   $self->{'host'}   = $host;
   $self->{'events'} = \@list;

   bless $self;
   return $self;
}


# Get/Set methods, events() is a little fancy in that it return a list rather
# then just the list reference (easier for others to deal with).

sub host { my $var = 'host';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub events { my $var = 'events';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return @{$_[0]->{$var}}; }


# Used to add history events to the list.  Typically used in the case where
# you want to show history events on a variety of hosts and you start with
# a blank list.

sub add_history {
   my( $self, $add ) = @_;

   if( ref( $add ) eq "Spong::HistoryList" ) {
      foreach( $add->events() ) { 
	 push( @{$self->{'events'}}, $_ );
      }
   } elsif( ref( $add ) eq "Spong::History" ) {
      push( @{$self->{'events'}}, $add );
   }
}


# Display summary.  Does both text and html, does rely on the History object
# to display itself, but this adds to it by showing header information in
# tables, and breaking output up by days.
#
# brief:      Shows events that have happened in the last 24 hours
# standard:   Shows events that have happened in the last 7 days
# full:       Shows events that have happened in the last 30 days

sub display {
   my( $self, $type, $view ) = @_;

   if( $type eq "text" ) { return $self->display_text( $view ); }
   if( $type eq "html" ) { return $self->display_html( $view ); }
}

sub display_text {
   my( $self, $format ) = @_;
   my( $event, $date );
   my( $day ) = 24*60*60;

   sub bytime { return $b->time() <=> $a->time(); }

   foreach $event ( sort bytime $self->events() ) {
      if( ($format eq "brief" && $event->time() > (time() - $day)) ||
	  ($format eq "standard" && $event->time() > (time() - 7*$day)) ||
	  ($format eq "full" && $event->time() > (time() - 31*$day)) ) {

	 my $dstr = POSIX::strftime( "%A, $main::DATEFMT", localtime($event->time()) );
	 if( $dstr ne $date ) { 
	    if( $date ne "" ) { print "\n"; }
	    print "$dstr\n", "-"x78, "\n";
	    $date = $dstr; 
	 }
	 
	 $event->display_text();
      }
   }
}

sub display_html {
   my( $self, $format ) = @_;
   my( $event, $date );
   my( $day ) = 24*60*60;

   sub bytime { return $b->time() <=> $a->time(); }

   foreach $event ( sort bytime $self->events() ) {
      if( ($format eq "brief" && $event->time() > (time() - $day)) ||
	  ($format eq "standard" && $event->time() > (time() - 7*$day)) ||
	  ($format eq "full" && $event->time() > (time() - 31*$day)) ) {

	 my $dstr = POSIX::strftime( "%A, $main::DATEFMT", localtime($event->time()) );
	 if( $dstr ne $date ) { 
	    if( $date ne "" ) { print "</table><br>"; }
	    print "$dstr\n<hr border=1 noshade>\n";
	    print "<table width=100% border=0 cellspacing=1 cellpadding=0>\n";
	    $date = $dstr; 
	 }

	 $event->display_html();
      }
   }
   print "</table>\n";
}

1;
