/* StarDict - A international dictionary for GNOME.
 * http://stardict.sourceforge.net
 * Copyright (C) 2003-2003 HuZheng <huzheng_001@163.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Modified by the StarDict Team and others 2003-2003.  See the AUTHORS
 * file for a list of people on the StarDict Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * StarDict at http://stardict.sourceforge.net.
 */


#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <string.h>
#include <stdlib.h>

#include "stardict.h"
#ifndef _WIN32
#  include "stardict-application-server.h"
#  include "GNOME_Stardict.h"
#endif
#include "distance.h"
#include "splash.h"

#ifndef _WIN32
#  include <libgnome/libgnome.h>
#  include <libgnomeui/libgnomeui.h>
#endif

#ifdef _WIN32
	#include "win32/intl.h"
	#include <gdk/gdkwin32.h>
	#include <windows.h>
	#include <gdk/gdkkeysyms.h>
	HINSTANCE stardictexe_hInstance;
	gchar stardict_data_dir[256];
#endif


AppFrame * gpAppFrame;

static gboolean hide_option = FALSE;

#ifndef _WIN32
static gint debug = 0;
static gboolean quit_option = FALSE;

static const struct poptOption options [] =
{
	{ "debug", 'g', POPT_ARG_NONE, &debug, 0,
	  N_("Turn on all debugging messages"), NULL },

	{ "hide", 'h', POPT_ARG_NONE, &hide_option, 1,
	  N_("Hide the main window"), NULL },

	{ "quit", 'q', POPT_ARG_NONE, &quit_option, 1,
	  N_("Quit an existing instance of stardict"), NULL },

	{NULL, '\0', 0, NULL, 0}
};
#endif

/********************************************************************/
AppCore::AppCore(AppFrame *pAppFrame)
{
	poAppFrame = pAppFrame;
	iMaxFuzzyDistance  = MAX_FUZZY_DISTANCE; //need to read from cfg.
	window = NULL; //need by save_yourself_cb().
	prefs_dlg = NULL;
	dict_manage_dlg = NULL;
}

AppCore::~AppCore()
{
	if (iCurrentIndex)
		g_free(iCurrentIndex);
	if (prefs_dlg)
		delete prefs_dlg;
	if (dict_manage_dlg)
		delete dict_manage_dlg;
}

void AppCore::Create(gchar *queryword)
{		
	oLibs.Load();
	iCurrentIndex = (glong*)g_malloc0(sizeof(glong) * oLibs.total_libs());

	gboolean use_custom_font;
#ifdef _WIN32
	rw_cfg_read_boolean (usercfgfile, "preferences/dictionary", "use_custom_font", &use_custom_font);	
#else
	gpAppFrame->oAppConf.read_bool("/apps/stardict/preferences/dictionary/use_custom_font", &use_custom_font, false);
#endif
	if (use_custom_font)
	{
		gchar *custom_font;
#ifdef _WIN32
		rw_cfg_read_string (usercfgfile, "preferences/dictionary", "custom_font", &custom_font);
#else
		gpAppFrame->oAppConf.read_string("/apps/stardict/preferences/dictionary/custom_font", &custom_font);
#endif
		if (custom_font && custom_font[0])
		{
			gchar *aa;
			aa = g_strdup_printf("style \"custom-font\" { font_name= \"%s\" }\nclass \"GtkWidget\" style \"custom-font\"\n", custom_font);
			gtk_rc_parse_string(aa);
			g_free(aa);
		}
		g_free(custom_font);
	}

	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_container_set_border_width(GTK_CONTAINER(window),2);
	gboolean maximized;
#ifdef _WIN32
	rw_cfg_read_boolean (usercfgfile, "preferences/main_window", "maximized", &maximized);
#else
	gpAppFrame->oAppConf.read_bool("/apps/stardict/preferences/main_window/maximized", &maximized, false);
#endif
	gint width,height;
#ifdef _WIN32
	rw_cfg_read_int (usercfgfile, "preferences/main_window", "window_width", &width);
	rw_cfg_read_int (usercfgfile, "preferences/main_window", "window_height", &height);
#else
	gpAppFrame->oAppConf.read_int("/apps/stardict/preferences/main_window/window_width", &width, DEFAULT_WINDOW_WIDTH);
	gpAppFrame->oAppConf.read_int("/apps/stardict/preferences/main_window/window_height", &height, DEFAULT_WINDOW_HEIGHT);
#endif
	if (width < MIN_WINDOW_WIDTH)
		width = MIN_WINDOW_WIDTH;
	if (height < MIN_WINDOW_HEIGHT)
		height = MIN_WINDOW_HEIGHT;
	gtk_window_set_default_size (GTK_WINDOW(window), width, height);
	if (maximized)
		gtk_window_maximize(GTK_WINDOW(window));
	gtk_window_set_title (GTK_WINDOW (window), _("StarDict"));
	gtk_window_set_icon(GTK_WINDOW (window), gpAppFrame->oAppSkin.stardict.icon.p[0]);
	gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);
	g_signal_connect (G_OBJECT (window), "delete_event", G_CALLBACK (on_delete_event), this);
	g_signal_connect (G_OBJECT (window), "window_state_event", G_CALLBACK (on_window_state_event), this);
	g_signal_connect (G_OBJECT (window), "key_press_event", G_CALLBACK (vKeyPressReleaseCallback), this);
	g_signal_connect (G_OBJECT (window), "key_release_event", G_CALLBACK (vKeyPressReleaseCallback), this);

	tooltips = gtk_tooltips_new ();

	GtkWidget *vbox = gtk_vbox_new(false,0);
	gtk_widget_show(vbox);
	gtk_container_add(GTK_CONTAINER(window),vbox);

	oTopWin.Create(vbox);
	oMidWin.Create(vbox);
	oBottomWin.Create(vbox);
	
	oFloatWin.Create();	
#ifdef _WIN32
	oDockLet.init();
#else
	oDockLet.Create();	
#endif
	oSelection.Start();
#ifdef _WIN32
	oClipboard.Start();
#endif

#ifndef _WIN32
	poAppFrame->oAppConf.EnableNotify();    
#endif

	gboolean hide;
#ifdef _WIN32
	rw_cfg_read_boolean (usercfgfile, "preferences/main_window", "hide_on_startup", &hide);
#else
	gpAppFrame->oAppConf.read_bool("/apps/stardict/preferences/main_window/hide_on_startup", &hide, false);
#endif
	
	//NOTICE: when docklet embedded failed,it should always show the window,but,how to detect the failure?
	// As stardict is FOR GNOME,so i don't want to consider the case that haven't the Notification area applet.
	if ((!hide_option) && (queryword || (!hide))) {
		gtk_widget_show(window);	
	}
	else {
		gdk_notify_startup_complete ();
		if (gpAppFrame->oAppCore.oSelection.bEnable)
			gpAppFrame->oAppCore.oDockLet.SetIcon(DOCKLET_SCAN_ICON);
		else
			gpAppFrame->oAppCore.oDockLet.SetIcon(DOCKLET_STOP_ICON);
	}
		
	if (oLibs.total_libs()) {
		if (queryword) {
			Query(queryword);
			g_free(queryword);
			//don't set queryword to NULL here,need by DockLet::EmbeddedCallback().
		}
		else {
			oMidWin.oTextWin.ShowTips();
		}
	}
	else {
		oMidWin.oTextWin.ShowInitFailed();
	}
}

gboolean AppCore::on_delete_event(GtkWidget * window, GdkEvent *event , AppCore *oAppCore)
{
#ifdef _WIN32
	oAppCore->oDockLet.stardict_systray_minimize(oAppCore->window);
	gtk_widget_hide(oAppCore->window);
#else
	if (oAppCore->oDockLet.embedded)
		gtk_widget_hide(oAppCore->window);
	else
		gpAppFrame->Quit();
#endif
	return true;
}

gboolean AppCore::on_window_state_event(GtkWidget * window, GdkEventWindowState *event , AppCore *oAppCore)
{
	if (event->changed_mask == GDK_WINDOW_STATE_WITHDRAWN) {
		if (event->new_window_state & GDK_WINDOW_STATE_WITHDRAWN) {
			if (gpAppFrame->oAppCore.oSelection.bEnable) {
				gpAppFrame->oAppCore.oDockLet.SetIcon(DOCKLET_SCAN_ICON);
			}
			else {
				gpAppFrame->oAppCore.oDockLet.SetIcon(DOCKLET_STOP_ICON);
			}
		}
		else {
			gpAppFrame->oAppCore.oDockLet.SetIcon(DOCKLET_NORMAL_ICON);
			//gtk_widget_grab_focus(oAppCore->oMidWin.oTextWin.textview); //it is better do it manually when click docklet icon or when main windows pop up the focus is not in the combo.
		}	
	}
	else if (event->changed_mask == GDK_WINDOW_STATE_ICONIFIED) {
		if (event->new_window_state & GDK_WINDOW_STATE_ICONIFIED) {
		}
		else {			
			if (gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry))[0]) {
				gtk_widget_grab_focus(oAppCore->oMidWin.oTextWin.textview); //this is better than the next two line because it don't change selection.
				//gtk_widget_grab_focus(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry);
				//gtk_editable_select_region(GTK_EDITABLE(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry), 0, -1);
			}
			else {
				gtk_widget_grab_focus(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry);
			}
		}
	}
	else if (event->changed_mask == GDK_WINDOW_STATE_MAXIMIZED) {
#ifdef _WIN32
		rw_cfg_write_boolean (usercfgfile, "preferences/main_window", "maximized", (event->new_window_state & GDK_WINDOW_STATE_MAXIMIZED));
#else
		gpAppFrame->oAppConf.write_bool("/apps/stardict/preferences/main_window/maximized", (event->new_window_state & GDK_WINDOW_STATE_MAXIMIZED));
#endif
	}
	return false;
}

gboolean AppCore::vKeyPressReleaseCallback(GtkWidget * window, GdkEventKey *event , AppCore *oAppCore)
{
	gboolean return_val=true;  //if return TRUE,the widget which in the main window will not receive any keyboard event.

	gboolean only_ctrl_pressed = ((event->state & GDK_CONTROL_MASK)&&(!(event->state & GDK_MOD1_MASK))&&(!(event->state & GDK_SHIFT_MASK)));
	gboolean only_mod1_pressed = ((event->state & GDK_MOD1_MASK)&&(!(event->state & GDK_CONTROL_MASK))&&(!(event->state & GDK_SHIFT_MASK)));
	if ((event->keyval==GDK_q || event->keyval==GDK_Q) && only_ctrl_pressed) {
		if (event->type==GDK_KEY_PRESS)
			gpAppFrame->Quit();
	}
	else if ((event->keyval==GDK_x || event->keyval==GDK_X) && only_mod1_pressed) {
		if (event->type==GDK_KEY_PRESS) {
#ifdef _WIN32
			oAppCore->oDockLet.stardict_systray_minimize(oAppCore->window);
			gtk_widget_hide(window);
#else
			if (oAppCore->oDockLet.embedded)
				gtk_widget_hide(window);
			else
				gpAppFrame->Quit();
#endif
		}
	}
	else if ((event->keyval==GDK_z || event->keyval==GDK_Z) && only_mod1_pressed) {
		if (event->type==GDK_KEY_PRESS) {
			gtk_window_iconify(GTK_WINDOW(window));
		}
	}
	else if (event->keyval==GDK_F1) {
		if (event->type==GDK_KEY_PRESS) {
#ifdef _WIN32
			gchar *filename = g_strdup_printf(_("file:///%s/help/C/stardict.html"), stardict_data_dir);
			ShellExecute((HWND)(GDK_WINDOW_HWND(gpAppFrame->oAppCore.window->window)), "OPEN", filename, NULL, NULL, SW_SHOWNORMAL);
			g_free(filename);
#else
			gnome_help_display ("stardict.xml", NULL, NULL);
#endif
		}
	}
	else if ((event->keyval==GDK_f || event->keyval==GDK_F) && only_ctrl_pressed) {
		if (event->type==GDK_KEY_PRESS)
			oAppCore->oMidWin.oToolWin.do_search();
	}
	else if ((event->keyval==GDK_Left) && only_mod1_pressed) {
		if (event->type==GDK_KEY_PRESS)
			oAppCore->oTopWin.do_back();
	}
	else if ((event->keyval==GDK_Up) && only_mod1_pressed) {
		if (event->type==GDK_KEY_PRESS)
			oAppCore->oTopWin.do_previous();
	}
	else if ((event->keyval==GDK_Down) && only_mod1_pressed) {
		if (event->type==GDK_KEY_PRESS)
			oAppCore->oTopWin.do_next();
	}	
	else if ((event->keyval==GDK_m || event->keyval==GDK_M) && only_mod1_pressed) {
		if (event->type==GDK_KEY_PRESS)
			oAppCore->oTopWin.do_menu();
	}
	else if ((event->type==GDK_KEY_PRESS)&&(event->keyval >= 0x21 && event->keyval <= 0x7E)&&(!(event->state & GDK_CONTROL_MASK))&&(!(event->state & GDK_MOD1_MASK))&&(!GTK_WIDGET_HAS_FOCUS(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry))) {
		oAppCore->oTopWin.InsertHisList(oAppCore->oTopWin.GetText());
		oAppCore->oTopWin.InsertBackList();
		gchar str[2];
		str[0]= event->keyval;
		str[1]= '\0';		
		gtk_widget_grab_focus(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry);
		oAppCore->oTopWin.SetText(str);
		gtk_editable_set_position(GTK_EDITABLE(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry), 1);
	}
	else if ((event->type==GDK_KEY_PRESS)&&(event->keyval == GDK_Return)&&(!GTK_WIDGET_HAS_FOCUS(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry))) {
		if (GTK_WIDGET_HAS_FOCUS(oAppCore->oMidWin.oIndexWin.oListWin.treeview)) {
			GtkTreeModel *model;
			GtkTreeIter iter;
		
			GtkTreeSelection *selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (oAppCore->oMidWin.oIndexWin.oListWin.treeview));
			if (gtk_tree_selection_get_selected (selection, &model, &iter))
			{
				gchar *word;
				gtk_tree_model_get (model, &iter, 0, &word, -1);
				gpAppFrame->oAppCore.ListClick(word);
				g_free(word);
			}
		}
		else {
			gtk_widget_grab_focus(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry);
			oAppCore->TopWinEnterWord(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry)));
		}
	}
	else if ((event->type==GDK_KEY_PRESS)&&(event->keyval == 0x20)&&(!GTK_WIDGET_HAS_FOCUS(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry))) {
		gtk_widget_grab_focus(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry);
	}
	else {
		switch (event->keyval)
		{
			case GDK_Escape:
				if (event->type==GDK_KEY_PRESS) {
					if (gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry))[0]) {
						oAppCore->oTopWin.InsertHisList(oAppCore->oTopWin.GetText());
						oAppCore->oTopWin.InsertBackList();
						oAppCore->oTopWin.SetText("");
						gtk_widget_grab_focus(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry);
					}
					else {
						if (GTK_WIDGET_HAS_FOCUS(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry)) {
							//gtk_window_iconify(GTK_WINDOW(window));
						}
						else {
							gtk_widget_grab_focus(GTK_COMBO(oAppCore->oTopWin.WordCombo)->entry);
						}
					}
				}
				break;
			default:
				return_val=false;
				break;
		}
	}
	return return_val;
}

gboolean AppCore::SimpleLookupToFloat(const char* sWord,gboolean bShowIfNotFound)
{
    if ( sWord==NULL || sWord[0]=='\0')
        return true;
    char *SearchWord = g_strdup(sWord);
    char * EndPointer,*P1,*P2;
    P1=(char *)sWord;
    P2=SearchWord;
    // delete chinese space at the begining
    while( *P1 && g_unichar_isspace(g_utf8_get_char(P1)))
        P1 = g_utf8_next_char(P1);
    //format word, delete any spilth blanks.    
    while(*P1)
    {
        if (g_unichar_isspace(g_utf8_get_char(P1)))
        {
            *P2++=' ';
            P1 = g_utf8_next_char(P1);
            while(g_unichar_isspace(g_utf8_get_char(P1)))
                P1 = g_utf8_next_char(P1);
        }
        else
		{
			g_utf8_strncpy(P2,P1,1);
			P1 = g_utf8_next_char(P1);
			P2 = g_utf8_next_char(P2);
		}
    }
    *P2='\0';
    EndPointer=SearchWord+strlen(SearchWord);

	gchar **ppWord = (gchar **)g_malloc(sizeof(gchar *) * oLibs.total_libs());
	gchar **ppWordData = (gchar **)g_malloc(sizeof(gchar *) * oLibs.total_libs());

    //find the word use most biggest length
    while (EndPointer>SearchWord)
    {
        // delete end spaces
        while ( EndPointer>SearchWord && *EndPointer==' ' )
            *EndPointer--='\0';
        
		glong iIndex;		

        gboolean bFound = false;
        for (int iLib=0;iLib<oLibs.total_libs();iLib++)
        {
            if (oLibs.SimpleLookupWord(SearchWord,iIndex,iLib))
            {
				ppWord[iLib] = oLibs.poGetWord(iIndex,iLib);
                ppWordData[iLib] = oLibs.poGetWordData(iIndex,iLib);
                bFound = true;
            }
            else {
				ppWord[iLib] = NULL;
                ppWordData[iLib] = NULL;
			}
        }
        if (bFound==true)
        {
            ShowDataToFloatWin(ppWord, ppWordData,SearchWord);
            oTopWin.InsertHisList(SearchWord);
			g_free(ppWord);
			g_free(ppWordData);
			g_free(SearchWord);
            return true;
        }
        // delete last word
        if (bIsPureEnglish(SearchWord))
        {
            while ( EndPointer>=SearchWord && *EndPointer!=' ' )
                EndPointer--;
            if (EndPointer>=SearchWord)
                *EndPointer='\0';
        }
        else // delete one character per time
		{
			EndPointer = g_utf8_find_prev_char(SearchWord,EndPointer);
			if (EndPointer)
            	*EndPointer='\0';
			else
				EndPointer = SearchWord-1; // so < SearchWord
		}
    }
	g_free(ppWord);
	g_free(ppWordData);

    // not found
    if (bShowIfNotFound)
    {
        ShowNotFoundToFloatWin((SearchWord[0])?SearchWord:sWord,_("<Not Found!>"), false);
        oTopWin.InsertHisList(sWord); //really need?
    }
	g_free(SearchWord);
    return false;
}

gboolean AppCore::SimpleLookupToTextWin(const char* sWord,glong* piIndex,gboolean piIndexValid, gboolean bTryMoreIfNotFound)
{
	//the input can be:
	// (sWord,NULL,false) look up the sWord.
	// (sWord,piIndex,false),look up the sWord,and set piIndex to the new indexes that found.
	// (sWord,piIndex,true), show word by piIndex's information. it will always found, so bTryMoreIfNotFound is useless.
	gboolean bFound = false;
	gchar **ppWord = (gchar **)g_malloc(sizeof(gchar *) * oLibs.total_libs());
	gchar **ppWordData = (gchar **)g_malloc(sizeof(gchar *) * oLibs.total_libs());
	glong *iIndex;
	if (!piIndex)
		iIndex = (glong *)g_malloc(sizeof(glong) * oLibs.total_libs());
	else
		iIndex = piIndex;
	    
    for (int iLib=0;iLib<oLibs.total_libs();iLib++) {
		if (!piIndexValid) {
        	if (oLibs.LookupWord(sWord,iIndex[iLib],iLib)) {
				ppWord[iLib] = oLibs.poGetWord(iIndex[iLib],iLib);
				ppWordData[iLib] = oLibs.poGetWordData(iIndex[iLib],iLib);
            	bFound = true;
        	}
        	else {
				ppWord[iLib] = NULL;
            	ppWordData[iLib] = NULL;
			}
		}
		else {
			if (piIndex[iLib] != INVALID_INDEX && (!strcmp((ppWord[iLib] = oLibs.poGetWord(piIndex[iLib],iLib)),sWord))) {				
				ppWordData[iLib] = oLibs.poGetWordData(piIndex[iLib],iLib);
				bFound = true;
			}
			else {
				ppWord[iLib] = NULL;
				ppWordData[iLib] = NULL;
			}
		}
    }
	if ((!bFound)&&(!piIndexValid)) {
		for (int iLib=0;iLib<oLibs.total_libs();iLib++) {
        	if (oLibs.LookupSimilarWord(sWord,iIndex[iLib],iLib)) {
				ppWord[iLib] = oLibs.poGetWord(iIndex[iLib],iLib);
				ppWordData[iLib] = oLibs.poGetWordData(iIndex[iLib],iLib);
            	bFound = true;
        	}
        	else {
				ppWord[iLib] = NULL;
            	ppWordData[iLib] = NULL;
			}
		}
	}
    if (bFound)
		ShowDataToTextWin(ppWord, ppWordData,sWord);
    else {		
		if (bTryMoreIfNotFound) {		
			gchar *word = g_strdup(sWord);
			gchar *hword;
			hword = GetHeadWord(word);
			if (*hword) {
				if (!strcmp(hword,sWord)) {
					ShowNotFoundToTextWin(sWord,_("<Not Found!>"), TEXT_WIN_NOT_FOUND);
				}
				else {
					for (int iLib=0;iLib<oLibs.total_libs();iLib++) {
			        	if (oLibs.LookupWord(hword,iIndex[iLib],iLib)) {
							ppWord[iLib] = oLibs.poGetWord(iIndex[iLib],iLib);
							ppWordData[iLib] = oLibs.poGetWordData(iIndex[iLib],iLib);
            				bFound = true;
        				}
        				else {
							ppWord[iLib] = NULL;
            				ppWordData[iLib] = NULL;
						}
					}
					if (!bFound) {
						for (int iLib=0;iLib<oLibs.total_libs();iLib++) {
				        	if (oLibs.LookupSimilarWord(hword,iIndex[iLib],iLib)) {
								ppWord[iLib] = oLibs.poGetWord(iIndex[iLib],iLib);
								ppWordData[iLib] = oLibs.poGetWordData(iIndex[iLib],iLib);
            					bFound = true;
        					}
				        	else {
								ppWord[iLib] = NULL;
				            	ppWordData[iLib] = NULL;
							}
						}
					}
					if (bFound)
						ShowDataToTextWin(ppWord, ppWordData,sWord);
					else
						ShowNotFoundToTextWin(sWord,_("<Not Found!>"), TEXT_WIN_NOT_FOUND);		
				}
			}			
			else {
				ShowNotFoundToTextWin(sWord,_("<Not Found!>"), TEXT_WIN_NOT_FOUND);
			}
			g_free(word);
		}
		else {
			ShowNotFoundToTextWin(sWord,_("<Not Found!>"), TEXT_WIN_NOT_FOUND);
		}
	}
	
	if (!piIndex)
		g_free(iIndex);
	
	g_free(ppWord);
	g_free(ppWordData);
	return bFound;
}

int AppCore::FuzzystructCompare(const void * s1, const void * s2)
{
    if (s1==NULL || s2==NULL)
        return 0;
    const struct Fuzzystruct * o1 = (struct Fuzzystruct*)s1;
    const struct Fuzzystruct * o2 = (struct Fuzzystruct*)s2;
    
    if ( o1->iMatchWordDistance > o2->iMatchWordDistance )
        return 1;
    else if ( o1->iMatchWordDistance < o2->iMatchWordDistance )
        return -1;
    else if ( o1->pMatchWord && o2->pMatchWord )
        return stardict_strcmp(o1->pMatchWord, o2->pMatchWord);
    else
        return 0;
}

void AppCore::LookupWithFuzzyToMainWin(const gchar *sWord)
{
    if (sWord[0] == '\0')
        return;
   
	gdk_window_set_cursor (window->window, gpAppFrame->oAppSkin.stardict.watch_cursor.cursor);
        
    struct Fuzzystruct oFuzzystruct[MAX_FUZZY_MATCH_ITEM];
    for (int i=0;i<MAX_FUZZY_MATCH_ITEM;i++)
    {
        oFuzzystruct[i].pMatchWord = NULL;
        oFuzzystruct[i].iMatchWordDistance = iMaxFuzzyDistance;
    }
    int iMaxDistance = iMaxFuzzyDistance;
    int iDistance;
    gboolean Found = false;
    class EditDistance oEditDistance;

    glong iCheckWordLen;
	int sCheckLen;
	const char *sCheck;
	gunichar *ucs4_str1,*ucs4_str2;
	glong ucs4_str2_len;
	char *sLowerCheckWord;
	gchar *sLowerWord = g_utf8_strdown(sWord, -1);	
	ucs4_str2 = g_utf8_to_ucs4_fast(sLowerWord,-1,&ucs4_str2_len);
	g_free(sLowerWord);
    for(int iLib=0;iLib<oLibs.total_libs();iLib++)
    {
        ProcessGtkEvent();
        if((stardict_strcmp(sWord,oLibs.poGetWord(0,iLib))>=0) && (stardict_strcmp(sWord,oLibs.poGetWord(oLibs.iLength(iLib)-1,iLib))<=0)) //there are Chinese dicts and English dicts...
        {
            const int iwords = oLibs.iLength(iLib);
            for (int index=0;index<iwords;index++)
            {
                //ProcessGtkEvent(); // too slow if here
                sCheck = oLibs.poGetWord(index,iLib);
                // tolower and skip too long or too short words
				sCheckLen = strlen(sCheck);
                iCheckWordLen = g_utf8_strlen(sCheck, sCheckLen);
                if (iCheckWordLen-ucs4_str2_len>=iMaxDistance || ucs4_str2_len-iCheckWordLen>=iMaxDistance )
                    continue;				
				sLowerCheckWord = g_utf8_strdown(sCheck, sCheckLen);
				if (iCheckWordLen > ucs4_str2_len)
					(*g_utf8_offset_to_pointer(sLowerCheckWord, ucs4_str2_len)) = '\0';
				ucs4_str1 = g_utf8_to_ucs4_fast(sLowerCheckWord, -1,NULL);
				g_free(sLowerCheckWord);
                iDistance = oEditDistance.CalEditDistance(ucs4_str1,ucs4_str2,iMaxDistance);				
				g_free(ucs4_str1);								
                if ( iDistance < iMaxDistance && iDistance < ucs4_str2_len) // when ucs4_str2_len=1,2 we need less fuzzy.
                {
                    Found = true;
                    gboolean bAlreadyInList = false;
                    int iMaxDistanceAt=0;
                    for (int j=0;j<MAX_FUZZY_MATCH_ITEM;j++)
                    {
                      	if ( oFuzzystruct[j].pMatchWord && strcmp(oFuzzystruct[j].pMatchWord,sCheck)==0 )   //already in list
						{
                            bAlreadyInList = true;
							break;
						}
						//find the position,it will certainly be found (include the first time) as iMaxDistance is set by last time.
                        if ( oFuzzystruct[j].iMatchWordDistance == iMaxDistance )
						{
                            iMaxDistanceAt = j;
						}
                    }
                    if (!bAlreadyInList )
                    {
						if (oFuzzystruct[iMaxDistanceAt].pMatchWord)
							g_free(oFuzzystruct[iMaxDistanceAt].pMatchWord);
                        oFuzzystruct[iMaxDistanceAt].pMatchWord = g_strdup(sCheck);
                        oFuzzystruct[iMaxDistanceAt].iMatchWordDistance = iDistance;
                        // calc new iMaxDistance
                        iMaxDistance = iDistance;
                        for (int j=0;j<MAX_FUZZY_MATCH_ITEM;j++)
                        {
                            if ( oFuzzystruct[j].iMatchWordDistance > iMaxDistance )
                                iMaxDistance = oFuzzystruct[j].iMatchWordDistance;
                        } // calc new iMaxDistance
                    }   // add to list
                }   // find one
            }   // each word
        }   // ok for search
    }   // each lib
    g_free(ucs4_str2);
	
    // show
    oMidWin.oIndexWin.oListWin.Clear();
    if (Found) {		
		oMidWin.oIndexWin.oListWin.list_word_type = LIST_WIN_FUZZY_LIST;
        // sort with distance
        qsort(oFuzzystruct,MAX_FUZZY_MATCH_ITEM,sizeof(Fuzzystruct),FuzzystructCompare);

		//SimpleLookupToTextWin(oFuzzystruct[0].pMatchWord,NULL);
		SimpleLookupToTextWin(oFuzzystruct[0].pMatchWord, iCurrentIndex, false); // so iCurrentIndex is refreshed.

        for (int i=0;i<MAX_FUZZY_MATCH_ITEM;i++) {
            if ( oFuzzystruct[i].pMatchWord ) {
                oMidWin.oIndexWin.oListWin.InsertLast(oFuzzystruct[i].pMatchWord);
				g_free(oFuzzystruct[i].pMatchWord);
                //printf("fuzzy %s,%d\n",oFuzzystruct[i].pMatchWord,oFuzzystruct[i].iMatchWordDistance);
            }
			else {
				break;
			}
        }
		oMidWin.oIndexWin.oListWin.ReScroll();
    }
    else {
		oMidWin.oIndexWin.oListWin.list_word_type = LIST_WIN_EMPTY;
        ShowNotFoundToTextWin(sWord,_("There are too many spelling errors :-("), TEXT_WIN_FUZZY_NOT_FOUND);
	}	
	gdk_window_set_cursor (window->window, gpAppFrame->oAppSkin.stardict.normal_cursor.cursor);
}

void AppCore::LookupWithFuzzyToFloatWin(const gchar *sWord)
{
    if (sWord[0] == '\0')
        return;
    
	gdk_window_set_cursor (oFloatWin.FloatWindow->window, gpAppFrame->oAppSkin.stardict.watch_cursor.cursor);
	    
    struct Fuzzystruct oFuzzystruct[MAX_FLOAT_WINDOW_FUZZY_MATCH_ITEM];
    for (int i=0;i<MAX_FLOAT_WINDOW_FUZZY_MATCH_ITEM;i++)
    {
        oFuzzystruct[i].pMatchWord = NULL;
        oFuzzystruct[i].iMatchWordDistance = iMaxFuzzyDistance;
    }
    int iMaxDistance = iMaxFuzzyDistance;
    int iDistance;
    gboolean Found = false;
    class EditDistance oEditDistance;

	glong iCheckWordLen;
	int sCheckLen;
	const char * sCheck;
	gunichar *ucs4_str1,*ucs4_str2;
	glong ucs4_str2_len;    
    gchar *sLowerCheckWord;
	gchar *sLowerWord = g_utf8_strdown(sWord, -1);
	ucs4_str2 = g_utf8_to_ucs4_fast(sLowerWord, -1,&ucs4_str2_len);
	g_free(sLowerWord);
    for(int iLib=0;iLib<oLibs.total_libs();iLib++)
    {
        ProcessGtkEvent();
        if((stardict_strcmp(sWord,oLibs.poGetWord(0,iLib))>=0) && (stardict_strcmp(sWord,oLibs.poGetWord(oLibs.iLength(iLib)-1,iLib))<=0)) //there are Chinese dicts and English dicts...
        {
            const int iwords = oLibs.iLength(iLib);
            for (int index=0;index<iwords;index++)
            {
                //ProcessGtkEvent(); // too slow if here
                sCheck = oLibs.poGetWord(index,iLib);
                // tolower and skip too long or too short words
				sCheckLen = strlen(sCheck);
                iCheckWordLen = g_utf8_strlen(sCheck, sCheckLen);
                if (iCheckWordLen-ucs4_str2_len>=iMaxDistance || ucs4_str2_len-iCheckWordLen>=iMaxDistance )
                    continue;
				sLowerCheckWord = g_utf8_strdown(sCheck, sCheckLen);
				if (iCheckWordLen > ucs4_str2_len)
					(*g_utf8_offset_to_pointer(sLowerCheckWord, ucs4_str2_len)) = '\0';
				ucs4_str1 = g_utf8_to_ucs4_fast(sLowerCheckWord, -1,NULL);
				g_free(sLowerCheckWord);
                iDistance = oEditDistance.CalEditDistance(ucs4_str1,ucs4_str2,iMaxDistance);				
				g_free(ucs4_str1);				
                if ( iDistance < iMaxDistance && iDistance < ucs4_str2_len) // when ucs4_str2_len=1,2 we need less fuzzy.
                {
                    Found = true;
                    gboolean bAlreadyInList = false;
                    int iMaxDistanceAt=0;
                    for (int j=0;j<MAX_FLOAT_WINDOW_FUZZY_MATCH_ITEM;j++)
                    {
                      	if ( oFuzzystruct[j].pMatchWord && strcmp(oFuzzystruct[j].pMatchWord,sCheck)==0 )   //already in list
						{
                            bAlreadyInList = true;
							break;
						}
                        if ( oFuzzystruct[j].iMatchWordDistance == iMaxDistance )
						{
                            iMaxDistanceAt = j;
						}
                    }
                    if (!bAlreadyInList )
                    {
						if (oFuzzystruct[iMaxDistanceAt].pMatchWord)
							g_free(oFuzzystruct[iMaxDistanceAt].pMatchWord);
                        oFuzzystruct[iMaxDistanceAt].pMatchWord = g_strdup(sCheck);
                        oFuzzystruct[iMaxDistanceAt].iMatchWordDistance = iDistance;
                        // calc new iMaxDistance
                        iMaxDistance = iDistance;
                        for (int j=0;j<MAX_FLOAT_WINDOW_FUZZY_MATCH_ITEM;j++)
                        {
                            if ( oFuzzystruct[j].iMatchWordDistance > iMaxDistance )
                                iMaxDistance = oFuzzystruct[j].iMatchWordDistance;
                        } // calc new iMaxDistance
                    }   // add to list
                }   // find one
            }   // each word
        }   // ok for search
    }   // each lib
    g_free(ucs4_str2);
	
    if (Found)
    {
        // sort with distance
        qsort(oFuzzystruct,MAX_FLOAT_WINDOW_FUZZY_MATCH_ITEM,sizeof(Fuzzystruct),FuzzystructCompare);
		int i,count=0;
        for (i=0;i<MAX_FLOAT_WINDOW_FUZZY_MATCH_ITEM;i++)
        {
            if (oFuzzystruct[i].pMatchWord)
				count++;
			else
				break;
		}
		gchar ***pppWord = (gchar ***)g_malloc(sizeof(gchar **) * count);
		gchar ***pppWordData = (gchar ***)g_malloc(sizeof(gchar **) * count);
		const gchar **ppOriginWord = (const gchar **)g_malloc(sizeof(gchar *) * count);
		gchar **ppWord;
		gchar **ppWordData;
		glong iRetIndex;
		for (i=0;i<count;i++)
		{
    		gboolean bFound = false;
			ppWord = (gchar **)g_malloc(sizeof(gchar *) * oLibs.total_libs());
			ppWordData = (gchar **)g_malloc(sizeof(gchar *) * oLibs.total_libs());    		
	    
			ppOriginWord[i] = oFuzzystruct[i].pMatchWord;
    		for (int iLib=0;iLib<oLibs.total_libs();iLib++)
    		{
        		if (oLibs.SimpleLookupWord(oFuzzystruct[i].pMatchWord, iRetIndex,iLib))
        		{
					ppWord[iLib] = oLibs.poGetWord(iRetIndex,iLib);
					ppWordData[iLib] = oLibs.poGetWordData(iRetIndex,iLib);
            		bFound = true;
        		}
        		else {
					ppWord[iLib] = NULL;
            		ppWordData[iLib] = NULL;
				}
    		}
    		if (bFound==true) // it is certainly be true.
			{
				pppWord[i]=ppWord;
				pppWordData[i]=ppWordData;
			}
			else
			{
				g_free(ppWord);
				pppWord[i]=NULL;
				g_free(ppWordData);
				pppWordData[i]=NULL;
			}			
		}
		ShowDatasToFloatWin(pppWord, pppWordData, ppOriginWord, count,sWord);
		for (i=0;i<count;i++)
		{
			if (pppWord[i])
				g_free(pppWord[i]);
			if (pppWordData[i])
				g_free(pppWordData[i]);
		}
		g_free(pppWord);
		g_free(pppWordData);
		g_free(ppOriginWord);
		
		for (i=0;i<count;i++)
			g_free(oFuzzystruct[i].pMatchWord);
    }
    else
		ShowNotFoundToFloatWin(sWord,_("Fuzzy query failed, too :-("), true);
	gdk_window_set_cursor (oFloatWin.FloatWindow->window, gpAppFrame->oAppSkin.stardict.normal_cursor.cursor);
}

int AppCore::MatchWordCompare(const void * s1, const void * s2)
{
	const gchar **o1 = (const gchar **)s1;
	const gchar **o2 = (const gchar **)s2;
	return stardict_strcmp(*o1,*o2);
}

void AppCore::LookupWithRuleToMainWin(const gchar *word)
{
	gdk_window_set_cursor (window->window, gpAppFrame->oAppSkin.stardict.watch_cursor.cursor);
	
    glong aiIndex[MAX_MATCH_ITEM_PER_LIB+1];
    int iMatchCount = 0;
	gchar **ppMatchWord = (gchar **)g_malloc(sizeof(gchar *) * (MAX_MATCH_ITEM_PER_LIB) * oLibs.total_libs());
    GPatternSpec *pspec = g_pattern_spec_new(word);
	
    for(int iLib=0;iLib<oLibs.total_libs();iLib++)
    {
        //if(oLibs.LookdupWordsWithRule(pspec,aiIndex,MAX_MATCH_ITEM_PER_LIB+1-iMatchCount,iLib)) // -iMatchCount,so save time,but may got less result and the word may repeat.
		if(oLibs.LookdupWordsWithRule(pspec,aiIndex,MAX_MATCH_ITEM_PER_LIB+1,iLib))
        {
            ProcessGtkEvent();
            for(int i=0;aiIndex[i]!=-1;i++)
            {
                gchar * sMatchWord = oLibs.poGetWord(aiIndex[i],iLib);
                gboolean bAlreadyInList = false;
                for (int j=0;j<iMatchCount;j++)
                {
                    if ( strcmp(ppMatchWord[j],sMatchWord) == 0 )   //already in list
                    {
                        bAlreadyInList = true;
                        break;
                    }
                }
                if ( !bAlreadyInList)
                    ppMatchWord[iMatchCount++] = g_strdup(sMatchWord);
            }
        }
    }
	g_pattern_spec_free(pspec);
	oMidWin.oIndexWin.oListWin.Clear();
    if (iMatchCount)
    {		
		oMidWin.oIndexWin.oListWin.list_word_type = LIST_WIN_PATTERN_LIST;
		qsort(ppMatchWord,iMatchCount,sizeof(gchar *),MatchWordCompare); // sort it.
        for(int i=0;i<iMatchCount;i++)
            oMidWin.oIndexWin.oListWin.InsertLast(ppMatchWord[i]);
        //memset(iCurrentIndex,'\0',sizeof(iCurrentIndex));    // iCurrentIndex is ineffective now.
        
        // show the first word.
        //SimpleLookupToTextWin(ppMatchWord[0],NULL);
		SimpleLookupToTextWin(ppMatchWord[0], iCurrentIndex, false); // so iCurrentIndex is refreshed.
		oMidWin.oIndexWin.oListWin.ReScroll();
		for(int i=0;i<iMatchCount;i++)
			g_free(ppMatchWord[i]);
    }    
    else
    {
		oMidWin.oIndexWin.oListWin.list_word_type = LIST_WIN_EMPTY;
        ShowNotFoundToTextWin(word,_("Found no words matching this pattern!"), TEXT_WIN_PATTERN_NOT_FOUND);
    }	
	g_free(ppMatchWord);
	
	gdk_window_set_cursor (window->window, gpAppFrame->oAppSkin.stardict.normal_cursor.cursor);
}

void AppCore::ShowDataToTextWin(gchar ** ppWord, gchar ** ppWordData,const gchar * sOriginWord)
{	
	oMidWin.oTextWin.Show(ppWord, ppWordData,sOriginWord);
	oMidWin.oTextWin.query_result = TEXT_WIN_FOUND;
	oMidWin.oTextWin.queryWord = sOriginWord;
	gboolean canRead = gpAppFrame->oReadWord.canRead(sOriginWord);
	if (canRead) {
		oMidWin.oTextWin.pronounceWord = sOriginWord;
	}
	else {
		for (int i=0;i< gpAppFrame->oAppCore.oLibs.total_libs(); i++) {
			if (ppWord[i] && strcmp(ppWord[i], sOriginWord)) {
				if (gpAppFrame->oReadWord.canRead(ppWord[i])) {
					canRead = true;
					oMidWin.oTextWin.pronounceWord = ppWord[i];
				}
				break;
			}
		}
	}
	gtk_widget_set_sensitive(oMidWin.oToolWin.PronounceWordButton, canRead);
}

void AppCore::ShowTreeDictDataToTextWin(glong offset, glong size, gint iTreeDict)
{
	oMidWin.oTextWin.ShowTreeDictData(oTreeDicts.poGetWordData(offset, size, iTreeDict));
	oMidWin.oTextWin.query_result = TEXT_WIN_TREEDICT;	
}

void AppCore::ShowNotFoundToTextWin(const char* sWord,const char* sReason, TextWinQueryResult query_result)
{
    oMidWin.oTextWin.Show(sReason);
	oMidWin.oTextWin.query_result = query_result;
	oMidWin.oTextWin.queryWord = sWord;
	gboolean canRead = gpAppFrame->oReadWord.canRead(sWord);
	if (canRead)
		oMidWin.oTextWin.pronounceWord = sWord;
	gtk_widget_set_sensitive(oMidWin.oToolWin.PronounceWordButton, canRead);
}

void AppCore::ShowDataToFloatWin(gchar ** ppWord, gchar ** ppWordData,const gchar * sOriginWord)
{
	oFloatWin.ShowText(ppWord, ppWordData,sOriginWord);
}

void AppCore::ShowDatasToFloatWin(gchar *** pppWord, gchar *** pppWordData, const gchar ** ppOriginWord, gint count,const gchar * sOriginWord)
{
	oFloatWin.ShowText(pppWord, pppWordData, ppOriginWord, count, sOriginWord);
}

void AppCore::ShowNotFoundToFloatWin(const char* sWord,const char* sReason, gboolean fuzzy)
{
    oFloatWin.ShowNotFound(sWord, sReason, fuzzy);
}

void AppCore::TopWinEnterWord(const gchar *text)
{
	if (text[0]=='\0')
		return;
	
	if (text[0]=='/')
		LookupWithFuzzyToMainWin(text+1);
    else if( bContainRule(text) )
        LookupWithRuleToMainWin(text);
	else if (oMidWin.oTextWin.query_result == TEXT_WIN_NOT_FOUND)
		LookupWithFuzzyToMainWin(text);
	else if (oMidWin.oTextWin.query_result == TEXT_WIN_INFO || oMidWin.oTextWin.query_result == TEXT_WIN_TREEDICT) {
		GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(oMidWin.oIndexWin.oListWin.treeview));
		GtkTreeModel *model;
		GtkTreeIter iter;
		gboolean selected = gtk_tree_selection_get_selected(selection,&model,&iter);
		gboolean not_first_row;
		if (selected) {
			gchar *path_str = gtk_tree_model_get_string_from_iter(model,&iter);
			if (!strcmp(path_str,"0"))
				not_first_row = false;
			else
				not_first_row = true;
			g_free(path_str);
		}
		if ((!selected) || not_first_row) {
			// now select the first row.
			GtkTreePath* path = gtk_tree_path_new_first();
			gtk_tree_model_get_iter(model,&iter,path);
			gtk_tree_selection_select_iter(selection,&iter);
			gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(oMidWin.oIndexWin.oListWin.treeview),path,NULL,false,0,0);
			gtk_tree_path_free(path);
		}
		else
			SimpleLookupToTextWin(text, iCurrentIndex,true); //text 's index is already cached.
	}
	else if (oMidWin.oTextWin.query_result == TEXT_WIN_FOUND) {
		if (oMidWin.oTextWin.queryWord != text) {
			//user have selected some other word in the list,now select the first word again.
			GtkTreePath* path = gtk_tree_path_new_first();
			GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(oMidWin.oIndexWin.oListWin.treeview));
			GtkTreeIter iter;
			gtk_tree_model_get_iter(model,&iter,path);
			GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(oMidWin.oIndexWin.oListWin.treeview));
			gtk_tree_selection_select_iter(selection,&iter);
			gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(oMidWin.oIndexWin.oListWin.treeview),path,NULL,false,0,0);
			gtk_tree_path_free(path);
		}
	}
		
	 //when TEXT_WIN_TIPS,the text[0]=='\0',already returned.
	
	gtk_editable_select_region(GTK_EDITABLE(GTK_COMBO(oTopWin.WordCombo)->entry),0,-1);
	oTopWin.InsertHisList(text);
	oTopWin.InsertBackList();
	if (GTK_WIDGET_SENSITIVE(oMidWin.oToolWin.PronounceWordButton))
		gpAppFrame->oReadWord.read(oMidWin.oTextWin.pronounceWord.c_str());
}

void AppCore::TopWinWordChange(const gchar* sWord)
{
   	if( bContainRule(sWord) ) {
       	oMidWin.oTextWin.Show(_("Press Enter to list the words that match the pattern."));
       	return;
   	}
       
   	if (sWord[0] == '/') {
		// Fuzzy Search
       	if ( strlen(sWord) == 1 ) {
           	oMidWin.oTextWin.Show(_("Fuzzy query..."));
       	}
   	}
   	else {
		glong *iIndex = (glong *)g_malloc(sizeof(glong) * oLibs.total_libs());
		SimpleLookupToTextWin(sWord,iIndex,false,true);
       	ListWords(iIndex);
		g_free(iIndex);
   	}
}

void AppCore::ListWords(glong* iIndex)
{
    if (iIndex != iCurrentIndex)
        memcpy(iCurrentIndex,iIndex,sizeof(glong)* oLibs.total_libs());
    
    glong *iCurrent;
	iCurrent = (glong*)g_memdup(iIndex,sizeof(glong)* oLibs.total_libs());
	
	oMidWin.oIndexWin.oListWin.Clear();
    
	int iWordCount=0;
	gchar * poCurrentWord;
	poCurrentWord=oLibs.poGetCurrentWord(iCurrent);
	if (poCurrentWord) {
		oMidWin.oIndexWin.oListWin.InsertLast(poCurrentWord);
		iWordCount++;
	}
	while ( iWordCount<LIST_WIN_ROW_NUM && (poCurrentWord=oLibs.poGetNextWord(NULL,iCurrent)) )
    {
		oMidWin.oIndexWin.oListWin.InsertLast(poCurrentWord);
		iWordCount++;
    }	
	if (iWordCount) {
		oMidWin.oIndexWin.oListWin.ReScroll();
		oMidWin.oIndexWin.oListWin.list_word_type = LIST_WIN_NORMAL_LIST;
	}
	else
		oMidWin.oIndexWin.oListWin.list_word_type = LIST_WIN_EMPTY;
	
	g_free(iCurrent);
}

void AppCore::Query(const gchar *word)
{
	oTopWin.InsertHisList(oTopWin.GetText());
	oTopWin.InsertBackList();
	oTopWin.SetText(word);
	if (word[0]=='/')
		LookupWithFuzzyToMainWin(word+1);
    else if( bContainRule(word) )
        LookupWithRuleToMainWin(word);

	oTopWin.TextSelectAll();
	oTopWin.GrabFocus();
	oTopWin.InsertHisList(word);
	oTopWin.InsertBackList(word);
}

void AppCore::ListClick(const gchar *word)
{
	oTopWin.InsertHisList(oTopWin.GetText());
	oTopWin.InsertBackList();	
	oTopWin.SetText(word);
	oTopWin.InsertHisList(word);
	oTopWin.InsertBackList(word);	
	gtk_widget_grab_focus(GTK_COMBO(oTopWin.WordCombo)->entry);
}

void AppCore::PopupPrefsDlg()
{
	if (!prefs_dlg)
		prefs_dlg = new PrefsDlg;
	prefs_dlg->Show();
}

void AppCore::PopupDictManageDlg()
{
	if (!dict_manage_dlg)
		dict_manage_dlg = new DictManageDlg;
	dict_manage_dlg->Show();
}

void AppCore::End()
{
	oSelection.End();
#ifdef _WIN32
	oClipboard.End();
#endif
	oFloatWin.End();
#ifdef _WIN32
	oDockLet.cleanup();
#else
	oDockLet.End();
#endif	
	if (prefs_dlg)
		prefs_dlg->Close();
	if (dict_manage_dlg)
		dict_manage_dlg->Close();
	
	if (oTopWin.MainMenu)
		gtk_widget_destroy(oTopWin.MainMenu);
	if (oTopWin.HistoryMenu)
		gtk_widget_destroy(oTopWin.HistoryMenu);
	if (oTopWin.BackMenu)
		gtk_widget_destroy(oTopWin.BackMenu);
	if (oBottomWin.SearchWebsiteMenu)
		gtk_widget_destroy(oBottomWin.SearchWebsiteMenu);
	gtk_widget_destroy(window);
}

void AppCore::ProcessGtkEvent()
{
	poAppFrame->ProcessGtkEvent();
}


/***************************************************/
AppFrame::AppFrame():oAppCore(this)
{
#ifndef _WIN32
    gnome_sound_init(NULL);
#endif
}

AppFrame::~AppFrame()
{
#ifndef _WIN32
	gnome_sound_shutdown();
#endif
}

void AppFrame::Init(gchar *queryword)
{		
	if (!hide_option)
		show_splash_screen();

#ifdef _WIN32
	init_conf();
#endif

#ifdef _WIN32
	rw_cfg_read_int (usercfgfile, "preferences/dictionary", "enable_sound_event", &enable_sound_event);
#else
	oAppConf.read_bool("/apps/stardict/preferences/dictionary/enable_sound_event", &enable_sound_event, true);
#endif
	//if (enable_sound_event)
	//	gnome_sound_play(STARDICT_DATA_DIR "/sounds/startup.wav");
	
	oAppSkin.load();
		
	oAppCore.Create(queryword);

#ifndef _WIN32	
	stardict_app_server = stardict_application_server_new (gdk_screen_get_default ());
#endif
	
	gtk_main();
}

void AppFrame::ProcessGtkEvent()
{
	while (gtk_events_pending())
	{
		gtk_main_iteration();
	}
}

void AppFrame::Quit()
{   
#ifndef _WIN32
	oAppConf.DisableNotify();
#endif
	
	gboolean maximized;
#ifdef _WIN32
	rw_cfg_read_boolean (usercfgfile, "preferences/main_window", "maximized", &maximized);
#else
	oAppConf.read_bool("/apps/stardict/preferences/main_window/maximized", &maximized, false);
#endif
	if (!maximized) {
    	gint width,height;
		gtk_window_get_size(GTK_WINDOW (oAppCore.window), &width, &height);
#ifdef _WIN32
		rw_cfg_write_int (usercfgfile, "preferences/main_window", "window_width", width);
		rw_cfg_write_int (usercfgfile, "preferences/main_window", "window_height", height);
#else
		oAppConf.write_int("/apps/stardict/preferences/main_window/window_width",width);
		oAppConf.write_int("/apps/stardict/preferences/main_window/window_height",height);
#endif
	}
	gint pos = gtk_paned_get_position(GTK_PANED(oAppCore.oMidWin.hpaned));
#ifdef _WIN32
	rw_cfg_write_int (usercfgfile, "preferences/main_window", "hpaned_pos", pos);
#else
	oAppConf.write_int("/apps/stardict/preferences/main_window/hpaned_pos",pos);
#endif
	if (oAppCore.oFloatWin.bIsLocked) {
		gint x,y;
		gtk_window_get_position(GTK_WINDOW(oAppCore.oFloatWin.FloatWindow),&x,&y);
#ifdef _WIN32
		rw_cfg_write_int (usercfgfile, "preferences/floating_window", "lock_x", x);
		rw_cfg_write_int (usercfgfile, "preferences/floating_window", "lock_y", y);
#else
		oAppConf.write_int("/apps/stardict/preferences/floating_window/lock_x",x);
		oAppConf.write_int("/apps/stardict/preferences/floating_window/lock_y",y);
#endif
	}
    
	oAppCore.End();

#ifndef _WIN32	
	bonobo_object_unref (stardict_app_server);
#endif
	
#ifdef _WIN32
	save_conf();
#endif
    gtk_main_quit ();
}

gint stardict_strcmp(const gchar *s1, const gchar *s2)
{
	gint a;
	a = g_ascii_strcasecmp(s1, s2);
	if (a == 0)
		return strcmp(s1, s2);
	else
		return a;
}

gboolean bContainRule(const char* sWord)
{
    int i;
    for(i=0;sWord[i]!=0;i++)
    {
        if(sWord[i]=='*' || sWord[i]=='?')
        {
            return(true);
        }
    }

    return(false);
}

gboolean bIsPureEnglish(const gchar *str) // i think this should work even when it is UTF8 string :).
{
    int i;
    for(i=0;str[i]!=0;i++)
    {
		//use isascii()?
		if(str[i]<0)
        //if(str[i]<32 || str[i]>126) // tab equal 9,so this is not OK.
        {
            return(false);
        }
    }
    return(true);	
}

gchar* GetPureEnglishAlpha(gchar *str)
{
	while (*str && (!((*str >= 'a' && *str <='z')||(*str >= 'A' && *str <='Z'))))
		str++;
	gchar *p = str;
	while (*p && ((*p >= 'a' && *p <='z')||(*p >= 'A' && *p <='Z')||(*p==' ')))
		p++;
	*p='\0';
	return str;
}

gchar* GetHeadWord(gchar *str)
{
	while (g_ascii_isspace(*str))
		str++;
	glong len = g_utf8_strlen(str, -1);
	if (len) {
		gchar *last_str = g_utf8_offset_to_pointer(str, len-1);
		gunichar last = g_utf8_get_char(last_str);
		while ((g_unichar_isspace(last) || g_unichar_ispunct(last)) || g_unichar_isdigit(last)) {
			*last_str = '\0';
			last_str = g_utf8_find_prev_char(str, last_str);
			if (!last_str)
				break;
			last = g_utf8_get_char(last_str);
		}
	}
	return str;
}

gboolean stardict_on_enter_notify (GtkWidget * widget, GdkEventCrossing * event, gpointer data)
{
	if (gpAppFrame->enable_sound_event && event->mode == GDK_CROSSING_NORMAL) {
#ifdef _WIN32
		gchar *filename = g_build_filename(stardict_data_dir, "sounds", "buttonactive.wav", NULL);
		PlaySound(filename, 0, SND_ASYNC | SND_FILENAME);
		g_free(filename);
#else
		gnome_sound_play(STARDICT_DATA_DIR "/sounds/buttonactive.wav");
#endif
	}
	return false;
}

#ifndef _WIN32
static void
stardict_handle_automation_cmdline (gchar *queryword)
{
	CORBA_Environment env;
	GNOME_Stardict_Application server;
	
	CORBA_exception_init (&env);

	server = bonobo_activation_activate_from_id ("OAFIID:GNOME_Stardict_Application",
                                                     0, NULL, &env);
	if (!server) {
		g_free(queryword);
		gdk_notify_startup_complete ();
		return;
	}
	//g_return_if_fail (server != NULL);
	
	if (quit_option) {
		GNOME_Stardict_Application_quit (server, &env);
	}
	else {
		if (queryword) {
			GNOME_Stardict_Application_queryWord (server, queryword, &env);
		}		
		if (hide_option) {
			GNOME_Stardict_Application_hide (server, &env);
		}
		else {
			GNOME_Stardict_Application_grabFocus (server, &env);
			g_message(_("StarDict is already running. Using the running process."));
		}
	}


	bonobo_object_release_unref (server, &env);
	CORBA_exception_free (&env);

	g_free(queryword);
	
	/* we never popup a window, so tell startup-notification that
	 * we're done */
	gdk_notify_startup_complete ();
}

static gint client_die_cb (GnomeClient *client, gpointer client_data)
{
	gpAppFrame->Quit();
}

static gint save_yourself_cb (GnomeClient       *client,
                              gint               phase,
                              GnomeRestartStyle  save_style,
                              gint               shutdown,
                              GnomeInteractStyle interact_style,
                              gint               fast,
                              gpointer           client_data)
{
    gchar *argv[] = {NULL, NULL, NULL};
	gchar *word = NULL;
    gint argc = 1;

    argv[0] = (gchar *)client_data;
	
	if (gpAppFrame->oAppCore.window) {
		if (!GTK_WIDGET_VISIBLE(gpAppFrame->oAppCore.window))
			argv[argc++] = "-h";
	}

	if (gpAppFrame->oAppCore.oTopWin.WordCombo) {
	    const gchar *text;
		text = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(gpAppFrame->oAppCore.oTopWin.WordCombo)->entry));
    	if (text[0]) {
			word = g_strdup(text);
        	argv[argc++] = word;
		}
	}

    gnome_client_set_restart_command(client, argc, argv);
    gnome_client_set_clone_command(client, argc, argv);
	if (word)
		g_free(word);

    return true;
}
#endif

#ifdef _WIN32
static void stardict_dummy_print( const gchar* string ) {
	return;
}

static void stardict_dummy_log_handler (const gchar    *domain,
				    GLogLevelFlags  flags,
				    const gchar    *msg,
				    gpointer        user_data) {
	return;
}

static gboolean set_stardict_data_dir(void) {
	HMODULE hmod;

	hmod = GetModuleHandle(NULL);
	if( hmod == 0 ) {
		return true;
	}
	if(GetModuleFileName( hmod, (char*)&stardict_data_dir, 256 ) == 0) {
		return true;
	}
	gchar* buf;
	buf = g_path_get_dirname( stardict_data_dir );
	strcpy( (char*)&stardict_data_dir, buf );
	g_free( buf );
	return false;
}
#endif

#ifdef _WIN32
int stardict_main(int argc,char **argv)
#else
int main(int argc,char **argv)
#endif
{			
#ifdef _WIN32
	if (set_stardict_data_dir())
		return 0;
	gchar *locale_dir;
	locale_dir = g_strdup_printf("%s" G_DIR_SEPARATOR_S "locale", stardict_data_dir);
	bindtextdomain (GETTEXT_PACKAGE, locale_dir);
	g_free(locale_dir);
#else
	bindtextdomain (GETTEXT_PACKAGE, STARDICT_LOCALEDIR);
#endif
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
		
#ifdef _WIN32
	gchar *title=g_locale_from_utf8(_("StarDict"), -1, NULL, NULL, NULL);
	HWND ll_winhandle = FindWindowA(0, title);
	g_free(title);
	if (ll_winhandle > 0) {
		SetForegroundWindow(ll_winhandle);
		//BringWindowToTop(ll_winhandle); //don't work :(
		//ShowWindow( ll_winhandle, SW_SHOW); //don't work fine, as gtk widgets may still be hiden.
		return 0;
	}

    gtk_set_locale ();
    gtk_init (&argc, &argv);

	g_log_set_handler (NULL, (enum GLogLevelFlags)(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION),
			   stardict_dummy_log_handler, NULL);	
	g_log_set_handler ("Gdk", (enum GLogLevelFlags)(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION),
			   stardict_dummy_log_handler, NULL);
	g_log_set_handler ("Gtk", (enum GLogLevelFlags)(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION),
			   stardict_dummy_log_handler, NULL);
	g_log_set_handler ("GLib", (enum GLogLevelFlags)(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION),
			   stardict_dummy_log_handler, NULL);
	g_log_set_handler ("GModule", (enum GLogLevelFlags)(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION),
			   stardict_dummy_log_handler, NULL);
	g_log_set_handler ("GLib-GObject", (enum GLogLevelFlags)(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION),
			   stardict_dummy_log_handler, NULL);
	g_log_set_handler ("GThread", (enum GLogLevelFlags)(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION),
			   stardict_dummy_log_handler, NULL);
	g_set_print_handler( stardict_dummy_print );

	gchar *queryword = NULL;
	for (int i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			/* is an option */
			if (strcmp(argv[i], "-h") == 0) {
				hide_option = TRUE;
			}
		}
		else {
		    if (!queryword) {
		    	if (g_utf8_validate (argv[i], -1, NULL)) {
		    		queryword= g_strdup(argv[i]);			
	    		}
	    		else
	    			queryword = g_locale_to_utf8(argv[i],-1,NULL,NULL,NULL);
   			}
		}
	}

#else
	GnomeProgram *program;
	program = gnome_program_init ("stardict", VERSION,
			    LIBGNOMEUI_MODULE, argc, argv,
			    GNOME_PARAM_POPT_TABLE, options,			    
			    GNOME_PARAM_HUMAN_READABLE_NAME,
		            _("Dictionary"),
			    GNOME_PARAM_APP_DATADIR, DATADIR,
			    NULL);
		
	
	
	GValue        value = { 0 };
	poptContext   pctx;

	g_object_get_property (G_OBJECT (program),
			       GNOME_PARAM_POPT_CONTEXT,
			       g_value_init (&value, G_TYPE_POINTER));
	pctx = (poptContext) g_value_get_pointer (&value);
	g_value_unset (&value);

	char **args;
	args = (char**) poptGetArgs(pctx);

	gchar *queryword = NULL;
	if (args && args[0]) 
	{	
		//only look up the first word should OK.
		if (g_utf8_validate (args[0], -1, NULL)) {
			queryword= g_strdup(args[0]);			
		}
		else
			queryword = g_locale_to_utf8(args[0],-1,NULL,NULL,NULL);
	}	
	poptFreeContext (pctx);

	CORBA_Object factory;
	factory = bonobo_activation_activate_from_id
		("OAFIID:GNOME_Stardict_Factory",
		Bonobo_ACTIVATION_FLAG_EXISTING_ONLY,
		NULL, NULL);

	if (factory != NULL) 
	{
		/* there is an instance already running, so send
		 * commands to it if needed
		 */
                stardict_handle_automation_cmdline (queryword);				
                /* and we're done */
                exit (0);
	}
	
	GnomeClient *client;
	if ((client = gnome_master_client()) != NULL) {
		g_signal_connect (client, "save_yourself", G_CALLBACK (save_yourself_cb), (gpointer) argv[0]);
		g_signal_connect (client, "die", G_CALLBACK (client_die_cb), NULL);
	}		
#endif		
	
    AppFrame oAppFrame;
	gpAppFrame = &oAppFrame;
    oAppFrame.Init(queryword);
	return 0;
}

#ifdef _WIN32

#ifdef __GNUC__
#  ifndef _stdcall
#    define _stdcall  __attribute__((stdcall))
#  endif
#endif

int _stdcall
WinMain (struct HINSTANCE__ *hInstance, 
	 struct HINSTANCE__ *hPrevInstance,
	 char               *lpszCmdLine,
	 int                 nCmdShow)
{
	stardictexe_hInstance = hInstance;
	return stardict_main (__argc, __argv);
}
#endif
