// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


//
// stararray.cc - The Star Array class.  Not much more than a container class
//  for stars, with the ability to read and filter a text data file on the fly.
//

#include "stararray.h"

#define NEED_FULL_NAMES
#include "constellations.h"

// StarArray private functions -------------------------------------

// The usual functions to switch between coordinate systems; as always,
//  they don't check to see what coordinate system things are already
//  in before blindly applying the transform.  These are private, so the
//  check will be done by StarArray::SetRules().

void StarArray::toCelestial()
{
  for (unsigned int i = 0; i < Array.size(); i++)
    Array[i].toCelestial();
}

void StarArray::toGalactic()
{
  for (unsigned int i = 0; i < Array.size(); i++)
    Array[i].toGalactic();
}


// Function to read an input file (assumes input is a valid opened ifstream)
//  and APPEND onto Array all the stars which pass the filters specified
//  in ArrayRules.
//  Data files are in the following format, with records separated by
//  the character RECORD_DELIMITER defined in "stararray.h":
//
// File header: may not contain the character RECORD_DELIMITER
// RECORD_DELIMITER first record (may contain line breaks)
// RECORD_DELIMITER second record (likewise)
// RECORD_DELIMITER etc.
//
// [See comment above Star::Star(const char *) in star.cc, or example
// StarPlot data file "sample.stars", for detailed structure of records.]

void StarArray::Read(ifstream &input)
{
  double chartdist = ArrayRules.ChartLocation.magnitude();
  string garbage;
  char record[RECORD_MAX_LENGTH + 1];
  vector<Star> companions = vector<Star>();
  companions.reserve(100);
  Star tempstar, faststar;

  // First, discard all characters before the first record
  getline(input, garbage, RECORD_DELIMITER);

  // Then loop over records.  std::getline(istream, string, char) would be
  //  much simpler, but it's _slow_.
  while (input.get(record, RECORD_MAX_LENGTH + 1, RECORD_DELIMITER)) {

    // Purge the record delimiter from input and check to see if we've
    //  exceeded the size of the record string buffer.  If so, ignore
    //  the rest of that record.
    if (input.get() != RECORD_DELIMITER) 
      getline(input, garbage, RECORD_DELIMITER);

    // keep track of total # of records in all files we've scanned in,
    //  and beware of memory overflows
    TotalStars++;

    // in the following we want "continue" instead of "return" so that
    //  TotalStars will be tallied correctly.
    if (Array.size() >= MAX_STARS) continue;

    // Filter out stars by distance and magnitude.  This is the fastest
    //  filter because it doesn't create any temporary objects.  So it 
    //  comes first.
    {
      double starmag, stardist;
      unsigned int posn = 0, field = 0, len = strlen(record);

      while (posn < len && field < 3)
	if (record[posn++] == FIELD_DELIMITER) field++;
      stardist = myatof(record + posn);
      if (stardist > chartdist + ArrayRules.ChartRadius ||
	  stardist < chartdist - ArrayRules.ChartRadius) continue;

      while (posn < len && field < 6)
	if (record[posn++] == FIELD_DELIMITER) field++;
      starmag = myatof(record + posn);
      if (starmag > ArrayRules.ChartDimmestMagnitude ||
	  starmag < ArrayRules.ChartBrightestMagnitude) continue;
    }

    // Convert each record which passes the fast filter into a Star and see
    //  whether it should go into the STL vector of Stars.
    faststar = Star(record, true /* fast conversion */);

    if (! ArrayRules.CelestialCoords)  // we assume the data file is in
      faststar.toGalactic();           // celestial, not galactic, coords

    if (faststar.PassesFilter(ArrayRules)) {
      tempstar = Star(record);
      if (! ArrayRules.CelestialCoords)	tempstar.toGalactic();

      // If it passes all the filters and is a singleton, primary, or
      //  sufficiently far from its primary, put it into the STL vector.
      if ((tempstar.GetStarPrimaryDistance() == 0.0)
	  || ((tempstar.GetStarPrimaryDistance() * (200 /*pixels*/) 
	       / ArrayRules.ChartRadius) > (double)MIN_SECONDARY_DISTANCE))
	Array.push_back(tempstar);

      // If it is a companion star, but no membership is given, we have
      //  no way of telling whether its primary has been filtered out,
      //  so toss it.
      else if (tempstar.GetStarMembership().strlen(0) == 0) continue;

      // Put everything else into a separate STL vector to examine later.
      else companions.push_back(tempstar);
    }
  }

  // Now, go through all the stars in the array of companions.
  //  Put stars in this array into the main array only if their primaries
  //  are not already in the main array.

  unsigned int arraysize = Array.size();
  for (unsigned int i = 0; i < companions.size(); i++) {
    unsigned int j = 0;

    if (Array.size() >= MAX_STARS) return;
    while ((j < arraysize)
	   && (Array[j].GetStarPrimaryDistance() != 0.0
	       || (strcmp(companions[i].GetStarMembership()[0],
			  Array[j].GetStarMembership()[0]) != 0)))
      j++;
    if (j == arraysize) /* no match found */ Array.push_back(companions[i]);
  }
  return;
}


// Functions to sort stars in order of increasing "local" x-coordinate

// First, a typedef to hold all the information the compare function will need

typedef struct {
  double xposition;
  Star   star;
} sortable;

// Next, the function to compare for qsort().

int compare_function(const void *p, const void *q)
{
  double x1 = ((const sortable *)p)->xposition;
  double x2 = ((const sortable *)q)->xposition;
  return (x1 - x2 >= 0.0) ? 1 : -1;
}

// Finally, the main function which calls qsort()

void StarArray::Sort()
{
  unsigned int size = Array.size();
  Vector relativeLocation;
  SolidAngle orientation = ArrayRules.ChartOrientation;
  sortable *temparray = new sortable[size];

  // Make a temporary array for qsort(), consisting of "sortable" structs
  //  which each contain a Star and a position in local coordinates.
  for (unsigned int i = 0; i < size; i++) {
    relativeLocation = Array[i].GetStarXYZ() - ArrayRules.ChartLocation;
    temparray[i].xposition =
      relativeLocation.getX() * cos(orientation.getPhi())
      + relativeLocation.getY() * sin(orientation.getPhi());
    temparray[i].star = Array[i];
  }

  qsort(temparray, size, sizeof(sortable), compare_function);

  // Put the sorted temporary array back into the vector
  Array.clear();
  for (unsigned int i = 0; i < size; i++) {
    temparray[i].star.SetPlace(i+1);    // label stars starting at 1
    Array.push_back(temparray[i].star);
  }

  delete [] temparray;
  return;
}


// StarArray public functions --------------------------------------


// Search(): function to search for a given string in the names of all the
//  stars in a set of files.  For safety, this function refuses to work unless
//  the array is currently empty.  Stars containing the substring are added 
//  to Array.  This function uses the "rules" argument to set the correct
//  coordinate system (celestial / galactic), but does not bother to set 
//  its own Rules member.

// Options:
// If `casesensitive' is false, the function ignores the distinction
//  between upper and lower case.  Otherwise all cases must match.
// If `exactmatch' is true, `searchstring' must be identical to a star name
//  (possibly excluding case) for a match.  If `exactmatch' is false, the
//  function looks for `searchstring' as a substring of star names.  If, in
//  addition, `searchstring' contains spaces, it is tokenized and all of the
//  tokens must be substrings of the same star name for a match to occur.
//  These substrings may not overlap.
// If `exitonmatch' is true, the function returns only the first match found.
//  Otherwise all matches are returned.

void StarArray::Search(const char *searchstring, StringList filelist, 
		       const Rules &rules, bool casesensitive,
		       bool exactmatch /* i.e. not a substring */,
		       bool exitonmatch /* i.e. return at most one result */)
{
  if (Array.size() || isempty(searchstring)) return;

  StringList searchtokens = casesensitive ?
    StringList(searchstring, ' ') : StringList(searchstring, ' ').touppercase();
  unsigned int tokensize = searchtokens.size();
  searchtokens.stripspace();
  
  // Replace constellation names, if any, with abbreviations.
  // n.b. Abbreviations must be searched for preceded by a space, so that
  // (e.g.) we don't find Polaris on a search for 'arietis' or Tau Ceti on a 
  // search for 'Tauri'
  for (unsigned int i = 0; i < NUM_CONSTELLATIONS; i++) {
    string name = string(constelnames[i]);
    string abbr = string(constellations[i]);
    if (! casesensitive) { uppercase(name); uppercase(abbr); }
    
    if (name.find(' ') < name.size()) { // two-word constellation name
      for (unsigned int j = 0; j + 1 < tokensize; j++) {
	StringList name_tok = StringList(name.c_str(), ' ');
	if (strcmp(searchtokens[j], name_tok[0]) == 0 && 
	    strcmp(searchtokens[j + 1], name_tok[1]) == 0) {
	  searchtokens.set(string(" ") + abbr, j);
	  searchtokens.remove(j + 1);
	  tokensize--;
	  goto replaced;
	}
      }
    }
    else { // one-word constellation name
      for (unsigned int j = 0; j < tokensize; j++)
	if (strcmp(searchtokens[j], name.c_str()) == 0) {
	  searchtokens.set(string(" ") + abbr, j);
	  goto replaced;
	}
    }
  }
  
 replaced:
  
  // loop over data files
  for (unsigned int i = 0; i < filelist.size(); i++) {
    ifstream input;
    string garbage;
    char record[RECORD_MAX_LENGTH + 1];
    int nameposn = 0;

    input.open(filelist[i]);
    if (!input.good()) continue;
    
    // First, discard all characters before the first record
    getline(input, garbage, RECORD_DELIMITER);

    // Then loop over records.  std::getline(istream, string, char) would be
    //  much simpler, but it's _slow_.
    while (input.get(record, RECORD_MAX_LENGTH + 1, RECORD_DELIMITER)) {
      bool result = false;
      
      // Purge the record delimiter from input and check to see if we've
      //  exceeded the size of the record string buffer.  If so, ignore
      //  the rest of that record.
      if (input.get() != RECORD_DELIMITER) 
	getline(input, garbage, RECORD_DELIMITER);
      
      StringList namelist = StringList(StringList(record, FIELD_DELIMITER)[0],
				       SUBFIELD_DELIMITER);
      namelist.stripspace();
      if (! casesensitive && ! exactmatch) namelist.touppercase();
      unsigned int namelistsize = namelist.size();

      for (unsigned int j = 0; j < namelistsize; j++) {
	if (exactmatch)
	  // str[case]cmp returns 0 if there's a match, so flip the result
	  result = ! (casesensitive ? strcasecmp(searchstring, namelist[j])
		      : strcmp(searchstring, namelist[j]));
	else { // search on substrings
	  result = true;
	  char *temp = new char[strlen(namelist[j]) + 1];
	  strcpy(temp, namelist[j]);
	  
	  // loop over tokens in the search string
	  for (unsigned int k = 0; k < tokensize; k++) {
	    char *substring = strstr(temp, searchtokens[k]);
	    unsigned int len = strlen(searchtokens[k]);

	    // use the fact that strstr() returns NULL if 2nd arg not found
	    result = (result && substring);
	    if (!result) break;

	    // Now erase the found substring.  This makes certain that found
	    //  substrings won't overlap.  Need to do this in case someone
	    //  is searching for, for example, "Tau Tau" (Tau Tauri)
	    for (unsigned int l = 0; l < len; l++)
	      substring[l] = ' ';
	  }
	  delete [] temp;
	}
	
	if (result) { nameposn = j; break; }

      } // closes "for (unsigned int j = 0; j < namelistsize; j++)"

      if (result) {
	Star tempstar = Star(record);
	// use Star.sPlace to hold the position of the name in which a match
	//  was found:
	tempstar.SetPlace(nameposn);
	if (! rules.CelestialCoords) tempstar.toGalactic();
	Array.push_back(tempstar);

	if (exitonmatch || Array.size() >= MAX_STARS) {
	  input.close(); 
	  return; 
	}
      }

    } // closes "while (input.get(record, ...))"

    input.close();
  } // closes "for (unsigned int i = 0; i < filelist.size(); i++)"

  return;
}


// function to set the rules for the ArrayRules member of StarArray.
//  Will also update the Array itself by re-reading the files specified
//  in the rules (if necessary) and/or re-sorting the array (if necessary).
//  This does not, however, call StarArray::Display(); that's up to the
//  main program.

bool StarArray::SetRules(const Rules &rules, Changetype ruleschange)
{
  ArrayRules = rules;

 beginning: // if ruleschange is not a valid Changetype, set it to be
            //  type FILE_CHANGE and jump back here from default: label below

  switch (ruleschange) {
    case FILE_CHANGE:
    case LOCATION_CHANGE: case RADIUS_CHANGE:
    case COORDINATE_CHANGE: case FILTER_CHANGE:
      // reload and refilter the file(s) in ArrayRules.ChartFileNames
      //  (return with an error if none of the file(s) can be read)
      {
	unsigned int numValidFiles = 0;
	ifstream data;

	TotalStars = 0;
	Array.clear();
	// will probably not usually display more stars than this at once:
	Array.reserve(100);

	for (unsigned int i = 0; i < ArrayRules.ChartFileNames.size(); i++) {
	  data.open(ArrayRules.ChartFileNames[i]);
	  if (data.good()) {
	    Read(data);
	    data.close();
	    numValidFiles++;
	  }
	}

	// Ideally, we would have a check here to make sure there are not
	//  duplicate entries if more than one file was opened.

	if (numValidFiles == 0)
	  return false;
      }
      // fall through...

    case ORIENTATION_CHANGE: // re-sort Array
      Sort();
      // fall through...

    case DECORATIONS_CHANGE: // search for brightest 5 stars, if necessary
      if (ArrayRules.StarLabels == LANDMARK_LABEL) {

	if (Array.size() <= DISPLAY_N_STARS)
	  for (unsigned int i = 0; i <DISPLAY_N_STARS && i < Array.size(); i++)
	    Array[i].SetLabel(true);

	else {
	  Star *brightarray[DISPLAY_N_STARS];
	  double dimmag = -10000;

	  // The following variable is set "volatile" to work around a gcc
	  //  loop optimization bug which sometimes occurs when compiling
	  //  at -O2 or higher.  See one of these URLs for details:
	  //  http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=83363
	  //  http://www.winehq.com/hypermail/wine-patches/2000/11/0076.html
	  //
	  //  Supposedly it's fixed in the latest CVS versions.  However,
	  //  I still observe it in the Debian-packaged version of the
	  //  CVS of g++-3.0.2 of 2001-09-22.
	  //
	  //  Instead of declaring this "volatile", you can instead compile
	  //  this file with the g++ flag -fno-strength-reduce if you want.
#if defined __GNUC__ // && (__GNUC__ < 3)
	  volatile
#endif
	  unsigned int dimptr = 0;

	  // find the brightest DISPLAY_N_STARS stars and put pointers to them
	  //  in brightarray

	  for (unsigned int i = 0; i < DISPLAY_N_STARS; i++) {
	    Array[i].SetLabel(false);
	    brightarray[i] = &(Array[i]);
	    if (brightarray[i]->GetStarMagnitude() > dimmag) {
	      dimmag = brightarray[i]->GetStarMagnitude();
	      dimptr = i;
	    }
	  }

	  for (unsigned int i = DISPLAY_N_STARS; i < Array.size(); i++) {
	    Array[i].SetLabel(false);

	    if (Array[i].GetStarMagnitude() < dimmag) {
	      brightarray[dimptr] = &(Array[i]);
	    
	      // now find new dimmest star in brightarray
	      dimmag = -10000;
	      for (unsigned int j = 0; j < DISPLAY_N_STARS; j++) {
		if (brightarray[j]->GetStarMagnitude() > dimmag) {
		  dimmag = brightarray[j]->GetStarMagnitude();
		  dimptr = j;
		}
	      }
	    }
	  }
	  
	  // now set those DISPLAY_N_STARS stars to be labeled, as well as
	  //  any other stars which have the same magnitude as the dimmest
	  //  star in brightarray.
	  for (unsigned int i = 0; i < Array.size(); i++)
	    if (Array[i].GetStarMagnitude() <= dimmag)
	      Array[i].SetLabel(true);
	}
      }
    case NO_CHANGE:
      // do nothing; these changes are taken care of in StarArray::Display()
      break;

    default:  // assume the most radical type of change, to be safe
      ruleschange = FILE_CHANGE;
      goto beginning;            // yes, a goto statement.  I'm sorry, ok?
  }
  return true;
}


// Drawing the StarArray display -----------------------------------

// private functions to draw the chart grid

void StarArray::drawgridnorth(StarViewer *sv, unsigned int wincenterX,
			      unsigned int wincenterY, 
			      unsigned int pixelradius) const
{
  sv->setfill(false);
  sv->setcolor(POSITIVE);
  sv->drawarc(wincenterX, wincenterY, pixelradius, pixelradius, 0, M_PI);
  return;
}

void StarArray::drawgridsouth(StarViewer *sv, unsigned int wincenterX,
			      unsigned int wincenterY, 
			      unsigned int pixelradius) const
{
  sv->setfill(false);
  sv->setcolor(NEGATIVE);
  sv->drawarc(wincenterX, wincenterY, pixelradius, pixelradius,
	      M_PI, 2 * M_PI);
  return;
}

void StarArray::drawgridequator(StarViewer *sv, unsigned int wincenterX,
				unsigned int wincenterY, 
				unsigned int pixelradius) const
{
  double sra = sin(ArrayRules.ChartOrientation.getPhi());
  double cra = cos(ArrayRules.ChartOrientation.getPhi());
  double sdec = sin(ArrayRules.ChartOrientation.getTheta());

  // equator
  sv->setfill(false);
  if (sdec >= 0.0) sv->setcolor(POSITIVE);
  else sv->setcolor(NEGATIVE);
  sv->drawellipse(wincenterX, wincenterY,
  		  pixelradius, (int)(pixelradius * sdec));

  // 0 - 12 hr line
  sv->drawline(wincenterX - (int)(pixelradius * sra),
	       wincenterY + (int)(pixelradius * cra * sdec),
	       wincenterX + (int)(pixelradius * sra),
	       wincenterY - (int)(pixelradius * cra * sdec));

  // 6 - 18 hr line
  sv->drawline(wincenterX + (int)(pixelradius * cra),
	       wincenterY + (int)(pixelradius * sra * sdec),
	       wincenterX - (int)(pixelradius * cra),
	       wincenterY - (int)(pixelradius * sra * sdec));

  // RA / GalLong labels
  char gridlabels[4][5];
  sv->setcolor(POSITIVE);
  if (ArrayRules.CelestialCoords)
    for (unsigned int i = 0; i < 4; i++)
      sprintf(gridlabels[i], "%dh", 6 * i);
  else
    for (unsigned int i = 0; i < 4; i++)
      sprintf(gridlabels[i], "%d" DEGREE_SYMBOL, 90 * i);

  sv->drawtext(gridlabels[0], wincenterX - (int)(1.1 * pixelradius * sra) - 5,
	       wincenterY + (int)(1.1 * pixelradius * cra * sdec) + 5, 10);
  sv->drawtext(gridlabels[1], wincenterX + (int)(1.1 * pixelradius * cra) - 5,
	       wincenterY + (int)(1.1 * pixelradius * sra * sdec) + 5, 10);
  sv->drawtext(gridlabels[2], wincenterX + (int)(1.1 * pixelradius * sra) - 5,
	       wincenterY - (int)(1.1 * pixelradius * cra * sdec) + 5, 10);
  sv->drawtext(gridlabels[3], wincenterX - (int)(1.1 * pixelradius * cra) - 5,
	       wincenterY - (int)(1.1 * pixelradius * sra * sdec) + 5, 10);

  return;
}


// private function to draw a single chart legend item
void StarArray::drawlegenditem(StarViewer *sv, unsigned int color, 
			       unsigned int number, unsigned int x,
			       unsigned int y, unsigned int r,
			       const char *text, int relx, int rely) const
{
  sv->setcolor(color);
  if (ArrayRules.StarClasses[number])
    sv->drawstar(x, y, r);
  else {
    sv->setfill(false);
    sv->drawcircle(x, y, r);
  }

  sv->setcolor(LABEL_COLOR);
  sv->drawtext(text, x + relx, y + rely, 10);
}


// private function to draw the legend
void StarArray::drawlegend(StarViewer *sv) const
{
  unsigned int xbase, ybase = 0;
  char sra[30], sdec[30], sdist[30], srad[30], sdmag[30], sbmag[30];
    
  // first draw the chart information in the upper left
  if (ArrayRules.CelestialCoords) {
    strcpy(sra, "R.A. ");
    strcpy(sdec, "Dec ");
  }
  else {
    strcpy(sra, "Long ");
    strcpy(sdec, "Lat ");
  }
  SolidAngle gridposn = ArrayRules.ChartLocation.toSpherical();
  RadiansToRAString(gridposn.getPhi(), sra + 5, ArrayRules.CelestialCoords);
  RadiansToDecString(gridposn.getTheta(), sdec + 4);
  snprintf(sdist, 29, "%-.6g LY from Earth", 
	   ArrayRules.ChartLocation.magnitude());
  snprintf(srad, 29, "Chart Radius: %-.6g LY", ArrayRules.ChartRadius);
  snprintf(sdmag, 29, "Dim Mag: %+2.1f", 
	   ArrayRules.ChartDimmestMagnitude);
  snprintf(sbmag, 29, "Bright Mag: %+2.1f", 
	   ArrayRules.ChartBrightestMagnitude);
  
  sv->setcolor(BOLD_COLOR); sv->drawtext("CHART STATUS", 15, ybase += 15, 12);
  sv->setcolor(LABEL_COLOR);
  sv->drawtext("Location of Chart Origin:", 5, ybase += 15, 10);
  if (ArrayRules.ChartLocation.magnitude() > 0.0) {
    sv->drawtext(sra, 10, ybase += 15, 10);
    sv->drawtext(sdec, 10, ybase += 15, 10);
  }
  sv->drawtext(sdist, 10, ybase += 15, 10);
  
  if (ArrayRules.ChartRadius <= 0.0) sv->setcolor(ERROR_COLOR);
  sv->drawtext(srad, 5, ybase += 15, 10);
  
  if (ArrayRules.ChartDimmestMagnitude - ArrayRules.ChartBrightestMagnitude
      <= 0.0)
    sv->setcolor(ERROR_COLOR);
  else sv->setcolor(LABEL_COLOR);
  sv->drawtext(sbmag, 5, ybase += 15, 10);
  sv->drawtext(sdmag, 5, ybase += 15, 10);

  // then draw the legend (which doubles as a display of which spectral
  //  classes are currently turned on) in the upper right
  
  xbase = (sv->width() > 80) ? sv->width() - 75 : 5;
  sv->setcolor(BOLD_COLOR); sv->drawtext("LEGEND", xbase + 5, 15, 12);

#define LO LEGEND_OFFSETS

  drawlegenditem(sv, WOLF_RAYET_COLOR, 0, 
  		 xbase + LO[10][0], LO[10][1], 5, "Wolf-Rayet");
  drawlegenditem(sv, O_COLOR, 0, xbase + LO[0][0], LO[0][1], 5, "O");
  drawlegenditem(sv, B_COLOR, 1, xbase + LO[1][0], LO[1][1], 5, "B");
  drawlegenditem(sv, A_COLOR, 2, xbase + LO[2][0], LO[2][1], 5, "A");
  drawlegenditem(sv, F_COLOR, 3, xbase + LO[3][0], LO[3][1], 5, "F");
  drawlegenditem(sv, G_COLOR, 4, xbase + LO[4][0], LO[4][1], 5, "G");
  drawlegenditem(sv, K_COLOR, 5, xbase + LO[5][0], LO[5][1], 5, "K");
  drawlegenditem(sv, M_COLOR, 6, xbase + LO[6][0], LO[6][1], 5, "M");
  drawlegenditem(sv, D_COLOR, 7, xbase + LO[7][0], LO[7][1], 3, "wd");
  drawlegenditem(sv, DEFAULT_COLOR, 8, xbase + LO[8][0], LO[8][1], 5,
		 "Unknown", -65, 5);
  drawlegenditem(sv, NON_STELLAR_COLOR, 9, xbase + LO[9][0], LO[9][1], 5,
		 "Non-stellar", -65, 5);
  return;
}


// public function to display all the stars in Array onto the pixmap,
//  as well as a grid outline and some other decorations

void StarArray::Display(StarViewer *sv) const
{
  int windowsize, pixelradius, wincenterX, wincenterY;
  double ChartDec = ArrayRules.ChartOrientation.getTheta();
  double starZ;

  // Determine radius and center of chart, in pixels
  windowsize = (sv->width() > sv->height()) ? sv->height() : sv->width();
  pixelradius = (int)(0.4 * windowsize);
  wincenterX = sv->width() / 2;
  wincenterY = sv->height() / 2;

  // clear pixmap with the background color (defined in "specclass.h")
  sv->fill(BACKGROUND);

  // draw the hemisphere (N/S) of the grid facing AWAY from the screen
  if (ArrayRules.ChartGrid) {
    if (ChartDec >= 0.0)
      drawgridsouth(sv, wincenterX, wincenterY, pixelradius);
    else
      drawgridnorth(sv, wincenterX, wincenterY, pixelradius);
  }

  // Draw all the stars which are in the hemisphere of the chart facing
  //  away from the screen (e.g. the southern hemisphere for
  //  ArrayRules.ChartOrientation.getTheta() > 0)
  for (unsigned int i = 0; i < Array.size(); i++) {
    starZ = Array[i].GetStarXYZ().getZ() - ArrayRules.ChartLocation.getZ();
    if (starZ * ChartDec < 0.0)
      Array[i].Display(ArrayRules, sv);
  }

  // Draw the chart equatorial plane and crosshairs
  if (ArrayRules.ChartGrid) 
    drawgridequator(sv, wincenterX, wincenterY, pixelradius);

  // Draw all the stars which are in the hemisphere of the chart facing
  //  toward the screen (e.g. the northern hemisphere for
  //  ArrayRules.ChartOrientation.getTheta() > 0)
  for (unsigned int i = 0; i < Array.size(); i++) {
    starZ = Array[i].GetStarXYZ().getZ() - ArrayRules.ChartLocation.getZ();
    if (starZ * ChartDec >= 0.0)
      Array[i].Display(ArrayRules, sv);
  }

  // draw the hemisphere (N/S) of the grid facing TOWARD the screen
  if (ArrayRules.ChartGrid) {
    if (ChartDec < 0.0)
      drawgridsouth(sv, wincenterX, wincenterY, pixelradius);
    else
      drawgridnorth(sv, wincenterX, wincenterY, pixelradius);
  }

  // put in a legend, if desired
  if (ArrayRules.ChartLegend) drawlegend(sv);

  return;
}


// public function to plot the stars in the StarArray onto a
//  Hertzsprung-Russell diagram

void StarArray::Diagram(StarViewer *sv, double brightmag, double dimmag) const
{
  int width, height;
  double magscale = dimmag - brightmag;
  int scaling = (magscale < 0) ? -1 : 1;
  char *maglabel = "MAGNITUDE", *classes = "OBAFGKM";

  // if dimmag < brightmag, this will flip the vertical axis
  if (magscale < 0) 
    { double tmp = brightmag; brightmag = dimmag; dimmag = tmp; }

  // Determine size of HR diagram, in pixels, leaving a margin for axis labels
  width = sv->width() - 80;
  height = sv->height() - 60;

  // clear pixmap with the background color (defined in "star.h")
  sv->fill(BACKGROUND);

  // draw lines indicating the bright and dim magnitude cutoffs
  if (ArrayRules.ChartBrightestMagnitude > brightmag
      && ArrayRules.ChartBrightestMagnitude < dimmag) {
    int y = (magscale != 0.0) ? (int)((ArrayRules.ChartBrightestMagnitude - 
				       brightmag) * height / magscale) : 0;
    if (magscale < 0) y += height;
    sv->setcolor(NEGATIVE);
    sv->drawline(60, y, width + 80, y);
    sv->drawline(60, y + 5 * scaling, 64, y);
    sv->drawline(60, y + 4 * scaling, 63, y);
  }
  if (ArrayRules.ChartDimmestMagnitude < dimmag
      && ArrayRules.ChartDimmestMagnitude > brightmag) {
    int y = (magscale != 0.0) ? (int)((ArrayRules.ChartDimmestMagnitude - 
				       brightmag) * height / magscale) : 0;
    if (magscale < 0) y += height;
    sv->setcolor(POSITIVE);
    sv->drawline(60, y, width + 80, y);
    sv->drawline(60, y - 5 * scaling, 64, y);
    sv->drawline(60, y - 4 * scaling, 63, y);
  }

  // draw the chart axes
  sv->setcolor(LABEL_COLOR);
  sv->drawline(60, 0, 60, height);
  sv->drawline(60, height, width + 80, height);

  // Plot all the stars
  for (unsigned int i = 0; i < Array.size(); i++) {
    int xposn, yposn;
    double mag = Array[i].GetStarMagnitude();
    double numclass = SpecHash(Array[i].GetStarClass().getCoarseType());
    double numsubclass = Array[i].GetStarClass().getSubType();

    // don't plot stars which would be off-scale:
    if (mag > dimmag || mag < brightmag) continue;

    // don't plot non-stellar objects, white dwarfs, or objects without a
    //  known spectrum subclass (e.g. the '8' in "M8 V"):
    if (numclass >= 7.0 || numsubclass >= 10.0) continue; 

    numclass += 0.1 * numsubclass;
    xposn = (int)(numclass / 7.0 * width);
    yposn = (magscale != 0.0) ? (int)((mag - brightmag) * height / magscale):0;
    if (magscale < 0) yposn += height;

    // leave a margin for axis labels, hence the "+ 60"
    Array[i].Draw(ArrayRules, sv, xposn + 60, yposn);
  }

  // Draw horizontal axis labels.  Start at x position 56 instead of 60 to 
  //  account for width of letters, so they look better centered.
  for (unsigned int i = 0; i < 7; i++) {
    sv->setcolor(ArrayRules.StarClasses[i] ? LABEL_COLOR : 0x606060);
    sv->drawtext(classes[i], 56 + width * (2*i + 1) / 14, height + 20, 12);
  }

  // Draw horizontal axis tickmarks.
  sv->setcolor(LABEL_COLOR);
  for (unsigned int i = 1; i <= 7; i++)
    sv->drawline(60 + i * width / 7, height - 2,
		 60 + i * width / 7, height + 2);

  // Draw vertical axis labels and tickmarks.
  for (int i = (int)floor(brightmag) + 1; i < dimmag; i++) {
    int y = (magscale != 0.0) ? (int)((i - brightmag) * height / magscale) : 0;
    if (magscale < 0) y += height;
    if (i % 5)
      sv->drawline(58, y, 62, y);
    else
      sv->drawline(56, y, 64, y);
    if (magscale * scaling < 10 || ! (i % 5)) {
      char magtext[5];
      snprintf(magtext, 5, "%+3d", i);
      magtext[4] = 0;
      sv->drawtext(magtext, 30, y + 4, 12);
    }
  }

  // Draw axis titles.
  sv->setcolor(BOLD_COLOR);
  sv->drawtext("SPECTRAL CLASS", 60 + width / 2 - 50, height + 50, 12);

  if (magscale == 0.0) sv->setcolor(ERROR_COLOR);
  for (unsigned int posn = 0; posn < strlen(maglabel); posn++) {
    int y = height / 2 - 60 + 16 * posn;
    sv->drawtext(maglabel[posn], 10, y, 12);
  }

  return;
}
