/* Copyright (C) 2000-2004  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: Events.pmod,v 1.1.1.1 2005/02/23 14:47:21 cvs Exp $
 */

constant cvs_version="$Id: Events.pmod,v 1.1.1.1 2005/02/23 14:47:21 cvs Exp $";

#include <events.h>

//#define EVENT_DEBUG

#ifdef EVENT_DEBUG
#define DEBUG_EVENT(s,args...)  write(s+"\n",args)
#else
#define DEBUG_EVENT(s,args...) 
#endif

static mapping event_desc = ([ 
    EVENT_ENTER_INVENTORY: "enter-inventory",
    EVENT_LEAVE_INVENTORY: "leave-inventory",
    EVENT_UPLOAD: "upload",
    EVENT_DOWNLOAD: "download",
    EVENT_ATTRIBUTES_CHANGE: "attributes-change",
    EVENT_MOVE: "move",
    EVENT_SAY: "say",
    EVENT_TELL: "tell",
    EVENT_LOGIN: "login",
    EVENT_LOGOUT: "logout",
    EVENT_ATTRIBUTES_LOCK: "lock-attribute",
    EVENT_EXECUTE: "execute",
    EVENT_REGISTER_FACTORY: "register-factory",
    EVENT_REGISTER_MODULE: "register-module",
    EVENT_ATTRIBUTES_ACQUIRE: "acquire-attributes",
    EVENT_ATTRIBUTES_QUERY: "query-attributes",
    EVENT_REGISTER_ATTRIBUTE: "register-attribute",
    EVENT_DELETE: "delete",
    EVENT_ADD_MEMBER: "add-member",
    EVENT_REMOVE_MEMBER: "remove-member",
    EVENT_GRP_ADD_PERMISSION: "add-permissions-for-group",
    EVENT_USER_CHANGE_PW: "user-change-password",
    EVENT_SANCTION: "sanction",
    EVENT_SANCTION_META: "meta-sanction",
    EVENT_ARRANGE_OBJECT: "arrange-object",
    EVENT_ANNOTATE: "annotate",
    EVENT_LISTEN_EVENT: "listen-event",
    EVENT_IGNORE_EVENT: "ignore-event",
    EVENT_GET_INVENTORY: "get_inventory",
    EVENT_DUPLICATE: "duplicate",
    EVENT_REQ_SAVE: "save",
    EVENT_GRP_ADDMUTUAL: "group-add-mutual",
    EVENT_STATUS_CHANGED: "status-changed",
    EVENT_SAVE_OBJECT: "save",
    EVENT_REMOVE_ANNOTATION: "remove-annotation",
    EVENT_DOWNLOAD_FINISHED: "download-finished",
    EVENT_DB_REGISTER: "database-register",
    EVENT_DB_UNREGISTER: "database-unregister",
    EVENT_DB_QUERY: "database-query",
    EVENT_SERVER_SHUTDOWN: "server-shutdown",
    EVENT_CHANGE_QUOTA: "quota-change",
    EVENTS_SERVER: "SERVER",
    EVENTS_USER: "USER",
    EVENTS_MODULES: "MODULES",
    EVENTS_MONITORED: "MONITORED",
    ]);

class Listener {
    static int            event_id;
    static object        event_obj;
    static int         event_phase;
    static function event_callback;
    static object        listening;

    void set(int eid, int phase, object obj, function callback, object|void l) {
	event_id = eid;
	event_phase = phase;
	event_callback = callback;
	event_obj = obj;	
	listening = l;
    }

    void create(int eid, int phase, object obj,function callback,object|void l) {
	set(eid, phase, obj, callback, l);
    }

    int get_event() {
	return event_id;
    }
    object get_object() {
	return event_obj;
    }
    int get_phase() {
	return event_phase;
    }
    function get_callback() {
	return event_callback;
    }
    object get_listening() {
        return listening;
    }

    string describe() { 
	return "Listener("+event_obj->get_identifier()+
	    ", phase="+ (event_phase==PHASE_NOTIFY?"notify":"block")+","+
	    translate_eid(event_id)+")"; 
    }

    mixed `[] (mixed index) {
	switch ( index ) {
	case 0:
	    return event_callback;
	case 1:
	    return event_id;
	case 2:
	    return event_phase;
	case 3:
	    return event_obj;
	default:
	    return "unknown";
	}
    }
    void notify(int eid, mixed args) {
      if ( functionp(event_callback) )
	event_callback(eid, @args);
    }

    mixed `==(object l) {
      if ( !objectp(l) )
	return 0;

      return l->event_obj == event_obj &&
	l->event_id == event_id &&
	l->event_phase == event_phase &&
	l->event_callback == event_callback;
    }
}

class Event {

    static int            event_id;
    static array(object) listeners; 

    void create(int id) {
	event_id = id;
	listeners = ({ });
    }

    void add_listener(Listener l) {
	// already got such a listener
	foreach ( listeners, object listen )
	    if ( listen == l )
		return;

	listeners += ({ l });
    }

    void remove_listener(Listener l) {
	listeners -= ({ l });
    }

    array(object) get_listeners() {
	return listeners; 
    }

    void set_event(int id) {
	event_id = id;
    }

    int get_event() {
	return event_id;
    }

    static void notify_listener(object l, mixed args) {
	l->notify(event_id, args);
    }

    void run_event(int phase, mixed args) {
	listeners -= ({ 0 });
	foreach( listeners, object l ) {
	    if ( l->get_phase() == phase ) {
		notify_listener(l, args);
	    }
	}
    }

    string describe() { 
      int ilisten, idead;
      foreach(listeners, object l) {
	if ( !objectp(l) )
	  idead++;
	else if ( functionp(l->get_callback) && !functionp(l->get_callback()) )
	  idead++;
	else
	  ilisten++;
      }
      
      return "Event("+event_id+","+translate_eid(event_id)+", "+ilisten+ 
	"Listeners, "+idead+" dead)";
    }
}



/**
 * Returns a string description for a given event.
 *
 * @param int eid - event id (bits)  
 * @return string description
 */
string translate_eid(int eid) 
{
    string description = "";
    array index = indices(event_desc);
    
    foreach(index, int id) {
	if ( (id & eid) == id ) 
	    description += event_desc[id] + " ";
    }
    return description;
}

/**
 * Split a given integer bit array into segments. The EVENTS_MONITORED bit is 
 * preserved and set for each element of the resulting array.
 *
 * @param int event - the event to split
 * @return array of single event-id-bits.
 */
array(int) split_events(int event) 
{
  // second events
  int events_second = event & EVENTS_SECOND;

  array(int) events = ({ });
  int monitor = (event & EVENTS_MONITORED);
  
  for ( int i = 0; i <= 27; i++ ) {
    if ( event & (1<<i) ) 
      events += ({ (1<<i) | monitor | events_second });
  }
  return events;
}
