/*
 *
 * s y s t e m . c				-- System relative primitives
 *
 * Copyright  1994-2001 Erick Gallesio - I3S-CNRS/ESSI <eg@unice.fr>
 * 
 *
 * Permission to use, copy, modify, distribute,and license this
 * software and its documentation for any purpose is hereby granted,
 * provided that existing copyright notices are retained in all
 * copies and that this notice is included verbatim in any
 * distributions.  No written agreement, license, or royalty fee is
 * required for any of the authorized uses.
 * This software is provided ``AS IS'' without express or implied
 * warranty.
 *
 *           Author: Erick Gallesio [eg@kaolin.unice.fr]
 *    Creation date: 29-Mar-1994 10:57
 * Last file update: 24-Jan-2001 09:42 (eg)
 */

#ifndef WIN32
#  include <unistd.h>
#  include <pwd.h>
#else
#  include <windows.h>
#  include <io.h>
#  define F_OK   00
#  define X_OK   01
#  define W_OK   02
#  define R_OK   04
#endif

#include <sys/types.h>
#include <sys/stat.h>

#ifdef WIN32
   /* One of above files includes <stdarg.h> with BC++  (and stdarg and 
    * vararg are not compatible 
    */
#  undef __STDARG_H 

#  ifdef _MSC_VER
#  include <direct.h>
#  include <process.h>
#  include <sys/stat.h>
#  define S_ISDIR(mode) ((mode & _S_IFMT) == _S_IFDIR)
#  define S_ISREG(mode) ((mode & _S_IFMT) == _S_IFREG)
#  else
#     ifdef BCC
        /* Borland defines the opendir/readdir/closedir functions. Use them. */
#       include <dirent.h>
#     endif
#  endif
#else
#  include <dirent.h>
#endif

#include <time.h> 
#include "stklos.h"



/******************************************************************************
 *
 * Utilities
 *
 ******************************************************************************/
static void error_bad_path(SCM path)
{
  STk_error("~S is a bad pathname", path);
}

static void error_bad_string(SCM path)
{
  STk_error("~S is a bad string", path);
}


static SCM my_access(SCM path, int mode)
{
  if (!STRINGP(path)) error_bad_path(path);
  return MAKE_BOOLEAN(access(STRING_CHARS(path), mode) == 0);
}



int STk_dirp(const char *path)
{
  struct stat buf;
  
  if (stat(path, &buf) >= 0) 
    return S_ISDIR(buf.st_mode);
  return FALSE;
}


#ifdef FIXME
//EG: #define MAXLINK 50	/* Number max of link before declaring we have a loop */
//EG: 
//EG: SCM STk_resolve_link(char *path, int count)
//EG: {
//EG: #ifdef WIN32
//EG:   return STk_internal_expand_file_name(path);
//EG: #else
//EG:   char link[MAX_PATH_LENGTH], dst[MAX_PATH_LENGTH], *s, *d=dst;
//EG:   int n;
//EG:   SCM p;
//EG:   
//EG:   p  = STk_internal_expand_file_name(path);
//EG:   
//EG:   for (s=CHARS(p)+1, *d++='/' ;       ; s++, d++) {
//EG:     switch (*s) {
//EG:       case '\0':
//EG:       case '/' : *d = '\0';
//EG: 	if ((n=readlink(dst, link, MAX_PATH_LENGTH-1)) > 0) {
//EG: 	  link[n] = '\0';
//EG: 	  if (link[0] == '/') 
//EG: 	    /* link is absolute */
//EG: 	    d = dst;
//EG: 	  else {
//EG: 	    /* relative link. Delete last item */
//EG: 	    while (*--d != '/') {
//EG: 	    }
//EG: 	    d += 1;
//EG: 	  }
//EG: 		   
//EG: 	  /* d points the place where the link must be placed */
//EG: 	  if (d - dst + strlen(link) + strlen(s) < MAX_PATH_LENGTH - 1) {
//EG: 	    /* we have enough room */
//EG: 	    sprintf(d, "%s%s", link, s); 
//EG: 	    /* Recurse. Be careful for loops (a->b and b->a) */
//EG: 	    if (count < MAXLINK) 
//EG: 	      return STk_resolve_link(dst, count+1);
//EG: 	  }
//EG: 	  return STk_false;
//EG: 	}
//EG: 	else {
//EG: 	  if (errno != EINVAL) 
//EG: 	    /* EINVAL = file is not a symlink (i.e. it's a true error) */
//EG: 	    return STk_false;
//EG: 	  else
//EG: 	    if (*s) *d = '/'; 
//EG: 	    else return STk_makestring(dst);		       
//EG: 	}
//EG:       default:   *d = *s;
//EG:     }
//EG:   }
//EG: #endif
//EG: }
//EG: 
//EG:    
//EG: /*
//EG:  *----------------------------------------------------------------------
//EG:  *
//EG:  * fileglob --
//EG:  *      *****							        ******
//EG:  * 	***** This function is an adaptation of the Tcl function DoGlob ******
//EG:  *      ***** Adaptated to use true lists rather than string as in Tcl  ******
//EG:  *      *****							        ******
//EG:  *	
//EG:  *
//EG:  *      This recursive procedure forms the heart of the globbing
//EG:  *      code.  It performs a depth-first traversal of the tree
//EG:  *      given by the path name to be globbed.
//EG:  *
//EG:  * Results:
//EG:  *      The return value is a standard Tcl result indicating whether
//EG:  *      an error occurred in globbing.  After a normal return the
//EG:  *      result in interp will be set to hold all of the file names
//EG:  *      given by the dir and rem arguments.  After an error the
//EG:  *      result in interp will hold an error message.
//EG:  *
//EG:  * Side effects:
//EG:  *      None.
//EG:  *
//EG:  *----------------------------------------------------------------------
//EG:  */
//EG: 
//EG: static SCM fileglob(char *dir, char *rem, SCM result)
//EG: /* dir: Name of a directory at which to start glob expansion.  This name
//EG:  * is fixed: it doesn't contain any globbing chars. 
//EG:  * rem: Path to glob-expand.
//EG:  */
//EG: {
//EG:   /*
//EG:    * When this procedure is entered, the name to be globbed may
//EG:    * already have been partly expanded by ancestor invocations of
//EG:    * fileglob.  The part that's already been expanded is in "dir"
//EG:    * (this may initially be empty), and the part still to expand
//EG:    * is in "rem".  This procedure expands "rem" one level, making
//EG:    * recursive calls to itself if there's still more stuff left
//EG:    * in the remainder.
//EG:    */
//EG:   
//EG:   Tcl_DString newName;                /* Holds new name consisting of
//EG: 				       * dir plus the first part of rem. */
//EG:   register char *p;
//EG:   register char c;
//EG:   char *openBrace, *closeBrace, *name, *dirName;
//EG:   int gotSpecial, baseLength;
//EG:   struct stat statBuf;
//EG: 
//EG:   /*
//EG:    * Make sure that the directory part of the name really is a
//EG:    * directory.  If the directory name is "", use the name "."
//EG:    * instead, because some UNIX systems don't treat "" like "."
//EG:    * automatically. Keep the "" for use in generating file names,
//EG:    * otherwise "glob foo.c" would return "./foo.c".
//EG:    */
//EG:   
//EG:   dirName = (*dir == '\0') ? ".": dir;
//EG:   if ((stat(dirName, &statBuf) != 0) || !S_ISDIR(statBuf.st_mode))
//EG:     return result;
//EG: 
//EG:   Tcl_DStringInit(&newName);
//EG: 
//EG:   /*
//EG:    * First, find the end of the next element in rem, checking
//EG:    * along the way for special globbing characters.
//EG:    */
//EG:   
//EG:   gotSpecial = 0;
//EG:   openBrace = closeBrace = NULL;
//EG:   for (p = rem; ; p++) {
//EG:     c = *p;
//EG:     if ((c == '\0') || ((openBrace == NULL) && (c == '/'))) break;
//EG:     if ((c == '{')  && (openBrace == NULL)) openBrace = p;
//EG:     if ((c == '}')  && (openBrace != NULL) && (closeBrace == NULL)) closeBrace = p;
//EG:     if ((c == '*') || (c == '[') || (c == '\\') || (c == '?')) gotSpecial = 1;
//EG:   }
//EG: 
//EG:   /*
//EG:    * If there is an open brace in the argument, then make a recursive
//EG:    * call for each element between the braces.  In this case, the
//EG:    * recursive call to fileglob uses the same "dir" that we got.
//EG:    * If there are several brace-pairs in a single name, we just handle
//EG:    * one here, and the others will be handled in recursive calls.
//EG:    */
//EG: 
//EG:   if (openBrace != NULL) {
//EG:     char *element;
//EG:     
//EG:     if (closeBrace == NULL) {
//EG:       Tcl_DStringFree(&newName);      
//EG:       Err("unmatched open-brace in file name", STk_nil);
//EG:     }
//EG: 
//EG:     Tcl_DStringAppend(&newName, rem, openBrace-rem);
//EG:     baseLength = newName.length;
//EG:     for (p = openBrace; *p != '}'; ) {
//EG:       element = p+1;
//EG:       for (p = element; ((*p != '}') && (*p != ',')); p++) {}
//EG:       Tcl_DStringAppend(&newName, element, p-element);
//EG:       Tcl_DStringAppend(&newName, closeBrace+1, -1);
//EG:       result = fileglob(dir, newName.string, result);
//EG:       newName.length = baseLength;
//EG:     }
//EG:     goto done;
//EG:   }
//EG: 
//EG:   /*
//EG:    * Start building up the next-level name with dir plus a slash if
//EG:    * needed to separate it from the next file name.
//EG:    */
//EG: 
//EG:   Tcl_DStringAppend(&newName, dir, -1);
//EG:   if ((dir[0] != 0) && (newName.string[newName.length-1] != '/')) {
//EG:     Tcl_DStringAppend(&newName, SDIRSEP, 1);
//EG:   }
//EG:   baseLength = newName.length;
//EG: 
//EG:   /*
//EG:    * If there were any pattern-matching characters, then scan through
//EG:    * the directory to find all the matching names.
//EG:    */
//EG:   if (gotSpecial) {
//EG: #ifdef _MSC_VER
//EG:     WIN32_FIND_DATA wfd; /* VC++ support from Caleb Deupree <cdeupree@erinet.com> */
//EG:     HANDLE handle;
//EG:     Tcl_DString msvcname;
//EG:     char savedChar;
//EG:     BOOL bFound = TRUE;
//EG: 
//EG:     Tcl_DStringInit(&msvcname);
//EG:     Tcl_DStringAppend(&msvcname, dirName, -1);
//EG:     Tcl_DStringAppend(&msvcname, SDIRSEP, 1);
//EG:     Tcl_DStringAppend(&msvcname, "*\0", -1);
//EG:        
//EG:     handle = FindFirstFile(Tcl_DStringValue(&msvcname), &wfd);
//EG:     if (handle == INVALID_HANDLE_VALUE) {
//EG:       Tcl_DStringFree(&msvcname);
//EG:       Err("Cannot find files, error = ", STk_makestring((char *) GetLastError()));
//EG:     }
//EG:    
//EG:     savedChar = *p;
//EG:     *p = 0;
//EG:     
//EG:     while (bFound) {
//EG:       /*
//EG:        * Don't match names starting with "." unless the "." is
//EG:        * present in the pattern.
//EG:        */
//EG:       if ((wfd.cFileName == '.') && (*rem != '.')) continue;
//EG:       
//EG:       if (Tcl_StringMatch(wfd.cFileName, rem)) {
//EG: 	newName.length = baseLength;
//EG: 	Tcl_DStringAppend(&newName, wfd.cFileName, -1);
//EG: 	if (savedChar == 0)
//EG: 	  result = Cons(STk_makestring(newName.string), result);
//EG: 	else {
//EG: 	  result = fileglob(newName.string, p+1, result);
//EG: 	  if (result != TCL_OK) break;
//EG: 	}
//EG:       }
//EG:       bFound = FindNextFile(handle, &wfd);
//EG:     }
//EG:     FindClose(handle);
//EG:     Tcl_DStringFree(&msvcname);
//EG:     *p = savedChar;
//EG:     goto done;
//EG: #else
//EG:     DIR *d;
//EG:     struct dirent *entryPtr;
//EG:     char savedChar;
//EG: 
//EG:     d = opendir(dirName);
//EG:     if (d == NULL) {
//EG:       Tcl_DStringFree(&newName);
//EG:       Err("cannot read directory", STk_makestring(dirName));
//EG:     }
//EG: 
//EG:     /*
//EG:      * Temporarily store a null into rem so that the pattern string
//EG:      * is now null-terminated.
//EG:      */
//EG: 
//EG:     savedChar = *p;
//EG:     *p = 0;
//EG: 
//EG:     while (1) {
//EG:       entryPtr = readdir(d);
//EG:       if (entryPtr == NULL) break;
//EG:       /*
//EG:        * Don't match names starting with "." unless the "." is
//EG:        * present in the pattern.
//EG:        */
//EG:       if ((*entryPtr->d_name == '.') && (*rem != '.')) continue;
//EG: 
//EG:       if (Tcl_StringMatch(entryPtr->d_name, rem)) {
//EG: 	newName.length = baseLength;
//EG: 	Tcl_DStringAppend(&newName, entryPtr->d_name, -1);
//EG: 	if (savedChar == 0)
//EG: 	  result = Cons(STk_makestring(newName.string), result);
//EG: 	else {
//EG: 	  result = fileglob(newName.string, p+1, result);
//EG: 	  if (result != TCL_OK) break;
//EG: 	}
//EG:       }
//EG:       
//EG:     }
//EG:     closedir(d);
//EG:     *p = savedChar;
//EG:     goto done;
//EG: #endif
//EG:   }
//EG: 
//EG:   /*
//EG:    * The current element is a simple one with no fancy features.  Add
//EG:    * it to the new name.  If there are more elements still to come,
//EG:    * then recurse to process them.
//EG:    */
//EG:   
//EG:   Tcl_DStringAppend(&newName, rem, p-rem);
//EG:   if (*p != 0) {
//EG:     result = fileglob(newName.string, p+1, result);
//EG:     goto done;
//EG:   }
//EG: 
//EG:   /*
//EG:    * There are no more elements in the pattern.  Check to be sure the
//EG:    * file actually exists, then add its name to the list being formed
//EG:    * in main_interp-result.
//EG:    */
//EG: 
//EG:   name = newName.string;
//EG:   if (*name == 0) name = ".";
//EG:   if (access(name, F_OK) != 0) goto done;
//EG:   result = Cons(STk_makestring(name), result);
//EG: done:
//EG:   Tcl_DStringFree(&newName);
//EG:   return result;
//EG: }
//EG: 
#endif 

SCM STk_int_expand_file_name(char *s)
{
  return STk_Cstring2string(s);			//FIXME
#ifdef FIXME
//  char expanded[2 * MAX_PATH_LENGTH], abs[2 * MAX_PATH_LENGTH];  
//   /* Warning: absolute makes no control about path overflow. Hence the "2 *" */
// 
//   absolute(tilde_expand(s, expanded), abs);
//   return STk_makestring(abs);
#endif
}


#ifdef FIXME
//EG: void STk_whence(char *exec, char *path)
//EG: {
//EG:   char *p, *q, dir[MAX_PATH_LENGTH];
//EG:   struct stat buf;
//EG:  
//EG:   if (ISABSOLUTE(exec)) {
//EG:     strncpy(path, exec, MAX_PATH_LENGTH);
//EG:     return;
//EG:   }
//EG:   
//EG:   /* the executable path may be specified by relative path from the cwd. */
//EG:   /* Patch suggested by Shiro Kawai <shiro@squareusa.com> */
//EG:   if (strchr(exec, DIRSEP) != NULL) {
//EG:     getcwd(dir, MAX_PATH_LENGTH);
//EG:     sprintf(dir + strlen(dir), "%c%s", DIRSEP, exec);
//EG:     absolute(dir, path);
//EG:     return;
//EG:   }
//EG: 
//EG: #ifdef FREEBSD 
//EG:   /* I don't understand why this is needed */
//EG:   if (access(path, X_OK) == 0) {
//EG:     stat(path, &buf);
//EG:     if (!S_ISDIR(buf.st_mode)) return;
//EG:   }  
//EG: #endif
//EG: 
//EG:   p = getenv("PATH");
//EG:   if (p == NULL) {
//EG:     p = "/bin:/usr/bin";
//EG:   }
//EG: 
//EG:   while (*p) {
//EG:     /* Copy the stuck of path in dir */
//EG:     for (q = dir; *p && *p != PATHSEP; p++, q++) *q = *p;
//EG:     *q = '\000';
//EG: 
//EG:     if (!*dir) { 
//EG:       /* patch suggested by Erik Ostrom <eostrom@vesuvius.ccs.neu.edu> */
//EG:       getcwd(path, MAX_PATH_LENGTH);
//EG:       sprintf(path + strlen(path), "%c%s", DIRSEP, exec);
//EG:     }
//EG:     else
//EG:       sprintf(path, "%s%c%s", dir, DIRSEP, exec);
//EG: 
//EG:     sprintf(path, "%s%c%s", dir, DIRSEP, exec);
//EG:     if (access(path, X_OK) == 0) {
//EG:       stat(path, &buf);
//EG:       if (!S_ISDIR(buf.st_mode)) return;
//EG:     }
//EG: 	 
//EG:     /* Try next path */
//EG:     if (*p) p++;
//EG:   }
//EG:   /* Not found. Set path to "" */
//EG:   path[0] = '\0';
//EG: }
#endif

/******************************************************************************
 *
 * Primitives
 *
 ******************************************************************************/

#ifdef FIXME 
//EG: PRIMITIVE STk_expand_file_name(SCM s)
//EG: {
//EG:   if (NSTRINGP(s)) Err("expand-file-name: bad string", s);
//EG:   return STk_internal_expand_file_name(CHARS(s));
//EG: }
//EG: 
//EG: PRIMITIVE STk_canonical_path(SCM str)
//EG: {
//EG:   if (NSTRINGP(str)) Err("canonical-path: not a string", str);
//EG: #ifdef WIN32
//EG:   return str;
//EG: #else
//EG:   return STk_resolve_link(CHARS(str), 0);
//EG: #endif
//EG: }
#endif

/*
<doc EXT getcwd
 * (getcwd)
 *
 * Returns a string containing the current working directory.
doc>
*/
DEFINE_PRIMITIVE("getcwd", getcwd, subr0, (void))
{
  char buf[MAX_PATH_LENGTH], *s;
  SCM z;

  ENTER_PRIMITIVE(getcwd);

  s = getcwd(buf, MAX_PATH_LENGTH);
  if (!s) STk_error("cannot determine current directory");
  z = STk_Cstring2string(buf);

  return z;
}


/*
<doc EXT chdir
 * (chdir dir)
 *
 * Changes the current directory to the directory given in string |dir|.
doc>
*/
DEFINE_PRIMITIVE("chdir", chdir, subr1, (SCM s))
{
  ENTER_PRIMITIVE(chdir);

  if (!STRINGP(s)) error_bad_path(s);
  
  if (chdir(STRING_CHARS(STk_int_expand_file_name(STRING_CHARS(s)))))
    STk_error("cannot change directory to ~S", s);
 
  return STk_void;
}


/*
<doc EXT getpid
 * (getpid)
 *
 * Returns the system process number of the current program (i.e. the
 * Unix @i{pid}) as an integer.
doc>
*/
DEFINE_PRIMITIVE("getpid", getpid, subr0, (void))
{
  return (MAKE_INT((int) getpid()));
}


/*
<doc EXT system
 * (system string)
 *
 * Sends the given |string| to the system shell |/bin/sh|. The result of
 * |system| is the integer status code the shell returns.
doc>
*/
DEFINE_PRIMITIVE("system", system, subr1, (SCM com))
{
  ENTER_PRIMITIVE(system);
  
  if (!STRINGP(com)) error_bad_string(com);
  return MAKE_INT(system(STRING_CHARS(com)));
}

/*
<doc EXT file-is-directory? file-is-regular? file-is-writable? file-is-readable? file-is-executable? file-exists?
 * (file-is-directory?  string)
 * (file-is-regular?    string)
 * (file-is-readable?   string)
 * (file-is-writable?   string)
 * (file-is-executable? string)
 * (file-exists?        string)
 *
 * Returns |#t| if the predicate is true for the path name given in
 * |string|; returns |#f| otherwise (or if |string| denotes a file
 * which does not exist).
doc>
 */
DEFINE_PRIMITIVE("file-is-directory?", file_is_directoryp, subr1, (SCM f))
{
  struct stat info;

  ENTER_PRIMITIVE(file_is_directoryp);

  if (!STRINGP(f)) error_bad_path(f);
  if (stat(STRING_CHARS(f), &info) != 0) return STk_false;

  return MAKE_BOOLEAN((S_ISDIR(info.st_mode)));
}


DEFINE_PRIMITIVE("file-is-regular?", file_is_regularp, subr1, (SCM f))
{
  struct stat info;

  ENTER_PRIMITIVE(file_is_regularp);

  if (!STRINGP(f)) error_bad_path(f);
  if (stat(STRING_CHARS(f), &info) != 0) return STk_false;

  return MAKE_BOOLEAN((S_ISREG(info.st_mode)));
}


DEFINE_PRIMITIVE("file-is-readable?", file_is_readablep, subr1, (SCM f))
{
  ENTER_PRIMITIVE(file_is_readablep);
  return my_access(f, R_OK);
}


DEFINE_PRIMITIVE("file-is-writable?", file_is_writablep, subr1, (SCM f))
{
  ENTER_PRIMITIVE(file_is_writablep);
  return my_access(f, W_OK);
}


DEFINE_PRIMITIVE("file-is-executable?", file_is_executablep, subr1, (SCM f))
{
  ENTER_PRIMITIVE(file_is_executablep);
  return my_access(f, X_OK);
}


DEFINE_PRIMITIVE("file-exists?", file_existsp, subr1, (SCM f))
{
  ENTER_PRIMITIVE(file_existsp)
  return my_access(f, F_OK);
}

#ifdef FIXME
//EG: PRIMITIVE STk_file_glob(SCM l, int len) /* len is unused here */
//EG: {
//EG:   SCM res = STk_nil;
//EG:   char s[2*MAX_PATH_LENGTH];
//EG:   
//EG:   for ( ; NNULLP(l); l = CDR(l)) {
//EG:     if (NSTRINGP(CAR(l))) Err("glob: bad string", CAR(l));
//EG:     
//EG:     tilde_expand(CHARS(CAR(l)), s);
//EG: 
//EG:     res = STk_append2(res, (ISDIRSEP(*s)) ? fileglob(SDIRSEP, s+1, STk_nil) :
//EG: 		                            fileglob("", s, STk_nil));
//EG:   }
//EG:   return res;
//EG: }
#endif


/*
<doc EXT remove-file
 * (remove-file string)
 *
 * Removes the file whose path name is given in |string|.
 * The result of |remove-file| is @emph{void}.
doc>
*/
DEFINE_PRIMITIVE("remove-file", remove_file, subr1, (SCM filename))
{
  ENTER_PRIMITIVE(remove_file);
  
  if (!STRINGP(filename)) error_bad_string(filename);
  if (remove(STRING_CHARS(filename)) != 0)
    STk_error("cannot remove ~S", filename);
  return STk_void;
}
 

/*
<doc EXT rename-file
 * (rename-file string1 string2)
 *
 * Renames the file whose path-name is |string1| to a file whose path-name is
 * |string2|. The result of |rename-file| is @emph{void}.
doc>
*/
DEFINE_PRIMITIVE("rename-file", rename_file, subr2, (SCM filename1, SCM filename2))
{
  ENTER_PRIMITIVE(rename_file);
  
  if (!STRINGP(filename1)) error_bad_string(filename1);
  if (!STRINGP(filename2)) error_bad_string(filename2);
  if (rename(STRING_CHARS(filename1), STRING_CHARS(filename2)) != 0)
    STk_error("cannot rename file ~S in ~S", filename1, filename2);
  return STk_void;
}
 
/*
<doc EXT temporary-file-name
 * (temporary-file-name)
 *
 * Generates a unique temporary file name. The value returned by
 * |temporary-file-name| is the newly generated name of |#f|
 * if a unique name cannot be generated.
doc>
*/
DEFINE_PRIMITIVE("temporary-file-name", tmp_file, subr0, (void))
{
  char buff[MAX_PATH_LENGTH], *s;
  
  s = tmpnam(buff);
  return s ? STk_Cstring2string(s) : STk_false;
}


/*
<doc EXT exit
 * (exit) 
 * (exit ret-code)
 *
 * Exits the program with the specified integer return code. If |ret-code|
 * is omitted, the program terminates with a return code of 0.
doc>
*/
DEFINE_PRIMITIVE("exit", quit, subr01, (SCM retcode))
{
  long ret = 0;
  
  ENTER_PRIMITIVE(quit);

  if (retcode) {
    ret = STk_integer_value(retcode);
    if (retcode == LONG_MIN) STk_error("bad return code ~S", retcode);
  }
#ifdef FIXME
//EG:  /* Execute all the terminal thunks of pending dynamic-wind */
//EG:  STk_unwind_all();
//EG:
//EG:  /* call user finalization code */
//EG:  STk_user_cleanup();
//EG:
//EG:#if defined(WIN32) && defined(USE_SOCKET)
//EG:  /* Unregister the interpreter from Winsock */
//EG:  WSACleanup();  
//EG:#endif
#endif
  exit(ret);
  return STk_void; /* never reached */
}


/*
<doc EXT machine-type
 * (machine-type)
 *
 * Returns a string identifying the kind of machine which is running the
 * program. The result string is of the form 
 * |[os-name]-[os-version]-[processor-type]|.
doc>
*/
DEFINE_PRIMITIVE("machine-type", machine_type, subr0, (void))
{
  return STk_Cstring2string(BUILD_MACHINE);
}


#ifdef FIXME
//EG: PRIMITIVE STk_random(SCM n)
//EG: {
//EG:   if (NEXACTP(n) || STk_negativep(n) == STk_true || STk_zerop(n) == STk_true)
//EG:     Err("random: bad number", n);
//EG:   return STk_modulo(STk_makeinteger(rand()), n);
//EG: }
//EG: 
//EG: PRIMITIVE STk_set_random_seed(SCM n)
//EG: {
//EG:   if (NEXACTP(n)) Err("set-random-seed!: bad number", n);
//EG:   srand((unsigned int) STk_integer_value_no_overflow(n));
//EG:   return STk_unsepecified;
//EG: }
//EG: 
//EG: #ifndef HZ
//EG: #define HZ 60.0
//EG: #endif
//EG: 
//EG: #ifdef CLOCKS_PER_SEC
//EG: #  define TIC CLOCKS_PER_SEC
//EG: #else 
//EG: #  define TIC HZ
//EG: #endif
//EG: 
//EG: PRIMITIVE STk_get_internal_info(void)
//EG: {
//EG:   SCM z = STk_makevect(7, STk_nil);
//EG:   long allocated, used, calls;
//EG: 
//EG:   /* The result is a vector which contains
//EG:    *	0 The total cpu used in ms
//EG:    *	1 The number of cells currently in use.
//EG:    *    2 Total number of allocated cells
//EG:    *	3 The number of cells used since the last call to get-internal-info
//EG:    *	4 Number of gc calls
//EG:    *    5 Total time used in the gc
//EG:    *	6 A boolean indicating if Tk is initialized
//EG:    */
//EG: 
//EG:   STk_gc_count_cells(&allocated, &used, &calls);
//EG: 
//EG:   VECT(z)[0] = STk_makenumber(STk_my_time());
//EG:   VECT(z)[1] = STk_makeinteger(used);
//EG:   VECT(z)[2] = STk_makeinteger(allocated);
//EG:   VECT(z)[3] = STk_makenumber((double) STk_alloc_cells);
//EG:   VECT(z)[4] = STk_makeinteger(calls);
//EG:   VECT(z)[5] = STk_makenumber((double) STk_total_gc_time);
//EG: #ifdef USE_TK
//EG:   VECT(z)[6] = Tk_initialized ? STk_true: STk_false;
//EG: #else
//EG:   VECT(z)[6] = STk_false;
//EG: #endif
//EG:   
//EG:   STk_alloc_cells = 0;
//EG:   return z;
//EG: }
#endif


/*
<doc EXT clock 
 * (clock)
 *
 * Returns an approximation of processor time, in milliseconds, used so far by the
 * program.
doc>
 */
DEFINE_PRIMITIVE("clock", clock, subr0, (void))
{
  return STk_double2real((double) clock() / CLOCKS_PER_SEC * 1000.0);
}

/*
<doc EXT current-time 
 * (current-time)
 *
 * Returns the time since the Epoch (that is 00:00:00 UTC, January 1, 1970), 
 * measured in seconds.
doc>
 */
DEFINE_PRIMITIVE("current-time", current_time, subr0, (void))
{
  return STk_long2integer(time(NULL));
}


/*
<doc EXT running-os
 * (running-os)
 * 
 * Returns the name of the underlying Operating System which is running 
 * the program. 
 * The value returned by |runnin-os| is a symbol. For now, this procedure 
 * returns either |unix| or |windows|.
doc>
*/
DEFINE_PRIMITIVE("running-os", running_os, subr0, (void))
{
#ifdef WIN32
  return STk_intern("windows");
#else
  return STk_intern("unix");
#endif
}



/*
<doc EXT getenv
 * (getenv str)
 *
 * Looks for the environment variable named |str| and returns its
 * value as a string, if it exists. Otherwise, |getenv| returns |#f|.
 * @lisp
 * (getenv "SHELL")   => "/bin/zsh"
 * @end lisp
doc>
 */
DEFINE_PRIMITIVE("getenv", getenv, subr1, (SCM str))
{
  char *tmp;
  
  ENTER_PRIMITIVE(getenv);
  if (!STRINGP(str)) error_bad_string(str);
  
  tmp = getenv(STRING_CHARS(str));
  return tmp ? STk_Cstring2string(tmp) : STk_false;
}

/*
<doc EXT setenv!
 * (setenv! var value)
 *
 * Sets the environment variable |var| to |value|. |Var| and
 * |value| must be strings. The result of |setenv!| is @emph{void}.
doc>
 */
DEFINE_PRIMITIVE("setenv!", setenv, subr2, (SCM var, SCM value))
{
  char *s;
  ENTER_PRIMITIVE(setenv);
  if (!STRINGP(var)) 		      STk_error("variable ~S is not a string", var);
  if (strchr(STRING_CHARS(var), '=')) STk_error("variable ~S contains a '='", var);
  if (!STRINGP(value)) 		      STk_error("value ~S is not a string", value);

  s = STk_must_malloc(strlen(STRING_CHARS(var))   + 
		      strlen(STRING_CHARS(value)) + 2); /* 2 because of '=' & \0 */
  sprintf(s, "%s=%s", STRING_CHARS(var), STRING_CHARS(value));
  putenv(s);
  return STk_void;
}


/*
 * Undocumented primitives
 *
 */

DEFINE_PRIMITIVE("%library-prefix", library_prefix, subr0, (void))
{
  return STk_Cstring2string(PREFIXDIR);
}



int STk_init_system(void)
{
  ADD_PRIMITIVE(clock);
  ADD_PRIMITIVE(current_time);
  ADD_PRIMITIVE(running_os);
  ADD_PRIMITIVE(getenv);
  ADD_PRIMITIVE(setenv);
  ADD_PRIMITIVE(library_prefix);

  ADD_PRIMITIVE(getcwd);
  ADD_PRIMITIVE(chdir);
  ADD_PRIMITIVE(getpid);
  ADD_PRIMITIVE(system);
    
  ADD_PRIMITIVE(file_is_directoryp);
  ADD_PRIMITIVE(file_is_regularp);
  ADD_PRIMITIVE(file_is_readablep);
  ADD_PRIMITIVE(file_is_writablep);
  ADD_PRIMITIVE(file_is_executablep);
  ADD_PRIMITIVE(file_existsp);
  ADD_PRIMITIVE(remove_file);
  ADD_PRIMITIVE(rename_file);
  ADD_PRIMITIVE(tmp_file);
  ADD_PRIMITIVE(quit);
  ADD_PRIMITIVE(machine_type);

  return TRUE;
}
