/*==================================================================
 * SwamiWavetbl.c - Swami Wavetable object (base class for drivers)
 *
 * Swami
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Swami homepage: http://swami.sourceforge.net
 *==================================================================*/
#include "config.h"

#include <stdio.h>
#include <glib.h>

#include "SwamiWavetbl.h"
#include "SwamiObject.h"

/* --- signals and properties --- */

enum {
  STATE_CHANGE,
  LAST_SIGNAL
};

enum {
  PROP_0,
  PROP_ACTIVE
};

/* --- private function prototypes --- */

static void swami_wavetbl_class_init (SwamiWavetblClass *klass);
static void swami_wavetbl_init (SwamiWavetbl *wavetbl);
static void swami_wavetbl_get_property (GObject *object, guint property_id,
					GValue *value, GParamSpec *pspec);

/* --- private data --- */

SwamiWavetblClass *wavetbl_class = NULL;
static guint wavetbl_signals[LAST_SIGNAL] = { 0 };


/* --- public data --- */

int swami_wavetbl_temp_bank = 127;
int swami_wavetbl_temp_psetnum = 127;


/* --- functions --- */

GType
swami_wavetbl_get_type (void)
{
  static GType item_type = 0;

  if (!item_type) {
    static const GTypeInfo item_info = {
      sizeof (SwamiWavetblClass),
      NULL,
      NULL,
      (GClassInitFunc) swami_wavetbl_class_init,
      NULL,
      NULL,
      sizeof (SwamiWavetbl),
      0,
      (GInstanceInitFunc) swami_wavetbl_init,
    };

    item_type = g_type_register_static (G_TYPE_OBJECT, "SwamiWavetbl",
					&item_info, G_TYPE_FLAG_ABSTRACT);
  }

  return (item_type);
}

static void
swami_wavetbl_class_init (SwamiWavetblClass *klass)
{
  GObjectClass *objclass;

  wavetbl_class = g_type_class_ref (SWAMI_TYPE_WAVETBL);

  objclass = (GObjectClass *)klass;

  wavetbl_signals[STATE_CHANGE] =
    g_signal_new ("state_change", G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_FIRST,
		  0, NULL, NULL,
		  g_cclosure_marshal_VOID__BOOLEAN, G_TYPE_NONE, 1,
		  G_TYPE_BOOLEAN);

  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_ACTIVE,
				   g_param_spec_boolean ("active", "Active",
							 "State of driver",
							 FALSE,
							 G_PARAM_READABLE));
  objclass->get_property = swami_wavetbl_get_property;
}

static void
swami_wavetbl_init (SwamiWavetbl *wavetbl)
{
  wavetbl->active = FALSE;
}

static void
swami_wavetbl_get_property (GObject *object, guint property_id, GValue *value,
			    GParamSpec *pspec)
{
  SwamiWavetbl *wavetbl;

  g_return_if_fail (object != NULL);
  g_return_if_fail (SWAMI_IS_WAVETBL (object));

  wavetbl = SWAMI_WAVETBL (object);

  switch (property_id)
    {
    case PROP_ACTIVE:
      g_value_set_boolean (value, wavetbl->active);
      break;
    default:
      break;
    }
}

/**
 * swami_wavetbl_new:
 *
 * Create a new wavetable object
 *
 * Returns: New Swami Wavetable object
 */
SwamiWavetbl *
swami_wavetbl_new (void)
{
  return (SWAMI_WAVETBL (g_object_new (SWAMI_TYPE_WAVETBL, NULL)));
}

/**
 * swami_wavetbl_init_driver:
 * @wavetbl: Swami Wavetable object
 *
 * Initialize Wavetbl driver
 *
 * Returns: SWAMI_OK on success, SWAMI_FAIL otherwise
 */
int
swami_wavetbl_init_driver (SwamiWavetbl *wavetbl)
{
  SwamiWavetblClass *oclass;
  int retval;

  g_return_val_if_fail (wavetbl != NULL, SWAMI_FAIL);
  g_return_val_if_fail (SWAMI_IS_WAVETBL (wavetbl), SWAMI_FAIL);

  oclass = SWAMI_WAVETBL_CLASS (G_OBJECT_GET_CLASS (wavetbl));

  if (wavetbl->active || !oclass->init_driver)
    return (SWAMI_OK);

  if ((retval = (*oclass->init_driver) (wavetbl)) == SWAMI_OK)
    {
      wavetbl->active = TRUE;
      g_signal_emit (G_OBJECT (wavetbl), wavetbl_signals[STATE_CHANGE],
		     0, TRUE);
    }

  return (retval);
}

/**
 * swami_wavetbl_close_driver:
 * @wavetbl: Swami Wavetable object
 *
 * Close driver, has no effect if already closed.
 */
void
swami_wavetbl_close_driver (SwamiWavetbl *wavetbl)
{
  SwamiWavetblClass *oclass;

  g_return_if_fail (wavetbl != NULL);
  g_return_if_fail (SWAMI_IS_WAVETBL (wavetbl));

  if (!wavetbl->active) return;

  oclass = SWAMI_WAVETBL_CLASS (G_OBJECT_GET_CLASS (wavetbl));

  if (oclass->close_driver)
    (*oclass->close_driver) (wavetbl);

  wavetbl->active = FALSE;
  g_signal_emit (G_OBJECT (wavetbl), wavetbl_signals[STATE_CHANGE], 0, FALSE);
}

/**
 * swami_wavetbl_load_patch:
 * @wavetbl: Swami Wavetable object
 * @patch: Patch object to load
 *
 * Load a patch into a wavetable device
 *
 * Returns: SWAMI_OK on success, SWAMI_FAIL otherwise
 */
int
swami_wavetbl_load_patch (SwamiWavetbl *wavetbl, IPItem *patch)
{
  SwamiWavetblClass *oclass;

  g_return_val_if_fail (wavetbl != NULL, SWAMI_FAIL);
  g_return_val_if_fail (SWAMI_IS_WAVETBL (wavetbl), SWAMI_FAIL);
  g_return_val_if_fail (wavetbl->active, SWAMI_FAIL);
  g_return_val_if_fail (patch != NULL, SWAMI_FAIL);

  oclass = SWAMI_WAVETBL_CLASS (G_OBJECT_GET_CLASS (wavetbl));

  if (oclass->load_patch)
    return ((*oclass->load_patch) (wavetbl, patch));
  else return (SWAMI_FAIL);
}

/**
 * swami_wavetbl_load_temp_item:
 * @wavetbl: Swami Wavetable object
 * @item: Patch item to set as temporary preset.  Can be an
 *   #IPPreset, #IPInst or #IPSample.
 *
 * Load an item as the temporary preset
 *
 * Returns: SWAMI_OK on success, SWAMI_FAIL otherwise
 */
int
swami_wavetbl_load_temp_item (SwamiWavetbl *wavetbl, IPItem *item)
{
  SwamiWavetblClass *oclass;

  g_return_val_if_fail (wavetbl != NULL, SWAMI_FAIL);
  g_return_val_if_fail (SWAMI_IS_WAVETBL (wavetbl), SWAMI_FAIL);
  g_return_val_if_fail (wavetbl->active, SWAMI_FAIL);
  g_return_val_if_fail (item != NULL, SWAMI_FAIL);

  oclass = SWAMI_WAVETBL_CLASS (G_OBJECT_GET_CLASS (wavetbl));

  if (oclass->load_temp_item)
    return ((*oclass->load_temp_item) (wavetbl, item));
  else return (SWAMI_FAIL);
}

/**
 * swami_wavetbl_set_gen_realtime:
 * @wavetbl: Swami Wavetable object
 * @item: Base audible item to set generator of (IPPreset, IPInst or
 *   IPSample).
 * @layer: Layer item within base @item (an IPZone if @item is
 *   an IPPreset or IPInst, NULL if @item is an IPSample)
 * @genid: ID of generator to set
 * @val: Value to set generator to
 *
 * Set a generator in real time
 */
void
swami_wavetbl_set_gen_realtime (SwamiWavetbl *wavetbl, IPItem *item,
				IPItem *layer, guint16 genid, int val)
{
  SwamiWavetblClass *oclass;

  g_return_if_fail (wavetbl != NULL);
  g_return_if_fail (SWAMI_IS_WAVETBL (wavetbl));
  g_return_if_fail (wavetbl->active);
  g_return_if_fail (item != NULL);

  oclass = SWAMI_WAVETBL_CLASS (G_OBJECT_GET_CLASS (wavetbl));

  if (oclass->set_gen_realtime)
    (*oclass->set_gen_realtime) (wavetbl, item, layer, genid, val);
}
