/*==================================================================
 * instpatch.c - Main libInstPatch routines
 *
 * libInstPatch
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 *==================================================================*/
#include <stdio.h>
#include "instpatch.h"
#include "ippriv.h"
#include "soundfont_priv.h"
#include "i18n.h"

/* -- private function prototypes -- */

static IPItem *instp_item_alloc_func (IPItemType type);
static void instp_item_free_func (IPItem *item);


/* -- private global data -- */

/* a lock for voice synthesis related code (hack to minimize MT crashes) */
G_LOCK_DEFINE (instp_voice_lock);

static gboolean instp_initialized = FALSE; /* libInstPatch initialized? */

/* master global modulator list */
static IPMod *global_mod_list = NULL;

static IPItemTypeInfo item_type_info[IPITEM_COUNT] =
{
  { N_("NULL"), sizeof (IPItem), NULL, NULL, NULL, NULL }, /* IPITEM_NONE */
  { N_("SoundFont"), sizeof (IPSFont), /* IPITEM_SFONT */
    instp_item_alloc_func,
    instp_item_free_func,
    instp_sfont_init_func,
    instp_sfont_done_func,
    instp_sfont_destroy_func,
    NULL },			/* duplicate */
  { N_("Preset"), sizeof (IPPreset), /* IPITEM_PRESET */
    instp_item_alloc_func,
    instp_item_free_func,
    NULL,			/* init */
    instp_preset_done_func,
    instp_preset_destroy_func,
    instp_preset_duplicate_func },
  { N_("Instrument"), sizeof (IPInst), /* IPITEM_INST */
    instp_item_alloc_func,
    instp_item_free_func,
    NULL,			/* init */
    instp_inst_done_func,
    instp_inst_destroy_func,
    instp_inst_duplicate_func },
  { N_("Sample"), sizeof (IPSample), /* IPITEM_SAMPLE */
    instp_item_alloc_func,
    instp_item_free_func,
    NULL,			/* init */
    instp_sample_done_func,
    instp_sample_destroy_func,
    instp_sample_duplicate_func },
  { N_("Sample Data"), sizeof (IPSampleData),	/* IPITEM_SAMPLE_DATA */
    instp_item_alloc_func,
    instp_item_free_func,
    instp_sample_data_init_func,
    instp_sample_data_done_func,
    NULL,		/* IPItem_DestroyFunc (not directly destroyed) */
    NULL },			/* duplicate */
  { N_("Zone"), sizeof (IPZone),	/* IPITEM_ZONE */
    instp_item_alloc_func,
    instp_item_free_func,
    NULL,			/* init */
    instp_zone_done_func,
    instp_item_unlink,
    instp_zone_duplicate_func },
  { N_("Virtual Bank"), 0, NULL, NULL, NULL, NULL,
    NULL, NULL },		/* IPITEM_VBANK */
  { N_("Virtual Bank Map"), 0, NULL, NULL, NULL, NULL,
    NULL, NULL }		/* IPITEM_VBANK_MAP */
};


/* predefined error message strings that are used multiple times */

#define IPERR_MSG_INVALID_ITEM_TYPE_1 "Invalid IPItem type (%d) for this operation"


/**
 * Initialize libInstPatch library
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 *
 * Initialize libInstPatch library, call before using any other libInstPatch
 * routines
 */
int
instp_init (void)
{
  if (instp_initialized) return (INSTP_OK);
  instp_initialized = TRUE;

  return (INSTP_OK);
}

/**
 * Get item type info for a given type
 * @type Item type to get info on.
 * Returns: Item type info structure (should \b NOT be modified directly) or
 *   NULL if invalid type.
 */
IPItemTypeInfo *
instp_get_item_type_info (IPItemType type)
{
  g_return_val_if_fail (type >= IPITEM_NONE && type < IPITEM_COUNT, NULL);
  return (&item_type_info[type]);
}

/**
 * Get the descriptive name for a given item type
 * @type Item type to get name of
 * Returns: Name of type or NULL if invalid type (should \b NOT be modified).
 */
char *
instp_get_item_type_name (IPItemType type)
{
  g_return_val_if_fail (type >= IPITEM_NONE && type < IPITEM_COUNT, NULL);
  return (item_type_info[type].name);
}

/**
 * Get the size of an instance of a given item type
 * @type Item type to get size of
 * Returns: Size of an instance of type (in bytes), 0 if invalid type.
 */
int
instp_get_item_type_size (IPItemType type)
{
  g_return_val_if_fail (type >= IPITEM_NONE && type < IPITEM_COUNT, 0);
  return (item_type_info[type].instance_size);
}

/**
 * Get next IPItem in list
 * @item IPItem to get next sibling of
 * Returns: The next IPItem or NULL if end of list reached
 * \see instp_item_prev
 *
 * Gets an IPItem's next sibling in a list of IPItems
 */
IPItem *
instp_item_next (const IPItem *item)
{
  g_return_val_if_fail (item != NULL, NULL);
  return (item->next);
}

/**
 * Get previous IPItem in list
 * @item IPItem to get previous sibling of
 * Returns: The previous IPItem or NULL if begining of list reached
 * \see instp_item_next
 *
 * Gets an IPItem's previous sibling in a list of IPItems
 */
IPItem *
instp_item_prev (const IPItem *item)
{
  g_return_val_if_fail (item != NULL, NULL);
  return (item->prev);
}

/**
 * Get parent of an IPItem
 * @item IPItem to get parent of
 * Returns: The parent IPItem or NULL if no parent
 *
 * Gets an IPItem's parent item in IPItem tree
 */
IPItem *
instp_item_parent (const IPItem *item)
{
  g_return_val_if_fail (item != NULL, NULL);
  return (item->parent);
}

/**
 * Increase the reference count of an item
 * @item Item to increment reference of
 */
void
instp_item_ref (IPItem *item)
{
  g_return_if_fail (item != NULL);

  ITEM_WRITE_LOCK (item);

  if (log_if_fail (item->ref_count > 0))
    {
      ITEM_WRITE_UNLOCK (item);
      return;
    }
  item->ref_count++;

  ITEM_WRITE_UNLOCK (item);
}

/**
 * Decrease the reference count of an item
 * @item Item to decrement reference of
 *
 * Decrease an item's reference count. Item is freed if its reference
 * count reaches zero.
 */
void
instp_item_unref (IPItem *item)
{
  g_return_if_fail (item != NULL);

  ITEM_WRITE_LOCK (item);

  if (log_if_fail (item->ref_count > 0))
    {
      ITEM_WRITE_UNLOCK (item);
      return;
    }
  if (--item->ref_count == 0)
    instp_item_free (item);

  ITEM_WRITE_UNLOCK (item);
}

/**
 * Decrement the initial floating count given to an item
 * @item Instrument patch item
 *
 * When an item is created it is given a ref_count of 1 and flagged as
 * "floating". This is so the item does not get de-allocated after being
 * created. When an item's parent is set, the parent will "sink" the item
 * and add its own reference. This routine can be called multiple times,
 * additional invocations have no effect.
 */
void
instp_item_sink (IPItem *item)
{
  ITEM_WRITE_LOCK (item);

  if (!(item->flags & IPITEM_FLAG_FLOAT)) return;

  item->flags = item->flags & ~IPITEM_FLAG_FLOAT;
  instp_item_unref (item);

  ITEM_WRITE_UNLOCK (item);
}

/**
 * Get count of items in an IPItem list
 * @item First item in list
 * Returns: Count of items
 *
 * NOTE: Multi-thread locking on item list must be provided by caller
 */
int
instp_item_count (const IPItem *item)
{
  int count = 0;

  while (item)
    {
      count++;
      item = item->next;
    }
  return (count);
}

/**
 * Create a new instrument patch item
 * @type Type of item to create
 * Returns: The new item or NULL on error
 *
 * Creates a new item of the specified type.
 */
IPItem *
instp_item_new (IPItemType type)
{
  IPItemTypeInfo *item_info;
  IPItem *item;

  g_return_val_if_fail (type > IPITEM_NONE && type < IPITEM_COUNT, NULL);

  item_info = &item_type_info[type];
  if (item_info->alloc)
    item = (*item_info->alloc)(type);
  else
    {
      g_critical (IPERR_MSG_INVALID_ITEM_TYPE_1, type);
      return (NULL);
    }

  item->type = type;
  item->flags = IPITEM_FLAG_FLOAT; /* item is floating */
  item->ref_count = 1;		/* 1 for float reference */

  if (item_info->init)		/* item type has init routine? */
    {
      if ((*item_info->init)(item) != INSTP_OK)	/* call init routine */
	{
	  instp_item_free (item); /* free item if error occurs */
	  return (NULL);
	}
    }

  return (item);
}

/**
 * Allocate memory for an item
 * @type Item type to allocate memory for.
 *
 * Used internally to allocate space for items, not normally
 * used otherwise.
 */
IPItem *
instp_item_alloc (IPItemType type)
{
  IPItemTypeInfo *item_info;
  IPItem *item;

  g_return_val_if_fail (type > IPITEM_NONE && type < IPITEM_COUNT, NULL);

  item_info = &item_type_info[type];
  if (item_info->free)
    item = (*item_info->alloc)(type);
  else g_critical (IPERR_MSG_INVALID_ITEM_TYPE_1, type);

  return (item);
}

/* Default #IPItem_AllocFunc used for all items */
static IPItem *
instp_item_alloc_func (IPItemType type)
{
  IPItem *item;
  int size;

  size = instp_get_item_type_size (type);
  if (!size) return (NULL);

  item = g_malloc0 (size);
  return (item);
}

/**
 * Free an item
 * @item Item to free (its ref count should be 0)
 *
 * Used internally to free items when their ref counts get to
 * zero, not normally used otherwise.
 */
void
instp_item_free (IPItem *item)
{
  IPItemTypeInfo *item_info;

  g_return_if_fail (item != NULL);
  g_return_if_fail (item->type > IPITEM_NONE && item->type < IPITEM_COUNT);
  g_return_if_fail (item->ref_count == 0);

  item_info = &item_type_info[item->type];
  if (item_info->free)
    (*item_info->free)(item);
  else g_critical (IPERR_MSG_INVALID_ITEM_TYPE_1, item->type);
}

/* Default #IPItem_FreeFunc used for all items */
void
instp_item_free_func (IPItem *item)
{
  g_return_if_fail (item != NULL);
  g_free (item);
}

/**
 * Destroy an item
 * @item Item to destroy
 *
 * Destroys an item by unlinking it and all its children. Objects
 * are only freed when they are no longer referenced, though.
 * Sets changed flag.
 */
void
instp_item_destroy (IPItem *item)
{
  IPItemTypeInfo *item_info;

  g_return_if_fail (item != NULL);
  g_return_if_fail (item->type > IPITEM_NONE && item->type < IPITEM_COUNT);

  item_info = &item_type_info[item->type];
  if (item_info->destroy)
    (*item_info->destroy)(item);
}

/**
 * Duplicate an item
 * @item Item to duplicate
 * Returns: The duplicated item or NULL on error
 */
IPItem *
instp_item_duplicate (const IPItem *item)
{
  IPItemTypeInfo *item_info;
  IPItem *dup;

  g_return_val_if_fail (item != NULL, NULL);
  g_return_val_if_fail (item->type > IPITEM_NONE && item->type < IPITEM_COUNT,
			NULL);

  item_info = &item_type_info[item->type];
  if (item_info->duplicate)
    dup = (*item_info->duplicate)(item);
  else
    {
      g_critical (IPERR_MSG_INVALID_ITEM_TYPE_1, item->type);
      return (NULL);
    }

  return (dup);
}

/**
 * Insert an IPItem at the given index in a tree of items
 * @parent New parent of the item to insert
 * @item Item to insert
 * @pos Index position to insert item into the list
 * (0 = first, < 0 last)
 *
 * Inserts an IPItem into a list at the given position and parents it.
 * Sets changed flag.
 */
void
instp_item_insert (IPItem *parent, IPItem *item, int pos)
{
  IPItem **list_root, *sibling;

  g_return_if_fail (parent != NULL);
  g_return_if_fail (item != NULL);

  list_root = instp_item_get_child_list (parent, item->type);
  g_return_if_fail (list_root != NULL);

  if (pos >= 0)
    {
      ITEM_READ_LOCK (parent);

      sibling = *list_root;
      while (sibling && pos-- > 0)
	sibling = instp_item_next (sibling);

      ITEM_READ_UNLOCK (parent);
    }
  else sibling = NULL;

  instp_item_insert_before (parent, item, sibling);
}

/**
 * Insert an IPItem before a given item in a tree of items
 * @parent New parent of the item to insert
 * @item Item to insert
 * @sibling A child of parent to insert the item before,
 * NULL will append the item
 *
 * Inserts an IPItem into a list before sibling and parents it.
 * Sets changed flag.
 */
void
instp_item_insert_before (IPItem *parent, IPItem *item, IPItem *sibling)
{
  IPItem **list_root, *newroot;
  gboolean parent_is_ok;

  g_return_if_fail (parent != NULL);
  g_return_if_fail (item != NULL);

  /* item types don't change, no need to lock */
  parent_is_ok = instp_item_is_parent_ok (parent->type, item->type);
  g_return_if_fail (parent_is_ok);

  list_root = instp_item_get_child_list (parent, item->type);
  g_return_if_fail (list_root != NULL);

  ITEM_WRITE_LOCK (item);

  if (log_if_fail (item->parent == NULL))
    {
      ITEM_WRITE_UNLOCK (item);
      return;
    }

  ITEM_WRITE_LOCK (parent);

  instp_item_ref (parent);	/* ++ inc parent ref count */
  item->parent = parent;

  newroot = instp_item_list_insert_before (*list_root, item, sibling);

  instp_item_ref (newroot);	/* ++ inc newroot ref count for parent */
  if (*list_root)
    instp_item_unref (*list_root); /* -- unref old root (maybe same as new) */

  *list_root = newroot;		/* set new root (possibly the old one) */

  instp_item_sink (item);

  instp_item_set_root_changed_flag (item, TRUE);

  ITEM_WRITE_UNLOCK (parent);
  ITEM_WRITE_UNLOCK (item);
}

/**
 * Insert an item into a list at a given index
 * @list_root Pointer to root of list (or NULL if empty list)
 * @item Item to insert
 * @pos Index position to insert item at
 * Returns: New list root
 *
 * NOTE: Multi-thread locking on item list must be provided by caller
 */
IPItem *
instp_item_list_insert (IPItem *list_root, IPItem *item, int pos)
{
  IPItem *sibling;

  g_return_val_if_fail (item != NULL, NULL);

  if (pos != -1)
    {
      sibling = list_root;
      while (sibling && pos-- > 0)
	sibling = instp_item_next (sibling);
    }
  else sibling = NULL;

  return (instp_item_list_insert_before (list_root, item, sibling));
}

/**
 * Insert an item into a list before another item
 * @list_root Pointer to root of list (or NULL if empty list)
 * @item Item to insert
 * @sibling Item to insert item before (or NULL to append item)
 * Returns: New list root
 *
 * NOTE: Multi-thread locking on 'list_root' list must be provided by caller
 */
IPItem *
instp_item_list_insert_before (IPItem *list_root, IPItem *item,
			       IPItem *sibling)
{
  IPItem *p;

  g_return_val_if_fail (item != NULL, NULL);

  ITEM_WRITE_LOCK (item);

  if (log_if_fail (item->next == NULL)
      || log_if_fail (item->prev == NULL))
    {
      ITEM_WRITE_UNLOCK (item);
      return (NULL);
    }

  /* "item" should be first in list? */
  if (!list_root || list_root == sibling)
    {
      item->prev = NULL;
      item->next = sibling;
      if (sibling)
	{
	  instp_item_ref (sibling); /* ++ inc sibling ref count */
	  instp_item_ref (item); /* ++ inc item ref count for sibling */
	  sibling->prev = item;
	}

      ITEM_WRITE_UNLOCK (item);

      return (item);		/* new list root */
    }
  else if (sibling)		/* insert "item" in middle of list? */
    {
      instp_item_ref (sibling);	/* ++ inc sibling ref count for item->next */
      item->next = sibling;
      instp_item_ref (sibling->prev); /* ++ inc sibling->prev for item->prev */
      item->prev = sibling->prev;
      instp_item_ref (item);	/* ++ inc item ref for item->prev->next */
      item->prev->next = item;
      instp_item_ref (item);	/* ++ inc item ref for item->next->prev */
      item->next->prev = item;
    }
  else				/* put "item" at end of list */
    {
      item->next = NULL;

      /* find last item */
      p = list_root;
      while (p->next)
	p = instp_item_next (p);

      instp_item_ref (p);	/* ++ inc p ref count for item->prev */
      item->prev = p;
      instp_item_ref (item);	/* ++ inc item ref count for p->next */
      p->next = item;
    }

  ITEM_WRITE_UNLOCK (item);

  return (list_root);
}

/**
 * Non-destructive removal of an IPItem from a tree of items
 * @item Item to unlink
 *
 * Removes an IPItem from a tree of items, unlinks next and previous siblings
 * and parent (does not destroy the item). Item is only destroyed when it is
 * no longer referenced (ref_count reaches 0).
 * Sets changed flag.
 */
void
instp_item_unlink (IPItem *item)
{
  IPItem **rootp;

  g_return_if_fail (item != NULL);

  instp_item_set_root_changed_flag (item, TRUE);
  instp_item_ref (item);	/* ++ ref item for duration of this func */

  ITEM_WRITE_LOCK (item);

  if (item->parent)
    {
      rootp = instp_item_get_child_list (item->parent, item->type);
      if (*rootp == item)
	{
	  instp_item_unref (item); /* -- unref item (parent's list root ref) */
	  *rootp = item->next;
	  if (*rootp)
	    instp_item_ref (*rootp); /* ++ ref new root item for parent */
	}

      instp_item_unref (item->parent); /* -- unref item's parent */
      item->parent = NULL;
    }

  if (item->next)
    {
      item->next->prev = item->prev;
      if (!item->prev)	/* -- unref next item if it has no more prev item */
	instp_item_unref (item->next);
      instp_item_unref (item);	/* -- unref item for next item's ref */
    }

  if (item->prev)
    {
      item->prev->next = item->next;
      if (!item->next)	/* -- unref prev item if it has no more next item */
	instp_item_unref (item->prev);
      instp_item_unref (item);	/* -- unref item for prev item's ref */
    }

  item->next = item->prev = NULL;

  ITEM_WRITE_UNLOCK (item);

  instp_item_unref (item);	/* -- unref temporary reference */
}

/**
 * Set the user definable pointer of an IPItem
 * @item Item to set the user pointer of
 * @ptr Value to set the user pointer to
 * \see instp_item_get_userptr
 *
 * Set the user pointer of an item (associate arbitrary data)
 * Does \b NOT set changed flag.
 */
void
instp_item_set_userptr (IPItem *item, const void *ptr)
{
  g_return_if_fail (item != NULL);
  item->userptr = (void *)ptr;
}

/**
 * Get the user definable pointer of an IPItem
 * @item Item to get the user pointer from
 * \see instp_item_set_userptr
 *
 * Get the user pointer of a sound font item.
 */
void *
instp_item_get_userptr (const IPItem *item)
{
  g_return_val_if_fail (item != NULL, NULL);
  return (item->userptr);
}

/* FIXME: Multi-thread locking? */
/**
 * Find parent item by item type
 * @item Item to start from
 * @type Type of parent to find
 * Returns: The found item or NULL if not found
 *
 * Find a parent of an item by type (can match itself), by traversing up
 * the item tree.
 */
IPItem *
instp_item_find_parent_by_type (const IPItem *item, IPItemType type)
{
  IPItem *p;

  g_return_val_if_fail (item != NULL, NULL);

  p = (IPItem *)item;
  do				/* climb sfitem tree */
    {
      if (p->type == type)	/* type matches? */
	return (p);
    }
  while ((p = instp_item_parent (p)));

  return (NULL);		/* not found */
}

/* FIXME: Multi-thread locking? */
/**
 * Find the root parent of an item
 * @item Item to start from
 * Returns: The root of item (parent of item that has no parent)
 *
 * Finds the top level root parent of an item in an item tree
 */
IPItem *
instp_item_find_root (const IPItem *item)
{
  IPItem *p;

  g_return_val_if_fail (item != NULL, NULL);

  p = (IPItem *)item;

  while (p->parent)		/* climb item tree */
    p = p->parent;

  return (p);
}

/**
 * Get child list pointer of an item by type
 * @item Item to get pointer from
 * @childtype Child list item type to get pointer for
 * Returns: Pointer within the IPItem that points to the root IPItem of
 * the given childtype or NULL if invalid childtype to item
 */
IPItem **
instp_item_get_child_list (IPItem *item, IPItemType childtype)
{
  if (item->type == IPITEM_SFONT)
    {
      IPSFont *sf = INSTP_SFONT (item);
      if (childtype == IPITEM_PRESET) return ((IPItem **)(&sf->preset));
      if (childtype == IPITEM_INST) return ((IPItem **)(&sf->inst));
      if (childtype == IPITEM_SAMPLE) return ((IPItem **)(&sf->sample));
    }
  else if (item->type == IPITEM_PRESET && childtype == IPITEM_ZONE)
    return ((IPItem **)(&INSTP_PRESET (item)->zone));
  else if (item->type == IPITEM_INST && childtype == IPITEM_ZONE)
    return ((IPItem **)(&INSTP_INST (item)->zone));
#if 0				/* FIXME: will be added soon (tm) */
  else if (item->type == IPITEM_VBANK && childtype == IPITEM_VBANK_MAP)
    return (blah);
#endif

  return (NULL);
}

/**
 * Check if an IPItem type can be parented to another type
 * @Parent item type
 * @Child item type
 * Returns: TRUE if parenting of these types is valid, FALSE if not
 */
gboolean
instp_item_is_parent_ok (IPItemType parent, IPItemType child)
{
  return ((parent == IPITEM_SFONT && (child == IPITEM_PRESET
				       || child == IPITEM_INST
				       || child == IPITEM_SAMPLE))
	  || (parent == IPITEM_PRESET && child == IPITEM_ZONE)
	  || (parent == IPITEM_INST && child == IPITEM_ZONE)
	  || (parent == IPITEM_VBANK && child == IPITEM_VBANK_MAP));
}

/**
 * Set the value of the root item's changed flag
 * @item Item to find the root item of
 * @val Value to set the root item's changed flag to
 *
 * Finds the root of the given item (#instp_item_find_root) and sets
 * its \b changed flag to the given value, to indicate that the patch
 * has changed.
 */
void
instp_item_set_root_changed_flag (IPItem *item, gboolean val)
{
  IPItem *root;

  g_return_if_fail (item != NULL);

  root = instp_item_find_root (item);
  if (root->type == IPITEM_SFONT)
    INSTP_SFONT (root)->flag_changed = val;
}

/* apply a global modulator list (overrides defaults) */
void
instp_mod_list_apply_global (IPMod *globalist)
{
  G_LOCK (instp_voice_lock);
  if (global_mod_list) instp_mod_list_free (global_mod_list);
  global_mod_list = instp_mod_list_copy (globalist);
  G_UNLOCK (instp_voice_lock);
}

/* FIXME! This function is pretty time critical, currently it is not
   programmed as such (use of malloc and modulator lists)
   Needs proper mt locking */
/**
 * Render an item into voices and call a function for each one
 * @item Item to extract voices from
 * @note MIDI note number to restrict voices to (-1 for wild card)
 * @velocity MIDI velocity to restrict voices to (-1 for wild card)
 * @func Function to call for each voice
 * @data User defined data to pass to callback
 * Returns: TRUE if all voices were processed, FALSE if callback func aborted
 *
 * For rendering items into voices (A.K.A. layers). This routine is
 * meant for #IPPreset, #IPInst and #IPSample items. Combines
 * generator and modulator layers and sends a generator array,
 * modulator list and associated sample to a user defined callback
 * function for each layer rendered.
 */
gboolean
instp_item_foreach_voice (IPItem *item, int note, int velocity,
			  IPItemForeachVoiceFunc func, void *data)
{
  IPZone *pzone, *izone;
  IPGenAmount *gpz, *pz, *giz, *iz;
  IPMod *gpmods, *pmods;	/* global and preset modulator lists */
  IPMod *gimods, *imods;	/* global and instrument modulator lists */
  IPMod *defmods, *mods;	/* default and final modulator list */
  IPMod *temp;

  g_return_val_if_fail (item != NULL, FALSE);
  g_return_val_if_fail (func != NULL, FALSE);

  /* allocate for the 4 generator arrays and assign the pointers */
  gpz = g_malloc (sizeof (IPGenAmount) * IPGEN_COUNT * 4);

  pz = &gpz[IPGEN_COUNT];
  giz = &pz[IPGEN_COUNT];
  iz = &giz[IPGEN_COUNT];

  defmods = instp_mod_list_default (); /* get default modulator list */

  G_LOCK (instp_voice_lock);	   /* hack to minimize multi-thread crashes */

  switch (item->type)
    {
    case IPITEM_PRESET:
      pzone = INSTP_PRESET (item)->zone;

      instp_gen_array_init (gpz, TRUE);	/* init preset global zone array */

      if (pzone && !pzone->refitem) /* global zone? */
	{
	  instp_gen_array_process_zone (gpz, pzone);
	  gpmods = pzone->mod;
	  pzone = instp_zone_next (pzone);
	}
      else gpmods = NULL;

      while (pzone)		/* loop over preset zones */
	{
	  if (!instp_zone_in_range (pzone, note, velocity))
	    {
	      pzone = instp_zone_next (pzone);
	      continue;
	    }

	  instp_gen_array_copy (pz, gpz); /* copy default gen values */
	  instp_gen_array_process_zone (pz, pzone); /* process preset zone */

	  if (gpmods)
	    pmods = instp_mod_list_combine (gpmods, pzone->mod);
	  else pmods = pzone->mod;

	  if (global_mod_list)	/* apply master global modulators */
	    {
	      if (pmods)
		{
		  temp = pmods;
		  pmods = instp_mod_list_combine (pmods, global_mod_list);
		  instp_mod_list_free (temp);
		}
	      else pmods = global_mod_list;
	    }

	  izone = INSTP_INST (pzone->refitem)->zone;

	  instp_gen_array_init (giz, FALSE); /* init inst global zone array */

	  if (izone && !izone->refitem)	/* global zone? */
	    {
	      instp_gen_array_process_zone (giz, izone);
	      gimods = izone->mod;
	      izone = instp_zone_next (izone);
	    }
	  else gimods = NULL;

	  while (izone)
	    {
	      if (!instp_zone_in_range (izone, note, velocity))
		{
		  izone = instp_zone_next (izone);
		  continue;
		}

	      instp_gen_array_copy (iz, giz); /* copy default gen values */
	      instp_gen_array_process_zone (iz, izone);	/* process inst zone */
	      instp_gen_array_offset (iz, pz);

	      if (gimods)
		{
		  temp = instp_mod_list_combine (defmods, gimods);
		  imods = instp_mod_list_combine (temp, izone->mod);
		  instp_mod_list_free (temp);
		}
	      else imods = instp_mod_list_combine (defmods, izone->mod);

	      mods = instp_mod_list_offset (imods, pmods);

	      if (!(func) (item, INSTP_SAMPLE (izone->refitem),
			   iz, mods, data))
		{
		  instp_mod_list_free (imods);
		  instp_mod_list_free (mods);
		  if (pmods != pzone->mod && pmods != global_mod_list)
		    instp_mod_list_free (pmods);
		  G_UNLOCK (instp_voice_lock);
		  g_free (gpz);
		  return (FALSE);
		}

	      instp_mod_list_free (imods);
	      instp_mod_list_free (mods);

	      izone = instp_zone_next (izone);
	    }

	  if (pmods != pzone->mod && pmods != global_mod_list)
	    instp_mod_list_free (pmods);

	  pzone = instp_zone_next (pzone);
	}
      break;
    case IPITEM_INST:
      izone = INSTP_INST (item)->zone;

      instp_gen_array_init (giz, FALSE); /* init inst global zone array */

      if (izone && !izone->refitem)	/* global zone? */
	{
	  instp_gen_array_process_zone (giz, izone);
	  gimods = izone->mod;
	  izone = instp_zone_next (izone);
	}
      else gimods = NULL;

      while (izone)
	{
	  if (!instp_zone_in_range (izone, note, velocity))
	    {
	      izone = instp_zone_next (izone);
	      continue;
	    }

	  instp_gen_array_copy (iz, giz); /* copy default gen values */
	  instp_gen_array_process_zone (iz, izone);	/* process inst zone */

	  if (gimods)
	    {
	      temp = instp_mod_list_combine (defmods, gimods);
	      imods = instp_mod_list_combine (temp, izone->mod);
	      instp_mod_list_free (temp);
	    }
	  else imods = instp_mod_list_combine (defmods, izone->mod);

	  if (global_mod_list)	/* apply master global modulators */
	    {
	      temp = imods;
	      imods = instp_mod_list_offset (imods, global_mod_list);
	      instp_mod_list_free (temp);
	    }

	  if (!(*func) (item, INSTP_SAMPLE (izone->refitem), iz, imods, data))
	    {
	      instp_mod_list_free (imods);
	      G_UNLOCK (instp_voice_lock);
	      g_free (gpz);
	      return (FALSE);
	    }

	  instp_mod_list_free (imods);

	  izone = instp_zone_next (izone);
	}
      break;
    case IPITEM_SAMPLE:
      instp_gen_array_init (iz, FALSE);

      if (!(*func) (item, INSTP_SAMPLE (item), iz, global_mod_list, data))
	{
	  G_UNLOCK (instp_voice_lock);
	  g_free (gpz);
	  return (FALSE);
	}
      break;
    }

  G_UNLOCK (instp_voice_lock);

  g_free (gpz);

  return (TRUE);
}
