/*==================================================================
 * sfsave.c - sound font loading functions
 *
 * libsoundfont
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *==================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "instpatch.h"
#include "ippriv.h"
#include "i18n.h"

struct _IPSFontSaveHandle
{
  IPFileContext *ctx;
  IPFileInfo *file_info;	/* new file info structure */
  IPSFont *sf;			/* SoundFont object to save */
  IPSFontSaveSampleDataCallback sample_callback; /* sampledata override func */
  gpointer sample_callback_data; /* data to pass to the sample callback */
  gboolean saved;		/* if SoundFont has been saved yet */
  gboolean migrated;		/* if samples have been migrated */
  gboolean tempstores;		/* if temp sample stores have been created */
  gboolean in_sample_callback;	/* in sample save override callback */
};

#define INSTP_ERRMSG_REFITEM_NOT_FOUND_0 \
   "Failed to find zone's refitem in the same SoundFont"

static int sfsave_callback (IPFileContext *ctx, void *userptr);
static int sfsave_infos (IPSFontSaveHandle *handle);
static int sfsave_samples (IPSFontSaveHandle *handle);
static int sfsave_phdrs (IPSFontSaveHandle *handle);
static int sfsave_pbags (IPSFontSaveHandle *handle);
static int sfsave_pmods (IPSFontSaveHandle *handle);
static int sfsave_pgens (IPSFontSaveHandle *handle);
static int sfsave_ihdrs (IPSFontSaveHandle *handle);
static int sfsave_ibags (IPSFontSaveHandle *handle);
static int sfsave_imods (IPSFontSaveHandle *handle);
static int sfsave_igens (IPSFontSaveHandle *handle);
static int sfsave_shdrs (IPSFontSaveHandle *handle);

/**
 * instp_sfont_save_new_handle:
 *
 * Creates a new save handle for use in saving a SoundFont file.
 *
 * Returns: An IPSFontSaveHandle or NULL on error.
 */
IPSFontSaveHandle *
instp_sfont_save_new_handle (void)
{
  IPSFontSaveHandle *handle;

  handle = g_malloc0 (sizeof (IPSFontSaveHandle));

  if (!(handle->ctx = ipfile_context_new ("w")))
    {
      g_free (handle);
      return (NULL);
    }

  /* create a new file info structure for new file handle */
  if (!(handle->file_info = instp_file_info_new ()))
    {
      ipfile_context_free (handle->ctx);
      g_free (handle);
      return (NULL);
    }

  instp_file_info_ref (handle->file_info); /* hold a ref for save context */

  return (handle);
}

/**
 * instp_sfont_save_close_handle:
 * @handle: SoundFont save handle
 *
 * Closes and frees a SoundFont save handle and cleans up after itself.
 */
void
instp_sfont_save_close_handle (IPSFontSaveHandle *handle)
{
  g_return_if_fail (handle != NULL);

  /* if samples weren't migrated, remove temp sample data objects */
  if (!handle->migrated && handle->tempstores)
    {
      IPSampleData *sampledata;
      IPSampleStore *store;
      IPFileInfo *file_info;

      sampledata = instp_sample_data_list;
      while (sampledata)	/* loop over sample data objects */
	{
	  store = instp_sample_data_first_store (sampledata);
	  while (store)		/* loop over sample stores */
	    {
	      if (store->method->type == IPSAMPLE_METHOD_SFONT)
		{
		  instp_sample_method_SFONT_get_params (sampledata, store,
							&file_info, NULL);
		  if (file_info == handle->file_info)
		    {  /* it is a temporary sample data */
		      instp_sample_store_destroy (sampledata, store);
		      break;	/* break out of store loop */
		    }
		}
	      store = instp_sample_store_next (store);
	    }
	  sampledata = instp_sample_data_next (sampledata);
	}
    }

  ipfile_context_free (handle->ctx);

  handle->file_info->fd = -1;
  instp_file_info_unref (handle->file_info);

  if (handle->sf)
    instp_item_unref (INSTP_ITEM (handle->sf));

  g_free (handle);
}

/**
 * instp_sfont_save_set_sfont:
 * @handle: SoundFont save handle
 * @sf: SoundFont to save.
 *
 * Sets the SoundFont object to save to in a SoundFont save handle.
 * A call to this function is required.
 */
void
instp_sfont_save_set_sfont (IPSFontSaveHandle *handle, IPSFont *sf)
{
  g_return_if_fail (handle != NULL);
  g_return_if_fail (sf != NULL);

  instp_item_ref (INSTP_ITEM (sf));
  handle->sf = sf;
}

/**
 * instp_sfont_save_get_sfont:
 * @handle: SoundFont save handle
 *
 * Accessor function to the SoundFont being saved in a save handle.
 *
 * Returns: SoundFont item set in the save handle or NULL if not set.
 */
IPSFont *
instp_sfont_save_get_sfont (IPSFontSaveHandle *handle)
{
  g_return_val_if_fail (handle != NULL, NULL);
  return (handle->sf);
}

/**
 * instp_sfont_save_set_fhandle:
 * @handle: SoundFont save handle
 * @file_handle: File handle to save to.
 *
 * Sets the file handle to save SoundFont to in a SoundFont save handle.
 * A call to this function is required if not overriding the default file I/O
 * functions.
 */
void
instp_sfont_save_set_fhandle (IPSFontSaveHandle *handle, int file_handle)
{
  g_return_if_fail (handle != NULL);
  g_return_if_fail (file_handle >= 0);

  ipfile_set_fhandle (handle->ctx, file_handle);
  handle->file_info->fd = file_handle;
}

/**
 * instp_sfont_save_get_fhandle:
 * @handle: SoundFont save handle
 *
 * Accessor function to the file handle being saved to in a save handle.
 *
 * Returns: File handle set in the save handle or -1 if not set.
 */
int
instp_sfont_save_get_fhandle (IPSFontSaveHandle *handle)
{
  g_return_val_if_fail (handle != NULL, -1);
  return (ipfile_get_fhandle (handle->ctx));
}

/**
 * instp_sfont_save_set_sample_callback:
 * @handle: SoundFont save handle
 * @callback: Sample data callback override function.
 * @callback_data: Programmer defined pointer to pass to callback.
 */
void
instp_sfont_save_set_sample_callback (IPSFontSaveHandle *handle,
				      IPSFontSaveSampleDataCallback callback,
				      gpointer callback_data)
{
  g_return_if_fail (handle != NULL);

  handle->sample_callback = callback;
  handle->sample_callback_data = callback_data;
}

/**
 * instp_sfont_save_get_file_context:
 * @handle: SoundFont save handle
 *
 * Get the #IPFileContext in a SoundFont save handle.
 *
 * Returns: The #IPFileContext in the SoundFont save @handle.
 */
IPFileContext *
instp_sfont_save_get_file_context (IPSFontSaveHandle *handle)
{
  g_return_val_if_fail (handle != NULL, NULL);

  return (handle->ctx);
}

/**
 * instp_sfont_save:
 *
 * Save a SoundFont file. Clears "changed" flag and sets "saved" flag.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
instp_sfont_save (IPSFontSaveHandle *handle)
{
  int fhandle;

  g_return_val_if_fail (handle != NULL, INSTP_FAIL);
  g_return_val_if_fail (handle->sf != NULL, INSTP_FAIL);
  g_return_val_if_fail (!handle->saved, INSTP_FAIL);

  ipfile_set_callback (handle->ctx, (IPFileCallback *)sfsave_callback, handle);

  if (ipfile_start_transfer (handle->ctx) != INSTP_OK)
    return (INSTP_FAIL);

  handle->saved = TRUE;

  handle->sf->flag_changed = FALSE; /* sfont in memory is in sync with file */
  handle->sf->flag_saved = TRUE; /* has not been saved yet */

  return (INSTP_OK);
}

/*
 * instp_sfont_save_migrate_samples:
 * @handle: SoundFont save handle
 *
 * This function can only be called after a successful save of a SoundFont.
 * Sample stores of the samples that have been saved are migrated to the new
 * file. Samples that reference the SoundFont's old file handle but are not in
 * the new file are moved to the sample buffer.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
instp_sfont_save_migrate_samples (IPSFontSaveHandle *handle)
{
  IPSampleData *sampledata;
  IPSampleStore *store, *remove;
  IPFileInfo *file_info;
  gboolean migrate;		/* migrate to sample buffer flag */

  g_return_val_if_fail (handle != NULL, INSTP_FAIL);
  g_return_val_if_fail (handle->saved, INSTP_FAIL);
  g_return_val_if_fail (!handle->migrated, INSTP_FAIL);

  instp_set_file_info (handle->sf, handle->file_info);

  sampledata = instp_sample_data_list;
  while (sampledata)		/* loop over all sample data objects */
    {
      remove = NULL;
      migrate = TRUE;
      store = instp_sample_data_first_store (sampledata);
      while (store)		/* loop over sample stores */
	{
	  if (store->method->type == IPSAMPLE_METHOD_SFONT)
	    {
	      instp_sample_method_SFONT_get_params (sampledata, store,
						    &file_info, NULL);
	      /* sample data references old file? */
	      if (file_info == handle->sf->file)
		remove = store;	/* ?:yes, remove */
	      else migrate = FALSE; /* ?: no, don't need to migrate */
	    }
	  else if (IPSAMPLE_STORE_IS_READABLE (store))
	    migrate = FALSE;	/* sample has a readable store, no migrate */

	  store = instp_sample_store_next (store);
	}

      if (remove)
	{
	  /* need to migrate to disk buffer? If duplication fails, we
	     keep the old file sample store which will keep the old file
	     handle open.. FIXME for non-UNIX OSes? Maybe should fail. */
	  if (migrate)
	    {
	      if (!instp_sample_store_duplicate (sampledata, remove,
						 IPSAMPLE_METHOD_BUFFER))
		remove = NULL;
	    }

	  if (remove) instp_sample_store_destroy (sampledata, remove);
	}
      sampledata = instp_sample_data_next (sampledata);
    }

  handle->migrated = TRUE;
  return (INSTP_OK);
}

static int
sfsave_callback (IPFileContext *ctx, void *userptr)
{
  IPSFontSaveHandle *handle = (IPSFontSaveHandle *)userptr;
  IPChunkType type;
  int retval = INSTP_OK;

  if (ipfile_get_chunk_state (ctx, -1, &type, NULL, NULL) != INSTP_OK)
    return (INSTP_FAIL);

  switch (type)
    {
    case IPCHUNK_INFO:		/* save SoundFont info */
      retval = sfsave_infos (handle);
      break;
    case IPCHUNK_SMPL:		/* save sample data */
      handle->in_sample_callback = TRUE;

      if (handle->sample_callback)
	retval = (*handle->sample_callback)(handle,
					    handle->sample_callback_data);
      else retval = sfsave_samples (handle);

      handle->in_sample_callback = FALSE;
      break;
    case IPCHUNK_PHDR:		/* save preset headers */
      retval = sfsave_phdrs (handle);
      break;
    case IPCHUNK_PBAG:		/* save preset bags */
      retval = sfsave_pbags (handle);
      break;
    case IPCHUNK_PMOD:		/* save preset modulators */
      retval = sfsave_pmods (handle);
      break;
    case IPCHUNK_PGEN:		/* save preset generators */
      retval = sfsave_pgens (handle);
      break;
    case IPCHUNK_IHDR:		/* save instrument headers */
      retval = sfsave_ihdrs (handle);
      break;
    case IPCHUNK_IBAG:		/* save instrument bags */
      retval = sfsave_ibags (handle);
      break;
    case IPCHUNK_IMOD:		/* save instrument modulators */
      retval = sfsave_imods (handle);
      break;
    case IPCHUNK_IGEN:		/* save instrument generators */
      retval = sfsave_igens (handle);
      break;
    case IPCHUNK_SHDR:		/* save sample headers */
      retval = sfsave_shdrs (handle);
      break;
    default:
      break;
    }

  return (retval);
}

/* save SoundFont info in the recommended order */
static int
sfsave_infos (IPSFontSaveHandle *handle)
{
  IPChunkType infotype;
  IPSFontInfo *info;
  char *val;

  /* save SoundFont version */
  if (ipfile_save_info_version (handle->ctx,
				ipfile_chunk_enum_to_str (IPINFO_VERSION),
				&handle->sf->version) != INSTP_OK)
    return (INSTP_FAIL);

  if (!(val = instp_get_info (handle->sf, IPINFO_ENGINE)))
    val = "EMU8000";
  if (ipfile_save_info (handle->ctx, ipfile_chunk_enum_to_str (IPINFO_ENGINE),
			val, strlen (val) + 1) != INSTP_OK)
    return (INSTP_FAIL);

  /* SoundFont has ROM name set? */
  if ((val = instp_get_info (handle->sf, IPINFO_ROM_NAME)))
    {
      if (ipfile_save_info (handle->ctx,
			    ipfile_chunk_enum_to_str (IPINFO_ROM_NAME),
			    val, strlen (val) + 1) != INSTP_OK)
	return (INSTP_FAIL);
      
      /* save the ROM version too */
      if (ipfile_save_info_version (handle->ctx,
			ipfile_chunk_enum_to_str (IPINFO_ROM_VERSION),
				    &handle->sf->rom_version) != INSTP_OK)
	return (INSTP_FAIL);
    }

  if (!(val = instp_get_info (handle->sf, IPINFO_NAME)))
    val = "Untitled";
  if (ipfile_save_info (handle->ctx, ipfile_chunk_enum_to_str (IPINFO_NAME), val,
			strlen (val) + 1) != INSTP_OK)
    return (INSTP_FAIL);

  infotype = IPINFO_DATE;
  while (infotype <= IPINFO_LAST)
    {
      val = instp_get_info (handle->sf, infotype);
      if (val)
	if (ipfile_save_info (handle->ctx,
			      ipfile_chunk_enum_to_str (infotype), val,
			      strlen (val) + 1) != INSTP_OK)
	  return (INSTP_FAIL);
      infotype++;
    }

  /* save custom (unknown) info IDs if any */
  info = instp_first_info (handle->sf);
  while (info)
    {
      if (ipfile_chunk_str_to_enum (info->id) == IPCHUNK_UNKNOWN)
	if (ipfile_save_info (handle->ctx, info->id, info->val,
			      strlen (info->val) + 1) != INSTP_OK)
	  return (INSTP_FAIL);
      info = instp_info_next (info);
    }

  return (INSTP_OK);
}

/* save sample data */
static int
sfsave_samples (IPSFontSaveHandle *handle)
{
  IPSample *sam;
  guint start;

  sam = instp_first_sample (handle->sf);
  while (sam)
    {
      /* ignore ROM samples */
      if (!(sam->sampletype & IPSAMPLE_TYPE_ROM))
	{
	  ipfile_get_chunk_state (handle->ctx, -1, NULL, NULL, &start);

	  if (ipfile_save_sample (handle->ctx, sam) != INSTP_OK)
	    return (INSTP_FAIL);

	  instp_sfont_save_report_sample_offset (handle, sam, start / 2);
	}

      sam = instp_sample_next (sam);
    }

  return (INSTP_OK);
}


/**
 * instp_sfont_save_report_sample_offset:
 * handle: SoundFont save handle.
 * sample: Sample object to report new offset of.
 * start: New sample start offset (in samples).
 *
 * Function to report the new offset of a sample in a SoundFont sample
 * chunk being saved. Only useful when overriding the default sample save
 * routine and can only be called from the overriding function.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
instp_sfont_save_report_sample_offset (IPSFontSaveHandle *handle,
				       IPSample *sample, guint start)
{
  IPSampleStore *store;

  g_return_val_if_fail (handle != NULL, INSTP_FAIL);
  g_return_val_if_fail (handle->in_sample_callback, INSTP_FAIL);
  g_return_val_if_fail (sample != NULL, INSTP_FAIL);

  /* create temp sample store to keep track of new location in file */
  if (!(store = instp_sample_store_new (sample->sampledata,
					IPSAMPLE_METHOD_SFONT)))
    return (INSTP_FAIL);
  instp_sample_method_SFONT_set_params (sample->sampledata, store,
					handle->file_info, start);
  handle->tempstores = TRUE;

  return (INSTP_OK);
}

/* save preset headers */
static int
sfsave_phdrs (IPSFontSaveHandle *handle)
{
  IPPreset *p;
  guint16 pbagndx = 0;
  IPFilePHDR phdr;

  /* loop over all presets */
  p = instp_first_preset (handle->sf);

  if (!p) g_warning (_("Saving SoundFont with no presets"));

  while (p)
    {
      memset (&phdr.name, 0, IPFILE_NAME_SIZE);
      strncpy (phdr.name, p->name, IPFILE_NAME_SIZE - 1);
      phdr.psetnum = p->psetnum;
      phdr.bank = p->bank;
      phdr.pbagndx = pbagndx;
      phdr.library = p->library;
      phdr.genre = p->genre;
      phdr.morphology = p->morphology;

      if (ipfile_save_phdr (handle->ctx, &phdr) != INSTP_OK)
	return (INSTP_FAIL);

      pbagndx += instp_item_count (INSTP_ITEM (p->zone));
      p = instp_preset_next (p);
    }

  /* create terminal record */
  memset (&phdr, 0, sizeof (phdr));
  strcpy (phdr.name, "EOP");
  phdr.pbagndx = pbagndx;

  if (ipfile_save_phdr (handle->ctx, &phdr) != INSTP_OK)
    return (INSTP_FAIL);

  return (INSTP_OK);
}

/* save preset bags */
static int
sfsave_pbags (IPSFontSaveHandle *handle)
{
  IPPreset *p;
  IPZone *z;
  IPFileBag bag;
  guint16 genndx = 0, modndx = 0;

  p = instp_first_preset (handle->sf);
  while (p)			/* traverse through presets */
    {
      z = instp_preset_first_zone (p);
      while (z)			/* traverse preset's zones */
	{
	  bag.genndx = genndx;
	  bag.modndx = modndx;

	  if (ipfile_save_bag (handle->ctx, &bag) != INSTP_OK)
	    return (INSTP_FAIL);

	  genndx += instp_zone_gen_count (z);
	  if (z->refitem) genndx++; /* increment for IPGEN_INSTRUMENT_ID */

	  modndx += instp_zone_mod_count (z);

	  z = instp_zone_next (z);
	}
      p = instp_preset_next (p);
    }

  /* terminal record */
  bag.genndx = genndx;
  bag.modndx = modndx;

  if (ipfile_save_bag (handle->ctx, &bag) != INSTP_OK)
    return (INSTP_FAIL);

  return (INSTP_OK);
}

/* save preset modulators */
static int
sfsave_pmods (IPSFontSaveHandle *handle)
{
  IPPreset *p;
  IPZone *z;
  IPMod *m;
  IPFileMod fmod;

  p = instp_first_preset (handle->sf);
  while (p)			/* traverse through all presets */
    {
      z = instp_preset_first_zone (p);
      while (z)			/* traverse this preset's zones */
	{
	  m = instp_zone_first_mod (z);
	  while (m)		/* save zone's modulators */
	    {
	      fmod.src = m->src;
	      fmod.dest = m->dest;
	      fmod.amount = m->amount;
	      fmod.amtsrc = m->amtsrc;
	      fmod.trans = m->trans;

	      if (ipfile_save_mod (handle->ctx, &fmod) != INSTP_OK)
		return (INSTP_FAIL);

	      m = instp_mod_next (m);
	    }
	  z = instp_zone_next (z);
	}
      p = instp_preset_next (p);
    }

  /* terminal record */
  memset (&fmod, 0, sizeof (fmod));

  if (ipfile_save_mod (handle->ctx, &fmod) != INSTP_OK)
    return (INSTP_FAIL);

  return (INSTP_OK);
}

/* save preset generators */
static int
sfsave_pgens (IPSFontSaveHandle *handle)
{
  IPPreset *p;
  IPInst *inst;
  IPZone *z;
  IPGen *g;
  IPFileGen fgen;
  int instid;

  p = instp_first_preset (handle->sf);
  while (p)			/* traverse through all presets */
    {
      z = instp_preset_first_zone (p);
      while (z)			/* traverse preset's zones */
	{
	  g = instp_zone_first_gen (z);
	  while (g)
	    {
	      fgen.id = g->id;
	      fgen.amount = g->amount;

	      if (ipfile_save_gen (handle->ctx, &fgen) != INSTP_OK)
		return (INSTP_FAIL);

	      g = instp_gen_next (g); /* next generator */
	    }

	  /* save instrument ID if any */
	  if (z->refitem)
	    {
	      instid = 0;
	      inst = instp_first_inst (handle->sf);
	      while (inst)
		{
		  if ((void *)inst == (void *)z->refitem) break;
		  instid++;
		  inst = instp_inst_next (inst);
		}

	      if (!inst)
		{
		  g_critical (INSTP_ERRMSG_REFITEM_NOT_FOUND_0);
		  return (INSTP_FAIL);
		}

	      fgen.id = IPGEN_INSTRUMENT_ID;
	      fgen.amount.uword = instid;

	      if (ipfile_save_gen (handle->ctx, &fgen) != INSTP_OK)
		return (INSTP_FAIL);
	    }

	  z = instp_zone_next (z); /* next zone */
	}
      p = instp_preset_next (p); /* next preset */
    }

  /* terminal record */
  memset (&fgen, 0, sizeof (IPFileGen));

  if (ipfile_save_gen (handle->ctx, &fgen) != INSTP_OK)
    return (INSTP_FAIL);

  return (INSTP_OK);
}

/* save instrument headers */
static int
sfsave_ihdrs (IPSFontSaveHandle *handle)
{
  IPInst *inst;
  IPFileIHDR ihdr;
  int ibagndx = 0;

  inst = instp_first_inst (handle->sf);
  while (inst)			/* loop over instruments */
    {
      memset (ihdr.name, 0, IPFILE_NAME_SIZE);
      strncpy (ihdr.name, inst->name, IPFILE_NAME_SIZE - 1);
      ihdr.ibagndx = ibagndx;

      if (ipfile_save_ihdr (handle->ctx, &ihdr) != INSTP_OK)
	return (INSTP_FAIL);

      ibagndx += instp_item_count (INSTP_ITEM (inst->zone));

      inst = instp_inst_next (inst);
    }

  /* terminal record */
  memset (&ihdr, 0, sizeof (IPFileIHDR));
  strcpy (ihdr.name, "EOI");
  ihdr.ibagndx = ibagndx;

  if (ipfile_save_ihdr (handle->ctx, &ihdr) != INSTP_OK)
    return (INSTP_FAIL);

  return (INSTP_OK);
}

/* save instrument bags */
static int
sfsave_ibags (IPSFontSaveHandle *handle)
{
  IPInst *inst;
  IPZone *z;
  IPFileBag bag;
  int genndx = 0, modndx = 0;

  inst = instp_first_inst (handle->sf);
  while (inst)			/* traverse through instruments */
    {
      z = instp_inst_first_zone (inst);
      while (z)			/* traverse instrument's zones */
	{
	  bag.genndx = genndx;
	  bag.modndx = modndx;

	  if (ipfile_save_bag (handle->ctx, &bag) != INSTP_OK)
	    return (INSTP_FAIL);

	  genndx += instp_zone_gen_count (z);
	  if (z->refitem) genndx++; /* increment for IPGEN_SAMPLE_ID */

	  modndx += instp_zone_mod_count (z);

	  z = instp_zone_next (z);
	}
      inst = instp_inst_next (inst);
    }

  /* terminal record */
  bag.genndx = genndx;
  bag.modndx = modndx;

  if (ipfile_save_bag (handle->ctx, &bag) != INSTP_OK)
    return (INSTP_FAIL);

  return (INSTP_OK);
}

/* save instrument modulators */
static int
sfsave_imods (IPSFontSaveHandle *handle)
{
  IPInst *inst;
  IPZone *z;
  IPMod *m;
  IPFileMod fmod;

  inst = instp_first_inst (handle->sf);
  while (inst)			/* traverse through all instruments */
    {
      z = instp_inst_first_zone (inst);
      while (z)			/* traverse this instrument's zones */
	{
	  m = instp_zone_first_mod (z);
	  while (m)		/* save zone's modulators */
	    {
	      fmod.src = m->src;
	      fmod.dest = m->dest;
	      fmod.amount = m->amount;
	      fmod.amtsrc = m->amtsrc;
	      fmod.trans = m->trans;

	      if (ipfile_save_mod (handle->ctx, &fmod) != INSTP_OK)
		return (INSTP_FAIL);

	      m = instp_mod_next (m);
	    }
	  z = instp_zone_next (z);
	}
      inst = instp_inst_next (inst);
    }

  /* terminal record */
  memset (&fmod, 0, sizeof (fmod));

  if (ipfile_save_mod (handle->ctx, &fmod) != INSTP_OK)
    return (INSTP_FAIL);

  return (INSTP_OK);
}

/* save instrument generators */
static int
sfsave_igens (IPSFontSaveHandle *handle)
{
  IPInst *inst;
  IPSample *sample;
  IPZone *z;
  IPGen *g;
  IPFileGen fgen;
  int sampleid;

  inst = instp_first_inst (handle->sf);
  while (inst)			/* traverse through all instruments */
    {
      z = instp_inst_first_zone (inst);
      while (z)			/* traverse instrument's zones */
	{
	  g = instp_zone_first_gen (z);
	  while (g)
	    {
	      fgen.id = g->id;
	      fgen.amount = g->amount;

	      if (ipfile_save_gen (handle->ctx, &fgen) != INSTP_OK)
		return (INSTP_FAIL);

	      g = instp_gen_next (g); /* next generator */
	    }

	  /* save sample ID if any */
	  if (z->refitem)
	    {
	      sampleid = 0;
	      sample = instp_first_sample (handle->sf);
	      while (sample)
		{
		  if ((void *)sample == (void *)z->refitem) break;
		  sampleid++;
		  sample = instp_sample_next (sample);
		}

	      if (!sample)
		{
		  g_critical (_(INSTP_ERRMSG_REFITEM_NOT_FOUND_0));
		  return (INSTP_FAIL);
		}

	      fgen.id = IPGEN_SAMPLE_ID;
	      fgen.amount.uword = sampleid;

	      if (ipfile_save_gen (handle->ctx, &fgen) != INSTP_OK)
		return (INSTP_FAIL);
	    }

	  z = instp_zone_next (z); /* next zone */
	}
      inst = instp_inst_next (inst); /* next instrument */
    }

  /* terminal record */
  memset (&fgen, 0, sizeof (IPFileGen));

  if (ipfile_save_gen (handle->ctx, &fgen) != INSTP_OK)
    return (INSTP_FAIL);

  return (INSTP_OK);
}

/* save sample headers */
static int
sfsave_shdrs (IPSFontSaveHandle *handle)
{
  IPSample *sam;
  IPSampleStore *store;
  IPFileInfo *file_info;
  IPFileSHDR shdr;
  guint32 start;

  sam = instp_first_sample (handle->sf);
  while (sam)			/* traverse all samples */
    {
      if (!(sam->sampletype & IPSAMPLE_TYPE_ROM))
	{
	  /* search for sample store of the new stored sample */
	  store = instp_sample_data_first_store (sam->sampledata);
	  while (store)
	    {
	      if (store->method->type == IPSAMPLE_METHOD_SFONT)
		{
		  instp_sample_method_SFONT_get_params (sam->sampledata, store,
							&file_info, &start);
		  if (file_info == handle->file_info) break;
		}
	      store = instp_sample_store_next (store);
	    }

	  g_return_val_if_fail (store != NULL, INSTP_FAIL);
	}
      else			/* ROM sample */
	{
	  store = instp_sample_data_find_store (sam->sampledata,
						IPSAMPLE_METHOD_ROM, 0);
	  g_return_val_if_fail (store != NULL, INSTP_FAIL);
	  start = instp_sample_method_ROM_get_start (sam->sampledata, store);
	}

      memset (&shdr.name, 0, IPFILE_NAME_SIZE);
      strncpy (shdr.name, sam->name, IPFILE_NAME_SIZE - 1);

      shdr.start = start;
      shdr.end = instp_sample_get_size (sam) + start;
      shdr.loopstart = sam->loopstart + start;
      shdr.loopend = sam->loopend + start;
      shdr.samplerate = sam->samplerate;
      shdr.origpitch = sam->origpitch;
      shdr.pitchadj = sam->pitchadj;
      shdr.samplelink = 0;	/* FIXME */
      shdr.sampletype = sam->sampletype;

      if (ipfile_save_shdr (handle->ctx, &shdr) != INSTP_OK)
	return (INSTP_FAIL);

      sam = instp_sample_next (sam);
    }

  /* terminal record */
  memset (&shdr, 0, sizeof (IPFileSHDR));
  strcpy (shdr.name, "EOS");

  if (ipfile_save_shdr (handle->ctx, &shdr) != INSTP_OK)
    return (INSTP_FAIL);

  return (INSTP_OK);
}
