#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#include "ladspa-util.h"

#define BASE_BUFFER 2 // Base buffer length (s)

#define FADDELAY_DELAY                 0
#define FADDELAY_FB                    1
#define FADDELAY_INPUT                 2
#define FADDELAY_OUTPUT                3

LADSPA_Descriptor *fadDelayDescriptor = NULL;

typedef struct {
	LADSPA_Data *delay;
	LADSPA_Data *fb;
	LADSPA_Data *input;
	LADSPA_Data *output;
	LADSPA_Data *buffer;
	long         buffer_size;
	LADSPA_Data  last_in;
	int          last_phase;
	float        phase;
	long         sample_rate;
	LADSPA_Data run_adding_gain;
} FadDelay;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return fadDelayDescriptor;
	default:
		return NULL;
	}
}

void activateFadDelay(LADSPA_Handle instance) {
	FadDelay *plugin_data = (FadDelay *)instance;
	LADSPA_Data *buffer = plugin_data->buffer;
	long buffer_size = plugin_data->buffer_size;
	LADSPA_Data last_in = plugin_data->last_in;
	int last_phase = plugin_data->last_phase;
	float phase = plugin_data->phase;
	long sample_rate = plugin_data->sample_rate;
	int i;

	for (i = 0; i < buffer_size; i++) {
	        buffer[i] = 0;
	}
	phase = 0;
	last_phase = 0;
	last_in = 0.0f;
	sample_rate = sample_rate;
	plugin_data->buffer = buffer;
	plugin_data->buffer_size = buffer_size;
	plugin_data->last_in = last_in;
	plugin_data->last_phase = last_phase;
	plugin_data->phase = phase;
	plugin_data->sample_rate = sample_rate;

}

void cleanupFadDelay(LADSPA_Handle instance) {
	free(instance);
}

void connectPortFadDelay(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	FadDelay *plugin;

	plugin = (FadDelay *)instance;
	switch (port) {
	case FADDELAY_DELAY:
		plugin->delay = data;
		break;
	case FADDELAY_FB:
		plugin->fb = data;
		break;
	case FADDELAY_INPUT:
		plugin->input = data;
		break;
	case FADDELAY_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateFadDelay(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	FadDelay *plugin_data = (FadDelay *)malloc(sizeof(FadDelay));
	LADSPA_Data *buffer = NULL;
	long buffer_size;
	LADSPA_Data last_in;
	int last_phase;
	float phase;
	long sample_rate;

	sample_rate = s_rate;
	buffer_size = BASE_BUFFER * s_rate;
	buffer = calloc(buffer_size, sizeof(LADSPA_Data));
	phase = 0;
	last_phase = 0;
	last_in = 0.0f;

	plugin_data->buffer = buffer;
	plugin_data->buffer_size = buffer_size;
	plugin_data->last_in = last_in;
	plugin_data->last_phase = last_phase;
	plugin_data->phase = phase;
	plugin_data->sample_rate = sample_rate;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runFadDelay(LADSPA_Handle instance, unsigned long sample_count) {
	FadDelay *plugin_data = (FadDelay *)instance;

	/* Delay (seconds) (float value) */
	LADSPA_Data delay = *(plugin_data->delay);

	/* Feedback (dB) (float value) */
	LADSPA_Data fb = *(plugin_data->fb);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	LADSPA_Data * buffer = plugin_data->buffer;
	long buffer_size = plugin_data->buffer_size;
	LADSPA_Data last_in = plugin_data->last_in;
	int last_phase = plugin_data->last_phase;
	float phase = plugin_data->phase;
	long sample_rate = plugin_data->sample_rate;

	long int pos;
	float increment = (float)buffer_size / ((float)sample_rate * delay);
	float lin_int, lin_inc;
	int track;
	int fph;
	LADSPA_Data out;
	
	fb = DB_CO(fb);
	
	for (pos = 0; pos < sample_count; pos++) {
	        fph = (int)floor(phase);
	        last_phase = fph;
	        lin_int = phase - (float)fph;
	        out = LIN_INTERP(lin_int, buffer[(fph+1) % buffer_size],
	         buffer[(fph+2) % buffer_size]);
	        phase += increment;
	        lin_inc = 1.0f / (floor(phase) - last_phase + 1);
	        lin_inc = lin_inc > 1.0f ? 1.0f : lin_inc;
	        lin_int = 0.0f;
	        for (track = last_phase; track < phase; track++) {
	                lin_int += lin_inc;
	                buffer[track % buffer_size] = out * fb +
	                 LIN_INTERP(lin_int, last_in, input[pos]);
	        }
	        last_in = input[pos];
	        buffer_write(output[pos], out);
	        if (phase >= buffer_size) {
	                phase -= buffer_size;
	        }
	}
	
	// Store current phase in instance
	plugin_data->phase = phase;
	plugin_data->last_phase = last_phase;
	plugin_data->last_in = last_in;
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainFadDelay(LADSPA_Handle instance, LADSPA_Data gain) {
	((FadDelay *)instance)->run_adding_gain = gain;
}

void runAddingFadDelay(LADSPA_Handle instance, unsigned long sample_count) {
	FadDelay *plugin_data = (FadDelay *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Delay (seconds) (float value) */
	LADSPA_Data delay = *(plugin_data->delay);

	/* Feedback (dB) (float value) */
	LADSPA_Data fb = *(plugin_data->fb);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	LADSPA_Data * buffer = plugin_data->buffer;
	long buffer_size = plugin_data->buffer_size;
	LADSPA_Data last_in = plugin_data->last_in;
	int last_phase = plugin_data->last_phase;
	float phase = plugin_data->phase;
	long sample_rate = plugin_data->sample_rate;

	long int pos;
	float increment = (float)buffer_size / ((float)sample_rate * delay);
	float lin_int, lin_inc;
	int track;
	int fph;
	LADSPA_Data out;
	
	fb = DB_CO(fb);
	
	for (pos = 0; pos < sample_count; pos++) {
	        fph = (int)floor(phase);
	        last_phase = fph;
	        lin_int = phase - (float)fph;
	        out = LIN_INTERP(lin_int, buffer[(fph+1) % buffer_size],
	         buffer[(fph+2) % buffer_size]);
	        phase += increment;
	        lin_inc = 1.0f / (floor(phase) - last_phase + 1);
	        lin_inc = lin_inc > 1.0f ? 1.0f : lin_inc;
	        lin_int = 0.0f;
	        for (track = last_phase; track < phase; track++) {
	                lin_int += lin_inc;
	                buffer[track % buffer_size] = out * fb +
	                 LIN_INTERP(lin_int, last_in, input[pos]);
	        }
	        last_in = input[pos];
	        buffer_write(output[pos], out);
	        if (phase >= buffer_size) {
	                phase -= buffer_size;
	        }
	}
	
	// Store current phase in instance
	plugin_data->phase = phase;
	plugin_data->last_phase = last_phase;
	plugin_data->last_in = last_in;
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	fadDelayDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (fadDelayDescriptor) {
		fadDelayDescriptor->UniqueID = 1192;
		fadDelayDescriptor->Label = strdup("fadDelay");
		fadDelayDescriptor->Properties =
		 0;
		fadDelayDescriptor->Name =
		 strdup("Fractionally Addressed Delay Line");
		fadDelayDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		fadDelayDescriptor->Copyright =
		 strdup("GPL");
		fadDelayDescriptor->PortCount = 4;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(4,
		 sizeof(LADSPA_PortDescriptor));
		fadDelayDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(4,
		 sizeof(LADSPA_PortRangeHint));
		fadDelayDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(4, sizeof(char*));
		fadDelayDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Delay (seconds) */
		port_descriptors[FADDELAY_DELAY] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[FADDELAY_DELAY] =
		 strdup("Delay (seconds)");
		port_range_hints[FADDELAY_DELAY].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[FADDELAY_DELAY].LowerBound = 0.1;
		port_range_hints[FADDELAY_DELAY].UpperBound = 10;

		/* Parameters for Feedback (dB) */
		port_descriptors[FADDELAY_FB] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[FADDELAY_FB] =
		 strdup("Feedback (dB)");
		port_range_hints[FADDELAY_FB].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[FADDELAY_FB].LowerBound = -70;
		port_range_hints[FADDELAY_FB].UpperBound = 0;

		/* Parameters for Input */
		port_descriptors[FADDELAY_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[FADDELAY_INPUT] =
		 strdup("Input");
		port_range_hints[FADDELAY_INPUT].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[FADDELAY_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[FADDELAY_OUTPUT] =
		 strdup("Output");
		port_range_hints[FADDELAY_OUTPUT].HintDescriptor = 0;

		fadDelayDescriptor->activate = activateFadDelay;
		fadDelayDescriptor->cleanup = cleanupFadDelay;
		fadDelayDescriptor->connect_port = connectPortFadDelay;
		fadDelayDescriptor->deactivate = NULL;
		fadDelayDescriptor->instantiate = instantiateFadDelay;
		fadDelayDescriptor->run = runFadDelay;
		fadDelayDescriptor->run_adding = runAddingFadDelay;
		fadDelayDescriptor->set_run_adding_gain = setRunAddingGainFadDelay;
	}
}

void _fini() {
	int i;

	if (fadDelayDescriptor) {
		free((char *)fadDelayDescriptor->Label);
		free((char *)fadDelayDescriptor->Name);
		free((char *)fadDelayDescriptor->Maker);
		free((char *)fadDelayDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)fadDelayDescriptor->PortDescriptors);
		for (i = 0; i < fadDelayDescriptor->PortCount; i++)
			free((char *)(fadDelayDescriptor->PortNames[i]));
		free((char **)fadDelayDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)fadDelayDescriptor->PortRangeHints);
		free(fadDelayDescriptor);
	}

}
