#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#include "ladspa-util.h"

inline void rc_window(float *window, float *meta_window, long law_p, long max_law_p) {
        long i;
        float step, track = 0.0f;

        step = (float)max_law_p / (float)law_p;

        // Fill window with a raised cosine function
        for (i=0; i<law_p; i++) {
                window[i] = meta_window[f_round(track)];
                window[law_p * 2 - i] = window[i];
                track += step;
        }
        window[law_p] = 1.0f;
}

#define FLANGER_DELAY_BASE             0
#define FLANGER_DETUNE                 1
#define FLANGER_LAW_FREQ               2
#define FLANGER_FB                     3
#define FLANGER_INPUT                  4
#define FLANGER_OUTPUT                 5

LADSPA_Descriptor *flangerDescriptor = NULL;

typedef struct {
	LADSPA_Data *delay_base;
	LADSPA_Data *detune;
	LADSPA_Data *law_freq;
	LADSPA_Data *fb;
	LADSPA_Data *input;
	LADSPA_Data *output;
	long         count;
	long         d_max_law_p;
	long         delay_pos;
	long         delay_size;
	float *      delay_tbl;
	long         last_law_p;
	long         law_pos;
	float *      law_tbl;
	long         max_law_p;
	float *      meta_window;
	long         old_d_base;
	long         sample_rate;
	float *      window;
	LADSPA_Data run_adding_gain;
} Flanger;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return flangerDescriptor;
	default:
		return NULL;
	}
}

void cleanupFlanger(LADSPA_Handle instance) {
	Flanger *plugin_data = (Flanger *)instance;
	free(plugin_data->law_tbl);
	free(plugin_data->delay_tbl);
	free(plugin_data->window);
	free(plugin_data->meta_window);
	free(instance);
}

void connectPortFlanger(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	Flanger *plugin;

	plugin = (Flanger *)instance;
	switch (port) {
	case FLANGER_DELAY_BASE:
		plugin->delay_base = data;
		break;
	case FLANGER_DETUNE:
		plugin->detune = data;
		break;
	case FLANGER_LAW_FREQ:
		plugin->law_freq = data;
		break;
	case FLANGER_FB:
		plugin->fb = data;
		break;
	case FLANGER_INPUT:
		plugin->input = data;
		break;
	case FLANGER_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateFlanger(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	Flanger *plugin_data = (Flanger *)malloc(sizeof(Flanger));
	long count;
	long d_max_law_p;
	long delay_pos;
	long delay_size;
	float *delay_tbl = NULL;
	long last_law_p;
	long law_pos;
	float *law_tbl = NULL;
	long max_law_p;
	float *meta_window = NULL;
	long old_d_base;
	long sample_rate;
	float *window = NULL;

	long i;
	
	sample_rate = s_rate;
	
	max_law_p = s_rate*2;
	d_max_law_p = max_law_p * 2;
	last_law_p = -1;
	law_tbl = calloc(sizeof(float), max_law_p * 2);
	law_pos = 0;
	
	meta_window = calloc(sizeof(float), max_law_p);
	window = calloc(sizeof(float), max_law_p * 2);
	for (i=0; i<max_law_p; i++) {
	        meta_window[i] = cos(M_PI * (float)(max_law_p - i) / (float)d_max_law_p);
	        meta_window[i] *= meta_window[i];
	}
	
	delay_size = sample_rate/2;
	delay_tbl = calloc(sizeof(float), delay_size);
	delay_pos = 0;
	
	count = 0;
	
	old_d_base = 0;

	plugin_data->count = count;
	plugin_data->d_max_law_p = d_max_law_p;
	plugin_data->delay_pos = delay_pos;
	plugin_data->delay_size = delay_size;
	plugin_data->delay_tbl = delay_tbl;
	plugin_data->last_law_p = last_law_p;
	plugin_data->law_pos = law_pos;
	plugin_data->law_tbl = law_tbl;
	plugin_data->max_law_p = max_law_p;
	plugin_data->meta_window = meta_window;
	plugin_data->old_d_base = old_d_base;
	plugin_data->sample_rate = sample_rate;
	plugin_data->window = window;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runFlanger(LADSPA_Handle instance, unsigned long sample_count) {
	Flanger *plugin_data = (Flanger *)instance;

	/* Delay base (ms) (float value) */
	LADSPA_Data delay_base = *(plugin_data->delay_base);

	/* Max slowdown (ms) (float value) */
	LADSPA_Data detune = *(plugin_data->detune);

	/* LFO frequency (Hz) (float value) */
	LADSPA_Data law_freq = *(plugin_data->law_freq);

	/* Feedback (float value) */
	LADSPA_Data fb = *(plugin_data->fb);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	long count = plugin_data->count;
	long d_max_law_p = plugin_data->d_max_law_p;
	long delay_pos = plugin_data->delay_pos;
	long delay_size = plugin_data->delay_size;
	float * delay_tbl = plugin_data->delay_tbl;
	long last_law_p = plugin_data->last_law_p;
	long law_pos = plugin_data->law_pos;
	float * law_tbl = plugin_data->law_tbl;
	long max_law_p = plugin_data->max_law_p;
	float * meta_window = plugin_data->meta_window;
	long old_d_base = plugin_data->old_d_base;
	long sample_rate = plugin_data->sample_rate;
	float * window = plugin_data->window;

	unsigned long pos;
	long i, d_base, new_d_base;
	LADSPA_Data out;
	float delay_depth;
	float dp; // float delay position
	float dp_frac; // fractional part
	long dp_idx; // Integer delay index
	long law_p; // Period of law
	float frac = 0.0f, step; // Portion the way through the block
	
	// Set law params
	law_p = LIMIT((float)sample_rate / law_freq, 1, max_law_p);
	
	// Calculate base delay size in samples
	new_d_base = (LIMIT(delay_base, 0, 25) * sample_rate) / 1000;
	
	// Calculate delay depth in samples
	delay_depth = LIMIT(detune * (float)sample_rate * 0.001f, 0, delay_size - new_d_base - 1);
	
	step = 1.0f/sample_count;
	for (pos = 0; pos < sample_count; pos++) {
	        // Once per law 'frequency' splurge a new set of windowed data
	        // into the law buffer.
	        if (count++ > law_p) {
	                // Value for amplitude of law peak
	                float n = (float)rand() / (float)RAND_MAX;
	
	                count = 0;
	
	                // Is the law table up to date?
	                if (last_law_p != law_p) {
	                        rc_window(window, meta_window, law_p, max_law_p);
	                        last_law_p = law_p;
	                }
	
	                // Interpolate into law table
	                for (i = 0; i<law_p*2; i++) {
	                        law_tbl[(i+law_pos) % (d_max_law_p)] += window[i] * n;
	                }
	        }
	
	        // Calculate position in delay table
	        d_base = LIN_INTERP(frac, old_d_base, new_d_base);
	        dp = (float)(delay_pos - d_base) - (delay_depth * law_tbl[law_pos]);
	        // Get the integer part
	        dp_idx = (long)dp;
	        // Get the fractional part
	        dp_frac = dp - dp_idx;
	        // Calculate the modulo'd table index
	        dp_idx = MOD(dp_idx, delay_size);
	
	        // Zero the used law buffer
	        law_tbl[law_pos] = 0.0f;
	
	        // Accumulate into output buffer
	        out = cube_interp(dp_frac, delay_tbl[MOD(dp_idx-1, delay_size)], delay_tbl[dp_idx], delay_tbl[(dp_idx+1) % delay_size], delay_tbl[(dp_idx+2) % delay_size]);
	
	        // Increment law pos
	        law_pos = (law_pos + 1) % (d_max_law_p);
	
	        // Store new delay value
	        delay_tbl[delay_pos] = input[pos] + (fb * out);
	        delay_pos = (delay_pos + 1) % delay_size;
	
	        buffer_write(output[pos], (out + input[pos]) * 0.75f);
	        frac += step;
	}
	
	plugin_data->count = count;
	plugin_data->law_pos = law_pos;
	plugin_data->last_law_p = last_law_p;
	plugin_data->delay_pos = delay_pos;
	plugin_data->old_d_base = new_d_base;
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainFlanger(LADSPA_Handle instance, LADSPA_Data gain) {
	((Flanger *)instance)->run_adding_gain = gain;
}

void runAddingFlanger(LADSPA_Handle instance, unsigned long sample_count) {
	Flanger *plugin_data = (Flanger *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Delay base (ms) (float value) */
	LADSPA_Data delay_base = *(plugin_data->delay_base);

	/* Max slowdown (ms) (float value) */
	LADSPA_Data detune = *(plugin_data->detune);

	/* LFO frequency (Hz) (float value) */
	LADSPA_Data law_freq = *(plugin_data->law_freq);

	/* Feedback (float value) */
	LADSPA_Data fb = *(plugin_data->fb);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	long count = plugin_data->count;
	long d_max_law_p = plugin_data->d_max_law_p;
	long delay_pos = plugin_data->delay_pos;
	long delay_size = plugin_data->delay_size;
	float * delay_tbl = plugin_data->delay_tbl;
	long last_law_p = plugin_data->last_law_p;
	long law_pos = plugin_data->law_pos;
	float * law_tbl = plugin_data->law_tbl;
	long max_law_p = plugin_data->max_law_p;
	float * meta_window = plugin_data->meta_window;
	long old_d_base = plugin_data->old_d_base;
	long sample_rate = plugin_data->sample_rate;
	float * window = plugin_data->window;

	unsigned long pos;
	long i, d_base, new_d_base;
	LADSPA_Data out;
	float delay_depth;
	float dp; // float delay position
	float dp_frac; // fractional part
	long dp_idx; // Integer delay index
	long law_p; // Period of law
	float frac = 0.0f, step; // Portion the way through the block
	
	// Set law params
	law_p = LIMIT((float)sample_rate / law_freq, 1, max_law_p);
	
	// Calculate base delay size in samples
	new_d_base = (LIMIT(delay_base, 0, 25) * sample_rate) / 1000;
	
	// Calculate delay depth in samples
	delay_depth = LIMIT(detune * (float)sample_rate * 0.001f, 0, delay_size - new_d_base - 1);
	
	step = 1.0f/sample_count;
	for (pos = 0; pos < sample_count; pos++) {
	        // Once per law 'frequency' splurge a new set of windowed data
	        // into the law buffer.
	        if (count++ > law_p) {
	                // Value for amplitude of law peak
	                float n = (float)rand() / (float)RAND_MAX;
	
	                count = 0;
	
	                // Is the law table up to date?
	                if (last_law_p != law_p) {
	                        rc_window(window, meta_window, law_p, max_law_p);
	                        last_law_p = law_p;
	                }
	
	                // Interpolate into law table
	                for (i = 0; i<law_p*2; i++) {
	                        law_tbl[(i+law_pos) % (d_max_law_p)] += window[i] * n;
	                }
	        }
	
	        // Calculate position in delay table
	        d_base = LIN_INTERP(frac, old_d_base, new_d_base);
	        dp = (float)(delay_pos - d_base) - (delay_depth * law_tbl[law_pos]);
	        // Get the integer part
	        dp_idx = (long)dp;
	        // Get the fractional part
	        dp_frac = dp - dp_idx;
	        // Calculate the modulo'd table index
	        dp_idx = MOD(dp_idx, delay_size);
	
	        // Zero the used law buffer
	        law_tbl[law_pos] = 0.0f;
	
	        // Accumulate into output buffer
	        out = cube_interp(dp_frac, delay_tbl[MOD(dp_idx-1, delay_size)], delay_tbl[dp_idx], delay_tbl[(dp_idx+1) % delay_size], delay_tbl[(dp_idx+2) % delay_size]);
	
	        // Increment law pos
	        law_pos = (law_pos + 1) % (d_max_law_p);
	
	        // Store new delay value
	        delay_tbl[delay_pos] = input[pos] + (fb * out);
	        delay_pos = (delay_pos + 1) % delay_size;
	
	        buffer_write(output[pos], (out + input[pos]) * 0.75f);
	        frac += step;
	}
	
	plugin_data->count = count;
	plugin_data->law_pos = law_pos;
	plugin_data->last_law_p = last_law_p;
	plugin_data->delay_pos = delay_pos;
	plugin_data->old_d_base = new_d_base;
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	flangerDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (flangerDescriptor) {
		flangerDescriptor->UniqueID = 1191;
		flangerDescriptor->Label = strdup("flanger");
		flangerDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		flangerDescriptor->Name =
		 strdup("Flanger");
		flangerDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		flangerDescriptor->Copyright =
		 strdup("GPL");
		flangerDescriptor->PortCount = 6;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(6,
		 sizeof(LADSPA_PortDescriptor));
		flangerDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(6,
		 sizeof(LADSPA_PortRangeHint));
		flangerDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(6, sizeof(char*));
		flangerDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Delay base (ms) */
		port_descriptors[FLANGER_DELAY_BASE] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[FLANGER_DELAY_BASE] =
		 strdup("Delay base (ms)");
		port_range_hints[FLANGER_DELAY_BASE].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[FLANGER_DELAY_BASE].LowerBound = 0.1;
		port_range_hints[FLANGER_DELAY_BASE].UpperBound = 25;

		/* Parameters for Max slowdown (ms) */
		port_descriptors[FLANGER_DETUNE] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[FLANGER_DETUNE] =
		 strdup("Max slowdown (ms)");
		port_range_hints[FLANGER_DETUNE].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[FLANGER_DETUNE].LowerBound = 0;
		port_range_hints[FLANGER_DETUNE].UpperBound = 10;

		/* Parameters for LFO frequency (Hz) */
		port_descriptors[FLANGER_LAW_FREQ] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[FLANGER_LAW_FREQ] =
		 strdup("LFO frequency (Hz)");
		port_range_hints[FLANGER_LAW_FREQ].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[FLANGER_LAW_FREQ].LowerBound = 0.5;
		port_range_hints[FLANGER_LAW_FREQ].UpperBound = 100;

		/* Parameters for Feedback */
		port_descriptors[FLANGER_FB] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[FLANGER_FB] =
		 strdup("Feedback");
		port_range_hints[FLANGER_FB].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[FLANGER_FB].LowerBound = -1;
		port_range_hints[FLANGER_FB].UpperBound = 1;

		/* Parameters for Input */
		port_descriptors[FLANGER_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[FLANGER_INPUT] =
		 strdup("Input");
		port_range_hints[FLANGER_INPUT].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[FLANGER_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[FLANGER_OUTPUT] =
		 strdup("Output");
		port_range_hints[FLANGER_OUTPUT].HintDescriptor = 0;

		flangerDescriptor->activate = NULL;
		flangerDescriptor->cleanup = cleanupFlanger;
		flangerDescriptor->connect_port = connectPortFlanger;
		flangerDescriptor->deactivate = NULL;
		flangerDescriptor->instantiate = instantiateFlanger;
		flangerDescriptor->run = runFlanger;
		flangerDescriptor->run_adding = runAddingFlanger;
		flangerDescriptor->set_run_adding_gain = setRunAddingGainFlanger;
	}
}

void _fini() {
	int i;

	if (flangerDescriptor) {
		free((char *)flangerDescriptor->Label);
		free((char *)flangerDescriptor->Name);
		free((char *)flangerDescriptor->Maker);
		free((char *)flangerDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)flangerDescriptor->PortDescriptors);
		for (i = 0; i < flangerDescriptor->PortCount; i++)
			free((char *)(flangerDescriptor->PortNames[i]));
		free((char **)flangerDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)flangerDescriptor->PortRangeHints);
		free(flangerDescriptor);
	}

}
