#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#include "ladspa-util.h"

#define LN_2_2 0.34657359 // ln(2)/2

#define SINGLEPARA_GAIN                0
#define SINGLEPARA_FC                  1
#define SINGLEPARA_BW                  2
#define SINGLEPARA_INPUT               3
#define SINGLEPARA_OUTPUT              4

LADSPA_Descriptor *singleParaDescriptor = NULL;

typedef struct {
	LADSPA_Data *gain;
	LADSPA_Data *fc;
	LADSPA_Data *bw;
	LADSPA_Data *input;
	LADSPA_Data *output;
	long         sample_rate;
	LADSPA_Data  z0;
	LADSPA_Data  z1;
	LADSPA_Data  z2;
	LADSPA_Data run_adding_gain;
} SinglePara;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return singleParaDescriptor;
	default:
		return NULL;
	}
}

void cleanupSinglePara(LADSPA_Handle instance) {
	free(instance);
}

void connectPortSinglePara(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	SinglePara *plugin;

	plugin = (SinglePara *)instance;
	switch (port) {
	case SINGLEPARA_GAIN:
		plugin->gain = data;
		break;
	case SINGLEPARA_FC:
		plugin->fc = data;
		break;
	case SINGLEPARA_BW:
		plugin->bw = data;
		break;
	case SINGLEPARA_INPUT:
		plugin->input = data;
		break;
	case SINGLEPARA_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateSinglePara(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	SinglePara *plugin_data = (SinglePara *)malloc(sizeof(SinglePara));
	long sample_rate;
	LADSPA_Data z0;
	LADSPA_Data z1;
	LADSPA_Data z2;

	sample_rate = s_rate;
	
	z0 = 0.0f;
	z1 = 0.0f;
	z2 = 0.0f;

	plugin_data->sample_rate = sample_rate;
	plugin_data->z0 = z0;
	plugin_data->z1 = z1;
	plugin_data->z2 = z2;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runSinglePara(LADSPA_Handle instance, unsigned long sample_count) {
	SinglePara *plugin_data = (SinglePara *)instance;

	/* Gain (dB) (float value) */
	LADSPA_Data gain = *(plugin_data->gain);

	/* Frequency (Hz) (float value) */
	LADSPA_Data fc = *(plugin_data->fc);

	/* Bandwidth (octaves) (float value) */
	LADSPA_Data bw = *(plugin_data->bw);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	long sample_rate = plugin_data->sample_rate;
	LADSPA_Data z0 = plugin_data->z0;
	LADSPA_Data z1 = plugin_data->z1;
	LADSPA_Data z2 = plugin_data->z2;

	unsigned long pos;
	
	/* Biquad filter (adapted from code by Eli Brandt,
	   http://www.cs.cmu.edu/~eli/) */
	
	float w = 2.0f * M_PI * LIMIT(fc, 0.01f, sample_rate/2.0f) / (float)sample_rate;
	float cw = cos(w);
	float sw = sin(w);
	float J = pow(10.0f, gain / 40.0f);
	float g = sw * sinh(LN_2_2 * bw * w / sw);
	float a0r = 1.0f / (1.0 + (g / J));
	float b0 = (1.0 + (g * J)) * a0r;
	float b1 = (-2.0 * cw) * a0r;
	float b2 = (1.0 - (g * J)) * a0r;
	float a1 = -b1;
	float a2 = ((g / J) - 1.0) * a0r;
	
	for (pos = 0; pos < sample_count; pos++) {
	        z0 = input[pos] + a1*z1 + a2*z2;
	        buffer_write(output[pos], z0*b0 + z1*b1 + z2*b2);
	        z2 = z1; z1 = z0;
	}
	
	plugin_data->z0 = z0;
	plugin_data->z1 = z1;
	plugin_data->z2 = z2;
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainSinglePara(LADSPA_Handle instance, LADSPA_Data gain) {
	((SinglePara *)instance)->run_adding_gain = gain;
}

void runAddingSinglePara(LADSPA_Handle instance, unsigned long sample_count) {
	SinglePara *plugin_data = (SinglePara *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Gain (dB) (float value) */
	LADSPA_Data gain = *(plugin_data->gain);

	/* Frequency (Hz) (float value) */
	LADSPA_Data fc = *(plugin_data->fc);

	/* Bandwidth (octaves) (float value) */
	LADSPA_Data bw = *(plugin_data->bw);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	long sample_rate = plugin_data->sample_rate;
	LADSPA_Data z0 = plugin_data->z0;
	LADSPA_Data z1 = plugin_data->z1;
	LADSPA_Data z2 = plugin_data->z2;

	unsigned long pos;
	
	/* Biquad filter (adapted from code by Eli Brandt,
	   http://www.cs.cmu.edu/~eli/) */
	
	float w = 2.0f * M_PI * LIMIT(fc, 0.01f, sample_rate/2.0f) / (float)sample_rate;
	float cw = cos(w);
	float sw = sin(w);
	float J = pow(10.0f, gain / 40.0f);
	float g = sw * sinh(LN_2_2 * bw * w / sw);
	float a0r = 1.0f / (1.0 + (g / J));
	float b0 = (1.0 + (g * J)) * a0r;
	float b1 = (-2.0 * cw) * a0r;
	float b2 = (1.0 - (g * J)) * a0r;
	float a1 = -b1;
	float a2 = ((g / J) - 1.0) * a0r;
	
	for (pos = 0; pos < sample_count; pos++) {
	        z0 = input[pos] + a1*z1 + a2*z2;
	        buffer_write(output[pos], z0*b0 + z1*b1 + z2*b2);
	        z2 = z1; z1 = z0;
	}
	
	plugin_data->z0 = z0;
	plugin_data->z1 = z1;
	plugin_data->z2 = z2;
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	singleParaDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (singleParaDescriptor) {
		singleParaDescriptor->UniqueID = 1203;
		singleParaDescriptor->Label = strdup("singlePara");
		singleParaDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		singleParaDescriptor->Name =
		 strdup("Single band parametric");
		singleParaDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		singleParaDescriptor->Copyright =
		 strdup("GPL");
		singleParaDescriptor->PortCount = 5;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(5,
		 sizeof(LADSPA_PortDescriptor));
		singleParaDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(5,
		 sizeof(LADSPA_PortRangeHint));
		singleParaDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(5, sizeof(char*));
		singleParaDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Gain (dB) */
		port_descriptors[SINGLEPARA_GAIN] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[SINGLEPARA_GAIN] =
		 strdup("Gain (dB)");
		port_range_hints[SINGLEPARA_GAIN].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[SINGLEPARA_GAIN].LowerBound = -70;
		port_range_hints[SINGLEPARA_GAIN].UpperBound = +30;

		/* Parameters for Frequency (Hz) */
		port_descriptors[SINGLEPARA_FC] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[SINGLEPARA_FC] =
		 strdup("Frequency (Hz)");
		port_range_hints[SINGLEPARA_FC].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_SAMPLE_RATE;
		port_range_hints[SINGLEPARA_FC].LowerBound = 0;
		port_range_hints[SINGLEPARA_FC].UpperBound = 0.5;

		/* Parameters for Bandwidth (octaves) */
		port_descriptors[SINGLEPARA_BW] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[SINGLEPARA_BW] =
		 strdup("Bandwidth (octaves)");
		port_range_hints[SINGLEPARA_BW].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[SINGLEPARA_BW].LowerBound = 0;
		port_range_hints[SINGLEPARA_BW].UpperBound = 4;

		/* Parameters for Input */
		port_descriptors[SINGLEPARA_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[SINGLEPARA_INPUT] =
		 strdup("Input");
		port_range_hints[SINGLEPARA_INPUT].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[SINGLEPARA_INPUT].LowerBound = -1.0;
		port_range_hints[SINGLEPARA_INPUT].UpperBound = +1.0;

		/* Parameters for Output */
		port_descriptors[SINGLEPARA_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[SINGLEPARA_OUTPUT] =
		 strdup("Output");
		port_range_hints[SINGLEPARA_OUTPUT].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[SINGLEPARA_OUTPUT].LowerBound = -1.0;
		port_range_hints[SINGLEPARA_OUTPUT].UpperBound = +1.0;

		singleParaDescriptor->activate = NULL;
		singleParaDescriptor->cleanup = cleanupSinglePara;
		singleParaDescriptor->connect_port = connectPortSinglePara;
		singleParaDescriptor->deactivate = NULL;
		singleParaDescriptor->instantiate = instantiateSinglePara;
		singleParaDescriptor->run = runSinglePara;
		singleParaDescriptor->run_adding = runAddingSinglePara;
		singleParaDescriptor->set_run_adding_gain = setRunAddingGainSinglePara;
	}
}

void _fini() {
	int i;

	if (singleParaDescriptor) {
		free((char *)singleParaDescriptor->Label);
		free((char *)singleParaDescriptor->Name);
		free((char *)singleParaDescriptor->Maker);
		free((char *)singleParaDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)singleParaDescriptor->PortDescriptors);
		for (i = 0; i < singleParaDescriptor->PortCount; i++)
			free((char *)(singleParaDescriptor->PortNames[i]));
		free((char **)singleParaDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)singleParaDescriptor->PortRangeHints);
		free(singleParaDescriptor);
	}

}
