// Copyright (C) 2009-2009 Martin Sandve Alnæs
// Licensed under the GNU LGPL Version 3.0.
//
// First added:  2009-01-01
// Last changed: 2009-04-01
//
// This demo program solves Poissons equation in 1D.

#include <dolfin.h>
#include "generated_code/Poisson1D.h"

using namespace dolfin;
using namespace Poisson1D;

class Weight: public Function
{
public:
  Weight(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    values[0] = 1.0;
  }
};

class Source: public Function
{
public:
  Source(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    double dx = x[0] - 0.5;
    values[0] = 10.0;
  }
};

class BoundarySource: public Function
{
public:
  BoundarySource(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    values[0] = 10.0;
  }
};

class BoundaryValue: public Function
{
public:
  BoundaryValue(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    values[0] = x[0];
  }
};

class EntireBoundary: public SubDomain
{
  bool inside(const double* x, bool on_boundary) const
  {
    return x[0] < DOLFIN_EPS || x[0] > 1.0-DOLFIN_EPS;
  }
};

int main(int argc, char ** argv)
{
    UnitInterval mesh(10);
    
    BilinearForm::TrialSpace V(mesh);
    CoefficientSpace_c C(mesh);
    CoefficientSpace_f F(mesh);
    CoefficientSpace_g G(mesh);
    
    BilinearForm a(V, V);
    LinearForm L(V);
    
    Weight c(C);
    Source f(F);
    BoundarySource g(G);
    BoundaryValue ubc(V);
    
    a.c = c;
    L.f = f;
    L.g = g;
    
    EntireBoundary boundary;
    DirichletBC bc(V, ubc, boundary);
    
    VariationalProblem problem(a, L, bc);
    problem.set("linear solver", "iterative");
    problem.set("symmetric", true);
    Function u(V);
    problem.solve(u);
    
    File ufile("u.pvd");
    ufile << u;
    
    //plot(u);
}

