// Copyright (C) 2009-2009 Martin Sandve Alnæs
// Licensed under the GNU LGPL Version 3.0.
//
// First added:  2009-01-01
// Last changed: 2009-04-01
//
// This demo program solves Poissons equation in 2D with
// the solution projected to a fixed value in the middle.

#include <vector>
#include <dolfin.h>
#include "generated_code/Forms.h"

using std::vector;
using namespace dolfin;
using namespace Forms;

class Source: public Function
{
public:
  Source(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    double dx = x[0] - 0.5;
    double dy = x[1] - 0.5;
    values[0] = 0.0*dx*dy;
  }
};

class Value: public Function
{
public:
  Value(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    double dx = x[0] - 0.5;
    double dy = x[1] - 0.5;
    values[0] = 2.0*exp(-10.0*(dx*dx+dy*dy))-1;
  }
};

class BoundaryValue: public Function
{
public:
  BoundaryValue(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    double dx = x[0] - 0.5;
    double dy = x[1] - 0.5;
    values[0] = dx + dy;
  }
};

class ValueDomain: public SubDomain
{
  bool inside(const double* x, bool on_boundary) const
  {
    double dx = x[0] - 0.5;
    double dy = x[1] - 0.5;
    double r = 0.2;
    return (dx*dx + dy*dy) < r*r;
  }
};

class DirichletBoundary: public SubDomain
{
  bool inside(const double* x, bool on_boundary) const
  {
    return x[1] < DOLFIN_EPS || x[1] > 1.0-DOLFIN_EPS;
  }
};

int main()
{
    unsigned n = 100;
    UnitSquare mesh(n, n);

    MeshFunction<unsigned> domains(mesh, 2);
    ValueDomain vdomain;
    vdomain.mark(domains, 1);
    
    BilinearForm::TrialSpace V(mesh);
    CoefficientSpace_f F(mesh);
    CoefficientSpace_g G(mesh);
    
    BilinearForm a(V, V);
    LinearForm L(V);
    
    Source f(F);
    Value g(G);
    BoundaryValue ubc(V);
    
    L.f = f;
    L.g = g;
    
    DirichletBoundary boundary;
    DirichletBC bc(V, ubc, boundary);
    
    vector<BoundaryCondition*> bcs;
    bcs.push_back(&bc);

    VariationalProblem problem(a, L, bcs, &domains, 0, 0, false);
    Function u(V);
    problem.solve(u);
    
    File ufile("u.pvd");
    ufile << u;
    
    //plot(u);
}

