package Initrd::Ramdisk;

#   $Header: /cvsroot/systemconfig/systemconfig/lib/Initrd/Ramdisk.pm,v 1.14 2002/02/25 20:08:06 sdague Exp $

#   Copyright (c) 2001 International Business Machines

#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
 
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
 
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#   Vasilios Hoffman <greekboy@users.sourceforge.net>

use strict;
use Carp;
use Util::Log qw(:all);
use Util::Cmd qw(:all);
use File::Copy;
use Data::Dumper;

use vars qw($VERSION);

$VERSION = sprintf("%d.%02d", q$Revision: 1.14 $ =~ /(\d+)\.(\d+)/);

=head1 NAME

Initrd::Ramdisk - ramdisk object used by SystemConfg::Initrd

=head1 DESCRIPTION

This package represents the ramdisk objects used by Initrd while
building kernel initrds.

=head1 AUTHOR

  Vasilios Hoffman <greekboy@users.sourceforge.net>

=head1 SEE ALSO

L<Config>, 
L<perl>

=cut

sub new {
    my $class = shift;
    my %this = (
                root => "",
                # preset containers for information
                format_cmds => {
                                ext2 => "mke2fs",
                                cramfs => "mkcramfs",
                               },
                # empty containers for information
                filesmod => [],
                module_list => [],
                file_list => {}, #has everything that needs to keep
                                 #a path structure, i.e. modules and libs
                linuxrc => "#!/sbin/sh\n\n",
                filesystem => "",
                imgfile => "",
                mount_point => "",
                initrd => 0,
                #set the defaults for tools here, these should get overridden
                mknod => "mknod",
                mount => "mount",
                umount => "umount",
                losetup => "losetup",
                dd => "dd",
                loop => "/dev/loop0",
                #set the defaults for things that get copied
                insmod => "/sbin/insmod",
                shell => "/bin/bash",
                libs => "/lib/libc.so.6 /lib/ld-linux.so.2",
                @_,
                #stuff that can't be overridden
               );

    %{ $this{file_list} } = map { $_ => 1 } split('\s+',$this{libs});

    return bless \%this, $class;
}

sub configure {

    my $this = shift;

    # here we gather the rest of the kernel-specific information
    #     -  version number, location of modules, modules.dep
    #     -  what file system can we use?  ext2 and cramfs seem to be the popular ones and all we
    #         support for now, but we can support others pretty easily (like minix?).
    #     -  what modules do we need? right now we check for scsi and filesystem modules
    #     -  is the ramdisk too big?  we're putting a 4meg cap on the ramdisk

    #version number
    $this->{kernel_version} = kernel_version($this->{kernel_path});

    if ( $this->{kernel_version} eq "0.0.0" ) {
        croak("Problem detecting kernel version.");
    }

    #gather module paths
    $this->set_module_paths();

    # what file system does the kernel support natively?
    # i.e. what filesystem do we make the initrd
    $this->set_filesystem();

    # what modules do we need?
    $this->set_modules();

    # if module_list is empty, we don't need to build one
    if (not scalar(@{$this->{module_list}})) {
        croak("Don't need any modules loaded, skipping $this->{kernel}.");
    }

    # check the size
    if (not $this->check_size()) {
        croak("Ramdisk size limit reached!  Perhaps your shell and libraries are too big.");
    }

    #set the name and mount point 
    $this->{filename} = "sc_initrd_" . $this->{kernel};
    $this->{mount_point} = $this->{root} . "/tmp/" . $this->{filename} . "_mounted";

    return 1;

}

sub create {

    my $this = shift;

    $this->create_filesystem();
    $this->mount_filesystem();
    $this->copy_files();
    $this->write_linuxrc();
    $this->umount_filesystem();
    $this->gzip_image();
    $this->move_image();

    return $this->{initrd};

}

sub files {

    my $this = shift;

    return @{ $this->{filesmod} };

}

sub has_module {

    my $this = shift;
    my $module = shift;

    if (exists $this->{modules_hash}->{$module}) {
        return 1;
    }

    return 0;

}

sub set_module_paths {

    my $this = shift;

    #module path and modules.dep
    debug("$this->{kernel} is version $this->{kernel_version}");
    $this->{modules_path} = "$this->{root}/lib/modules/$this->{kernel_version}";
    $this->{"modules.dep"} = "$this->{modules_path}/modules.dep";

    if (not -d $this->{modules_path}) {
        croak("$this->{modules_path} missing.");
    }

    if (not -e $this->{'modules.dep'}) {
        croak("$this->{'modules.dep'} missing.");
    }

    # now we turn modules.dep into a hash structure, so we can query it easier.
    # structure is as follows:
    #     module => { module_path => "/lib/whatever/blah.o",
    #                        module_deps => [ "module1", "module2" ] }

    $this->{modules_hash} = create_modules_hash($this->{'modules.dep'});

    #print Dumper($this->{modules_hash});

}

sub set_filesystem {

    my $this = shift;

    if ($this->has_module("ext2")) {
        debug("Ext2 is modular, searching for alternative filesystem type.");
        if ($this->{kernel_version} =~ /^2\.2\./) {
            croak("Unable to detect initrd root filesystem.");
        } elsif ($this->has_module("cramfs")) {
            croak("Unable to detect initrd root filesystem.");
        } else {
            $this->{filesystem} = "cramfs";
            debug("Filesystem is cramfs.");
            return;
        }
    }

    #cool, ext2 it is.
    $this->{filesystem} = "ext2";
    debug("Filesystem is ext2.");

    return;

}

sub set_modules {

    my $this = shift;

    #first we determine if we need to load a module for the root filesystem
    my $root_type = $this->find_root_type();

    #and then add the modules iff we need them
    if ($this->has_module($root_type)) {
        $this->add_module($root_type);
    }

    #print Dumper($this->{file_list},$this->{module_list});

    #Now we figure out if we need any hardware modules to boot
    my $module;
    foreach $module ($this->find_scsi_modules()) {
        if ($this->has_module($module)) {
            $this->add_module($module);
        } else {
            debug("Missing $module -- assuming in kernel.");
        }
        #
        # sd_mod needs to have some things depend on it, dammit.
        # it can't hurt to always add it, since you need it for mounting
        # scsi block devices.  i.e. root!
        #
        if ($this->has_module("sd_mod")) {
            $this->add_module("sd_mod");
        }
    }

    debug("Here is the module list:" . Dumper($this->{module_list}));

    return;

}

sub add_module {

    # this adds a new module plus any module it depends on
    # it's recursive and takes a scalar or array as an argument

    my $this = shift;
    my @modules = @_;

    my $size = scalar(@modules);

    if ($size > 1) {
        foreach (@modules) {
            $this->add_module($_);
        }
        return;
    } elsif (  $size == 1 ) {
        #here we do most of the work:
        # 1) return if the module is already there
        # 2) add the module to the list of modules
        #       if it is already there, you have might have to re-order the list because
        #       2 things might depend on it
        # 3) add the module_path to the list of files
        # 4) add any dependancies by calling ourself
        my $module = $modules[0];
        for(my $i = scalar(@{$this->{module_list}}) - 1; $i >= 0; $i--) {
            if ($module eq $this->{module_list}->[$i]) {
                debug("reodering with $module");
                debug("before reorder list is: @{$this->{module_list}}");
                $this->reorder_module_list($module,$i);
                debug("after reorder list is: @{$this->{module_list}}");
                return;
            }
        }
        unshift @{ $this->{module_list} }, $module;
        my $path = $this->module_path($module);
        $this->{file_list}->{$path}++;
        my @deps = $this->module_deps($module);
        $this->add_module(@deps);
        return;
    } else {
        return;
    }

}

sub reorder_module_list {

    my $this = shift;
    my $module = shift;
    my $index = shift;

    # first we remove it
    splice (@{$this->{module_list}},$index,1);

    # now we walk the list:  each time we see something that depends on us,
    # we make sure that we get inserted before it by setting $insert

    # start with $insert at the end
    my $end = scalar(@{$this->{module_list}}) - 1;
    my $insert = $end;

    debug("insert is $insert");

    # walk the list
    for(my $i=$insert;$i >= 0; $i--) {
        my @deps = $this->module_deps($this->{module_list}->[$i]);
        debug("$this->{module_list}->[$i] has dependancies of @deps");
        foreach (@deps) {
            if ( $module eq $_ ) {
                debug("$this->{module_list}->[$i] depends on $module");
                $insert = $i - 1;
            }
        }
    }

    debug("inserting $module at $insert");

    #now place the $module in the list at $insert
    if ( $insert < 0 ) {
        debug("unshifting");
        unshift @{$this->{module_list}},$module;
    } elsif ( $insert == $end ) {
        debug("pushing");
        push @{$this->{module_list}},$module;
    } else {
        debug("splicing");
        my $element = $this->{module_list}->[$insert];
        splice(@{$this->{module_list}},$insert,1,($module, $element));
    }

    return 1;

}

sub check_size {

    my $this = shift;

    my $total = 0;

    #this is the easy way to do it
    #map { $total += -s $_ }  (keys %{$this->{file_list}});

    #this is the way I have to do it.  Which is longer?
    my $file;
    foreach $file (keys %{$this->{file_list}}) {
        $total += ( -s $file );
    }

    # 2831155 is (in bytes) 10% less than 3 megs, which is the
    # largest we are going to allow our ramdisks to be, uncompressed.
    # initrds on x86 systems have some limit around 4 megs due to a 16
    # meg limit from the BIOS where you need room for kernel and friends.
    # Actually, this might be gone from modern systems
    # but backwards compatability is the land we live in.
    if ( $total <= 2831155 ) {
        return 1;
    }

    return 0;

}

sub find_root_type {

    my $this = shift;

    my $fstab = $this->{root} . "/etc/fstab";

    open(FSTAB,"<$fstab") or croak("Failed to open $fstab");
    while (<FSTAB>) {
        /^\#/ and next;

        if ( /(\/\w+)+\s+\/\s+(\w+)/ ) {
            if ($2 ne "auto") {
                return $2;
            }
            last;
        }

    }
    close(FSTAB) or croak("Failed to close $fstab");

    return "";

}

sub find_scsi_modules {

    my $this = shift;

    my $file = $this->{root} . "/etc/modules.conf";

    (not -e $file) and ($file = $this->{root} . "/etc/conf.modules");

    if (not -e $file) {
        croak("Missing $file.\n");
    }

    my @modules;

    open(MODCONF,"<$file") or croak("Failed to open $file.");
    while(<MODCONF>) {
        /^\#/ and next;
        if ( /alias\s+scsi\w+\s+(\w+)/ ) {
            push @modules,$1;
        }
    }
    close(MODCONF) or croak("Failed to close $file.");

    return @modules;

}

sub create_modules_hash {

    # structure is as follows:
    #     module => { module_path => "/lib/whatever/blah.o",
    #                        module_deps => [ "module1", "module2" ] }

    my $file = shift;
    my $modules = {};

    open (MODTREE,"<$file") or croak("Failed to open $file.");

    my $line = "";

    while (<MODTREE>) {
        /^\s+$/ and next;
        chomp;
        $line .= $_;
        if ( $line =~ /\\$/ ) {
            $line =~ s/\\$//;
            next;
        }
        my ($name,$path);
        my (@depnames) = ();

        @depnames = split('\s+',$line);
        $path = shift @depnames;
        $path =~ s/://g;
        $path =~ /(\w+)\.o(\.gz)?$/;
        $name = $1;

        for (my $i = 0; $i < scalar(@depnames); $i++) {
            $depnames[$i] =~ /(\w+)\.o(\.gz)?$/;
            $depnames[$i] = $1;
        }

        $modules->{"$name"} = {
                              "module_path" => $path,
                              "module_deps" => \@depnames,
                             };

        $line = "";
    }

    close (MODTREE) or croak("Failed to close $file.");

    return $modules;

}

sub create_filesystem {

    my $this = shift;

    my $imgfile = "$this->{root}/tmp/$this->{filename}";
    $this->{imgfile} = $imgfile;

    if ( $this->{filesystem} eq "cramfs" ) {
        #special case, it's handled in mount_filesystem and umount_filesystem!
        #so do nothing
        return 1;
    }

    system("$this->{dd} if=/dev/zero of=$imgfile bs=1024 count=3072  > /dev/null 2>&1");

    ( -e "$imgfile" ) or croak("Failed to create $imgfile, aborting.");

    if ( system("$this->{losetup} $this->{loop} $imgfile  > /dev/null 2>&1") != 0 ) {
        croak("Failed to loopback setup $imgfile on $this->{loop}");
    }

    my $cmd = $this->format_cmd();
    if ( system("$cmd $this->{loop}   > /dev/null 2>&1") != 0 ) {
        croak("Failed to format $this->{loop} with $cmd");
    }


    return 1;

}

sub mount_filesystem {

    my $this = shift;

    #mkdir here
    if ( not mkpath($this->{mount_point}) ) {
        croak("Failed to create $this->{mount_point} directory");
    }

    if ( $this->{filesystem} eq "cramfs" ) {
        #cramfs creates a file out of a directory structure, and that file is the initrd
        #so no mounting involved and we already made the directory
        return 1;
    }

    #mount here
    if ( system("$this->{mount} -t $this->{filesystem} $this->{loop} $this->{mount_point}") != 0 ) {
        deltree($this->{mount_point});
        croak("Failed to mount $this->{loop}");
    }


}

sub copy_files {

    my $this = shift;

    my ($path,$directory,$file);

    foreach $path ( keys %{$this->{file_list}} ) {
        $directory = $path;
        $directory =~ /.*\/(.*)$/;
        $file = $1;
        $directory =~ s/\/[^\/]+$//;
        #print "directory: $directory file: $file\n";
        mkpath("$this->{mount_point}/$directory");
        copy($path,"$this->{mount_point}/$directory/$file");
    }

    #system("ls $this->{mount_point}");

    #now do shell and insmod
    mkpath("$this->{mount_point}/sbin");
    copy($this->{shell},"$this->{mount_point}/sbin/sh");
    copy($this->{insmod},"$this->{mount_point}/sbin/insmod");
    chmod 0755,("$this->{mount_point}/sbin/sh","$this->{mount_point}/sbin/insmod");

    #now create devices
    mkpath("$this->{mount_point}/dev");
    system("$this->{mknod} $this->{mount_point}/dev/console c 5 1");
    chmod 0777,("$this->{mount_point}/dev/console");

}

sub write_linuxrc {

    my $this = shift;

    debug("This is the module list: " . Dumper($this->{module_list}));

    foreach (@{ $this->{module_list} } ) {
        $this->{linuxrc} .= "/sbin/insmod $_\n";
    }

    open(RC,">$this->{mount_point}/linuxrc") or croak("Failed to open $this->{mount_point}/linuxrc for writing");
    print RC $this->{linuxrc};
    close(RC) or croak("Failed to close $this->{mount_point}/linuxrc");

    chmod 0755,"$this->{mount_point}/linuxrc";

    return 1;

}

sub umount_filesystem {

    my $this = shift;

    if ( $this->{filesystem} eq "cramfs" ) {

        #actually make the cramfs file here
        my $cmd = $this->format_cmd();
        if ( system("$cmd $this->{mount_point} $this->{imgfile} > /dev/null 2>&1") != 0 ) {
            croak("Failed to run $cmd $this->{mount_point} $this->{imgfile}");
        }

    }  else {

        #umount
        if ( system("$this->{umount} $this->{mount_point} > /dev/null 2>&1") != 0 ) {
            deltree($this->{mount_point});
            croak("Failed to umount $this->{mount_point}");
        }

        #detach device
        if ( system("$this->{losetup} -d $this->{loop} ") != 0 ) {
            croak("Failed to detach $this->{loop} device");
        }

    }

    #now cleanup the directories
    deltree($this->{mount_point});

    return 1;

}

sub gzip_image {

    my $this = shift;

    if ($this->{filesystem} eq "cramfs") {
        return 1;
    }

    if (defined $this->{gzip}) {
        system("$this->{gzip} $this->{imgfile} > /dev/null 2>&1");
        if ( -e "$this->{imgfile}.gz" ) {
            $this->{imgfile} = "$this->{imgfile}.gz";
            return 1;
        }
    }

    return 0;

}

sub move_image {

    my $this = shift;

    my $initrd = $this->{imgfile};
    $initrd =~ s/^.*\///;
    $initrd = "$this->{root}/boot/$initrd";
    $this->{initrd} = $initrd;
    if ( not move($this->{imgfile},$this->{initrd}) ) {
        croak("$!");
    }

    return 1;

}

sub format_cmd {

    my $this = shift;

    return $this->{format_cmds}->{$this->{filesystem}};

}

sub module_deps {

    my $this = shift;
    my $module = shift;

    return @{ $this->{modules_hash}->{$module}->{module_deps} };

}

sub module_path {

    my $this = shift;
    my $module = shift;

    return $this->{modules_hash}->{$module}->{module_path};

}

sub kernel_version {

    # all that remains of Sean's original code.
    # bwahahahahahahahahahahahahahaah
    # I am evil incarnate.  Where's my donut? (doughnut whatever)

    my $file = shift;

    my $version = '0.0.0';

    if(_is_gzip($file)) {
        open(IN,"gzip -dc $file |") or croak("$!");
    } else {
        open(IN,"<$file") or croak("Failed to open $file.");
    }
        
    while(<IN>) {
        # When Linux Kernel 4.0pre1 comes out, we'll have to change this
        if(/([123]\.\d+\.[\w\-]+)/) {
            $version = $1;
            last;
        }
    }
    close(IN) or croak("Failed to close $file.");

    return $version;
}

sub _is_gzip {
    my $file = shift;
    open(IN,"<$file") or (carp($!), return undef);
    my $chr1 = getc IN;
    my $chr2 = getc IN;
    close(IN) or (carp($!), return undef);
    if($chr1 eq 0x8b and $chr2 eq 0x1f) {
        return 1;
    } else {
        return 0;
    }
}

1;





