/*
 * Detect a Library for hardware detection
 *
 * Copyright (C) 1998-2000 MandrakeSoft
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <fcntl.h>
#include <linux/pci.h>
/*#include <linux/usb.h>*/

#include "pcmcia/cs_types.h"
#include "pcmcia/cistpl.h"
#include "usb.h"
#include "discover.h"
#include "utils.h"



/**********************************************************************/
/* class2device: Convert a PCI class type to DeviceType               */
/*                                                                    */
/*     bus: type of bus to be converted to DeviceType                 */
/**********************************************************************/
extern DeviceType pciclass2device(int class) {
  
  switch(class) {
  case PCI_CLASS_NETWORK_ETHERNET: return ETHERNETCARD;
  case PCI_CLASS_DISPLAY_VGA:
  case PCI_CLASS_DISPLAY_XGA:
  case PCI_CLASS_DISPLAY_OTHER: return VIDEOCARD;
  case PCI_CLASS_MULTIMEDIA_VIDEO: return TVCARD;
  case PCI_CLASS_MULTIMEDIA_AUDIO: return SOUNDCARD;
  case PCI_CLASS_SERIAL_USB: return USBINTERFACE;
  case PCI_CLASS_STORAGE_IDE: return IDEINTERFACE;
  case PCI_CLASS_STORAGE_FLOPPY: return FLOPPY;
  case PCI_CLASS_STORAGE_SCSI: return SCSIINTERFACE;
  case PCI_CLASS_INPUT_MOUSE: return MOUSE;
  case PCI_CLASS_COMMUNICATION_SERIAL: return MODEM;
  case PCI_CLASS_BRIDGE_HOST:
  case PCI_CLASS_BRIDGE_ISA :
  case PCI_CLASS_BRIDGE_EISA:
  case PCI_CLASS_BRIDGE_MC:
  case PCI_CLASS_BRIDGE_PCI:
  case PCI_CLASS_BRIDGE_PCMCIA:
  case PCI_CLASS_BRIDGE_NUBUS:
  case PCI_CLASS_BRIDGE_CARDBUS:
  case PCI_CLASS_BRIDGE_OTHER:return BRIDGE;
  default: return UNKNOWN_DEVICE;
  }
}

/**********************************************************************/
/* pcmciaclass2device: Convert a PCMCIA class type to DeviceType      */
/*                                                                    */
/*     bus: type of bus to be converted to DeviceType                 */
/**********************************************************************/
extern DeviceType pcmciaclass2device(unsigned char class) {

  switch(class) {
  case CISTPL_FUNCID_SERIAL: return MODEM;
  case CISTPL_FUNCID_VIDEO: return VIDEOCARD;
  case CISTPL_FUNCID_NETWORK: return ETHERNETCARD;
  case CISTPL_FUNCID_SCSI: return SCSIINTERFACE;
  default: return UNKNOWN_DEVICE;
  }
}


/**********************************************************************/
/* usbclass2device: Convert a PCMCIA class type to DeviceType      */
/*                                                                    */
/*     bus: type of bus to be converted to DeviceType                 */
/**********************************************************************/
extern DeviceType usbclass2device(unsigned char class) {
  
  switch(class) {
  case USB_CLASS_AUDIO: return SOUNDCARD;
  case USB_CLASS_COMM:  return MODEM;
  case USB_CLASS_PRINTER: return PRINTER;
  default: return UNKNOWN_DEVICE;
  }
}

/**********************************************************************/
/* bus2str: Convert BusType to string                                 */
/*                                                                    */
/*     bus: type of bus to be converted to string.                    */
/**********************************************************************/
extern char *bus2str(BusType bus){
  switch(bus){
  case IDE: return "ATAPI/IDE";
  case ISA: return "ISA";
  case PCI: return "PCI";
  case PCMCIA: return "PCMCIA";
  case SERIAL: return "Serial";
  case USB: return "USB";
  case PS2: return "PS/2";
  case VLB: return "VLB";
  case SCSI: return "SCSI";
  case FDC: return "Floppy Drive Controller";
  case PARALLEL: return "Parallel";
  case GAMEPORT: return "Game Port";	
  default: return s_unknown;
  }/*endswitch*/
}/*endfunc bus2str*/


/**********************************************************************/
/* str2bus: Convert string to BusType                                 */
/*                                                                    */
/*     str: String to be converted.                                   */
/**********************************************************************/
extern BusType str2bus(char *str){
  if(str == (char *)NULL){
    return UNKNOWN_BUS;
  }/*endif*/
  if(!strcmp(str, "ATAPI/IDE")) return IDE;
  else if(!strcmp(str, "ISA")) return ISA;
  else if(!strcmp(str, "PCI")) return PCI;
  else if(!strcmp(str, "PCMCIA")) return PCMCIA;
  else if(!strcmp(str, "Serial")) return SERIAL;
  else if(!strcmp(str, "USB")) return USB;
  else if(!strcmp(str, "PS/2")) return PS2;
  else if(!strcmp(str, "VLB")) return VLB;
  else if(!strcmp(str, "SCSI")) return SCSI;
  else if(!strcmp(str, "Floppy Drive Controller")) return FDC;
  else if(!strcmp(str, "Parallel")) return  PARALLEL;
  else if(!strcmp(str, "Game Port")) return GAMEPORT;
  else return UNKNOWN_BUS;
}/*endfunc str2bus*/


/**********************************************************************/
/* str2device: Convert string to device                               */
/*                                                                    */
/*     str: String to be converted.                                   */
/**********************************************************************/
extern DeviceType str2device(char *str){
  if(str == (char *)NULL){
    return UNKNOWN_DEVICE;
  }/*endif*/
  if(strstr(str, "cpu")) return CPU;
  else if(strstr(str, "memory")) return MEMORY;
  else if(strstr(str, "floppy")) return FLOPPY;
  else if(strstr(str, "disk")) return DISK;
  else if(strstr(str, "cdrom")) return CDROM;
  else if(strstr(str, "tape")) return TAPE;
  else if(strstr(str, "ethernet")) return ETHERNETCARD;
  else if(strstr(str, "modem")) return MODEM;
  else if(strstr(str, "video")) return VIDEOCARD;
  else if(strstr(str, "tvcard")) return TVCARD;
  else if(strstr(str, "sound")) return SOUNDCARD;
  else if(strstr(str, "printer")) return PRINTER;
  else if(strstr(str, "scanner")) return SCANNER;
  else if(strstr(str, "mouse")) return MOUSE;
  else if(strstr(str, "other")) return OTHERS;
  else if(strstr(str, "ide")) return IDEINTERFACE;
  else if(strstr(str, "scsi")) return SCSIINTERFACE;
  else if(strstr(str, "usb")) return USBINTERFACE;
  else if(strstr(str, "joystick")) return JOYSTICK;
  else if(strstr(str, "isdn")) return ISDN;
  else if(strstr(str, "webcam")) return WEBCAM;
  else if(strstr(str, "bridge")) return BRIDGE;	
  else return UNKNOWN_DEVICE;
}/*endfunc str2device*/

/**********************************************************************/
/* device2str: Convert a DeviceType to the string used in the report  */
/*                                                                    */
/*      type: DeviceType to be converted.                             */
/**********************************************************************/
extern char *device2str(DeviceType type){
  switch(type){
  case CPU: return "cpu";
  case MEMORY: return "memory";
  case FLOPPY: return "floppy";
  case DISK: return "disk";
  case CDROM: return "cdrom";
  case TAPE: return "tape";
  case ETHERNETCARD: return "ethernet";
  case MODEM: return "modem";
  case VIDEOCARD: return "video";
  case TVCARD: return "tvcard";
  case SOUNDCARD: return "sound";
  case PRINTER: return "printer";
  case SCANNER: return "scanner";
  case MOUSE: return "mouse";
  case OTHERS: return "other";
  case IDEINTERFACE: return "ide";
  case SCSIINTERFACE: return "scsi";
  case USBINTERFACE: return "usb";
  case JOYSTICK: return "joystick";
  case ISDN: return "isdn";
  case WEBCAM: return "webcam";
  case BRIDGE: return "bridge";
  default: return s_unknown;
  }/*endswitch*/
}/*endfunc device2str*/


/**********************************************************************/
/* device2text: Convert a DeviceType to the text used in other        */
/*              (onscreen) applications (=> i18n)                     */
/*                                                                    */
/*      type: DeviceType to be converted.                             */
/**********************************************************************/
extern char *device2text(DeviceType type){
  switch(type){
  /* Note to translators: This are the device types, as they will be  */
  /*                      shown e.g. in HardDrake's device list       */
  case CPU: return _("CPU");
  case MEMORY: return _("Memory");
  case FLOPPY: return _("Floppy Drives");
  case DISK: return _("Disks");
  case CDROM: return _("CD-ROMs");
  case TAPE: return _("Tapes");
  case ETHERNETCARD: return _("Network Device Cards");
  case MODEM: return _("Modem");
  case VIDEOCARD: return _("Videocards");
  case TVCARD: return _("TV Cards");
  case SOUNDCARD: return _("Soundcards");
  case MOUSE: return _("Mouse");
  case PRINTER: return _("Printer");
  case SCANNER: return _("Scanner");
  case OTHERS: return _("Other Devices");
  case IDEINTERFACE: return _("IDE Interfaces");
  case SCSIINTERFACE: return _("SCSI Interfaces");
  case USBINTERFACE: return _("USB Interfaces");
  case JOYSTICK: return _("Joystick");
  case ISDN: return _("ISDN Adapters");
  case WEBCAM: return _("WebCams");
  case BRIDGE: return _("Bridge");	
  default: return _("Unknown Devices");
  }/*endswitch*/
}/*endfunc device2text*/


/**********************************************************************/
/* str2options: Convert a string to options.                          */
/*                                                                    */
/*     str: String to be converted.                                   */
/**********************************************************************/
extern unsigned int str2options(char *str){
  unsigned int options = 0;

  if(str == (char *)NULL){
    return options;
  }/*endif*/
  if(strstr(str, "HAS_FPU")) options += HAS_FPU;
  if(strstr(str, "HAS_3DNOW")) options += HAS_TDNOW;
  if(strstr(str, "HAS_MMX")) options += HAS_MMX;
  if(strstr(str, "HAS_OPL3")) options += HAS_OPL3;
  if(strstr(str, "HAS_MPU401")) options += HAS_MPU401;
  if(strstr(str, "HAS_DMA16")) options += HAS_DMA16;
  if(strstr(str, "HAS_AWE")) options += HAS_AWE;
  if(strstr(str, "HAS_DREAM")) options += HAS_DREAM;
  if(strstr(str, "HAS_MAD16")) options += HAS_MAD16;
  if(strstr(str, "IS_SERIAL")) options += IS_SERIAL;
  if(strstr(str, s_unknown)) options = 0;
  return options;
}/*endfunc str2options*/



/**********************************************************************/
/* options2str: Convert options to string.                            */
/*                                                                    */
/*     options: Options to be converted.                              */
/**********************************************************************/
extern char *options2str(unsigned int options){
  char *str = (char *)my_malloc(256);

  strcpy(str, "");

  if(options & HAS_FPU)
    sprintf(str, "%s:%s", str, "HAS_FPU");
  if(options & HAS_TDNOW)
    sprintf(str, "%s:%s", str, "HAS_3DNOW");
  if(options & HAS_MMX)
    sprintf(str, "%s:%s", str, "HAS_MMX");
  if(options & HAS_OPL3)
    sprintf(str, "%s:%s", str, "HAS_OPL3");
  if(options & HAS_MPU401)
    sprintf(str, "%s:%s", str, "HAS_MPU401");
  if(options & HAS_DMA16)
    sprintf(str, "%s:%s", str, "HAS_DMA16");
  if(options & HAS_AWE)
    sprintf(str, "%s:%s", str, "HAS_AWE");
  if(options & HAS_DREAM)
    sprintf(str, "%s:%s", str, "HAS_DREAM");
  if(options & HAS_MAD16)
    sprintf(str, "%s:%s", str, "HAS_MAD16");
  if(options & IS_SERIAL)
    sprintf(str, "%s:%s", str, "IS_SERIAL");
  if(strlen(str)<1){
    sprintf(str, "[]");
  }else{
    sprintf(str, "[%s]", str+1);
  }/*endif*/
  return str;
}/*endfunc options2str*/

