// disasm.c - TMS9900-Series Symbolic Disassembler
//
// Copyright (c) 2002, Timothy M. Stark
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// TIMOTHY M STARK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Timothy M Stark shall not
// be used in advertising or otherwise to promote the sale, use or other 
// dealings in this Software without prior written authorization from
// Timothy M Stark.

#include "ti99/defs.h"
#include "emu/socket.h"

#ifdef DEBUG

extern TI99_INST ti99_Inst[];
static TI99_INST **d99_Opcodes = NULL;

char *d99_DecodeOperand(register TI99_CPU *ti99cpu,
	char *str, uint16 op, uint32 *addr, uint32 sw)
{
	uint16 type = (op >> 4) & 0x3;
	uint16 reg  = op & 0xF;

	switch (type) {
		case 0: // Rn  - General Register
			str += sprintf(str, "R%d", reg);
			break;

		case 1: // *Rn - Indirect Register
			str += sprintf(str, "*R%d", reg);
			break;

		case 2: // Immediate or Indexed Register
			     //   @>nnnn     if R0
			     //   @>nnnn(Rn) if Rn
			str += sprintf(str, "@>%04X", ReadW(*addr));
			*addr += 2;
			if (reg > 0)
				str += sprintf(str, "(R%d)", reg);
			break;

		case 3: // *Rn++ - Auto Increment Register
			str += sprintf(str, "*R%d+", reg);
			break;
	}

	return str;
}

void d99_Disasm(register TI99_CPU *ti99cpu, SOCKET *dbg,
	uint32 *addr, uint32 sw)
{
	TI99_INST *opInfo;
	char      strAddr[40];
	char      strOper[256], *opr;
	char      *strInst;
	uint16    opCode;

	sprintf(strAddr, "%04X", *addr);

	opCode = ReadW(*addr);
	*addr += 2;

	if (opInfo = d99_Opcodes[opCode]) {
		strInst = opInfo->Name;
		strOper[0] = '\0';
		switch (opInfo->opFormat) {
			case 1: // Format 1 - Two-register Instruction
				opr = d99_DecodeOperand(ti99cpu, strOper, opCode, addr, sw);
				*opr++ = ',';
				opr = d99_DecodeOperand(ti99cpu, opr, opCode >> 6, addr, sw);
				break;

			case 2: // Format 2 - Jump/CRU Instruction
				sprintf(strOper, ">%04X", *addr + (((int8)(opCode & 0xFF)) << 1));
				break;

			case 3: // Format 3
			case 9: // Format 9
				opr = d99_DecodeOperand(ti99cpu, strOper, opCode, addr, sw);
				*opr++ = ',';
				opr = d99_DecodeOperand(ti99cpu, opr, (opCode >> 6) & 0xF, addr, sw);
				break;

			case 4: // Format 4
				opr =  d99_DecodeOperand(ti99cpu, strOper, opCode, addr, sw);
				opr += sprintf(opr, ",%d", (opCode >> 6) & 0xF);
				break;

			case 5: // Format 5
				opr =  d99_DecodeOperand(ti99cpu, strOper, opCode & 0xF, addr, sw);
				opr += sprintf(opr, ",%d", (opCode >> 4) & 0xF);
				break;

			case 6: // Format 6 - One-register Instruction
				d99_DecodeOperand(ti99cpu, strOper, opCode, addr, sw);
				break;

			case 0: 
			case 7: // Format 7
				// Do nothing (no operands)
				break;

			case 8:
				if (opCode < 0x02E0) {
					opr = d99_DecodeOperand(ti99cpu, strOper, opCode, addr, sw);
					if (opCode < 0x02A0)
						*opr++ = ',';
				} else
					opr = strOper;
				if (opCode < 0x02A0 || opCode >= 0x02E0) {
					opr += sprintf(opr, ">%04X", ReadW(*addr));
					*addr += 2;
				}
				break;
		}
	} else {
		strInst = "DATA";
		sprintf(strOper, ">%04X", opCode);
	}

	SockPrintf(dbg, "%s %-4s %s\n", strAddr, strInst, strOper);
}

// *********************************************************

void d99_BuildOpcodes(void)
{
	int idx;

	// Build new instruction table.
	d99_Opcodes = (TI99_INST **)calloc(65536, sizeof(TI99_INST *));
	for (idx = 0; ti99_Inst[idx].Name; idx++) {
		uint16 opidx = ti99_Inst[idx].opCode;
		uint16 opcnt = ~ti99_Inst[idx].opMask + 1;

		while (opcnt--)
			d99_Opcodes[opidx++] = &ti99_Inst[idx];
	}
}

void d99_DropOpcodes(void)
{
	if (d99_Opcodes)
		free(d99_Opcodes);
}

#endif /* DEBUG */
