/***************************************************************************
 *   Copyright (C) 2006-2008 by Thomas Schweitzer                          *
 *   thomas-schweitzer(at)arcor.de                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2.0 as   *
 *   published by the Free Software Foundation.                            *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program in the file LICENSE.GPL; if not, write to the *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "mainwindow.h"

//! \defgroup grp_MainWindow All concerning main window functionality.

/*!
    \class MainWindow
    \ingroup grp_MainWindow
    \brief Is the main window of UniversalIndentGUI

    The MainWindow class is responsible for generating and displaying most of the gui elements.
    Its look is set in the file "indentgui.ui". An object for the indent handler is generated here
    and user actions are being controlled. Is responsible for file open dialogs and indenter selection.
 */

/*!
    \brief Constructs the main window.
 */
MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent) {
    // set the program version, revision and date, which is shown in the main window title and in the about dialog.
    version = "0.8.1";
    revision = "650";
    QDate buildDate(2008, 03, 27);
    buildDateStr = buildDate.toString("d. MMMM yyyy");

    // Get the applications binary path, with respect to MacOSXs use of the .app folder. 
	applicationBinaryPath = QCoreApplication::applicationDirPath();
#ifdef Q_OS_MAC
    // Because on Mac universal binaries are used, the binary path is not equal
	// to the applications (.app) path. So get the .apps path here.
    int indexOfDotApp = applicationBinaryPath.indexOf(".app");
    if ( indexOfDotApp != -1 ) {
		// Cut off after the dot of ".app".
	    applicationBinaryPath = applicationBinaryPath.left( indexOfDotApp-1 );
		// Cut off after the first slash that was in front of ".app" (normally this is the word "UniversalIndentGUI")
	    applicationBinaryPath = applicationBinaryPath.left( applicationBinaryPath.lastIndexOf("/") );
	}
#endif

   // If the "config" directory is a subdir of the applications binary path, use this one (portable mode)
    settingsDirctoryStr = applicationBinaryPath + "/config";
    if ( QFile::exists( settingsDirctoryStr ) ) {
        portableMode = true;
        QDir dirCreator;
        globalFilesDirectoryStr = applicationBinaryPath;
        indenterDirctoryStr = applicationBinaryPath + "/indenters";
        dirCreator.mkpath( settingsDirctoryStr );
        tempDirctoryStr = applicationBinaryPath + "/temp";
        //TODO: If the portable drive has write protection, use local temp path and clean it up on exit.
        dirCreator.mkpath( tempDirctoryStr );
    }
    // ... otherwise use the system specific global application data path.
    else {
        portableMode = false;
        QDir dirCreator;
#ifdef Q_OS_WIN
        // Get the local users application settings directory.
        settingsDirctoryStr = QDir::fromNativeSeparators( qgetenv("APPDATA") ) + "/UniversalIndentGUI";
        // On windows systems the directories "indenters", "translations" are subdirs of the applicationBinaryPath.
        globalFilesDirectoryStr = applicationBinaryPath;
#else
        settingsDirctoryStr = QDir::homePath() + "/.config/universalindentgui";
        globalFilesDirectoryStr = "/usr/share/universalindentgui";
#endif
        dirCreator.mkpath( settingsDirctoryStr );
        // If a highlighter config file does not exist in the users home config dir
        // copy the default config file over there.
        if ( !QFile::exists(settingsDirctoryStr+"/UiGuiSyntaxHighlightConfig.ini") ) {
            QFile::copy( globalFilesDirectoryStr+"/config/UiGuiSyntaxHighlightConfig.ini", settingsDirctoryStr+"/UiGuiSyntaxHighlightConfig.ini" );
        }
        indenterDirctoryStr = globalFilesDirectoryStr + "/indenters";
#ifdef Q_OS_WIN
        tempDirctoryStr = QDir::tempPath() + "/UniversalIndentGUI";
#else
        tempDirctoryStr = QDir::tempPath() + "UniversalIndentGUI";
#endif
        dirCreator.mkpath( tempDirctoryStr );
    }

    qDebug() << "Using directories:\nsettings = " << settingsDirctoryStr;
    qDebug() << "globalFiles = " << globalFilesDirectoryStr;
    qDebug() << "indenterDirctoryStr = " << indenterDirctoryStr;
    qDebug() << "tempDirctoryStr = " << tempDirctoryStr;

    // Init of some variables.
    sourceCodeChanged = false;
    scrollPositionChanged = false;

    // Set default values for all by UniversalIndentGUI used settings objects.
    QCoreApplication::setOrganizationName("UniversalIndentGUI");
    QCoreApplication::setOrganizationDomain("universalindent.sf.net");
    QCoreApplication::setApplicationName("UniversalIndentGUI");

    // Create the settings object, which loads all UiGui settings from a file.
	settings = new UiGuiSettings( portableMode, globalFilesDirectoryStr );

    // Initialize the language of the application.
    initApplicationLanguage();

    // Creates the main window and initializes it.
    initMainWindow();

    // Create toolbar and insert it into the main window.
    initToolBar();

    // Create the text edit component using the QScintilla widget.
    initTextEditor();

    // Create and init the syntax highlighter.
    initSyntaxHighlighter();

    // Create and init the indenter.
    initIndenter();

    // Create some menus.
    createEncodingMenu();
    createHighlighterMenu();
	

    // generate about dialog box
/*#if QT_VERSION >= 0x040400
    aboutDialog = new AboutDialog(this, Qt::SplashScreen, version, revision, buildDateStr);
    aboutDialogGraphicsView = new AboutDialogGraphicsView(aboutDialog, this);
    connect( toolBarWidget->pbAbout, SIGNAL(clicked()), aboutDialogGraphicsView, SLOT(show()) );
    connect( actionAbout_UniversalIndentGUI, SIGNAL(activated()), aboutDialogGraphicsView, SLOT(show()) );
#else */
    aboutDialog = new AboutDialog(this, Qt::Dialog, version, revision, buildDateStr);
    connect( actionAbout_UniversalIndentGUI, SIGNAL(activated()), aboutDialog, SLOT(exec()) );
    connect( toolBarWidget->pbAbout, SIGNAL(clicked()), aboutDialog, SLOT(exec()) );
//#endif

	// generate settings dialog box
	settingsDialog = new UiGuiSettingsDialog(this, settings);
    connect( actionShowSettings, SIGNAL(activated()), settingsDialog, SLOT(showDialog()) );

    // Loads the last opened file, if this is enabled in the settings.
    loadLastOpenedFile();

    updateSourceView();

    // Check if a newer version is available but only if the setting for that is enabled and today not already a check has been done.
    if ( settings->getValueByName("CheckForUpdate").toBool() && QDate::currentDate() != settings->getValueByName("LastUpdateCheck").toDate() ) {
        updateCheckDialog->checkForUpdate();
    }
}


/*!
    \brief Initializes the main window by creating the main gui and make some settings.
 */
void MainWindow::initMainWindow() {
    // Generate gui as it is build in the file "indentgui.ui"
    setupUi(this);

	// Handle last opened window size
	// ------------------------------
	bool maximized = settings->getValueByName("WindowIsMaximized").toBool();
	QPoint pos = settings->getValueByName("WindowPosition").toPoint();
	QSize size = settings->getValueByName("WindowSize").toSize();
	resize(size);
	move(pos);
	if ( maximized ) {
		showMaximized();
	}
    restoreState( settings->getValueByName("MainWindowState").toByteArray() );

    // Handle if first run of this version
    // -----------------------------------
    QString readVersion = settings->getValueByName("VersionInSettingsFile").toString();
	// If version strings are not equal set first run true.
	if ( readVersion != version ) {
		isFirstRunOfThisVersion = true;
	}
	else {
		isFirstRunOfThisVersion = false;
	}

    // Get last selected file encoding
	// -------------------------------
	currentEncoding = settings->getValueByName("FileEncoding").toString();

    updateCheckDialog = new UpdateCheckDialog(version, settings, this);

    // Register the syntax highlightning setting in the menu to the settings object.
    connect( uiGuiSyntaxHighlightningEnabled, SIGNAL(toggled(bool)), settings, SLOT(handleValueChangeFromExtern(bool)) );
    connect( settings, SIGNAL(syntaxHighlightningEnabled(bool)), uiGuiSyntaxHighlightningEnabled, SLOT(setChecked(bool)) );
    uiGuiSyntaxHighlightningEnabled->setChecked( settings->getValueByName("SyntaxHighlightningEnabled").toBool() );
    // Tell the highlighter if it has to be enabled or disabled.
    connect( settings, SIGNAL(syntaxHighlightningEnabled(bool)), this, SLOT(turnHighlightOnOff(bool)) );

    // Register the load last file setting in the menu to the settings object.
    connect( uiGuiLoadLastOpenedFileOnStartup, SIGNAL(toggled(bool)), settings, SLOT(handleValueChangeFromExtern(bool)) );
    connect( settings, SIGNAL(loadLastOpenedFileOnStartup(bool)), uiGuiLoadLastOpenedFileOnStartup, SLOT(setChecked(bool)) );
    uiGuiLoadLastOpenedFileOnStartup->setChecked( settings->getValueByName("LoadLastOpenedFileOnStartup").toBool() );

    // Register the white space setting in the menu to the settings object.
    connect( uiGuiWhiteSpaceIsVisible, SIGNAL(toggled(bool)), settings, SLOT(handleValueChangeFromExtern(bool)) );
    connect( settings, SIGNAL(whiteSpaceIsVisible(bool)), uiGuiWhiteSpaceIsVisible, SLOT(setChecked(bool)) );
    uiGuiWhiteSpaceIsVisible->setChecked( settings->getValueByName("WhiteSpaceIsVisible").toBool() );
    // Tell the QScintilla editor if it has to show white space.
    connect( settings, SIGNAL(whiteSpaceIsVisible(bool)), this, SLOT(setWhiteSpaceVisibility(bool)) );

    // Connect the remaining menu items.
    connect( actionOpen_Source_File, SIGNAL(activated()), this, SLOT(openSourceFileDialog()) );
    connect( actionSave_Source_File_As, SIGNAL(activated()), this, SLOT(saveasSourceFileDialog()) );
    connect( actionSave_Source_File, SIGNAL(activated()), this, SLOT(saveSourceFile()) );
    connect( actionExportPDF, SIGNAL(activated()), this, SLOT(exportToPDF()) );
    connect( actionExportHTML, SIGNAL(activated()), this, SLOT(exportToHTML()) );
    connect( actionLoad_Indenter_Config_File, SIGNAL(activated()), this, SLOT(openConfigFileDialog()) );
    connect( actionSave_Indenter_Config_File, SIGNAL(activated()), this, SLOT(saveasIndentCfgFileDialog()) );
    connect( actionCreateShellScript, SIGNAL(activated()), this, SLOT(createIndenterCallShellScript()) );
    connect( actionCheck_for_update, SIGNAL(activated()), updateCheckDialog, SLOT(checkForUpdateAndShowDialog()) );

    // Init the menu for selecting one of the recently opened files.
    updateRecentlyOpenedList();
    connect( menuRecently_Opened_Files, SIGNAL(triggered(QAction*)), this, SLOT(openFileFromRecentlyOpenedList(QAction*)) );
    connect( settings, SIGNAL(recentlyOpenedListSize(int)), this, SLOT(updateRecentlyOpenedList()) );
}


/*!
    \brief Creates and inits the tool bar. It is added to the main window.
 */
void MainWindow::initToolBar() {
    // Create the tool bar and add it to the main window.
    toolBarWidget = new Ui::toolBarWidget();
    QWidget* helpWidget = new QWidget();
    toolBarWidget->setupUi(helpWidget);
    toolBar->addWidget(helpWidget);
    toolBar->setAllowedAreas( Qt::TopToolBarArea | Qt::BottomToolBarArea );

    // Connect the tool bar widgets to their functions.
    connect( toolBarWidget->uiGuiSyntaxHighlightningEnabled, SIGNAL(toggled(bool)), settings, SLOT(handleValueChangeFromExtern(bool)) );
    connect( settings, SIGNAL(syntaxHighlightningEnabled(bool)), toolBarWidget->uiGuiSyntaxHighlightningEnabled, SLOT(setChecked(bool)) );
    toolBarWidget->uiGuiSyntaxHighlightningEnabled->setChecked( settings->getValueByName("SyntaxHighlightningEnabled").toBool() );
    toolBarWidget->uiGuiSyntaxHighlightningEnabled->hide();
    connect( toolBarWidget->pbOpen_Source_File, SIGNAL(clicked()), this, SLOT(openSourceFileDialog()) );
    connect( toolBarWidget->pbExit, SIGNAL(clicked()), this, SLOT(close()));
    connect( toolBarWidget->cmbBoxIndenters, SIGNAL(activated(int)), this, SLOT(selectIndenter(int)) );
    connect( toolBarWidget->cbLivePreview, SIGNAL(toggled(bool)), this, SLOT(previewTurnedOnOff(bool)) );
    connect( toolBarWidget->cbLivePreview, SIGNAL(toggled(bool)), actionLive_Indent_Preview, SLOT(setChecked(bool)) );
    connect( actionLive_Indent_Preview, SIGNAL(toggled(bool)), toolBarWidget->cbLivePreview, SLOT(setChecked(bool)) );
}


/*!
    \brief Create and initialize the text editor component. It uses the QScintilla widget.
 */
void MainWindow::initTextEditor() {
    // Create the QScintilla widget and add it to the layout.
    txtedSourceCode = new QsciScintilla(this);
	hboxLayout1->addWidget(txtedSourceCode);

    // Make some settings for the QScintilla widget.
    txtedSourceCode->setUtf8(true);
    txtedSourceCode->setMarginLineNumbers(1, true);
	txtedSourceCode->setMarginWidth(1, QString("10000") );
	txtedSourceCode->setBraceMatching(txtedSourceCode->SloppyBraceMatch);
	txtedSourceCode->setMatchedBraceForegroundColor( QColor("red") );
	txtedSourceCode->setFolding(QsciScintilla::BoxedTreeFoldStyle);
	txtedSourceCode->setAutoCompletionSource(QsciScintilla::AcsAll);
	txtedSourceCode->setAutoCompletionThreshold(3);

    // Handle if white space is set to be visible
	bool whiteSpaceIsVisible = settings->getValueByName("WhiteSpaceIsVisible").toBool();
    setWhiteSpaceVisibility( whiteSpaceIsVisible );

    // Handle the width of tabs in spaces
    int tabWidth = settings->getValueByName("TabWidth").toInt();
    txtedSourceCode->setTabWidth(tabWidth);

    // Remember a pointer to the scrollbar of the QScintilla widget used to keep
    // on the same line as before when turning preview on/off.
    textEditVScrollBar = txtedSourceCode->verticalScrollBar();

    // Connect the text editor to dependent functions.
    connect( txtedSourceCode, SIGNAL(textChanged()), this, SLOT(sourceCodeChangedHelperSlot()) );
	connect( txtedSourceCode, SIGNAL(linesChanged()), this, SLOT(numberOfLinesChanged()) );
    connect( settings, SIGNAL(tabWidth(int)), txtedSourceCode, SLOT(setTabWidth(int)) );
    txtedSourceCode->setTabWidth( settings->getValueByName("TabWidth").toInt() );
}


/*!
    \brief Create and init the syntax highlighter and set it to use the QScintilla edit component.
 */
void MainWindow::initSyntaxHighlighter() {
    // Create the highlighter.
    highlighter = new Highlighter(txtedSourceCode, portableMode, globalFilesDirectoryStr);

    // Handle if syntax highlighting is enabled
	bool syntaxHighlightningEnabled = settings->getValueByName("SyntaxHighlightningEnabled").toBool();
	if ( syntaxHighlightningEnabled ) {
        highlighter->turnHighlightOn();
    }
    else {
        highlighter->turnHighlightOff();
    }
}


/*!
    \brief Initializes the language of UniversalIndentGUI.

    If the program language is defined in the settings, the corresponding language
    file will be loaded and set for the application. If not set there, the system
    default language will be set, if a translation file for that language exists.
    Returns true, if the translation file could be loaded. Otherwise it returns
    false and uses the default language, which is English.
 */
bool MainWindow::initApplicationLanguage() {
    QString languageShort;

    // Get the language settings from the settings object.
	int languageIndex = settings->getValueByName("Language").toInt();

    // If no language was set, indicated by a negative index, use the system language.
    if ( languageIndex < 0 ) {
        languageShort = QLocale::system().name();

        // If no translation file for the systems local language exist, fall back to English.
        if ( settings->getAvailableTranslations().indexOf(languageShort) < 0 ) {
            languageShort = "en";
        }

        // Set the language setting to the new language.
        settings->setValueByName("Language", settings->getAvailableTranslations().indexOf(languageShort) );
    }
    // If a language was defined in the settings, get this language mnemonic.
    else {
        languageShort = settings->getAvailableTranslations().at(languageIndex);
    }

    // Load the Qt own translation file and set it for the application.
    qTTranslator = new QTranslator();
    bool translationFileLoaded;
    translationFileLoaded = qTTranslator->load( globalFilesDirectoryStr + "/translations/qt_" + languageShort );
    if ( translationFileLoaded ) {
        qApp->installTranslator(qTTranslator);
    }

    // Load the uigui translation file and set it for the application.
    uiGuiTranslator = new QTranslator();
    translationFileLoaded = uiGuiTranslator->load( globalFilesDirectoryStr + "/translations/universalindent_" + languageShort );
    if ( translationFileLoaded ) {
        qApp->installTranslator(uiGuiTranslator);
    }

    connect( settings, SIGNAL(language(int)), this, SLOT(languageChanged(int)) );

    return translationFileLoaded;
}


/*!
    \brief Creates and initializes the indenter.
 */
void MainWindow::initIndenter() {
    // Get Id of last selected indenter.
	currentIndenterID = settings->getValueByName("LastSelectedIndenterID").toInt();

    // Create the indenter widget with the ID and add it to the layout.
    indentHandler = new IndentHandler(indenterDirctoryStr, settingsDirctoryStr, tempDirctoryStr, currentIndenterID, this, centralwidget);
    vboxLayout->addWidget(indentHandler);

    // Check whether indenters are available.
	if ( !indentHandler->getAvailableIndenters().isEmpty() ) {
		toolBarWidget->cmbBoxIndenters->addItems( indentHandler->getAvailableIndenters() );
		// Take care if the selected indenterID is greater than the number of existing indenters
		if ( currentIndenterID >= indentHandler->getAvailableIndenters().count() ) {
			currentIndenterID = indentHandler->getAvailableIndenters().count() - 1;
		}
	}
    // If no indenter are found, show a warning message.
	else {
		currentIndenterID = 0;
	}

    // Set the combobox in the toolbar to show the selected indenter.
    toolBarWidget->cmbBoxIndenters->setCurrentIndex( currentIndenterID );

    // If settings for the indenter have changed, let the main window know aboud it.
    connect(indentHandler, SIGNAL(indenterSettingsChanged()), this, SLOT(indentSettingsChangedSlot()));

    // Set this true, so the indenter is called at first program start
    indentSettingsChanged = true;
    previewToggled = true;

    // Handle if indenter parameter tool tips are enabled
    connect( uiGuiIndenterParameterTooltipsEnabled, SIGNAL(toggled(bool)), settings, SLOT(handleValueChangeFromExtern(bool)) );
    connect( settings, SIGNAL(indenterParameterTooltipsEnabled(bool)), uiGuiIndenterParameterTooltipsEnabled, SLOT(setChecked(bool)) );
    uiGuiIndenterParameterTooltipsEnabled->setChecked( settings->getValueByName("IndenterParameterTooltipsEnabled").toBool() );

    // Handle if the indenter help/manual button is pressed
    connect( toolBarWidget->indenterManualButton, SIGNAL(clicked()), this, SLOT(showIndenterManual()) );
}


/*!
    \brief Creates the by \a indenterID selected indent handler object and adds the indent widget to its layout.
 */
void MainWindow::selectIndenter(int indenterID) {
    IndentHandler *oldIndentHandler = indentHandler;

    // Prevent unnecessary updates if same indenter as current has been selected
    if ( indenterID == currentIndenterID ) {
        return;
    }

    // Disconnect the old indent handler from the settings changed slot, because he will be deleted.
    disconnect(oldIndentHandler, SIGNAL(indenterSettingsChanged()), this, SLOT(indentSettingsChangedSlot()));

    QApplication::setOverrideCursor(Qt::WaitCursor);

    indentHandler = new IndentHandler(indenterDirctoryStr, settingsDirctoryStr, tempDirctoryStr, indenterID, this, centralwidget);
    indentHandler->hide();
    vboxLayout->insertWidget(0, indentHandler);
    oldIndentHandler->hide();
    indentHandler->show();
    vboxLayout->removeWidget(oldIndentHandler);
    delete oldIndentHandler;

    // Take care if the selected indenterID is smaller or greater than the number of existing indenters
    if ( indenterID < 0 ) {
        indenterID = 0;
    }
    if ( indenterID >= indentHandler->getAvailableIndenters().count() ) {
        indenterID = indentHandler->getAvailableIndenters().count() - 1;
    }

    // Set the combobox in the toolbar to show the selected indenter.
    toolBarWidget->cmbBoxIndenters->setCurrentIndex(indenterID);

    // If settings for the indenter have changed, let the main window know aboud it.
    connect(indentHandler, SIGNAL(indenterSettingsChanged()), this, SLOT(indentSettingsChangedSlot()));

    currentIndenterID = indenterID;
    if ( toolBarWidget->cbLivePreview->isChecked() ) {
        callIndenter();
    }

    previewToggled = true;
    indentSettingsChanged = true;
    updateSourceView();
    QApplication::restoreOverrideCursor();
}


/*!
    \brief Tries to load the by \a filePath defined file and returns its content as QString.

    If the file could not be loaded a error dialog will be shown.
 */
QString MainWindow::loadFile(QString filePath) {

    QFile inSrcFile(filePath);
    QString fileContent = "";

    if ( !inSrcFile.open(QFile::ReadOnly | QFile::Text) ) {
        QMessageBox::warning(NULL, tr("Error opening file"), tr("Cannot read the file ")+"\""+filePath+"\"." );
    }
    else {
        QTextStream inSrcStrm(&inSrcFile);
        QApplication::setOverrideCursor(Qt::WaitCursor);
        inSrcStrm.setCodec( QTextCodec::codecForName(currentEncoding.toAscii()) );
        fileContent = inSrcStrm.readAll();
        QApplication::restoreOverrideCursor();
        inSrcFile.close();

		QFileInfo fileInfo(filePath);
		currentSourceFileExtension = fileInfo.suffix();
		int indexOfHighlighter = highlighter->setLexerForExtension( currentSourceFileExtension );
        highlighterActionGroup->actions().at(indexOfHighlighter)->setChecked(true);
    }
    return fileContent;
}


/*!
    \brief Calls the source file open dialog to load a source file for the formatting preview.

    If the file was successfully loaded the indenter will be called to generate the formatted source code.
 */
void MainWindow::openSourceFileDialog(QString fileName) {
    // If the source code file is changed and the shown dialog for saving the file
    // is canceled, also stop opening another source file.
    if ( !maybeSave() ) {
        return;
    }
    QString openedSourceFileContent = "";
    QString fileExtensions = tr("Supported by indenter")+" ("+indentHandler->getPossibleIndenterFileExtensions()+
                             ");;"+tr("All files")+" (*.*)";

    //QString openedSourceFileContent = openFileDialog( tr("Choose source code file"), "./", fileExtensions );
	if ( fileName.isEmpty() ) {
		fileName = QFileDialog::getOpenFileName( this, tr("Choose source code file"), currentSourceFile, fileExtensions);
	}

    if (fileName != "") {
        currentSourceFile = fileName;
        QFileInfo fileInfo(fileName);
        currentSourceFileExtension = fileInfo.suffix();

        openedSourceFileContent = loadFile(fileName);
        sourceFileContent = openedSourceFileContent;
        if ( toolBarWidget->cbLivePreview->isChecked() ) {
            callIndenter();
        }
        sourceCodeChanged = true;
        previewToggled = true;
        updateSourceView();
        updateWindowTitle();
        updateRecentlyOpenedList();
        textEditLastScrollPos = 0;
        textEditVScrollBar->setValue( textEditLastScrollPos );

        savedSourceContent = openedSourceFileContent;
        txtedSourceCode->setModified( false );
        setWindowModified( false );
    }
}


/*!
    \brief Calls the source file save as dialog to save a source file under a chosen name.

    If the file already exists and it should be overwritten, a warning is shown before.
 */
bool MainWindow::saveasSourceFileDialog(QAction *chosenEncodingAction) {
    QString encoding;
    QString fileExtensions = tr("Supported by indenter")+" ("+indentHandler->getPossibleIndenterFileExtensions()+
                             ");;"+tr("All files")+" (*.*)";

    //QString openedSourceFileContent = openFileDialog( tr("Choose source code file"), "./", fileExtensions );
    QString fileName = QFileDialog::getSaveFileName( this, tr("Save source code file"), currentSourceFile, fileExtensions);

    // Saving has been canceled if the filename is empty
    if ( fileName.isEmpty() ) {
        return false;
    }

    savedSourceContent = txtedSourceCode->text();

    currentSourceFile = fileName;
    QFile::remove(fileName);
    QFile outSrcFile(fileName);
    outSrcFile.open( QFile::ReadWrite | QFile::Text );
    
    // Get current encoding.
    if ( chosenEncodingAction != NULL ) {
        encoding = chosenEncodingAction->text();
    }
    else {
        encoding = encodingActionGroup->checkedAction()->text();
    }
    QTextStream outSrcStrm(&outSrcFile);
    outSrcStrm.setCodec( QTextCodec::codecForName(encoding.toAscii()) );
    outSrcStrm << savedSourceContent;
    outSrcFile.close();

    QFileInfo fileInfo(fileName);
    currentSourceFileExtension = fileInfo.suffix();

    txtedSourceCode->setModified( false );
    setWindowModified( txtedSourceCode->isModified() );

    updateWindowTitle();
    return true;
}


/*!
    \brief Saves the currently shown source code to the last save or opened source file.

    If no source file has been opened, because only the static example has been loaded,
    the save as file dialog will be shown.
 */
bool MainWindow::saveSourceFile() {
    if ( currentSourceFile.isEmpty() ) {
        return saveasSourceFileDialog();
    }
    else {
        QFile::remove(currentSourceFile);
        QFile outSrcFile(currentSourceFile);
        savedSourceContent = txtedSourceCode->text();
        outSrcFile.open( QFile::ReadWrite | QFile::Text );

        // Get current encoding.
        QString currentEncoding = encodingActionGroup->checkedAction()->text();
        QTextStream outSrcStrm(&outSrcFile);
        outSrcStrm.setCodec( QTextCodec::codecForName(currentEncoding.toAscii()) );
        outSrcStrm << savedSourceContent;
        outSrcFile.close();

        txtedSourceCode->setModified( false );
        setWindowModified( txtedSourceCode->isModified() );
    }
    return true;
}


/*!
    \brief Calls the indenter config file save as dialog to save the config file under a chosen name.

    If the file already exists and it should be overwritten, a warning is shown before.
 */
void MainWindow::saveasIndentCfgFileDialog() {
    QString fileExtensions = tr("All files")+" (*.*)";

    //QString openedSourceFileContent = openFileDialog( tr("Choose source code file"), "./", fileExtensions );
    QString fileName = QFileDialog::getSaveFileName( this, tr("Save indent config file"), indentHandler->getIndenterCfgFile(), fileExtensions);

    if (fileName != "") {
        QFile::remove(fileName);
        QFile outCfgFile(fileName);
        outCfgFile.open( QFile::ReadWrite | QFile::Text );
        outCfgFile.write( indentHandler->getParameterString().toAscii() );
        outCfgFile.close();
    }
}


/*!
    \brief Shows a file open dialog to open an existing config file for the currently selected indenter.

    If the file was successfully opened the indent handler is called to load the settings and update itself.
 */
void MainWindow::openConfigFileDialog() {
    QString configFilePath;

    configFilePath = QFileDialog::getOpenFileName( NULL, tr("Choose indenter config file"), indentHandler->getIndenterCfgFile(), "All files (*.*)" );

    if (configFilePath != "") {
        indentHandler->loadConfigFile(configFilePath);
    }
}


/*!
    \brief Shows a file open dialog.

    Shows a file open dialog with the title \a dialogHeaderStr starting in the directory \a startPath
    and with a file mask defined by \a fileMaskStr. Returns the contents of the file as QString.
 */
QString MainWindow::openFileDialog(QString dialogHeaderStr, QString startPath, QString fileMaskStr) {

    QString fileContent = "";

    QString fileName = QFileDialog::getOpenFileName( NULL, dialogHeaderStr, startPath, fileMaskStr);

    if (fileName != "") {
        fileContent = loadFile(fileName);
    }

    return fileContent;
}


/*!
    \brief Updates the displaying of the source code.

    Updates the text edit field, which is showing the loaded, and if preview is enabled formatted, source code.
    Reassigns the line numbers and in case of switch between preview and none preview keeps the text field
    at the same line number.
 */
void MainWindow::updateSourceView()
{
    textEditLastScrollPos = textEditVScrollBar->value();

    if ( toolBarWidget->cbLivePreview->isChecked() ) {
        sourceViewContent = sourceFormattedContent;
    }
    else {
        sourceViewContent = sourceFileContent;
    }

    if (previewToggled) {
        disconnect( txtedSourceCode, SIGNAL(textChanged ()), this, SLOT(sourceCodeChangedHelperSlot()) );
		bool textIsModified = txtedSourceCode->isModified();
        txtedSourceCode->setText(sourceViewContent);
		txtedSourceCode->setModified(textIsModified);
        previewToggled = false;
        connect( txtedSourceCode, SIGNAL(textChanged ()), this, SLOT(sourceCodeChangedHelperSlot()) );
    }

    textEditVScrollBar->setValue( textEditLastScrollPos );
}


/*!
    \brief Calls the selected indenter with the currently loaded source code to retrieve the formatted source code.

    The original loaded source code file will not be changed.
 */
void MainWindow::callIndenter() {
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    sourceFormattedContent = indentHandler->callIndenter(sourceFileContent, currentSourceFileExtension);
    //updateSourceView();
    QApplication::restoreOverrideCursor();
}


/*!
    \brief Switches the syntax highlighting corresponding to the value \a turnOn either on or off.
 */
void MainWindow::turnHighlightOnOff(bool turnOn) {
    if ( turnOn ) {
        highlighter->turnHighlightOn();
    }
    else {
        highlighter->turnHighlightOff();
    }
    previewToggled = true;
    updateSourceView();
}


/*!
    \brief Added this slot to avoid multiple calls because of changed text.
 */
void MainWindow::sourceCodeChangedHelperSlot() {
	QTimer::singleShot(0, this, SLOT(sourceCodeChangedSlot()));
}


/*!
    \brief Is emitted whenever the text inside the source view window changes. Calls the indenter
    to format the changed source code.
 */
void MainWindow::sourceCodeChangedSlot() {
    QChar enteredCharacter;
	int cursorPos, cursorPosAbsolut, cursorLine;
	QString text;

    sourceCodeChanged = true;
    if ( scrollPositionChanged ) {
        scrollPositionChanged = false;
    }

    // Get the content text of the text editor.
    sourceFileContent = txtedSourceCode->text();
	
    // Get the position of the cursor in the unindented text.
    if ( sourceFileContent.isEmpty() ) {
        // Add this line feed, because AStyle has problems with a totally emtpy file.
        sourceFileContent += "\n";
        cursorPosAbsolut = 0;
        cursorPos = 0;
        cursorLine = 0;
        enteredCharacter = sourceFileContent.at(cursorPosAbsolut);
    }
    else {
        txtedSourceCode->getCursorPosition(&cursorLine, &cursorPos);
        cursorPosAbsolut = txtedSourceCode->SendScintilla(QsciScintillaBase::SCI_GETCURRENTPOS);
	    text = txtedSourceCode->text(cursorLine);
        if ( cursorPosAbsolut > 0 ) {
            cursorPosAbsolut--;
        }
        if ( cursorPos > 0 ) {
            cursorPos--;
        }
        enteredCharacter = sourceFileContent.at(cursorPosAbsolut);
    }

    // Call the indenter to reformat the text.
    if ( toolBarWidget->cbLivePreview->isChecked() ) {
        callIndenter();
        previewToggled = true;
    }

    // Update the text editor.
    updateSourceView();

    if ( toolBarWidget->cbLivePreview->isChecked() && !enteredCharacter.isNull() && enteredCharacter != 10 ) {
        //const char ch = enteredCharacter.toAscii();

        int saveCursorLine = cursorLine;
        int saveCursorPos = cursorPos;

        bool charFound = false;

        // Search forward
        int lineBreakCounter = 0;
        for ( cursorLine = saveCursorLine; cursorLine-saveCursorLine < 6 && cursorLine < txtedSourceCode->lines(); cursorLine++ ) {
            text = txtedSourceCode->text(cursorLine);
            while ( cursorPos < text.count() && enteredCharacter != text.at(cursorPos)) {
                cursorPos++;
            }
            if ( cursorPos >= text.count() ) {
                cursorPos = 0;
            } 
            else {
                charFound = true;
                break;
            }
        }

        // If foward search did not find the character, search backward
        if ( !charFound ) {
            text = txtedSourceCode->text(saveCursorLine);
            cursorPos = saveCursorPos;
            if ( cursorPos >= text.count() ) {
                cursorPos = text.count() - 1;
            }
            int lineBreakCounter = 0;
            for ( cursorLine = saveCursorLine; saveCursorLine-cursorLine < 6 && cursorLine >= 0; cursorLine-- ) {
                text = txtedSourceCode->text(cursorLine);
                while ( cursorPos >= 0 && enteredCharacter != text.at(cursorPos)) {
                    cursorPos--;
                }
                if ( cursorPos < 0 ) {
                    cursorPos = txtedSourceCode->lineLength(cursorLine-1) - 1;
                } 
                else {
                    charFound = true;
                    break;
                }
            }
        }

        // If the character was found set its new cursor position...
        if ( charFound ) {
            txtedSourceCode->setCursorPosition( cursorLine, cursorPos+1 );
        }
        // ...if it was not found, set the previous cursor position.
        else {
            txtedSourceCode->setCursorPosition( saveCursorLine, saveCursorPos+1 );
        }
    }
    // set the previous cursor position.
    else if ( enteredCharacter == 10 ) {
        txtedSourceCode->setCursorPosition( cursorLine, cursorPos );
    }


    if ( toolBarWidget->cbLivePreview->isChecked() ) {
        sourceCodeChanged = false;
    }

    if ( savedSourceContent == txtedSourceCode->text() ) {
        txtedSourceCode->setModified( false );
        setWindowModified( false );
    }
    else {
        txtedSourceCode->setModified( true );
        setWindowModified( true );
    }

    // Could set cursor this way and use normal linear search in text instead of columns and rows.
    //txtedSourceCode->SendScintilla(QsciScintillaBase::SCI_SETCURRENTPOS, 50);
    //txtedSourceCode->SendScintilla(QsciScintillaBase::SCI_SETANCHOR, 50);
}


/*!
    \brief This slot is called whenever one of the indenter settings are changed.

    It calls the selected indenter if the preview is turned on. If preview
    is not active a flag is set, that the settings have changed.
 */
void MainWindow::indentSettingsChangedSlot() {
    indentSettingsChanged = true;

	int cursorLine, cursorPos;
	txtedSourceCode->getCursorPosition(&cursorLine, &cursorPos);

    if ( toolBarWidget->cbLivePreview->isChecked() ) {
        callIndenter();
        previewToggled = true;

        updateSourceView();
        if (sourceCodeChanged) {
/*            savedCursor = txtedSourceCode->textCursor();
            if ( cursorPos >= txtedSourceCode->text().count() ) {
                cursorPos = txtedSourceCode->text().count() - 1;
            }
            savedCursor.setPosition( cursorPos );
            txtedSourceCode->setTextCursor( savedCursor );
*/
            sourceCodeChanged = false;
        }
        indentSettingsChanged = false;
    }
    else {
        updateSourceView();
    }

    if ( savedSourceContent == txtedSourceCode->text() ) {
        txtedSourceCode->setModified( false );
        setWindowModified( txtedSourceCode->isModified() );
    }
    else {
        txtedSourceCode->setModified( true );
        setWindowModified( txtedSourceCode->isModified() );
    }
}


/*!
    \brief This slot is called whenever the preview button is turned on or off.

    It calls the selected indenter to format the current source code if
    the code has been changed since the last indenter call.
 */
void MainWindow::previewTurnedOnOff(bool turnOn) {
    previewToggled = true;

	int cursorLine, cursorPos;
	txtedSourceCode->getCursorPosition(&cursorLine, &cursorPos);

    if ( turnOn && (indentSettingsChanged || sourceCodeChanged) ) {
        callIndenter();
    }
    updateSourceView();
    if (sourceCodeChanged) {
/*        savedCursor = txtedSourceCode->textCursor();
        if ( cursorPos >= txtedSourceCode->text().count() ) {
            cursorPos = txtedSourceCode->text().count() - 1;
        }
        savedCursor.setPosition( cursorPos );
        txtedSourceCode->setTextCursor( savedCursor );
*/
        sourceCodeChanged = false;
    }
    indentSettingsChanged = false;

    if ( savedSourceContent == txtedSourceCode->text() ) {
        txtedSourceCode->setModified( false );
        setWindowModified( txtedSourceCode->isModified() );
    }
    else {
        txtedSourceCode->setModified( true );
        setWindowModified( txtedSourceCode->isModified() );
    }
}


/*!
    \brief This slot updates the main window title to show the currently opened
    source code filename.
 */
void MainWindow::updateWindowTitle() {
    this->setWindowTitle( "UniversalIndentGUI " + version +" [*]"+ currentSourceFile);
}


/*!
    \brief Opens a dialog to save the current source code as a PDF document.
 */
void MainWindow::exportToPDF() {
    QString fileExtensions = tr("PDF Document")+" (*.pdf)";

    QString fileName = currentSourceFile;
    QFileInfo fileInfo(fileName);
    QString fileExtension = fileInfo.suffix();

    fileName.replace( fileName.length()-fileExtension.length(), fileExtension.length(), "pdf" );
    fileName = QFileDialog::getSaveFileName( this, tr("Export source code file"), fileName, fileExtensions);

    if ( !fileName.isEmpty() ) {
        QsciPrinter printer(QPrinter::HighResolution);
        printer.setOutputFormat(QPrinter::PdfFormat);
        printer.setOutputFileName(fileName);
		printer.printRange(txtedSourceCode);
    }
}


/*!
    \brief Opens a dialog to save the current source code as a HTML document.
 */
void MainWindow::exportToHTML() {
	QString fileExtensions = tr("HTML Document")+" (*.html)";

    QString fileName = currentSourceFile;
    QFileInfo fileInfo(fileName);
    QString fileExtension = fileInfo.suffix();

    fileName.replace( fileName.length()-fileExtension.length(), fileExtension.length(), "html" );
    fileName = QFileDialog::getSaveFileName( this, tr("Export source code file"), fileName, fileExtensions);

    if ( !fileName.isEmpty() ) {
        // Create a document from which HTML code can be generated.
        QTextDocument sourceCodeDocument( txtedSourceCode->text() );
        sourceCodeDocument.setDefaultFont( QFont("Courier", 12, QFont::Normal) );
        QString sourceCodeAsHTML = sourceCodeDocument.toHtml();
        // To ensure that empty lines are kept in the HTML code make this replacement.
        sourceCodeAsHTML.replace("\"></p>", "\"><br /></p>");

        // Write the HTML file.
        QFile::remove(fileName);
        QFile outSrcFile(fileName);
        outSrcFile.open( QFile::ReadWrite | QFile::Text );
        outSrcFile.write( sourceCodeAsHTML.toAscii() );
        outSrcFile.close();
    }
}


/*!
    \brief Loads the last opened file if this option is enabled in the settings. 
    
    If the file does not exist, the default example file is tried to be loaded. If even that
    fails a very small code example is shown.
    If the setting for opening the last file is disabled, the editor is empty on startup.
*/
void MainWindow::loadLastOpenedFile() {
    // Get setting for last opened source code file.
	loadLastSourceCodeFileOnStartup = settings->getValueByName("LoadLastOpenedFileOnStartup").toBool();

	// Only load last source code file if set to do so
	if ( loadLastSourceCodeFileOnStartup ) {
        // From the list of last opened files get the first one.
        currentSourceFile = settings->getValueByName("LastOpenedFiles").toString().split("|").first();

		// If source file exist load it.
		if ( QFile::exists(currentSourceFile) ) {
			QFileInfo fileInfo(currentSourceFile);
			currentSourceFile = fileInfo.absoluteFilePath();
			sourceFileContent = loadFile(currentSourceFile);
		}
        // If the last opened source code file does not exist, try to load the default example.cpp file.
        else if ( QFile::exists(indenterDirctoryStr + "/example.cpp") ) {
			QFileInfo fileInfo(indenterDirctoryStr + "/example.cpp");
			currentSourceFile = fileInfo.absoluteFilePath();
			sourceFileContent = loadFile(currentSourceFile);
		}
		// If neither the example source code file exists show some small code example.
		else {
			currentSourceFile = "untitled.cpp";
			currentSourceFileExtension = "cpp";
			sourceFileContent = "if(x==\"y\"){x=z;}";
		}
	}
	// if last opened source file should not be loaded make some default settings.
	else {
		currentSourceFile = "untitled.cpp";
		currentSourceFileExtension = "cpp";
		sourceFileContent = "";
	}
    savedSourceContent = sourceFileContent;

    // Update the mainwindow title to show the name of the loaded source code file.
    updateWindowTitle();
}


/*!
    \brief Saves the settings for the main application to the file "UniversalIndentGUI.ini".

    Settings are for example last selected indenter, last loaded config file and so on.
*/
void MainWindow::saveSettings() {
    //QFileInfo fileInfo(currentSourceFile);
    //if ( fileInfo.isFile() ) {
    //    settings->setValueByName( "LastOpenedFiles", currentSourceFile );
    //}
	//settings->setValueByName( "LoadLastOpenedFileOnStartup", uiGuiLoadLastOpenedFileOnStartup->isChecked() );
    settings->setValueByName( "LastSelectedIndenterID", currentIndenterID );
    //settings->setValueByName( "IndenterParameterTooltipsEnabled", uiGuiIndenterParameterTooltipsEnabled->isChecked() );
    //settings->setValueByName( "Language", language );
	settings->setValueByName( "FileEncoding", currentEncoding );
    settings->setValueByName( "VersionInSettingsFile", version );
	settings->setValueByName( "WindowIsMaximized", isMaximized() );
	if ( !isMaximized() ) {
		settings->setValueByName( "WindowPosition", pos() );
		settings->setValueByName( "WindowSize", size() );
	}
    settings->setValueByName( "MainWindowState", saveState() );
    //settings->setValueByName( "SyntaxHighlightningEnabled", uiGuiSyntaxHighlightningEnabled->isChecked() );
    //settings->setValueByName( "WhiteSpaceIsVisible", uiGuiWhiteSpaceIsVisible->isChecked() );
    //settings->setValueByName( "TabWidth", txtedSourceCode->tabWidth() );

    //FIXME: Needs to be called explicit here, because the destructor of UiGuiSettings doesn't do it.
	settings->saveSettings();

    highlighter->writeCurrentSettings("");
}


/*!
    \brief Is always called when the program is quit. Calls the saveSettings function before really quits.
*/
void MainWindow::closeEvent( QCloseEvent *event ) {
    if ( maybeSave() ) {
        saveSettings();
        event->accept();
    }
    else {
        event->ignore();
    }
}


/*!
    \brief This function is setup to capture tooltip events. 
    
    All widgets that are created by the indentHandler object and are responsible 
    for indenter parameters are connected with this event filter. 
    So depending on the settings the tooltips can be enabled and disabled for these widgets.
 */
bool MainWindow::eventFilter(QObject *obj, QEvent *event)
{
    if ( event->type() == QEvent::ToolTip) {
        if ( uiGuiIndenterParameterTooltipsEnabled->isChecked() ) {
            return QMainWindow::eventFilter(obj, event);
        }
        else {
            //QToolTip::showText( QPoint(100,100) , "Test1");
            return true;
        }
    } else {
        // pass the event on to the parent class
        return QMainWindow::eventFilter(obj, event);
    }
}


/*!
    \brief Is called at application exit and asks whether to save the source code file, if it has been changed.
 */
bool MainWindow::maybeSave()
{
    if ( txtedSourceCode->isModified() ) {
        int ret = QMessageBox::warning(this, tr("Modified code"),
            tr("The source code has been modified.\n"
            "Do you want to save your changes?"),
            QMessageBox::Yes | QMessageBox::Default,
            QMessageBox::No,
            QMessageBox::Cancel | QMessageBox::Escape);
        if (ret == QMessageBox::Yes) {
            return saveSourceFile();
        }
        else if (ret == QMessageBox::Cancel) {
            return false;
        }
    }
    return true;
}


/*!
    \brief This slot is called whenever a language is selected in the menu. It tries to find the
    corresponding action in the languageInfoList and sets the language.
 */
void MainWindow::languageChanged(int languageIndex) {
    // Get the mnemonic of the new selected language.
	QString languageShort = settings->getAvailableTranslations().at(languageIndex);

	// Remove the old qt translation.
	qApp->removeTranslator( qTTranslator );

    // Remove the old uigui translation.
	qApp->removeTranslator( uiGuiTranslator );

    // Load the Qt own translation file and set it for the application.
    bool translationFileLoaded;
    translationFileLoaded = qTTranslator->load( globalFilesDirectoryStr + "/translations/qt_" + languageShort );
    if ( translationFileLoaded ) {
        qApp->installTranslator(qTTranslator);
    }

    // Load the uigui translation file and set it for the application.
    translationFileLoaded = uiGuiTranslator->load( globalFilesDirectoryStr + "/translations/universalindent_" + languageShort );
    if ( translationFileLoaded ) {
        qApp->installTranslator(uiGuiTranslator);
    }
}


/*!
    \brief Creates a menu entries in the file menu for opening and saving a file with different encodings.
*/
void MainWindow::createEncodingMenu() {
    QAction *encodingAction;
    QString encodingName;

    encodingsList = QStringList() << "UTF-8" << "UTF-16" << "UTF-16BE" << "UTF-16LE"
            << "Apple Roman" << "Big5" << "Big5-HKSCS" << "EUC-JP" << "EUC-KR" << "GB18030-0"
            << "IBM 850" << "IBM 866" << "IBM 874" << "ISO 2022-JP" << "ISO 8859-1" << "ISO 8859-13"
            << "Iscii-Bng" << "JIS X 0201" << "JIS X 0208" << "KOI8-R" << "KOI8-U" << "MuleLao-1"
            << "ROMAN8" << "Shift-JIS" << "TIS-620" << "TSCII" << "Windows-1250" << "WINSAMI2";

    encodingActionGroup = new QActionGroup(this);
    saveEncodedActionGroup = new QActionGroup(this);

    // Loop for each available encoding
    foreach ( encodingName, encodingsList ) {
        // Create actions for the "reopen" menu
        encodingAction = new QAction(encodingName, encodingActionGroup);
        encodingAction->setStatusTip( tr("Reopen the currently opened source code file by using the text encoding scheme ") + encodingName );
        encodingAction->setCheckable(true);
        if ( encodingName == currentEncoding ) {
            encodingAction->setChecked(true);
        }

        // Create actions for the "save as encoded" menu
        encodingAction = new QAction(encodingName, saveEncodedActionGroup);
        encodingAction->setStatusTip( tr("Save the currently opened source code file by using the text encoding scheme ") + encodingName );
    }

    encodingMenu->addActions( encodingActionGroup->actions() );
    connect( encodingActionGroup, SIGNAL(triggered(QAction*)), this, SLOT(encodingChanged(QAction*)) );

    saveEncodedMenu->addActions( saveEncodedActionGroup->actions() );
    connect( saveEncodedActionGroup, SIGNAL(triggered(QAction*)), this, SLOT(saveAsOtherEncoding(QAction*)) );
}


/*!
    \brief This slot calls the save dialog to save the current source file with another encoding.

    If the saving is successful and not aborted, the currently used encoding, visible in the
    "reopen" menu, is also changed to the new encoding.
*/
void MainWindow::saveAsOtherEncoding(QAction *chosenEncodingAction) {
    bool fileWasSaved = saveasSourceFileDialog(chosenEncodingAction);

    // If the file was save with another encoding, change the selected encoding in the reopen menu.
    if ( fileWasSaved ) {
        foreach ( QAction *action, encodingActionGroup->actions() ) {
            if ( action->text() == chosenEncodingAction->text() ) {
                action->setChecked(true);
                return;
            }
        }
    }
}


/*!
    \brief This slot is called whenever an encoding is selected in the settings menu.
*/
void MainWindow::encodingChanged(QAction* encodingAction) {
    if ( maybeSave() ) {
        QFile inSrcFile(currentSourceFile);
        QString fileContent = "";

        if ( !inSrcFile.open(QFile::ReadOnly | QFile::Text) ) {
            QMessageBox::warning(NULL, tr("Error opening file"), tr("Cannot read the file ")+"\""+currentSourceFile+"\"." );
        }
        else {
            QTextStream inSrcStrm(&inSrcFile);
            QApplication::setOverrideCursor(Qt::WaitCursor);
            QString encodingName = encodingAction->text();
			currentEncoding = encodingName;
            inSrcStrm.setCodec( QTextCodec::codecForName(encodingName.toAscii()) );
            fileContent = inSrcStrm.readAll();
            QApplication::restoreOverrideCursor();
            inSrcFile.close();
            txtedSourceCode->setText( fileContent );
			txtedSourceCode->setModified(false);
        }
    }
}


/*!
	\brief Creates a menu entry under the settings menu for all available text encodings.
*/
void MainWindow::createHighlighterMenu() {
	QAction *highlighterAction;
	QString highlighterName;

	highlighterActionGroup = new QActionGroup(this);

	// Loop for each known highlighter
    foreach ( highlighterName, highlighter->getAvailableHighlighters() ) {
		highlighterAction = new QAction(highlighterName, highlighterActionGroup);
		highlighterAction->setStatusTip( tr("Set the syntax highlightning to ") + highlighterName );
		highlighterAction->setCheckable(true);
	}
	highlighterMenu->addActions( highlighterActionGroup->actions() );
    menuSettings->insertMenu(uiGuiIndenterParameterTooltipsEnabled, highlighterMenu );

	connect( highlighterActionGroup, SIGNAL(triggered(QAction*)), this, SLOT(highlighterChanged(QAction*)) );
}


/*!
    \brief This slot handles signals coming from selecting another syntax highlighter.
 */
void MainWindow::highlighterChanged(QAction* highlighterAction) {
	QString highlighterName = highlighterAction->text();
    highlighter->setLexerByName( highlighterName );
    //TODO: This is really no nice way. How do it better?
    // Need to do this "text update" to update the syntax highlighting. Otherwise highlighting is wrong.
    previewToggled = true;
    updateSourceView();
}


/*!
	\brief Is called whenever the white space visibility is being changed in the menu.
 */
void MainWindow::setWhiteSpaceVisibility(bool visible) {
	if ( visible ) {
		txtedSourceCode->setWhitespaceVisibility(QsciScintilla::WsVisible);
	}
	else {
		txtedSourceCode->setWhitespaceVisibility(QsciScintilla::WsInvisible);
	}
}

/*!
	\brief This slot is called whenever the number of lines in the editor changes
	and adapts the margin for the displayed line numbers.
*/
void MainWindow::numberOfLinesChanged() {
	QString lineNumbers;
	lineNumbers.setNum( txtedSourceCode->lines()*10 );
	txtedSourceCode->setMarginWidth(1, lineNumbers);
}


/*!
    \brief Catches language change events and retranslates all needed widgets.
 */
void MainWindow::changeEvent(QEvent *event) {
    int i = 0;

    if (event->type() == QEvent::LanguageChange) {
        QString languageName;

        // Translate the main window.
        retranslateUi(this);
        updateWindowTitle();

        // Translate the toolbar.
        toolBarWidget->retranslateUi(toolBar);

         // Translate the encoding menu.
        QList<QAction *> encodingActionList = encodingActionGroup->actions();
        for ( i = 0; i < encodingActionList.size(); i++ ) {
            encodingActionList.at(i)->setStatusTip( tr("Reopen the currently opened source code file by using the text encoding scheme ") + encodingsList.at(i) );
        }

        // Translate the highlighter menu.
        QList<QAction *> actionList = highlighterMenu->actions();
        i = 0;
        foreach ( QString highlighterName, highlighter->getAvailableHighlighters() ) {
            QAction *highlighterAction = actionList.at(i);
            highlighterAction->setStatusTip( tr("Set the syntax highlightning to ") + highlighterName );
            i++;
        }
    } 
    else {
        QWidget::changeEvent(event);
    }
}


/*!
    \brief Invokes the indenter to create a shell script.

    Lets the indenter create a shell script for calling the indenter out of any
    other application and open a save dialog for saving the shell script.
 */
void MainWindow::createIndenterCallShellScript() {
    QString indenterCallShellScript = indentHandler->generateCommandlineCall(currentSourceFileExtension);

    QString shellScriptExtension;
#if defined(Q_OS_WIN32)
        shellScriptExtension = "bat";
#else
        shellScriptExtension = "sh";
#endif

    QString fileExtensions = tr("Shell Script")+" (*."+shellScriptExtension+");;"+tr("All files")+" (*.*)";

    QString currentIndenterName = toolBarWidget->cmbBoxIndenters->currentText();
    currentIndenterName = currentIndenterName.replace(" ", "_");

    //QString openedSourceFileContent = openFileDialog( tr("Choose source code file"), "./", fileExtensions );
    QString fileName = QFileDialog::getSaveFileName( this, tr("Save shell script"), "call_"+currentIndenterName+"."+shellScriptExtension, fileExtensions);

    // Saving has been canceled if the filename is empty
    if ( fileName.isEmpty() ) {
        return;
    }

    QFile::remove(fileName);
    QFile outSrcFile(fileName);
    outSrcFile.open( QFile::ReadWrite | QFile::Text );
    outSrcFile.write( indenterCallShellScript.toAscii() );
    outSrcFile.close();
}


/*!
    \brief Updates the list of recently opened files. 
    
    Therefore the currently open file is set at the lists first position 
    regarding the in the settings set maximum list length. Overheads of the 
    list will be cut off. The new list will be updated to the settings and 
    the recently opened menu will be updated too.
 */
void MainWindow::updateRecentlyOpenedList() {

	QString fileName;
    QString filePath;
    QStringList recentlyOpenedList = settings->getValueByName("LastOpenedFiles").toString().split("|");
    QList<QAction*> recentlyOpenedActionList = menuRecently_Opened_Files->actions();

    // Check if the currently open file is in the list of recently opened.
    int indexOfCurrentFile = recentlyOpenedList.indexOf( currentSourceFile );

    // If it is in the list of recently opened files and not at the first position, move it to the first pos.
    if ( indexOfCurrentFile > 0 ) {
        recentlyOpenedList.move(indexOfCurrentFile, 0);
        recentlyOpenedActionList.move(indexOfCurrentFile, 0);
    }
    // Put the current file at the first position if it not already is and is not empty.
    else if ( indexOfCurrentFile == -1 && !currentSourceFile.isEmpty() ) {
        recentlyOpenedList.insert(0, currentSourceFile);
		QAction *recentlyOpenedAction = new QAction(QFileInfo(currentSourceFile).fileName(), menuRecently_Opened_Files);
		recentlyOpenedAction->setStatusTip(currentSourceFile);
        recentlyOpenedActionList.insert(0, recentlyOpenedAction );
    }

	// Get the maximum recently opened list size.
	int recentlyOpenedListMaxSize = settings->getValueByName("RecentlyOpenedListSize").toInt();

	// Loop for each filepath in the recently opened list, remove non existing files and
	// loop only as long as maximum allowed list entries are set.
    for ( int i = 0; i < recentlyOpenedList.size() && i < recentlyOpenedListMaxSize; ) {
		filePath = recentlyOpenedList.at(i);
        QFileInfo fileInfo(filePath);

		// If the file does no longer exist, remove it from the list.
		if ( !fileInfo.exists() ) {
			recentlyOpenedList.takeAt(i);
            if ( i < recentlyOpenedActionList.size()-2 ) {
                QAction* action = recentlyOpenedActionList.takeAt(i);
                delete action;
            }
		}
        // else if its not already in the menu, add it to the menu.
        else {
            if ( i >= recentlyOpenedActionList.size()-2 ) {
                QAction *recentlyOpenedAction = new QAction(fileInfo.fileName(), menuRecently_Opened_Files);
                recentlyOpenedAction->setStatusTip(filePath);
                recentlyOpenedActionList.insert( recentlyOpenedActionList.size()-2, recentlyOpenedAction );
            }
            i++;
        }
	}

	// Trim the list to its in the settings allowed maximum size.
	while ( recentlyOpenedList.size() > recentlyOpenedListMaxSize ) {
		recentlyOpenedList.takeLast();
        QAction* action = recentlyOpenedActionList.takeAt( recentlyOpenedActionList.size()-3 );
        delete action;
	}

    // Add all actions to the menu.
    menuRecently_Opened_Files->addActions(recentlyOpenedActionList);

    // Write the new recently opened list to the settings.
    settings->setValueByName( "LastOpenedFiles", recentlyOpenedList.join("|") );

    // Enable or disable "actionClear_Recently_Opened_List" if list is [not] emtpy
    if ( recentlyOpenedList.isEmpty() ) {
        actionClear_Recently_Opened_List->setEnabled(false);
    }
    else {
        actionClear_Recently_Opened_List->setEnabled(true);
    }
}


/*!
    \brief This slot empties the list of recently opened files.
 */
void MainWindow::clearRecentlyOpenedList() {
    QStringList recentlyOpenedList = settings->getValueByName("LastOpenedFiles").toString().split("|");
    QList<QAction*> recentlyOpenedActionList = menuRecently_Opened_Files->actions();

    while ( recentlyOpenedList.size() > 0 ) {
		recentlyOpenedList.takeLast();
        QAction* action = recentlyOpenedActionList.takeAt( recentlyOpenedActionList.size()-3 );
        delete action;
	}

    // Write the new recently opened list to the settings.
    settings->setValueByName( "LastOpenedFiles", recentlyOpenedList.join("|") );

    // Disable "actionClear_Recently_Opened_List"
    actionClear_Recently_Opened_List->setEnabled(false);
}


/*!
    \brief This slot is called if an entry from the list of recently opened files is
    being selected.
 */
void MainWindow::openFileFromRecentlyOpenedList(QAction* recentlyOpenedAction) {
    // If the selected action from the recently opened list menu is the clear action
    // call the slot to clear the list and then leave.
    if ( recentlyOpenedAction == actionClear_Recently_Opened_List ) {
        clearRecentlyOpenedList();
        return;
    }

    QString fileName = recentlyOpenedAction->text();
    int indexOfSelectedFile = menuRecently_Opened_Files->actions().indexOf( recentlyOpenedAction );
    QStringList recentlyOpenedList = settings->getValueByName("LastOpenedFiles").toString().split("|");
    QString filePath = recentlyOpenedList.at(indexOfSelectedFile);
	QFileInfo fileInfo(filePath);

	// If the file exists, open it.
	if ( fileInfo.exists() ) {
		openSourceFileDialog(filePath);
	}
	// If it does not exist, show a warning message and update the list of recently opened files.
	else {
		QMessageBox::warning(NULL, tr("File no longer exists"), tr("The file %1 in the list of recently opened files does no longer exist.") );
        // The function updateRecentlyOpenedList() has to be called via a singleShot so it is executed after this
        // function (openFileFromRecentlyOpenedList) has already been left. This has to be done because
        // a Qt3Support function tries to emit a signal based on the existing actions and deleting
        // any of these actions in updateRecentlyOpenedList() causes an error.
		QTimer::singleShot(0, this, SLOT(updateRecentlyOpenedList()) );
	}
}


/*!
    \brief This slot gets the reference to the indenters manual from the indenter handler and opens it.
 */
void MainWindow::showIndenterManual() {
    QString manualReference = indentHandler->getManual();
    QDesktopServices::openUrl( manualReference );
}
