/*
 *  update-cluster-maccollect - Collect MAC address and auto-generate cluster.xml
 *  Copyright 2002,2003 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 * This application requires root privilege.
 */


#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <netpacket/packet.h>
#include <getopt.h>
#include <string.h>
#include <sys/stat.h>


#ifndef ETH_P_ALL
/* from linux kernel source */
#define ETH_P_ALL 0x0003
#endif

#ifndef ETH_P_IP
/* from linux kernel source */
#define ETH_P_IP 0x0800
#endif

#include "config.h"
#include "update-cluster-i18n.h"

char * ipprefix="192.168.0.";
char * prefix="node";
int ipstart=2;
int hoststart=2;
char * domainsuffix=".cluster";
char * interface=NULL;
char * clusterxml= PKGSYSCONFDIR "/cluster.xml";

/**
   Show help message.
 */
static void 
showhelp ()
{
  printf (_("update-cluster-maccollect [--options] [item] ... \n"
	    "  collects mac address, and creates necessary hostnames \n"
	    "  and IP address\n"
	    "  for the MAC address, and auto-generates cluster.xml.\n"
	    "\n"));
}

/**
   Parse the command-line options and set parameters.

   @return 0 success
   @return 1 failure to parse
 */
static int
parse_options(int ac, char ** av )
{
  int index_point;  
  int c;
  
  static struct option long_options[]=
  {
    {"prefix", required_argument, 0, 'p'},
    {"ipprefix", required_argument, 0, 'i'},
    {"ipstart", required_argument, 0, 'I'},
    {"hoststart", required_argument, 0, 'H'},
    {"domainsuffix", required_argument, 0, 'd'},
    {"interface", required_argument, 0, 'e'},
    {"clusterxml", required_argument, 0, 'c'},
    {"help", no_argument, 0, 'h'},
    {0,0,0,0}
  };

  while((c = getopt_long (ac, av, 
			  "p:i:I:H:d:e:h", 
			  long_options, &index_point)) != -1)
    {
      switch (c)
	{
	case 'p': 			/* --prefix */
	  prefix=strdup(optarg);
	  break;
	case 'i': 			/* --ipprefix */
	  ipprefix=strdup(optarg);
	  break;
	case 'I': 			/* --ipstart */
	  ipstart=atoi(optarg);
	  break;
	case 'H': 			/* --hoststart */
	  hoststart=atoi(optarg);
	  break;
	case 'd': 			/* --domainsuffix */
	  domainsuffix=strdup(optarg);
	  break;
	case 'e': 			/* --interface */
	  interface=strdup(optarg);
	  break;
	case 'c': 			/* --clusterxml */
	  clusterxml=strdup(optarg);
	  break;
	case 'h': 			/* --help */
	default:
	  showhelp();
	  return 1;
	}
    }
  return 0;
}

/**
   Check the MAC address if it is a valid MAC address.

   Ignore broadcast, and non-MAC address values.
   ff, and 0.
   
   @return 0 on MAC address
   @return 1 not a MAC address
 */
static int
check_mac (const unsigned char * g)
{
  const unsigned char * f = g + 6;	/* get the src MAC */

  if (f[0] == 0 && 
      f[1] == 0 && 
      f[2] == 0 && 
      f[3] == 0 && 
      f[4] == 0 && 
      f[5] == 0)
    return 1;			/* ZERO */
  
  if (f[0] == 0xff && 
      f[1] == 0xff && 
      f[2] == 0xff && 
      f[3] == 0xff && 
      f[4] == 0xff && 
      f[5] == 0xff)
    return 1;			/* broadcast */

  return 0;
}


/** 
    The main function.

    Opens up socket, listens to it, and process MAC address.
    If MAC is not already registered in cluster.xml; add new information to 
    cluster.xml.
 */
int 
main(int ac, char ** av)
{
  int s = socket (AF_PACKET, SOCK_RAW, htons(ETH_P_ALL));
  struct sockaddr_ll sll;
  unsigned char f[BUFSIZ+2];
  int ifindex;
  int count;
  char * mac = NULL;
  char * execstring = NULL;
  struct stat stat_struct;
  
  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE_NAME, LOCALEDIR);
  textdomain(PACKAGE_NAME);

  if (parse_options(ac,av))
    exit (1);
  
  memset(&sll, 0, sizeof(sll));
  sll.sll_family          = AF_PACKET;
  if (interface) 
    {
      ifindex = if_nametoindex(interface);
      sll.sll_ifindex         = ifindex;
      printf(_("Interface set to %s\n"), interface);
    }
  sll.sll_protocol        = htons(ETH_P_ALL);
  if (bind(s, (struct sockaddr *) &sll, sizeof(sll)) == -1) 
    {
      perror("update-cluster-maccollect");
      exit(1);
    }

  /* Check if cluster.xml really exists */
  if (stat(clusterxml, &stat_struct))
    {
      fprintf (stderr,_("Error when trying to access [%s]\n"), clusterxml);      
      perror (_("update-cluster-maccollect; when trying to stat cluster.xml"));
      exit (1);
    }

  while ((count=read(s,f,BUFSIZ)) >= 0)
    {
      /* first 6 bytes contain the MAC address of the packet sender. */
      
      if (count > 12)
	{
	  if (asprintf(&mac,"%x:%x:%x:%x:%x:%x",f[6],f[7],f[8],f[9],f[10],f[11]) < 0)
	    {
	      /* error */
	      fprintf(stderr,_("Out of memory in asprintf\n"));
	      exit (1);
	    }
	  else
	    {
	      if (asprintf(&execstring, "update-cluster-remove --not --failfound --mac %s > /dev/null < %s", mac, clusterxml) < 0)
		{
		  /* error */
		  fprintf(stderr,_("Out of memory in asprintf\n"));
		  exit (1);
		}
	      else
		{
		  if (system (execstring))
		    {
		      printf (_("Mac address %s already exists\n"), mac);
		    }
		  else if (check_mac(f))
		    {
		      printf (_("Mac address %s is ignored\n"), mac);
		    }
		  else
		    {
		      printf (_("Add MAC address %s\n"), mac);
		      fflush (stdout);

		      asprintf(&execstring,
			       "update-cluster-add --ip %s%i --hostname %s%i%s --mac %s --clusterxml %s",
			       ipprefix, ipstart, prefix, hoststart, domainsuffix, 
			       mac,
			       clusterxml);
		      if (system(execstring))
			{
			  /* error */
			  fprintf(stderr,_("Error in system call to [%s]\n"), execstring);
			  exit (1);
			}
		      ipstart ++ ;
		      hoststart ++;
		    }
		}	      
	    }
	}
      else
	printf (_("W: Very small frame with size < 12 bytes found \n"));      
    }
  return 0;
}
