/*
 * GLX Hardware Device Driver for S3 ViRGE DX/GX/GX2
 * Copyright (C) 1999 Jim Duchek
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on Mach64 driver: mach64glx.h
 *
 *    Jim Duchek <jimduchek@ou.edu>
 */

#ifndef __S3VIRGE_GLX_H__
#define __S3VIRGE_GLX_H__

#include "mm.h"
#include "compiler.h"
#include "s3virgeregs.h"
#include "vga.h"
#include "s3virgelog.h"
#include "glx_symbols.h"
#include <stdio.h>
#include "hwtypes.h"

#include "types.h"		/* MESA types */
#ifndef MESA31
#error "The S3ViRGE driver requires Mesa 3.1 or higher"
#endif

#include "xsmesaP.h"	/* GLX header */
extern void xsmesa_setup_DD_pointers( GLcontext *ctx );


typedef unsigned int s3virgeUI32;
typedef unsigned short s3virgeUI16;
typedef unsigned char s3virgeUI8;
typedef int s3virgeI32;
typedef short s3virgeI16;
typedef char s3virgeI8;

typedef s3virgeUI8 s3virgeColor[4];


/*
 * main buffer structure for drawing destination
 */
#define s3virgeBufferMagic		0x050e011e
#define VALID_S3VIRGE_BUFFER( b )	( (b) && ( (b)->magic == s3virgeBufferMagic ) )

struct s3virge_buffer_t
{
    s3virgeUI32	magic;
    struct s3virge_buffer_t *next;

    void	*backBuffer;	/* pointer to buffer in VM */
    PMemBlock	backBufferBlock; /* location in card's space, NULL if software only */
    
    void	*depthBuffer;
    PMemBlock	depthBufferBlock;

    void	*lightmapBuffer;
    PMemBlock	lightmapBufferBlock;

    int		width, height;	/* in pixels */
    int		pitch;			/* in pixels */
};

typedef struct s3virge_buffer_t s3virgeBuffer;
typedef struct s3virge_buffer_t *s3virgeBufferPtr;

/*
 * main hardware context structure
 */
#define s3virgeContextMagic		0x086C3750
#define VALID_S3VIRGE_CONTEXT( ctx )	( (ctx) && ( (ctx)->magic == s3virgeContextMagic ) )

/* for SoftwareFallback */
#define FALLBACK_ALPHATEST	1
#define FALLBACK_BLENDING	2
#define FALLBACK_TEXTURE	4



/* For s3virgeCtx->new_state.
 */
#define S3VIRGE_NEW_DEPTH	0x1
#define S3VIRGE_NEW_ALPHA	0x2
#define S3VIRGE_NEW_FOG		0x4
#define S3VIRGE_NEW_CLIP		0x8
#define S3VIRGE_NEW_MASK		0x10
#define S3VIRGE_NEW_TEXTURE	0x20
#define S3VIRGE_NEW_CULL		0x40
#define S3VIRGE_NEW_CONTEXT	0x100

struct s3virge_context_t
{
    s3virgeUI32		magic;
    int			refcount;

    struct s3virge_buffer_t *DB;
    GLcontext		*gl_ctx;

    int			SoftwareFallback;  /* or'ed values of FALLBACK_* */

    s3virgeUI8		GlobalPalette[768];	

    /* Support for CVA and the fast paths */
    unsigned int	setupdone;
    unsigned int	setupindex;
    unsigned int	renderindex;
    unsigned int	using_fast_path;
    unsigned int	using_immediate_fast_path;

    /* Shortcircuit some state changes */
    points_func		PointsFunc;
    line_func		LineFunc;
    triangle_func	TriangleFunc;
    quad_func		QuadFunc;

    /* Manage our own state */
    GLuint		new_state;
    GLuint		reg_dirty;

    GLfloat		map[16];
};

struct s3virge_context_t;
typedef struct s3virge_context_t s3virgeContext;
typedef struct s3virge_context_t *s3virgeContextPtr;


/*
 * texture object
 */
#define S3VIRGE_TEXTURE_OBJECT_MAGIC 0x67cc93d6
#define S3VIRGE_TEX_MAXLEVELS 9
#define S3VIRGE_TEX_MAXWIDTH 512

typedef struct s3virge_texture_object_s {
	s3virgeUI32	magic;
	struct s3virge_texture_object_s *next;	/* driver global chain of textures */
	struct gl_texture_object *tObj;		/* mesa texture */
	s3virgeContextPtr ctx;
	
	PMemBlock	memBlock;				/* buffer for all miplevels */
	s3virgeUI32	totalSize;				/* for adding to c_textureUtilization */
	
	int		texelBytes;
	
	int		widthLog2;
	int		heightLog2;
	int		maxLog2;


	s3virgeUI8	Palette[768];	
	s3virgeUI32	offsets[S3VIRGE_TEX_MAXLEVELS];
	
	int		hasAlpha;
	
	s3virgeUI32 	age;   				/* set to mgaglx.swapBufferCount when bound */
} s3virgeTextureObject_t;

typedef s3virgeTextureObject_t *s3virgeTextureObjectPtr;



/*
 * main driver variables structure
 */
typedef struct
{
	s3virgeBuffer			*bufferList;
	s3virgeTextureObject_t	*textureList;

	s3virgeUI32	dmaActive;
	
	/* logging stuff */
	s3virgeUI32		logLevel;
	FILE		*logFile;

	/* dma stuff */
	s3virgeUI32		dmaDriver;
	s3virgeUI32		dmaAdr;
	s3virgeUI32		dmaSize;
	s3virgeUI32		cardCmds;
	s3virgeUI32		systemTexture;
	char*			dmaMemory;
	char			memoryFileName[HW_FILENAME_LENGTH];
	int			dmaMemoryFd;
	int			memoryRemapping;

	/* driver register state */
#if 0 /* FIXME: GH */
    /* bookkeeping for textureing */
    struct s3virge_texture_object_s *TexObjList;
    struct s3virge_texture_object_s *currentTextureObject[2];
    s3virgeUI32		deferTextureLoads;
    s3virgeUI32		default32BitTextures;
    s3virgeUI32		swapBuffersCount;
    struct gl_texture_object *currentMesaTexture;
#endif
    s3virgeUI8	 	currentTexturePalette[768];
    s3virgeUI32		updatedPalette;


    s3virgeUI32		currentline_cmd;
    s3virgeUI32		currenttri_cmd;
    s3virgeUI32		whitetri_cmd;

    /* options */
    s3virgeUI32		nullprims;	/* skip all primitive generation */
    s3virgeUI32		boxes;		/* draw performance boxes */
    s3virgeUI32		noFallback;	/* don't fall back to software, do best-effort rendering */
    s3virgeUI32		nosgram;	/* don't block clears */
    s3virgeUI32		skipDma;	/* don't send anything to the hardware */
    s3virgeUI32		lightmapHack;   /* Odd lightmap hack */

    /* performance counters */
    s3virgeUI32		swapBuffersCount;	/* also used for texture thrash checking */
    
    s3virgeUI32		c_textureUtilization;
    s3virgeUI32		c_textureSwaps;
    s3virgeUI32		c_setupPointers;
    s3virgeUI32		c_gtriangles;
    s3virgeUI32		c_ttriangles;
    s3virgeUI32		c_drawWaits;
    s3virgeUI32		c_signals;
    s3virgeUI32		c_dmaFlush;
    s3virgeUI32		c_overflows;

    s3virgeUI32		hardwareWentIdle; /* cleared each swapbuffers, set if
					     a waitfordmacompletion ever exited
					     without having to wait */

    s3virgeUI32		timeTemp;	/* for time counting in logfiles */
    
    /* stuff we get from the apropriate s3virgeinfo struct */
    s3virgeUI32		depth;		/* should be 15 or 16 */
    s3virgeUI32		virtualX;
    s3virgeUI32		virtualY;
    s3virgeUI32		displayWidth;
    
    s3virgeUI32		triHack;
    
    s3virgeUI32		videoRam;
    s3virgeUI32		linearBase;
    s3virgeUI32		linearPhysical;
    s3virgeUI32		MMIOBase;
    s3virgeUI32		bytesPerPixel;		/* should be 2 or 4 */
    
    
} s3virgeGlx_t;

#define S3HACK_ZERO_SRCCOL	0x1
#define S3HACK_NUMBER_2		0x2
#define S3HACK_SPEC_LIGHT	0x4
#define S3HACK_WHITE_TRI	0x8
#define S3HACK_WHITE_TRI_ALPHA	0x10

 
extern XSMesaContext	XSMesa;
extern s3virgeGlx_t		s3virgeglx;
extern s3virgeBufferPtr	s3virgeDB;
extern s3virgeContextPtr	s3virgeCtx;


#define S3VIRGEPACKCOLOR555( r, g, b, a )				\
    ((((r) & 0xf8) << 7) | (((g) & 0xf8) << 2) | (((b) & 0xf8) >> 3) |	\
     ((a) ? 0x8000 : 0))

#define S3VIRGEPACKCOLOR565( r, g, b )					\
    ((((r) & 0xf8) << 8) | (((g) & 0xfc) << 3) | (((b) & 0xf8) >> 3))

#define S3VIRGEPACKCOLOR888( r, g, b )					\
    (((r) << 16) | ((g) << 8) | (b))

#define S3VIRGEPACKCOLOR8888( r, g, b, a )				\
    (((a) << 24) | ((r) << 16) | ((g) << 8) | (b))

#define S3VIRGEPACKCOLOR4444( r, g, b, a )				\
    ((((a) & 0xf0) << 8) | (((r) & 0xf0) << 4) | ((g) & 0xf0) | ((b) >> 4))

static __inline__ s3virgeUI32 s3virgePackColor( s3virgeUI8 r, s3virgeUI8 g,
					      s3virgeUI8 b, s3virgeUI8 a )
{
#if 0	// JDC: FIXME
    switch ( s3virgeDB->Attrib & S3VIRGE_PF_MASK )
    {
    case S3VIRGE_PF_555:
	return S3VIRGEPACKCOLOR555( r, g, b, a );
    case S3VIRGE_PF_565:
	return S3VIRGEPACKCOLOR565( r, g, b );
    case S3VIRGE_PF_888:
	return S3VIRGEPACKCOLOR888( r, g, b );
    case S3VIRGE_PF_8888:
	return S3VIRGEPACKCOLOR8888( r, g, b, a );
    default:
	return 0;
    }
#else
	return S3VIRGEPACKCOLOR555( r, g, b, a );
#endif
 
}

/* ===== s3virgeglx.c ===== */


s3virgeContextPtr s3virgeCreateContext( GLcontext *ctx );
int s3virgeDestroyContext( s3virgeContextPtr ctx );

int usec( void );
void s3virgeLibInit();
void s3virgeSoftReset();

int s3virgeMakeCurrent( s3virgeContextPtr ctx, s3virgeBufferPtr buf );


void s3virgeGLXCreateDepthBuffer( GLcontext* ctx );
void s3virgeGLXDestroyImage( GLXImage* image );
GLXImage* s3virgeGLXCreateImage( WindowPtr pwindow, int depth,
				int width, int height, GLXImage *old_image );
GLboolean s3virgeGLXMakeCurrent( XSMesaContext c );
GLboolean s3virgeGLXBindBuffer( XSMesaContext c, XSMesaBuffer b );
XSMesaContext s3virgeGLXCreateContext( XSMesaVisual v,
				      XSMesaContext share_list );
void s3virgeGLXDestroyContext( XSMesaContext c );


/* ===== s3virgeclear.c ===== */

GLbitfield s3virgeClear( GLcontext *ctx, GLbitfield mask, GLboolean all,
			    GLint x, GLint y, GLint width, GLint height );


/* ===== s3virgeswap.c ===== */

void s3virgeGLXSwapBuffers( XSMesaBuffer b );
void s3virgePerformanceBoxes( int is_direct );
void s3virgeBackToFront( DrawablePtr drawable, s3virgeBufferPtr buf );

/* ===== s3virgedd.c ===== */

void s3virge_setup_DD_pointers( GLcontext *ctx );
void s3virgeDDExtensionsInit( GLcontext *ctx );


/* ===== s3virgestate.c ===== */

void s3virgeDDInitStatePointers( GLcontext *ctx );
void s3virgeDDUpdateState( GLcontext *ctx );


/* ===== s3virgetri.c ===== */

void s3virgeGouraudTriangle( GLcontext *ctx, GLuint e0, GLuint e1, GLuint e2, GLuint pv );
void s3virgeTextureTriangle( GLcontext *ctx, GLuint e0, GLuint e1, GLuint e2, GLuint pv );

/* ===== s3virgeline.c ==== */

void s3virgeLine( GLcontext *ctx, GLuint e0, GLuint e1, GLuint pv );


/* ===== s3virgetex.c ===== */

void s3virgeCreateTexObj( s3virgeContextPtr ctx, struct gl_texture_object *tObj );
GLboolean s3virgeIsTextureResident( GLcontext *ctx, struct gl_texture_object *tObj );
void s3virgeDeleteTexture( GLcontext *ctx, struct gl_texture_object *tObj );
void s3virgeTexSubImage( GLcontext *ctx, GLenum target,
		  struct gl_texture_object *tObj, GLint level,
		  GLint xoffset, GLint yoffset,
		  GLsizei width, GLsizei height,
		  GLint internalFormat,
		  const struct gl_texture_image *image );
void s3virgeTexImage( GLcontext *ctx, GLenum target,
		  struct gl_texture_object *tObj, GLint level,
		  GLint internalFormat,
		  const struct gl_texture_image *image );
int s3virgeUpdateTexturePalette( GLcontext *ctx, 
		  struct gl_texture_object *tObj);
void s3virgeLoadTexturePalette( s3virgeUI8 *pal );

/* ===== s3virgedma.c / s3virgedmainit.c ===== */

#include "s3virgedma.h"

/* ===== s3virgedirect.c ===== */

int s3virgeGLXVendorPrivate( ClientPtr client,
				XSMesaContext ctx,
				xGLXVendorPrivateReq *stuff);

int s3virgeGLXAllowDirect( ClientPtr client );

void s3virgeGLXClientSwapBuffers ( XSMesaBuffer b );

void s3virgeDirectClientDMAFlush ( int wait );
				


#endif


/*
 * Local Variables:
 * mode: c
 * c-basic-offset: 4
 * End:
 */
