(**
   This module implements a button.
   The button label  can be a text or any other graphical object.

   This module is well documented to show how to implement a simple
   gadget. It also demonstrates, how the VO-engine works.
**)

MODULE VOButton;

(*
    Implements a button.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D   := VODisplay,
       E   := VOEvent,
       F   := VOFrame,
       G   := VOGUIObject,
       K   := VOKeyHandler,
       O   := VOObject,
       T   := VOText,
       U   := VOUtil,

       str := Strings;


CONST
  (* Types *)

  normal  * = 0;
  small   * = 1;
  image   * = 2;
  toolBar * = 3;

  (* actions *)

  pressedMsg * = 0; (* The constant for the PressedMsg.
                       A PressedMsg will be generated everytime
                       our button gets pressed.
                     *)

  repeatTimeOut = 75; (* Time between button repeat *)

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 frame*,
                 sFrame*,
                 iFrame*,
                 tFrame*      : LONGINT; (* the frame to use for the button *)
                 highlight*   : BOOLEAN;
                 gridDisable* : BOOLEAN;
               END;


  Button*     = POINTER TO ButtonDesc;

  (**
    THe ButtonClass.
    Since a button needs some eventhandling, so we inherit
    from VOGUIObject.Gadget.
  **)

  ButtonDesc* = RECORD (G.GadgetDesc)
                  prefs    : Prefs;    (* A pointer to our prefs *)
                  frame    : F.Frame;  (* A button needs a frame *)
                  image    : G.Object; (* and an image *)
                  timeOut  : D.TimeOut;(* timeout info for pulse-mode *)
                  font     : LONGINT;  (* Special font for the button *)
                  type     : LONGINT;  (* normal, Small or image *)
                  state,               (* The state of the button. TRUE when currently selected *)
                  active,
                  pulse    : BOOLEAN;  (* send permanent pressed Msg on ButtonDown and none on ButtonUp *)
                END;


  (* messages *)

  PressedMsg*     = POINTER TO PressedMsgDesc;

  (**
    The PressedMsg generated everytime the button get clicked.
  **)

  PressedMsgDesc* = RECORD (O.MessageDesc)
                    END;


VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.double3DOut;
    p.sFrame:=F.single3DOut;
    p.iFrame:=F.double3DOut;
    p.tFrame:=F.double3DTool;
    p.highlight:=TRUE;
    p.gridDisable:=TRUE;
  END Init;

  PROCEDURE (p : Prefs) SetPrefs(b : Button);

  BEGIN
    b.prefs:=p;   (* We set the prefs *)

    IF p.background#NIL THEN
      b.SetBackgroundObject(p.background.Copy());
      b.backgroundObject.source:=b;
    END;
  END SetPrefs;

  (**
    The Init-method of our button.
  **)

  PROCEDURE (b : Button) Init*;

  BEGIN
    b.Init^;          (* We *must* call Init of our baseclass first *)

    b.SetBackground(D.buttonBackgroundColor);

    prefs.SetPrefs(b);

    b.timeOut:=NIL;   (* initialize the timeout info to NIL *)

    INCL(b.flags,G.canFocus); (* We can handle keyboard focus *)
    INCL(b.flags,G.handleSC); (* We can handle shortcuts *)
    b.type:=normal;   (* Normally we use normal button frames *)
    b.pulse:=FALSE;   (* normaly one pressedMsg on mousebutton up *)
    b.font:=D.normalFont; (* Just the global default font fo the button label *)

    NEW(b.frame);     (* We create a frame object to frame our button *)
    b.frame.Init;     (* We must of cause call Frame.Init of the frame *)
    (* The frame must be resizeable in all directions *)
    b.frame.SetFlags({G.horizontalFlex,G.verticalFlex});
    (* We need a frame-image for a button *)

    (* There is no default image *)
    b.image:=NIL;

    (* The button is by default unpressed *)
    b.state:=FALSE;
    b.active:=FALSE;
  END Init;

  (**
    Set a new font to be used by the string gadget.
  **)

  PROCEDURE (b : Button) SetFont*(font : LONGINT);

  BEGIN
    b.font:=font;
  END SetFont;

  (**
    Call this method, if you want the given text to be displayed in
    the button.

    This creates simply an VOText.Text instance for Button.image.
  **)

  PROCEDURE (b : Button) SetText*(string : ARRAY OF CHAR);

  VAR
    text   : T.Text;
    help   : U.Text;

  BEGIN
    NEW(text); (* Allocate a VOText.Text object for displaying text *)
    text.Init; (* We must call Text.Init asfor all objects *)
    help:=U.EscapeString(string);   (* We escape the string because it could contain escape-sequences.
                                         This is because Oberon-2 does not specify an escape-character for strings *)

    text.SetFlags({G.horizontalFlex,G.verticalFlex}); (* Our text should be resizeable in all directions *)
    text.SetDefault(T.centered,{},b.font); (* We want the text of tex textimage centered in its bounds *)
    text.SetText(help^);            (* Set the escaped text to the Text-object *)
    b.CopyBackground(text);         (* Copy our background to the text *)
    b.image:=text;                  (* Use it as our image *)
  END SetText;

  (**
    Call this method, if you want the given text to be displayed in
    the button and want the button to interpret the given string
    regarding shortcuts.

    This creates simply an VOText.Text instance for Button.image.
  **)

  PROCEDURE (b : Button) SetLabelText*(string : ARRAY OF CHAR; keyHandler : K.KeyHandler);

  VAR
    text     : T.Text;
    help     : U.Text;
    sc,x,
    length,
    offset   : INTEGER;
    shortCut : CHAR;
    scMode   : LONGINT;

  BEGIN
    NEW(text); (* Allocate a VOText.Text object for displaying text *)
    text.Init; (* We must call Text.Init asfor all objects *)

    offset:=0;
    sc:=-1;

    shortCut:=0X;   (* No shortcut key *)
    scMode:=K.none; (* No special key for keyboard shortcut *)

    length:=str.Length(string);
    x:=0;
    WHILE x<length DO
      CASE string[x] OF
        "\":
          CASE string[x+1] OF
            "_",
            "*",
            "#",
            "^": str.Delete(string,x,1);
                 DEC(length);
          ELSE
          END;
      | "_" : sc:=x;
              shortCut:=string[x+1];
      | "*" : scMode:=K.return;
      | "^" : scMode:=K.escape;
      | "#" : scMode:=K.default;
      ELSE
      END;
      INC(x);
    END;

    IF sc>=0 THEN
      INC(offset,5); (* "\eu" "\en" - underscore *)
    END;

    IF scMode#K.none THEN
      INC(offset,2); (* \eX - ( *|^|# ) *)
    END;

    IF (sc>=0) OR (scMode#K.none) THEN
      NEW(help,length+1+offset);
      COPY(string,help^);

      IF sc>=0 THEN
        str.Delete(help^,sc,1);
        str.Insert("\en",sc+1,help^);
        str.Insert("\eu",sc,help^);
        INC(length,5);
      END;

      IF scMode#K.none THEN
        str.Delete(help^,length-1,1);
        CASE scMode OF
          K.return:  str.Insert("\eR",length-1,help^);
        | K.escape:  str.Insert("\eE",length-1,help^);
        | K.default: str.Insert("\eD",length-1,help^);
        END;
      END;

      IF (keyHandler#NIL) & ((shortCut#0X) OR (scMode#K.none)) THEN
        keyHandler.AddShortcutObject(b,{},shortCut,0,scMode);
      END;

      help:=U.EscapeString(help^);
    ELSE
      help:=U.EscapeString(string);   (* We escape the string because it could contain escape-sequences.
                                         This is because Oberon-2 does not specify an escape-character for strings *)
    END;

    text.SetFlags({G.horizontalFlex,G.verticalFlex}); (* Our text should be resizeable in all directions *)
    text.SetDefault(T.centered,{},b.font); (* We want the text of tex textimage centered in its bounds *)
    text.SetText(help^);            (* Set the escaped text to the Text-object *)
    b.CopyBackground(text);         (* Copy our background to the text *)
    b.image:=text;                  (* Use it as our image *)
  END SetLabelText;


  (**
    Use this method if you do not want text displayed in the button.
    We use n external initialized image. Not that for Button there is no difference
    between text and any oher image after this point.
  **)

  PROCEDURE (b : Button) SetImage*(image : G.Object);

  BEGIN
    b.image:=image;
    b.CopyBackground(b.image);
  END SetImage;

  (**
    We can define special types of buttons. Currently supported are normal, small and image.
  **)

  PROCEDURE (b : Button) SetType*(type : LONGINT);

  BEGIN
    b.type:=type;
  END SetType;

  (**
    Is pulsemode is true, the button send permanent pressedMsg on mouse button down
    and none on the final button up.

    This is usefull for buttons in a scroller or similar.
  **)

  PROCEDURE (b : Button) SetPulse*(pulse : BOOLEAN);

  BEGIN
    b.pulse:=pulse;
  END SetPulse;

  (**
    Returns the object that coveres the given point and that supports
    drag and drop of data.

    If drag is TRUE, when want to find a object that we can drag data from,
    else we want an object to drop data on.
  **)

  PROCEDURE (b : Button) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;

  BEGIN
    IF b.visible & b.PointIsIn(x,y) & (b.image#NIL) THEN
      RETURN b.image.GetDnDObject(x,y,drag);
    ELSE
      RETURN NIL;
    END;
  END GetDnDObject;

  (**
    This method gets called by the parent object before the first call to Button.Draw.

    We have to calculate the bounds of our button.
  **)

  PROCEDURE (b : Button) CalcSize*(display : D.Display);

  BEGIN

    IF b.image#NIL THEN
      (*
        Tell the image not to highlight, if the prefs say it shouldn't.
      *)
      IF ~b.prefs.highlight THEN
        b.image.SetFlags({G.noHighlight});
      ELSE
        b.image.RemoveFlags({G.noHighlight});
      END;
    END;

    (*
      We check, if the image can show some kind of frame. If so, we do not display
      the frame ourself, but delegate it to the image.
    *)

    IF (b.image#NIL) & ~b.image.StdFocus() & b.MayFocus() THEN
      EXCL(b.flags,G.stdFocus);
      INCL(b.image.flags,G.mayFocus);
    END;

    (* Let the frame calculate its size *)
    CASE b.type OF
      normal:
        b.frame.SetInternalFrame(b.prefs.frame);
     | small:
        b.frame.SetInternalFrame(b.prefs.sFrame);
     | image:
        b.frame.SetInternalFrame(b.prefs.iFrame);
     | toolBar:
        b.frame.SetInternalFrame(b.prefs.tFrame);
    END;
    b.frame.CalcSize(display);
    (*
      Our size is the size of the frame plus a little space we want to have
      to have between the frame and the image.

      We use display.spaceXXX because it is fontrelated. If we have a bigger font,
      we automitically have a bigger space.
    *)
    b.width:=b.frame.leftBorder+b.frame.rightBorder+display.spaceWidth;
    b.height:=b.frame.topBorder+b.frame.bottomBorder+display.spaceHeight;

    (* Our minimal size is equal to the normal size *)
    b.minWidth:=b.width;
    b.minHeight:=b.height;

    IF b.image#NIL THEN
      (*
        Now we let the image calculate its bounds and simply add its size
        to the size of the button.
      *)
      b.image.CalcSize(display);
      INC(b.width,b.image.width);
      INC(b.height,b.image.height);
      INC(b.minWidth,b.image.minWidth);
      INC(b.minHeight,b.image.minHeight);
    END;

    (* We *must* call CalcSize of our superclass! *)
    b.CalcSize^(display);
  END CalcSize;

  (**
    This method gets called when the window gets an event and looks for
    someone that processes it.

    If GetFocus return an object, that objets HandleEvent-method
    get called untill it gives away the focus.
  **)

  PROCEDURE (b : Button) GetFocus*(event : E.Event):G.Object;

  VAR
    pressed : PressedMsg; (* We want to create a pressedMsg
                             when we found out we got pressed *)

  BEGIN
    (* It makes no sense to get the focus if we are currently not visible *)
    IF ~b.visible OR b.disabled THEN
      RETURN NIL;
    END;

    (*
      When the left mousebutton gets pressed without any qualifier
      in the bounds of our button...
   *)

    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseDown) & b.PointIsIn(event.x,event.y)
       & (event.qualifier={}) & (event.button=E.button1) THEN
        (* We change our state to pressed and redisplay ourself *)
        b.state:=TRUE;
        b.Redraw;

        IF b.pulse THEN
          NEW(pressed);
          b.Send(pressed,pressedMsg);
          b.timeOut:=b.display.AddTimeOut(0,repeatTimeOut,b);
        END;

        (*
          Since we want the focus for waiting for buttonup we return
          a pointer to ourself.
        *)
        RETURN b;
      END;
    | event : E.MotionEvent DO
      (* If we are a toolbar button and the mouse move above us we get activated, too *)
      IF (b.type=toolBar) & (b.PointIsIn(event.x,event.y)) THEN
        b.state:=FALSE;
        b.active:=TRUE;
        b.Redraw;
        RETURN b;
      END;
    ELSE
    END;
    RETURN NIL;
  END GetFocus;

  (**
    Handles all events after GetFocus catched the focus for us.
  **)

  PROCEDURE (b : Button) HandleEvent*(event : E.Event):BOOLEAN;

  VAR
    pressed : PressedMsg; (* We want to create a pressedMsg
                             when we found out we got pressed *)

  BEGIN
    (*
      If the user releases the left mousebutton...
    *)
    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseUp) & (event.button=E.button1) THEN
        (* We get unselected again and must redisplay ourself *)
        b.state:=FALSE;
        b.Redraw;
        (*
          If the users released the left mousebutton over our bounds we really
          got selected.
        *)
        IF b.PointIsIn(event.x,event.y) & ~b.pulse THEN
          (*
            We create a PressedMsg and send it away.
            Button.Send (inherited from Object) does the managing
            of the possible attached handlers for use.
          *)
          (* Action: Button pressed *)
          NEW(pressed);
          b.Send(pressed,pressedMsg);
        END;

        (*
          Clean up and remove possibly remaining timer event.
        *)
        IF b.timeOut#NIL THEN
          b.display.RemoveTimeOut(b.timeOut);
          b.timeOut:=NIL;
        END;

        IF ~b.PointIsIn(event.x,event.y) THEN
          IF b.active THEN
            b.active:=FALSE;
            b.Redraw;
          END;
          RETURN TRUE; (* We give way the focus *)
        ELSIF b.type#toolBar THEN
          RETURN TRUE;
        END;
      ELSIF (event.type=E.mouseDown) & b.PointIsIn(event.x,event.y)
       & (event.qualifier={}) & (event.button=E.button1) & ~b.state THEN
        (* We change our state to pressed and redisplay ourself *)
        b.state:=TRUE;
        b.Redraw;

        IF b.pulse THEN
          NEW(pressed);
          b.Send(pressed,pressedMsg);
          b.timeOut:=b.display.AddTimeOut(0,repeatTimeOut,b);
        END;
      END;

      (*
        We also watch mousemoves.
        If the cursor leaves our bounds we want to get unselected.
      *)
    | event : E.MotionEvent DO
      (* If cursor in in our bounds ... *)
      IF b.PointIsIn(event.x,event.y) THEN
        (* If button pressed *)
        IF event.qualifier*E.buttonMask={E.button1} THEN
          (* If we were unselected ... *)
          IF ~b.state THEN
            b.state:=TRUE;
            b.Redraw;
          END;
        END;
      (* Cursor in not in our bounds ... *)
      ELSE
        (* If we were selected ... *)
        IF b.state THEN
          b.state:=FALSE;
          b.Redraw;
        ELSIF b.type=toolBar THEN
          b.state:=FALSE;
          b.active:=FALSE;
          b.Redraw;
          RETURN TRUE;
        END;
      END;
      RETURN FALSE; (* We still wait for buttonup *)
    ELSE
    END;

    RETURN FALSE; (* No message we are interested in, get more msgs *)
  END HandleEvent;

  (**
    This method gets called when we have the keyboard focus and
    a key is pressed. We analyse the key. If it is the space-key,
    we've been selected.

    TODO
    Go into the selected state, when we get a KeyPress-event for
    the space key. Get unselected again, if we receive a KeyRelease-
    event and send a PressedMsg. Get unselected without sending
    a PressedMsg when Object.LostFocus got called.
  **)

  PROCEDURE (b : Button) HandleFocusEvent*(event : E.KeyEvent):BOOLEAN;

  VAR
    keysym  : LONGINT;

    pressed : PressedMsg; (* We want to create a pressedMsg
                             when we found out we got pressed *)

  BEGIN
    IF event.type=E.keyDown THEN
      keysym:=event.GetKey();
      IF keysym=E.space THEN
        b.state:=TRUE;
        b.Redraw;
        (* TODO: Add some delay here *)
        b.state:=FALSE;
        b.Redraw;
        NEW(pressed);
        b.Send(pressed,pressedMsg);
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END HandleFocusEvent;

  PROCEDURE (b : Button) HandleShortcutEvent*(id,state : LONGINT);

  VAR
    pressed : PressedMsg; (* We want to create a pressedMsg
                             when we found out we got pressed *)

  BEGIN
    IF state=G.pressed THEN
      b.state:=TRUE;
      b.Redraw;
    ELSE
      b.state:=FALSE;
      b.Redraw;
      IF state=G.released THEN
        NEW(pressed);
        b.Send(pressed,pressedMsg);
      END;
    END;
  END HandleShortcutEvent;

  (**
    Gets called if the engine whants us to draw ourself.
  **)

  PROCEDURE (b : Button) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    b.Draw^(x,y,draw); (* We must call Draw of our superclass *)

    (*
      We fill the entier region with the background color
    *)
    b.DrawBackground(b.x,b.y,b.width,b.height);

   (*
      We tell the frame and the image to resize themselfs to
      our current bounds. Our bounds could have changed
      because Resize may have been called by some layout-objects
      between Button.CalcSize and Button.Draw.
    *)

    IF b.image#NIL THEN
      (* Image must fit in the frame, so be make it smaller. *)

      b.image.Resize(b.width-b.frame.leftBorder-b.frame.rightBorder,
                     b.height-b.frame.topBorder-b.frame.bottomBorder);

      (*
        If we are selected, we set the drawmode to VODisplay.selected.

        It may be, that some objects draw themselfs different when they are
        selected. Others may ignore it.

        Check if disabling should be delegated to image and check
        if the image has special care for diabling, too.
      *)

      IF b.disabled & ~b.prefs.gridDisable & (G.canDisable IN b.image.flags) THEN
        b.draw.mode:={D.disabled};
      ELSIF b.state THEN
        draw.mode:={D.selected};
      ELSE
        draw.mode:={};
      END;

      (*
        Draw the image centered in the bounds of the button

        Note, that we assume that the frame is symetric. This is true
        for buttonFrame (wich we use), but for groupFrame with label this is
        *not* the case!
      *)
      b.image.Draw(b.x+(b.width-b.image.oWidth) DIV 2,b.y+(b.height-b.image.oHeight) DIV 2,draw);
    END;

    b.frame.Resize(b.width,b.height);

    IF b.active & ~b.state THEN
      draw.mode:={D.activated};
    ELSIF b.state THEN
      draw.mode:={D.selected};
    ELSE
      draw.mode:={};
    END;

    (* Draw the frame *)
    b.frame.Draw(b.x,b.y,draw);

    (* Change the drawmode to normal, since we share draw with our parent object. *)
    draw.mode:={};

    IF b.disabled & (b.prefs.gridDisable OR ~(G.canDisable IN b.image.flags)) THEN
      b.DrawDisabled;
    END;
  END Draw;

  (**
    Draw the keyboard focus.
  **)

  PROCEDURE (b : Button) DrawFocus*;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF (b.image#NIL) & ~b.image.StdFocus() THEN
      IF b.state THEN
        b.draw.mode:={D.selected};
      ELSE
        b.draw.mode:={};
      END;
      b.image.DrawFocus;
      b.draw.mode:={};
    ELSE
      (* Delegate drawing to the baseclass *)
      b.DrawFocus^;
    END;
  END DrawFocus;

  (**
    Hide the keyboard focus.
  **)

  PROCEDURE (b : Button) HideFocus*;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF (b.image#NIL) & ~b.image.StdFocus() THEN
      IF b.state THEN
        b.draw.mode:={D.selected};
      ELSE
        b.draw.mode:={};
      END;
      b.image.HideFocus;
      b.draw.mode:={};
    ELSE
      (* Delegate drawing to the baseclass *)
      b.HideFocus^;
    END;
  END HideFocus;

  (**
    Hides the button. The buttons hides itself by hiding its label and its frame. Note, that you must
    set Object.visible yourself when you overload Object.Hide.
  **)

  PROCEDURE (b : Button) Hide*;

  BEGIN
    IF b.visible THEN
      IF b.image#NIL THEN
        (* Hide the image *)
        b.image.Hide;
      END;
      (* hide the frame *)
      b.frame.Hide;
      b.DrawHide;
      b.Hide^;
    END;
  END Hide;

  PROCEDURE (b : Button) Receive*(message : O.Message);

  VAR
    pressed : PressedMsg; (* We want to create a pressedMsg
                             when we found out we got pressed *)

  BEGIN
    WITH
      message : D.TimeOutMsg DO
        IF b.state THEN
          (*
            Time to send a new pressed message.
          *)
          NEW(pressed);
          b.Send(pressed,pressedMsg);
          b.timeOut:=b.display.AddTimeOut(0,repeatTimeOut,b);
        ELSE
          b.timeOut:=NIL;
        END;
    ELSE
      b.Receive^(message);
    END;
  END Receive;

BEGIN
  NEW(prefs);
  prefs.Init;

END VOButton.